/*
*        NASA OPEN SOURCE AGREEMENT VERSION 1.3
*
*        THIS OPEN SOURCE AGREEMENT ("AGREEMENT") DEFINES THE RIGHTS OF USE,
*        REPRODUCTION, DISTRIBUTION, MODIFICATION AND REDISTRIBUTION OF CERTAIN
*        COMPUTER SOFTWARE ORIGINALLY RELEASED BY THE UNITED STATES GOVERNMENT
*        AS REPRESENTED BY THE GOVERNMENT AGENCY LISTED BELOW ("GOVERNMENT
*        AGENCY").  THE UNITED STATES GOVERNMENT, AS REPRESENTED BY GOVERNMENT
*        AGENCY, IS AN INTENDED THIRD-PARTY BENEFICIARY OF ALL SUBSEQUENT
*        DISTRIBUTIONS OR REDISTRIBUTIONS OF THE SUBJECT SOFTWARE.  ANYONE WHO
*        USES, REPRODUCES, DISTRIBUTES, MODIFIES OR REDISTRIBUTES THE SUBJECT
*        SOFTWARE, AS DEFINED HEREIN, OR ANY PART THEREOF, IS, BY THAT ACTION,
*        ACCEPTING IN FULL THE RESPONSIBILITIES AND OBLIGATIONS CONTAINED IN
*        THIS AGREEMENT.
*
*        Government Agency: National Aeronautics and Space Administration (NASA)
*        Government Agency Original Software Designation: ARC-15166-1
*        Government Agency Original Software Title: WorldWind Version 1.3
*        User Registration Requested.  Please Visit https://opensource.arc.nasa.gov/
*        Government Agency Point of Contact for Original Software: Patrick.Hogan@nasa.gov
*        ________________________________________________
*
*
*        1. DEFINITIONS
*
*        A. "Contributor" means Government Agency, as the developer of the
*        Original Software, and any entity that makes a Modification.
*        B. "Covered Patents" mean patent claims licensable by a Contributor
*        that are necessarily infringed by the use or sale of its Modification
*        alone or when combined with the Subject Software.
*        C. "Display" means the showing of a copy of the Subject Software,
*        either directly or by means of an image, or any other device.
*        D. "Distribution" means conveyance or transfer of the Subject
*        Software, regardless of means, to another.
*        E. "Larger Work" means computer software that combines Subject
*        Software, or portions thereof, with software separate from the Subject
*        Software that is not governed by the terms of this Agreement.
*        F.  "Modification" means any alteration of, including addition to or
*        deletion from, the substance or structure of either the Original
*        Software or Subject Software, and includes derivative works, as that
*        term is defined in the Copyright Statute, 17 USC 101.  However, the
*        act of including Subject Software as part of a Larger Work does not in
*        and of itself constitute a Modification.
*        G. "Original Software" means the computer software first released
*        under this Agreement by Government Agency with Government Agency
*        designation ARC-15166-1 and entitled WorldWind, including source code,
*        object code and accompanying documentation, if any.
*        H. "Recipient" means anyone who acquires the Subject Software under
*        this Agreement, including all Contributors.
*        I. "Redistribution" means Distribution of the Subject Software after a
*        Modification has been made.
*        J. "Reproduction" means the making of a counterpart, image or copy of
*        the Subject Software.
*        K. "Sale" means the exchange of the Subject Software for money or
*        equivalent value.
*        L. "Subject Software" means the Original Software, Modifications, or
*        any respective parts thereof.
*        M. "Use" means the application or employment of the Subject Software
*        for any purpose.
*
*        2. GRANT OF RIGHTS
*
*        A. Under Non-Patent Rights: Subject to the terms and conditions of
*        this Agreement, each Contributor, with respect to its own contribution
*        to the Subject Software, hereby grants to each Recipient a
*        non-exclusive, world-wide, royalty-free license to engage in the
*        following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Modification
*        5. Redistribution
*        6. Display
*
*        B. Under Patent Rights: Subject to the terms and conditions of this
*        Agreement, each Contributor, with respect to its own contribution to
*        the Subject Software, hereby grants to each Recipient under Covered
*        Patents a non-exclusive, world-wide, royalty-free license to engage in
*        the following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Sale
*        5. Offer for Sale
*
*        C. The rights granted under Paragraph B. also apply to the combination
*        of a Contributor's Modification and the Subject Software if, at the
*        time the Modification is added by the Contributor, the addition of
*        such Modification causes the combination to be covered by the Covered
*        Patents.  It does not apply to any other combinations that include a
*        Modification.
*
*        D. The rights granted in Paragraphs A. and B. allow the Recipient to
*        sublicense those same rights.  Such sublicense must be under the same
*        terms and conditions of this Agreement.
*
*        3. OBLIGATIONS OF RECIPIENT
*
*        A. Distribution or Redistribution of the Subject Software must be made
*        under this Agreement except for additions covered under paragraph 3H.
*
*        1. Whenever a Recipient distributes or redistributes the Subject
*        Software, a copy of this Agreement must be included with each copy
*        of the Subject Software; and
*        2. If Recipient distributes or redistributes the Subject Software in
*        any form other than source code, Recipient must also make the
*        source code freely available, and must provide with each copy of
*        the Subject Software information on how to obtain the source code
*        in a reasonable manner on or through a medium customarily used for
*        software exchange.
*
*        B. Each Recipient must ensure that the following copyright notice
*        appears prominently in the Subject Software:
*
*        Copyright (C) 2001 United States Government
*        as represented by the Administrator of the
*        National Aeronautics and Space Administration.
*        All Rights Reserved.
*
*        C. Each Contributor must characterize its alteration of the Subject
*        Software as a Modification and must identify itself as the originator
*        of its Modification in a manner that reasonably allows subsequent
*        Recipients to identify the originator of the Modification.  In
*        fulfillment of these requirements, Contributor must include a file
*        (e.g., a change log file) that describes the alterations made and the
*        date of the alterations, identifies Contributor as originator of the
*        alterations, and consents to characterization of the alterations as a
*        Modification, for example, by including a statement that the
*        Modification is derived, directly or indirectly, from Original
*        Software provided by Government Agency. Once consent is granted, it
*        may not thereafter be revoked.
*
*        D. A Contributor may add its own copyright notice to the Subject
*        Software.  Once a copyright notice has been added to the Subject
*        Software, a Recipient may not remove it without the express permission
*        of the Contributor who added the notice.
*
*        E. A Recipient may not make any representation in the Subject Software
*        or in any promotional, advertising or other material that may be
*        construed as an endorsement by Government Agency or by any prior
*        Recipient of any product or service provided by Recipient, or that may
*        seek to obtain commercial advantage by the fact of Government Agency's
*        or a prior Recipient's participation in this Agreement.
*
*        F. In an effort to track usage and maintain accurate records of the
*        Subject Software, each Recipient, upon receipt of the Subject
*        Software, is requested to register with Government Agency by visiting
*        the following website: https://opensource.arc.nasa.gov.  Recipient's
*        name and personal information shall be used for statistical purposes
*        only. Once a Recipient makes a Modification available, it is requested
*        that the Recipient inform Government Agency at the web site provided
*        above how to access the Modification.
*
*        G. Each Contributor represents that that its Modification is believed
*        to be Contributor's original creation and does not violate any
*        existing agreements, regulations, statutes or rules, and further that
*        Contributor has sufficient rights to grant the rights conveyed by this
*        Agreement.
*
*        H. A Recipient may choose to offer, and to charge a fee for, warranty,
*        support, indemnity and/or liability obligations to one or more other
*        Recipients of the Subject Software.  A Recipient may do so, however,
*        only on its own behalf and not on behalf of Government Agency or any
*        other Recipient.  Such a Recipient must make it absolutely clear that
*        any such warranty, support, indemnity and/or liability obligation is
*        offered by that Recipient alone.  Further, such Recipient agrees to
*        indemnify Government Agency and every other Recipient for any
*        liability incurred by them as a result of warranty, support, indemnity
*        and/or liability offered by such Recipient.
*
*        I. A Recipient may create a Larger Work by combining Subject Software
*        with separate software not governed by the terms of this agreement and
*        distribute the Larger Work as a single product. In such case, the
*        Recipient must make sure Subject Software, or portions thereof,
*        included in the Larger Work is subject to this Agreement.
*
*        J. Notwithstanding any provisions contained herein, Recipient is
*        hereby put on notice that export of any goods or technical data from
*        the United States may require some form of export license from the
*        U.S. Government.  Failure to obtain necessary export licenses may
*        result in criminal liability under U.S. laws.  Government Agency
*        neither represents that a license shall not be required nor that, if
*        required, it shall be issued.  Nothing granted herein provides any
*        such export license.
*
*        4. DISCLAIMER OF WARRANTIES AND LIABILITIES; WAIVER AND INDEMNIFICATION
*
*        A. No Warranty: THE SUBJECT SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY
*        WARRANTY OF ANY KIND, EITHER EXPRESSED, IMPLIED, OR STATUTORY,
*        INCLUDING, BUT NOT LIMITED TO, ANY WARRANTY THAT THE SUBJECT SOFTWARE
*        WILL CONFORM TO SPECIFICATIONS, ANY IMPLIED WARRANTIES OF
*        MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR FREEDOM FROM
*        INFRINGEMENT, ANY WARRANTY THAT THE SUBJECT SOFTWARE WILL BE ERROR
*        FREE, OR ANY WARRANTY THAT DOCUMENTATION, IF PROVIDED, WILL CONFORM TO
*        THE SUBJECT SOFTWARE. THIS AGREEMENT DOES NOT, IN ANY MANNER,
*        CONSTITUTE AN ENDORSEMENT BY GOVERNMENT AGENCY OR ANY PRIOR RECIPIENT
*        OF ANY RESULTS, RESULTING DESIGNS, HARDWARE, SOFTWARE PRODUCTS OR ANY
*        OTHER APPLICATIONS RESULTING FROM USE OF THE SUBJECT SOFTWARE.
*        FURTHER, GOVERNMENT AGENCY DISCLAIMS ALL WARRANTIES AND LIABILITIES
*        REGARDING THIRD-PARTY SOFTWARE, IF PRESENT IN THE ORIGINAL SOFTWARE,
*        AND DISTRIBUTES IT "AS IS."
*
*        B. Waiver and Indemnity: RECIPIENT AGREES TO WAIVE ANY AND ALL CLAIMS
*        AGAINST THE UNITED STATES GOVERNMENT, ITS CONTRACTORS AND
*        SUBCONTRACTORS, AS WELL AS ANY PRIOR RECIPIENT.  IF RECIPIENT'S USE OF
*        THE SUBJECT SOFTWARE RESULTS IN ANY LIABILITIES, DEMANDS, DAMAGES,
*        EXPENSES OR LOSSES ARISING FROM SUCH USE, INCLUDING ANY DAMAGES FROM
*        PRODUCTS BASED ON, OR RESULTING FROM, RECIPIENT'S USE OF THE SUBJECT
*        SOFTWARE, RECIPIENT SHALL INDEMNIFY AND HOLD HARMLESS THE UNITED
*        STATES GOVERNMENT, ITS CONTRACTORS AND SUBCONTRACTORS, AS WELL AS ANY
*        PRIOR RECIPIENT, TO THE EXTENT PERMITTED BY LAW.  RECIPIENT'S SOLE
*        REMEDY FOR ANY SUCH MATTER SHALL BE THE IMMEDIATE, UNILATERAL
*        TERMINATION OF THIS AGREEMENT.
*
*
*        5. GENERAL TERMS
*
*        A. Termination: This Agreement and the rights granted hereunder will
*        terminate automatically if a Recipient fails to comply with these
*        terms and conditions, and fails to cure such noncompliance within
*        thirty (30) days of becoming aware of such noncompliance.  Upon
*        termination, a Recipient agrees to immediately cease use and
*        distribution of the Subject Software.  All sublicenses to the Subject
*        Software properly granted by the breaching Recipient shall survive any
*        such termination of this Agreement.
*
*        B. Severability: If any provision of this Agreement is invalid or
*        unenforceable under applicable law, it shall not affect the validity
*        or enforceability of the remainder of the terms of this Agreement.
*
*        C. Applicable Law: This Agreement shall be subject to United States
*        federal law only for all purposes, including, but not limited to,
*        determining the validity of this Agreement, the meaning of its
*        provisions and the rights, obligations and remedies of the parties.
*
*        D. Entire Understanding: This Agreement constitutes the entire
*        understanding and agreement of the parties relating to release of the
*        Subject Software and may not be superseded, modified or amended except
*        by further written agreement duly executed by the parties.
*
*        E. Binding Authority: By accepting and using the Subject Software
*        under this Agreement, a Recipient affirms its authority to bind the
*        Recipient to all terms and conditions of this Agreement and that that
*        Recipient hereby agrees to all terms and conditions herein.
*
*        F. Point of Contact: Any Recipient contact with Government Agency is
*        to be directed to the designated representative as follows:
*        Patrick.Hogan@nasa.gov.
*    *//*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016:
 * adapted from nasa sources to support own 3d model loadings
 */
package eu.mavinci.desktop.gui.wwext;

import com.intel.missioncontrol.PublishSource;
import eu.mavinci.desktop.main.debug.Debug;
import eu.mavinci.desktop.main.debug.Debug;
import net.java.joglutils.model.ModelLoadException;
import net.java.joglutils.model.ResourceRetriever;
import net.java.joglutils.model.geometry.Bounds;
import net.java.joglutils.model.geometry.Face;
import net.java.joglutils.model.geometry.Material;
import net.java.joglutils.model.geometry.Mesh;
import net.java.joglutils.model.geometry.Model;
import net.java.joglutils.model.geometry.TexCoord;
import net.java.joglutils.model.geometry.Vec4;

import java.awt.Color;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

/** @author RodgersGB restructured by Peter Schau */

public class WaveFrontLoader2 {
    public static final String VERTEX_DATA = "v ";
    public static final String NORMAL_DATA = "vn ";
    public static final String TEXTURE_DATA = "vt ";
    public static final String FACE_DATA = "f ";
    public static final String SMOOTHING_GROUP = "s ";
    public static final String GROUP = "g ";
    public static final String OBJECT = "o ";
    public static final String COMMENT = "#";
    public static final String EMPTY = "";

    int vertexTotal = 0;
    int textureTotal = 0;
    int normalTotal = 0;

    // the model
    private Model model = null;
    /** Bounds of the model */
    private Bounds bounds = new Bounds();
    /** Center of the model */
    private Vec4 center = new Vec4(0.0f, 0.0f, 0.0f);

    private String baseDir = null;

    protected void init() {
        vertexTotal = 0;
        textureTotal = 0;
        normalTotal = 0;
        model = null;
        bounds = new Bounds();
        center = new Vec4(0.0f, 0.0f, 0.0f);
        baseDir = null;
    }

    /** Creates a new instance of myWaveFrontLoader */
    public WaveFrontLoader2() {}

    public void printVecArray(String name, Vector<Vec4> vs) {
        // System.out.println(name+":");
        // System.out.print("[");
        for (Vec4 v : vs) {
            System.out.format("v %.0f %.0f %.0f %.0f\n", v.x, v.y, v.z, v.w);
            // System.out.print("("+v[i].x+","+v[i].y+","+v[i].z+","+v[i].w+"),");
        }
        // System.out.print("[");
    }

    public void printFaceArray(String name, Vector<Face> fs) {
        // System.out.println(name+":");
        // System.out.print("[");
        for (Face f : fs) {
            System.out.format(
                "f %s %s %s\n",
                Arrays.toString(f.vertIndex), Arrays.toString(f.coordIndex), Arrays.toString(f.normalIndex));
            // System.out.print("("+v[i].x+","+v[i].y+","+v[i].z+","+v[i].w+"),");
        }
        // System.out.print("[");
    }

    /**
     * not group statement used
     *
     * @param br
     * @return
     * @throws ModelLoadException
     * @throws IOException
     */
    private Model load(BufferedReader br) throws ModelLoadException, IOException {
        Debug.getLog().fine("Loading mesh...");
        String line = null;
        Mesh mesh = new Mesh();
        Vector<Vec4> vertices = new Vector<Vec4>();
        Vector<TexCoord> texCoords = new Vector<TexCoord>();
        Vector<Vec4> normals = new Vector<Vec4>();
        Vector<Face> faces = new Vector<Face>();
        Set<String> groups = new HashSet<String>();
        Set<String> allGroups = new HashSet<String>();
        Map<String, Vector<Integer>> groupMap = new HashMap<String, Vector<Integer>>();
        Integer currentMaterial = -1;

        while ((line = br.readLine()) != null) {
            if (lineIs(COMMENT, line)) {
                // ignore comments
                numComments++;
                continue;
            } else if (line.length() == 0) {
                // ignore empty lines
                continue;
            } else if (lineIs(GROUP, line)) {
                groups = parseGroups(line);
                allGroups.addAll(groups);
            } else if (lineIs(OBJECT, line)) {
                groups = parseGroups(line); // handle object as group
                allGroups.addAll(groups);
            } else if (lineIs(VERTEX_DATA, line)) {
                Vec4 p = parsePoint(line);
                vertices.add(p);
                bounds.calc(p);
            } else if (lineIs(TEXTURE_DATA, line)) {
                String s[] = line.split("\\s+");
                TexCoord texCoord = new TexCoord();
                texCoord.u = Float.parseFloat(s[1]);
                texCoord.v = Float.parseFloat(s[2]);

                texCoords.add(texCoord);
            } else if (lineIs(NORMAL_DATA, line)) {
                normals.add(parsePoint(line));
            } else if (lineIs(FACE_DATA, line)) {
                Face f = parseFace(line);
                if (currentMaterial != -1) {
                    f.materialID = currentMaterial;
                }

                faces.add(f);
                for (String s : groups) {
                    if (groupMap.containsKey(s)) {
                        groupMap.get(s).add(faces.size() - 1);
                    } else {
                        Vector<Integer> l = new Vector<Integer>();
                        l.add(faces.size() - 1);
                        groupMap.put(s, l);
                    }
                }
            } else if (lineIs("mtllib ", line)) {
                processMaterialLib(line);
            } else if (lineIs("usemtl ", line)) {
                int materialId = processMaterialType(line, mesh);
                currentMaterial = materialId;
            }
        }

        mesh.faces = new Face[0];

        mesh.faces = faces.toArray(mesh.faces);
        mesh.numOfFaces = mesh.faces.length;

        mesh.normals = new Vec4[0];
        mesh.normals = normals.toArray(mesh.normals);
        mesh.vertices = new Vec4[0];
        mesh.vertices = vertices.toArray(mesh.vertices);
        mesh.numOfVerts = mesh.vertices.length;
        mesh.hasTexture = false;
        mesh.texCoords = new TexCoord[0];
        if (texCoords.size() > 0) {
            // mesh.hasTexture = true;
            mesh.texCoords = texCoords.toArray(mesh.texCoords);
        }

        mesh.numTexCoords = mesh.texCoords.length;
        mesh.bounds = bounds;

        if (Float.isInfinite(bounds.max.x) || Float.isNaN(bounds.max.x))
            throw new RuntimeException("bounds.max.x is out of bounds");
        if (Float.isInfinite(bounds.max.y) || Float.isNaN(bounds.max.y))
            throw new RuntimeException("bounds.max.y is out of bounds");
        if (Float.isInfinite(bounds.max.z) || Float.isNaN(bounds.max.z))
            throw new RuntimeException("bounds.max.z is out of bounds");

        if (Float.isNaN(bounds.min.x) || Float.isInfinite(bounds.min.x))
            throw new RuntimeException("bounds.min.x is out of bounds");
        if (Float.isNaN(bounds.min.y) || Float.isInfinite(bounds.min.y))
            throw new RuntimeException("bounds.min.y is out of bounds");
        if (Float.isNaN(bounds.min.z) || Float.isInfinite(bounds.min.z))
            throw new RuntimeException("bounds.min.z is out of bounds");

        // Calculate the center of the model
        center.x = 0.5f * (bounds.max.x + bounds.min.x);
        center.y = 0.5f * (bounds.max.y + bounds.min.y);
        center.z = 0.5f * (bounds.max.z + bounds.min.z);
        // mesh.hasTexture = false; // TOD remove
        model.addMesh(mesh);
        // mesh.hasTexture = true;
        Debug.getLog().fine("hasTexture: " + mesh.hasTexture);

        // System.out.println("mesh:"+mesh);
        // System.out.println("faces" + mesh.faces.length + " "+ Arrays.asList(mesh.faces));
        // System.out.println("normals" +mesh.normals.length + " "+ Arrays.asList(mesh.normals));
        // System.out.println("vertices" +mesh.vertices.length + " "+ Arrays.asList(mesh.vertices));
        // System.out.println("texCoords" + mesh.texCoords.length + " "+Arrays.asList(mesh.texCoords));

        // printVecArray("vertices",normals);
        // printFaceArray("faces",faces);
        Debug.getLog().fine(this.bounds.toString());
        model.setBounds(this.bounds);
        model.setCenterPoint(this.center);
        model.setUseLighting(true);
        model.setRenderModel(true);
        model.setRenderAsWireframe(false);
        model.setRenderModelBounds(false);
        model.setRenderObjectBounds(false);
        model.setUseTexture(true);
        model.setRenderPicker(false);
        model.centerModelOnPosition(true);
        // model.centerModelOnPosition(true); //do this only if it is meaningful for the current model

        return model;
    }

    private Set<String> parseGroups(String line) {
        final String s[] = line.split("\\s+");

        Set<String> set = new HashSet<String>();
        set.addAll(Arrays.asList(s));
        return set;
    }

    int numComments = 0;

    public synchronized Model load(String path) throws ModelLoadException {
        init();
        model = new Model(path);

        baseDir = "";
        String tokens[] = path.split("/");
        for (int i = 0; i < tokens.length - 1; i++) {
            baseDir += tokens[i] + "/";
        }

        try (InputStream stream = ResourceRetriever.getResourceAsInputStream(model.getSource());
             BufferedReader br = new BufferedReader(new InputStreamReader(stream))) {
            // Open a file handle and read the models data
            return load(br);
        } catch (IOException e) {
            throw new ModelLoadException("Caught IO exception: " + e);
        }

    }

    private boolean lineIs(String type, String line) {
        return line.startsWith(type);
    }

    private Face parseFace(String line) {
        String s[] = line.split("\\s+");
        if (line.contains("//")) { // Pattern is present if obj has no texture
            for (int loop = 1; loop < s.length; loop++) {
                s[loop] = s[loop].replaceAll("//", "/-1/"); // insert -1 for missing vt data
            }
        }

        Face face = new Face(s.length - 1);

        for (int loop = 1; loop < s.length; loop++) {
            String s1 = s[loop];
            String[] temp = s1.split("/");

            if (temp.length > 0) { // we have vertex data
                if (Integer.valueOf(temp[0]) < 0) {
                    // TODO handle relative vertex data
                } else {
                    face.vertIndex[loop - 1] = Integer.valueOf(temp[0]) - 1 - this.vertexTotal;
                    // System.out.println("found vertex index: " + face.vertIndex[loop-1]);
                }
            }

            if (temp.length > 1) { // we have texture data
                if (Integer.valueOf(temp[1]) < 0) {
                    face.coordIndex[loop - 1] = 0;
                } else {
                    face.coordIndex[loop - 1] = Integer.valueOf(temp[1]) - 1 - this.textureTotal;
                    // System.out.println("found texture index: " + face.coordIndex[loop-1]);
                }
            }

            if (temp.length > 2) { // we have normal data
                face.normalIndex[loop - 1] = Integer.valueOf(temp[2]) - 1 - this.normalTotal;
                // System.out.println("found normal index: " + face.normalIndex[loop-1]);
            }
        }

        return face;
    }

    private Vec4 parsePoint(String line) {
        Vec4 point = new Vec4();

        final String s[] = line.split("\\s+");

        point.x = Float.parseFloat(s[1]);
        point.y = Float.parseFloat(s[2]);
        point.z = Float.parseFloat(s[3]);

        return point;
    }

    private void processMaterialLib(String mtlData) {
        String s[] = mtlData.split("\\s+");

        // Material mat = new Material();
        Boolean bLoadMaterialFile = false;
        try (InputStream stream = ResourceRetriever.getResourceAsInputStream(baseDir + s[1])) {
            loadMaterialFile(stream);
            bLoadMaterialFile = true;
        } catch (IOException ex) {
            ex.printStackTrace();
        }

        if (!bLoadMaterialFile) {
            try (InputStream stream2 = new FileInputStream(baseDir + s[1])) {
                loadMaterialFile(stream2);
            } catch (FileNotFoundException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

    private int processMaterialType(String line, Mesh mesh) {
        String s[] = line.split("\\s+");

        int materialID = -1;

        for (int i = 0; i < model.getNumberOfMaterials(); i++) {
            Material mat = model.getMaterial(i);

            if (mat.strName.equals(s[1])) {
                materialID = i;
                // System.out.println("material "+mat.strName+":"+i+" assigned to mesh");
                if (mat.strFile != null) {
                    mesh.hasTexture |= true;
                }

                break;
            }
        }

        return materialID;
        // mesh.materialID = materialID;
    }

    public Material loadMaterialFile(InputStream stream) {
        Material mat = null;
        int texId = 0;

        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(stream));
            String line;

            while ((line = br.readLine()) != null) {
                String parts[] = line.trim().split("\\s+");

                if (parts[0].equals("newmtl")) {
                    if (mat != null) {
                        model.addMaterial(mat);
                    }

                    mat = new Material();
                    mat.strName = parts[1];
                    // System.out.println("newmatl: "+mat.strName);
                    mat.textureId = texId;
                    texId++;
                } else if (parts[0].equals("Ks") && mat != null) {
                    mat.specularColor = parseColor(line);
                } else if (parts[0].equals("Ns")) {
                    if (parts.length > 1 && mat != null) {
                        mat.shininess = Float.valueOf(parts[1]);
                    }
                } else if (parts[0].equals("d")) {;
                } else if (parts[0].equals("illum")) {;
                } else if (parts[0].equals("Ka") && mat != null) {
                    mat.ambientColor = parseColor(line);
                    // System.out.println("ambient color: "+mat.ambientColor);
                } else if (parts[0].equals("Kd") && mat != null) {
                    mat.diffuseColor = parseColor(line);
                    // mat.ambientColor = mat.diffuseColor; //TODO remove
                    // System.out.println("diffuseColor: "+mat.diffuseColor);
                } else if (parts[0].equals("map_Kd")) {
                    if (parts.length > 1 && mat != null) {
                        mat.strFile = /* baseDir + */ parts[1];
                    }
                } else if (parts[0].equals("map_Ka")) {
                    if (parts.length > 1 && mat != null) {
                        mat.strFile = /* baseDir + */ parts[1];
                    }
                }
            }

            br.close();
            model.addMaterial(mat);
            // System.out.println("added material "+mat.strName+" to model");

        } catch (FileNotFoundException ex) {
            ex.printStackTrace();
        } catch (IOException ioe) {
            ioe.printStackTrace();
        }

        return mat;
    }

    private Color parseColor(String line) {
        String parts[] = line.trim().split("\\s+");

        Color color = new Color(Float.valueOf(parts[1]), Float.valueOf(parts[2]), Float.valueOf(parts[3]));

        return color;
    }

    // public static void main(String[] args) {
    // WaveFrontLoader2 loader = new WaveFrontLoader2();
    // try {
    // loader.load("C:\\Documents and Settings\\RodgersGB\\My
    // Documents\\Projects\\JOGLUTILS\\src\\net\\java\\joglutils\\examples\\models\\obj\\penguin.obj");
    // } catch (ModelLoadException ex) {
    // ex.printStackTrace();
    // }
    // }
}
