/*
*        NASA OPEN SOURCE AGREEMENT VERSION 1.3
*
*        THIS OPEN SOURCE AGREEMENT ("AGREEMENT") DEFINES THE RIGHTS OF USE,
*        REPRODUCTION, DISTRIBUTION, MODIFICATION AND REDISTRIBUTION OF CERTAIN
*        COMPUTER SOFTWARE ORIGINALLY RELEASED BY THE UNITED STATES GOVERNMENT
*        AS REPRESENTED BY THE GOVERNMENT AGENCY LISTED BELOW ("GOVERNMENT
*        AGENCY").  THE UNITED STATES GOVERNMENT, AS REPRESENTED BY GOVERNMENT
*        AGENCY, IS AN INTENDED THIRD-PARTY BENEFICIARY OF ALL SUBSEQUENT
*        DISTRIBUTIONS OR REDISTRIBUTIONS OF THE SUBJECT SOFTWARE.  ANYONE WHO
*        USES, REPRODUCES, DISTRIBUTES, MODIFIES OR REDISTRIBUTES THE SUBJECT
*        SOFTWARE, AS DEFINED HEREIN, OR ANY PART THEREOF, IS, BY THAT ACTION,
*        ACCEPTING IN FULL THE RESPONSIBILITIES AND OBLIGATIONS CONTAINED IN
*        THIS AGREEMENT.
*
*        Government Agency: National Aeronautics and Space Administration (NASA)
*        Government Agency Original Software Designation: ARC-15166-1
*        Government Agency Original Software Title: WorldWind Version 1.3
*        User Registration Requested.  Please Visit https://opensource.arc.nasa.gov/
*        Government Agency Point of Contact for Original Software: Patrick.Hogan@nasa.gov
*        ________________________________________________
*
*
*        1. DEFINITIONS
*
*        A. "Contributor" means Government Agency, as the developer of the
*        Original Software, and any entity that makes a Modification.
*        B. "Covered Patents" mean patent claims licensable by a Contributor
*        that are necessarily infringed by the use or sale of its Modification
*        alone or when combined with the Subject Software.
*        C. "Display" means the showing of a copy of the Subject Software,
*        either directly or by means of an image, or any other device.
*        D. "Distribution" means conveyance or transfer of the Subject
*        Software, regardless of means, to another.
*        E. "Larger Work" means computer software that combines Subject
*        Software, or portions thereof, with software separate from the Subject
*        Software that is not governed by the terms of this Agreement.
*        F.  "Modification" means any alteration of, including addition to or
*        deletion from, the substance or structure of either the Original
*        Software or Subject Software, and includes derivative works, as that
*        term is defined in the Copyright Statute, 17 USC 101.  However, the
*        act of including Subject Software as part of a Larger Work does not in
*        and of itself constitute a Modification.
*        G. "Original Software" means the computer software first released
*        under this Agreement by Government Agency with Government Agency
*        designation ARC-15166-1 and entitled WorldWind, including source code,
*        object code and accompanying documentation, if any.
*        H. "Recipient" means anyone who acquires the Subject Software under
*        this Agreement, including all Contributors.
*        I. "Redistribution" means Distribution of the Subject Software after a
*        Modification has been made.
*        J. "Reproduction" means the making of a counterpart, image or copy of
*        the Subject Software.
*        K. "Sale" means the exchange of the Subject Software for money or
*        equivalent value.
*        L. "Subject Software" means the Original Software, Modifications, or
*        any respective parts thereof.
*        M. "Use" means the application or employment of the Subject Software
*        for any purpose.
*
*        2. GRANT OF RIGHTS
*
*        A. Under Non-Patent Rights: Subject to the terms and conditions of
*        this Agreement, each Contributor, with respect to its own contribution
*        to the Subject Software, hereby grants to each Recipient a
*        non-exclusive, world-wide, royalty-free license to engage in the
*        following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Modification
*        5. Redistribution
*        6. Display
*
*        B. Under Patent Rights: Subject to the terms and conditions of this
*        Agreement, each Contributor, with respect to its own contribution to
*        the Subject Software, hereby grants to each Recipient under Covered
*        Patents a non-exclusive, world-wide, royalty-free license to engage in
*        the following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Sale
*        5. Offer for Sale
*
*        C. The rights granted under Paragraph B. also apply to the combination
*        of a Contributor's Modification and the Subject Software if, at the
*        time the Modification is added by the Contributor, the addition of
*        such Modification causes the combination to be covered by the Covered
*        Patents.  It does not apply to any other combinations that include a
*        Modification.
*
*        D. The rights granted in Paragraphs A. and B. allow the Recipient to
*        sublicense those same rights.  Such sublicense must be under the same
*        terms and conditions of this Agreement.
*
*        3. OBLIGATIONS OF RECIPIENT
*
*        A. Distribution or Redistribution of the Subject Software must be made
*        under this Agreement except for additions covered under paragraph 3H.
*
*        1. Whenever a Recipient distributes or redistributes the Subject
*        Software, a copy of this Agreement must be included with each copy
*        of the Subject Software; and
*        2. If Recipient distributes or redistributes the Subject Software in
*        any form other than source code, Recipient must also make the
*        source code freely available, and must provide with each copy of
*        the Subject Software information on how to obtain the source code
*        in a reasonable manner on or through a medium customarily used for
*        software exchange.
*
*        B. Each Recipient must ensure that the following copyright notice
*        appears prominently in the Subject Software:
*
*        Copyright (C) 2001 United States Government
*        as represented by the Administrator of the
*        National Aeronautics and Space Administration.
*        All Rights Reserved.
*
*        C. Each Contributor must characterize its alteration of the Subject
*        Software as a Modification and must identify itself as the originator
*        of its Modification in a manner that reasonably allows subsequent
*        Recipients to identify the originator of the Modification.  In
*        fulfillment of these requirements, Contributor must include a file
*        (e.g., a change log file) that describes the alterations made and the
*        date of the alterations, identifies Contributor as originator of the
*        alterations, and consents to characterization of the alterations as a
*        Modification, for example, by including a statement that the
*        Modification is derived, directly or indirectly, from Original
*        Software provided by Government Agency. Once consent is granted, it
*        may not thereafter be revoked.
*
*        D. A Contributor may add its own copyright notice to the Subject
*        Software.  Once a copyright notice has been added to the Subject
*        Software, a Recipient may not remove it without the express permission
*        of the Contributor who added the notice.
*
*        E. A Recipient may not make any representation in the Subject Software
*        or in any promotional, advertising or other material that may be
*        construed as an endorsement by Government Agency or by any prior
*        Recipient of any product or service provided by Recipient, or that may
*        seek to obtain commercial advantage by the fact of Government Agency's
*        or a prior Recipient's participation in this Agreement.
*
*        F. In an effort to track usage and maintain accurate records of the
*        Subject Software, each Recipient, upon receipt of the Subject
*        Software, is requested to register with Government Agency by visiting
*        the following website: https://opensource.arc.nasa.gov.  Recipient's
*        name and personal information shall be used for statistical purposes
*        only. Once a Recipient makes a Modification available, it is requested
*        that the Recipient inform Government Agency at the web site provided
*        above how to access the Modification.
*
*        G. Each Contributor represents that that its Modification is believed
*        to be Contributor's original creation and does not violate any
*        existing agreements, regulations, statutes or rules, and further that
*        Contributor has sufficient rights to grant the rights conveyed by this
*        Agreement.
*
*        H. A Recipient may choose to offer, and to charge a fee for, warranty,
*        support, indemnity and/or liability obligations to one or more other
*        Recipients of the Subject Software.  A Recipient may do so, however,
*        only on its own behalf and not on behalf of Government Agency or any
*        other Recipient.  Such a Recipient must make it absolutely clear that
*        any such warranty, support, indemnity and/or liability obligation is
*        offered by that Recipient alone.  Further, such Recipient agrees to
*        indemnify Government Agency and every other Recipient for any
*        liability incurred by them as a result of warranty, support, indemnity
*        and/or liability offered by such Recipient.
*
*        I. A Recipient may create a Larger Work by combining Subject Software
*        with separate software not governed by the terms of this agreement and
*        distribute the Larger Work as a single product. In such case, the
*        Recipient must make sure Subject Software, or portions thereof,
*        included in the Larger Work is subject to this Agreement.
*
*        J. Notwithstanding any provisions contained herein, Recipient is
*        hereby put on notice that export of any goods or technical data from
*        the United States may require some form of export license from the
*        U.S. Government.  Failure to obtain necessary export licenses may
*        result in criminal liability under U.S. laws.  Government Agency
*        neither represents that a license shall not be required nor that, if
*        required, it shall be issued.  Nothing granted herein provides any
*        such export license.
*
*        4. DISCLAIMER OF WARRANTIES AND LIABILITIES; WAIVER AND INDEMNIFICATION
*
*        A. No Warranty: THE SUBJECT SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY
*        WARRANTY OF ANY KIND, EITHER EXPRESSED, IMPLIED, OR STATUTORY,
*        INCLUDING, BUT NOT LIMITED TO, ANY WARRANTY THAT THE SUBJECT SOFTWARE
*        WILL CONFORM TO SPECIFICATIONS, ANY IMPLIED WARRANTIES OF
*        MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR FREEDOM FROM
*        INFRINGEMENT, ANY WARRANTY THAT THE SUBJECT SOFTWARE WILL BE ERROR
*        FREE, OR ANY WARRANTY THAT DOCUMENTATION, IF PROVIDED, WILL CONFORM TO
*        THE SUBJECT SOFTWARE. THIS AGREEMENT DOES NOT, IN ANY MANNER,
*        CONSTITUTE AN ENDORSEMENT BY GOVERNMENT AGENCY OR ANY PRIOR RECIPIENT
*        OF ANY RESULTS, RESULTING DESIGNS, HARDWARE, SOFTWARE PRODUCTS OR ANY
*        OTHER APPLICATIONS RESULTING FROM USE OF THE SUBJECT SOFTWARE.
*        FURTHER, GOVERNMENT AGENCY DISCLAIMS ALL WARRANTIES AND LIABILITIES
*        REGARDING THIRD-PARTY SOFTWARE, IF PRESENT IN THE ORIGINAL SOFTWARE,
*        AND DISTRIBUTES IT "AS IS."
*
*        B. Waiver and Indemnity: RECIPIENT AGREES TO WAIVE ANY AND ALL CLAIMS
*        AGAINST THE UNITED STATES GOVERNMENT, ITS CONTRACTORS AND
*        SUBCONTRACTORS, AS WELL AS ANY PRIOR RECIPIENT.  IF RECIPIENT'S USE OF
*        THE SUBJECT SOFTWARE RESULTS IN ANY LIABILITIES, DEMANDS, DAMAGES,
*        EXPENSES OR LOSSES ARISING FROM SUCH USE, INCLUDING ANY DAMAGES FROM
*        PRODUCTS BASED ON, OR RESULTING FROM, RECIPIENT'S USE OF THE SUBJECT
*        SOFTWARE, RECIPIENT SHALL INDEMNIFY AND HOLD HARMLESS THE UNITED
*        STATES GOVERNMENT, ITS CONTRACTORS AND SUBCONTRACTORS, AS WELL AS ANY
*        PRIOR RECIPIENT, TO THE EXTENT PERMITTED BY LAW.  RECIPIENT'S SOLE
*        REMEDY FOR ANY SUCH MATTER SHALL BE THE IMMEDIATE, UNILATERAL
*        TERMINATION OF THIS AGREEMENT.
*
*
*        5. GENERAL TERMS
*
*        A. Termination: This Agreement and the rights granted hereunder will
*        terminate automatically if a Recipient fails to comply with these
*        terms and conditions, and fails to cure such noncompliance within
*        thirty (30) days of becoming aware of such noncompliance.  Upon
*        termination, a Recipient agrees to immediately cease use and
*        distribution of the Subject Software.  All sublicenses to the Subject
*        Software properly granted by the breaching Recipient shall survive any
*        such termination of this Agreement.
*
*        B. Severability: If any provision of this Agreement is invalid or
*        unenforceable under applicable law, it shall not affect the validity
*        or enforceability of the remainder of the terms of this Agreement.
*
*        C. Applicable Law: This Agreement shall be subject to United States
*        federal law only for all purposes, including, but not limited to,
*        determining the validity of this Agreement, the meaning of its
*        provisions and the rights, obligations and remedies of the parties.
*
*        D. Entire Understanding: This Agreement constitutes the entire
*        understanding and agreement of the parties relating to release of the
*        Subject Software and may not be superseded, modified or amended except
*        by further written agreement duly executed by the parties.
*
*        E. Binding Authority: By accepting and using the Subject Software
*        under this Agreement, a Recipient affirms its authority to bind the
*        Recipient to all terms and conditions of this Agreement and that that
*        Recipient hereby agrees to all terms and conditions herein.
*
*        F. Point of Contact: Any Recipient contact with Government Agency is
*        to be directed to the designated representative as follows:
*        Patrick.Hogan@nasa.gov.
*    *//*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016:
 * restored from old nasa sources
 */
package eu.mavinci.desktop.gui.wwext.sunlight;

import com.intel.missioncontrol.PublishSource;
import eu.mavinci.core.obfuscation.IKeepAll;
import gov.nasa.worldwind.View;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Matrix;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.layers.AbstractLayer;
import gov.nasa.worldwind.render.DrawContext;
import gov.nasa.worldwind.util.Logging;
import eu.mavinci.core.obfuscation.IKeepAll;

import com.jogamp.opengl.GL2;
import com.jogamp.opengl.GLContext;
import java.awt.Color;

/**
 * Renders an atmosphere around the globe and a sky dome at low altitude. Uses atmospheric scattering as color source.
 *
 * <p>Issue : Ellipsoidal globe doesnt match the spherical atmosphere everywhere.
 *
 * <p>TODO: Find a way to get a blue sky at ground level TODO: Increase dome geometry resolution and implement partial
 * sphere
 *
 * @author Patrick Murris
 * @version $Id: AtmosphereLayer.java 14741 2011-02-16 23:36:47Z tgaskins $
 */

public class AtmosphereLayer extends AbstractLayer implements IKeepAll {
    protected static final int STACKS = 24;
    protected static final int SLICES = 64;

    protected int glListId = -1; // GL list id
    protected double thickness = 60e3; // Atmosphere thickness
    protected double lastRebuildHorizon = 0;

    protected AtmosphericScatteringComputer asc;
    protected Vec4 sunDirection;
    protected boolean update = true;

    /** Renders an atmosphere around the globe */
    public AtmosphereLayer() {}

    /**
     * Get the atmosphere thickness in meter
     *
     * @return the atmosphere thickness in meter
     */
    public double getAtmosphereThickness() {
        return this.thickness;
    }

    /**
     * Set the atmosphere thickness in meter
     *
     * @param thickness the atmosphere thickness in meter
     */
    public void setAtmosphereThickness(double thickness) {
        if (thickness < 0) {
            String msg = Logging.getMessage("generic.ArgumentOutOfRange");
            Logging.logger().severe(msg);
            throw new IllegalArgumentException(msg);
        }

        this.thickness = thickness;
        this.asc = null; // invalidate atmospheric scattering computer
        this.update = true;
    }

    public Vec4 getSunDirection() {
        return this.sunDirection;
    }

    public void setSunDirection(Vec4 direction) {
        this.sunDirection = direction;
        this.update = true;
    }

    @Override
    public void doRender(DrawContext dc) {
        GL2 gl = dc.getGL().getGL2();
        boolean attribsPushed = false;
        boolean modelviewPushed = false;
        boolean projectionPushed = false;

        try {
            View view = dc.getView();
            Position camPos = dc.getGlobe().computePositionFromPoint(view.getEyePoint());
            double worldRadius = dc.getGlobe().getRadiusAt(camPos);
            double distToCenterOfPlanet = view.getEyePoint().getLength3();
            double camAlt = camPos.getElevation();
            double tangentalDistance = view.getFarClipDistance();
            // Dome radius
            double domeRadius = tangentalDistance;

            // horizon latitude degrees
            double horizonLat = (-Math.PI / 2 + Math.acos(tangentalDistance / distToCenterOfPlanet)) * 180 / Math.PI;
            // zenith latitude degrees
            double zenithLat = 90;
            if (camAlt >= thickness) {
                double tangentalDistanceZenith =
                    Math.sqrt(
                        distToCenterOfPlanet * distToCenterOfPlanet
                            - (worldRadius + thickness) * (worldRadius + thickness));
                zenithLat = (-Math.PI / 2 + Math.acos(tangentalDistanceZenith / distToCenterOfPlanet)) * 180 / Math.PI;
            }

            if (camAlt < thickness && camAlt > thickness * 0.7) {
                zenithLat = (thickness - camAlt) / (thickness - thickness * 0.7) * 90;
            }

            // Build or rebuild sky dome if horizon distance changed more then 100m
            if (this.update || this.glListId == -1 || Math.abs(this.lastRebuildHorizon - tangentalDistance) > 100) {
                if (this.glListId != -1) {
                    gl.glDeleteLists(this.glListId, 1);
                }

                this.makeSkyDome(dc, (float)(domeRadius), horizonLat, zenithLat, SLICES, STACKS);
                this.lastRebuildHorizon = tangentalDistance;
                this.update = false;
            }

            // GL set up
            gl.glPushAttrib(GL2.GL_POLYGON_BIT); // Temporary hack around aliased sky.
            gl.glPopAttrib();

            gl.glPushAttrib(
                GL2.GL_COLOR_BUFFER_BIT
                    | GL2.GL_DEPTH_BUFFER_BIT
                    | GL2.GL_TRANSFORM_BIT
                    | GL2.GL_POLYGON_BIT
                    | GL2.GL_ENABLE_BIT
                    | GL2.GL_CURRENT_BIT);
            attribsPushed = true;
            gl.glDisable(GL2.GL_DEPTH_TEST);
            gl.glDepthMask(false);

            Matrix projection =
                Matrix.fromPerspective(
                    view.getFieldOfView(),
                    Math.max(view.getViewport().getWidth(), 1),
                    Math.max(view.getViewport().getHeight(), 1),
                    10e3,
                    2 * distToCenterOfPlanet + 10e3);
            double[] matrixArray = new double[16];
            projection.toArray(matrixArray, 0, false);
            gl.glMatrixMode(GL2.GL_PROJECTION);
            gl.glPushMatrix();
            projectionPushed = true;
            gl.glLoadMatrixd(matrixArray, 0);

            gl.glMatrixMode(GL2.GL_MODELVIEW);
            gl.glPushMatrix();
            modelviewPushed = true;
            // Sky transform
            Matrix skyTransform = computeSkyTransform(dc);
            Matrix modelView = view.getModelviewMatrix().multiply(skyTransform);
            modelView.toArray(matrixArray, 0, false);
            gl.glLoadMatrixd(matrixArray, 0);
            // Draw sky
            if (this.glListId != -1) {
                gl.glCallList(this.glListId);
            }

        } finally {
            // Restore GL state
            if (modelviewPushed) {
                gl.glMatrixMode(GL2.GL_MODELVIEW);
                gl.glPopMatrix();
            }

            if (projectionPushed) {
                gl.glMatrixMode(GL2.GL_PROJECTION);
                gl.glPopMatrix();
            }

            if (attribsPushed) {
                gl.glPopAttrib();
            }
        }
    }

    /**
     * Build sky dome and draw into a glList
     *
     * @param dc the current DrawContext
     * @param radius the sky dome radius in meters.
     * @param startLat the horizon latitude in decimal degrees.
     * @param endLat the zenith latitude in decimal degrees.
     * @param slices the number of longitude divisions used for the dome geometry.
     * @param stacks the number of latitude divisions used for the dome geometry.
     */
    protected void makeSkyDome(DrawContext dc, float radius, double startLat, double endLat, int slices, int stacks) {
        if (this.sunDirection == null) {
            return;
        }

        GL2 gl = dc.getGL().getGL2();
        this.glListId = gl.glGenLists(1);
        gl.glNewList(this.glListId, GL2.GL_COMPILE);
        this.drawSkyGradient(dc, radius, startLat, endLat, slices, stacks);
        gl.glEndList();
    }

    /**
     * Draws the sky dome
     *
     * @param dc the current DrawContext
     * @param radius the sky dome radius
     * @param startLat the horizon latitude
     * @param endLat the zenith latitude
     * @param slices the number of slices - vertical divisions
     * @param stacks the nuber os stacks - horizontal divisions
     */
    protected void drawSkyGradient(
            DrawContext dc, float radius, double startLat, double endLat, int slices, int stacks) {
        // Init atmospheric scattering computer
        if (this.asc == null) {
            this.asc = new AtmosphericScatteringComputer(dc.getGlobe().getRadius(), this.thickness);
        }

        // Get sky dome transform
        Matrix skyTransform = computeSkyTransform(dc);

        // GL setup
        GL2 gl = dc.getGL().getGL2();
        gl.glBlendFunc(GL2.GL_SRC_ALPHA, GL2.GL_ONE_MINUS_SRC_ALPHA);
        gl.glEnable(GL2.GL_BLEND);
        // gl.glPolygonMode(GL2.GL_FRONT_AND_BACK, GL2.GL_LINE); // wireframe

        double latitude, longitude, latitudeTop = endLat;
        double linear, linearTop, k, kTop;
        Color color;
        Color[] stackColors = new Color[slices + 1];
        Vec4 eyePoint = dc.getView().getEyePoint();

        // bottom fade
        latitude = startLat - Math.max((endLat - startLat) / 4, 2);
        gl.glBegin(GL2.GL_QUAD_STRIP);
        for (int slice = 0; slice <= slices; slice++) {
            longitude = 180 - ((float)slice / slices * (float)360);
            Vec4 v1 = SphericalToCartesian(latitude, longitude, radius);
            Vec4 v2 = SphericalToCartesian(startLat, longitude, radius);
            color = this.asc.getAtmosphereColor(v2.transformBy4(skyTransform), eyePoint, this.sunDirection);
            gl.glColor4f(color.getRed() / 255f, color.getGreen() / 255f, color.getBlue() / 255f, 0f);
            gl.glVertex3d(v1.getX(), v1.getY(), v1.getZ());
            gl.glColor4f(
                color.getRed() / 255f, color.getGreen() / 255f, color.getBlue() / 255f, color.getAlpha() / 255f);
            gl.glVertex3d(v2.getX(), v2.getY(), v2.getZ());
            stackColors[slice] = color;
        }

        gl.glEnd();

        // stacks and slices
        for (int stack = 1; stack < stacks - 1; stack++) {
            // bottom vertex
            linear = (float)(stack - 1) / (stacks - 1f);
            k = 1 - Math.cos(linear * Math.PI / 2);
            latitude = startLat + Math.pow(k, 3) * (endLat - startLat);
            // top vertex
            linearTop = (float)(stack) / (stacks - 1f);
            kTop = 1 - Math.cos(linearTop * Math.PI / 2);
            latitudeTop = startLat + Math.pow(kTop, 3) * (endLat - startLat);
            // Draw stack
            gl.glBegin(GL2.GL_QUAD_STRIP);
            for (int slice = 0; slice <= slices; slice++) {
                longitude = 180 - ((float)slice / slices * (float)360);
                Vec4 v = SphericalToCartesian(latitude, longitude, radius);
                color = stackColors[slice];
                gl.glColor4f(
                    color.getRed() / 255f, color.getGreen() / 255f, color.getBlue() / 255f, color.getAlpha() / 255f);
                gl.glVertex3d(v.getX(), v.getY(), v.getZ());
                v = SphericalToCartesian(latitudeTop, longitude, radius);
                color = this.asc.getAtmosphereColor(v.transformBy4(skyTransform), eyePoint, this.sunDirection);
                gl.glColor4f(
                    color.getRed() / 255f, color.getGreen() / 255f, color.getBlue() / 255f, color.getAlpha() / 255f);
                gl.glVertex3d(v.getX(), v.getY(), v.getZ());
                stackColors[slice] = color;
            }

            gl.glEnd();
        }

        // Top fade
        if (endLat < 90) {
            gl.glBegin(GL2.GL_QUAD_STRIP);
            for (int slice = 0; slice <= slices; slice++) {
                longitude = 180 - ((float)slice / slices * (float)360);
                Vec4 v = SphericalToCartesian(latitudeTop, longitude, radius);
                color = stackColors[slice];
                gl.glColor4f(
                    color.getRed() / 255f, color.getGreen() / 255f, color.getBlue() / 255f, color.getAlpha() / 255f);
                gl.glVertex3d(v.getX(), v.getY(), v.getZ());
                v = SphericalToCartesian(endLat, longitude, radius);
                gl.glColor4f(color.getRed() / 255f, color.getGreen() / 255f, color.getBlue() / 255f, 0);
                gl.glVertex3d(v.getX(), v.getY(), v.getZ());
            }

            gl.glEnd();
        }

        gl.glDisable(GL2.GL_BLEND);
    }

    protected Matrix computeSkyTransform(DrawContext dc) {
        Matrix transform = Matrix.IDENTITY;
        transform =
            transform.multiply(dc.getGlobe().computeSurfaceOrientationAtPosition(dc.getView().getEyePosition()));
        transform = transform.multiply(Matrix.fromRotationX(Angle.POS90));
        return transform;
    }

    /**
     * Converts position in spherical coordinates (lat/lon/altitude) to cartesian (XYZ) coordinates.
     *
     * @param latitude Latitude in decimal degrees
     * @param longitude Longitude in decimal degrees
     * @param radius Radius
     * @return the corresponding Point
     */
    protected static Vec4 SphericalToCartesian(double latitude, double longitude, double radius) {
        latitude *= Math.PI / 180.0f;
        longitude *= Math.PI / 180.0f;

        double radCosLat = radius * Math.cos(latitude);

        return new Vec4(radCosLat * Math.sin(longitude), radius * Math.sin(latitude), radCosLat * Math.cos(longitude));
    }

    public void dispose() {
        if (this.glListId < 0) {
            return;
        }

        GLContext glc = GLContext.getCurrent();
        if (glc == null) {
            return;
        }

        glc.getGL().getGL2().glDeleteLists(this.glListId, 1);
        this.glListId = -1;
    }

    @Override
    public String toString() {
        return Logging.getMessage("layers.Earth.SkyGradientLayer.Name");
    }
}
