/*
*        NASA OPEN SOURCE AGREEMENT VERSION 1.3
*
*        THIS OPEN SOURCE AGREEMENT ("AGREEMENT") DEFINES THE RIGHTS OF USE,
*        REPRODUCTION, DISTRIBUTION, MODIFICATION AND REDISTRIBUTION OF CERTAIN
*        COMPUTER SOFTWARE ORIGINALLY RELEASED BY THE UNITED STATES GOVERNMENT
*        AS REPRESENTED BY THE GOVERNMENT AGENCY LISTED BELOW ("GOVERNMENT
*        AGENCY").  THE UNITED STATES GOVERNMENT, AS REPRESENTED BY GOVERNMENT
*        AGENCY, IS AN INTENDED THIRD-PARTY BENEFICIARY OF ALL SUBSEQUENT
*        DISTRIBUTIONS OR REDISTRIBUTIONS OF THE SUBJECT SOFTWARE.  ANYONE WHO
*        USES, REPRODUCES, DISTRIBUTES, MODIFIES OR REDISTRIBUTES THE SUBJECT
*        SOFTWARE, AS DEFINED HEREIN, OR ANY PART THEREOF, IS, BY THAT ACTION,
*        ACCEPTING IN FULL THE RESPONSIBILITIES AND OBLIGATIONS CONTAINED IN
*        THIS AGREEMENT.
*
*        Government Agency: National Aeronautics and Space Administration (NASA)
*        Government Agency Original Software Designation: ARC-15166-1
*        Government Agency Original Software Title: WorldWind Version 1.3
*        User Registration Requested.  Please Visit https://opensource.arc.nasa.gov/
*        Government Agency Point of Contact for Original Software: Patrick.Hogan@nasa.gov
*        ________________________________________________
*
*
*        1. DEFINITIONS
*
*        A. "Contributor" means Government Agency, as the developer of the
*        Original Software, and any entity that makes a Modification.
*        B. "Covered Patents" mean patent claims licensable by a Contributor
*        that are necessarily infringed by the use or sale of its Modification
*        alone or when combined with the Subject Software.
*        C. "Display" means the showing of a copy of the Subject Software,
*        either directly or by means of an image, or any other device.
*        D. "Distribution" means conveyance or transfer of the Subject
*        Software, regardless of means, to another.
*        E. "Larger Work" means computer software that combines Subject
*        Software, or portions thereof, with software separate from the Subject
*        Software that is not governed by the terms of this Agreement.
*        F.  "Modification" means any alteration of, including addition to or
*        deletion from, the substance or structure of either the Original
*        Software or Subject Software, and includes derivative works, as that
*        term is defined in the Copyright Statute, 17 USC 101.  However, the
*        act of including Subject Software as part of a Larger Work does not in
*        and of itself constitute a Modification.
*        G. "Original Software" means the computer software first released
*        under this Agreement by Government Agency with Government Agency
*        designation ARC-15166-1 and entitled WorldWind, including source code,
*        object code and accompanying documentation, if any.
*        H. "Recipient" means anyone who acquires the Subject Software under
*        this Agreement, including all Contributors.
*        I. "Redistribution" means Distribution of the Subject Software after a
*        Modification has been made.
*        J. "Reproduction" means the making of a counterpart, image or copy of
*        the Subject Software.
*        K. "Sale" means the exchange of the Subject Software for money or
*        equivalent value.
*        L. "Subject Software" means the Original Software, Modifications, or
*        any respective parts thereof.
*        M. "Use" means the application or employment of the Subject Software
*        for any purpose.
*
*        2. GRANT OF RIGHTS
*
*        A. Under Non-Patent Rights: Subject to the terms and conditions of
*        this Agreement, each Contributor, with respect to its own contribution
*        to the Subject Software, hereby grants to each Recipient a
*        non-exclusive, world-wide, royalty-free license to engage in the
*        following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Modification
*        5. Redistribution
*        6. Display
*
*        B. Under Patent Rights: Subject to the terms and conditions of this
*        Agreement, each Contributor, with respect to its own contribution to
*        the Subject Software, hereby grants to each Recipient under Covered
*        Patents a non-exclusive, world-wide, royalty-free license to engage in
*        the following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Sale
*        5. Offer for Sale
*
*        C. The rights granted under Paragraph B. also apply to the combination
*        of a Contributor's Modification and the Subject Software if, at the
*        time the Modification is added by the Contributor, the addition of
*        such Modification causes the combination to be covered by the Covered
*        Patents.  It does not apply to any other combinations that include a
*        Modification.
*
*        D. The rights granted in Paragraphs A. and B. allow the Recipient to
*        sublicense those same rights.  Such sublicense must be under the same
*        terms and conditions of this Agreement.
*
*        3. OBLIGATIONS OF RECIPIENT
*
*        A. Distribution or Redistribution of the Subject Software must be made
*        under this Agreement except for additions covered under paragraph 3H.
*
*        1. Whenever a Recipient distributes or redistributes the Subject
*        Software, a copy of this Agreement must be included with each copy
*        of the Subject Software; and
*        2. If Recipient distributes or redistributes the Subject Software in
*        any form other than source code, Recipient must also make the
*        source code freely available, and must provide with each copy of
*        the Subject Software information on how to obtain the source code
*        in a reasonable manner on or through a medium customarily used for
*        software exchange.
*
*        B. Each Recipient must ensure that the following copyright notice
*        appears prominently in the Subject Software:
*
*        Copyright (C) 2001 United States Government
*        as represented by the Administrator of the
*        National Aeronautics and Space Administration.
*        All Rights Reserved.
*
*        C. Each Contributor must characterize its alteration of the Subject
*        Software as a Modification and must identify itself as the originator
*        of its Modification in a manner that reasonably allows subsequent
*        Recipients to identify the originator of the Modification.  In
*        fulfillment of these requirements, Contributor must include a file
*        (e.g., a change log file) that describes the alterations made and the
*        date of the alterations, identifies Contributor as originator of the
*        alterations, and consents to characterization of the alterations as a
*        Modification, for example, by including a statement that the
*        Modification is derived, directly or indirectly, from Original
*        Software provided by Government Agency. Once consent is granted, it
*        may not thereafter be revoked.
*
*        D. A Contributor may add its own copyright notice to the Subject
*        Software.  Once a copyright notice has been added to the Subject
*        Software, a Recipient may not remove it without the express permission
*        of the Contributor who added the notice.
*
*        E. A Recipient may not make any representation in the Subject Software
*        or in any promotional, advertising or other material that may be
*        construed as an endorsement by Government Agency or by any prior
*        Recipient of any product or service provided by Recipient, or that may
*        seek to obtain commercial advantage by the fact of Government Agency's
*        or a prior Recipient's participation in this Agreement.
*
*        F. In an effort to track usage and maintain accurate records of the
*        Subject Software, each Recipient, upon receipt of the Subject
*        Software, is requested to register with Government Agency by visiting
*        the following website: https://opensource.arc.nasa.gov.  Recipient's
*        name and personal information shall be used for statistical purposes
*        only. Once a Recipient makes a Modification available, it is requested
*        that the Recipient inform Government Agency at the web site provided
*        above how to access the Modification.
*
*        G. Each Contributor represents that that its Modification is believed
*        to be Contributor's original creation and does not violate any
*        existing agreements, regulations, statutes or rules, and further that
*        Contributor has sufficient rights to grant the rights conveyed by this
*        Agreement.
*
*        H. A Recipient may choose to offer, and to charge a fee for, warranty,
*        support, indemnity and/or liability obligations to one or more other
*        Recipients of the Subject Software.  A Recipient may do so, however,
*        only on its own behalf and not on behalf of Government Agency or any
*        other Recipient.  Such a Recipient must make it absolutely clear that
*        any such warranty, support, indemnity and/or liability obligation is
*        offered by that Recipient alone.  Further, such Recipient agrees to
*        indemnify Government Agency and every other Recipient for any
*        liability incurred by them as a result of warranty, support, indemnity
*        and/or liability offered by such Recipient.
*
*        I. A Recipient may create a Larger Work by combining Subject Software
*        with separate software not governed by the terms of this agreement and
*        distribute the Larger Work as a single product. In such case, the
*        Recipient must make sure Subject Software, or portions thereof,
*        included in the Larger Work is subject to this Agreement.
*
*        J. Notwithstanding any provisions contained herein, Recipient is
*        hereby put on notice that export of any goods or technical data from
*        the United States may require some form of export license from the
*        U.S. Government.  Failure to obtain necessary export licenses may
*        result in criminal liability under U.S. laws.  Government Agency
*        neither represents that a license shall not be required nor that, if
*        required, it shall be issued.  Nothing granted herein provides any
*        such export license.
*
*        4. DISCLAIMER OF WARRANTIES AND LIABILITIES; WAIVER AND INDEMNIFICATION
*
*        A. No Warranty: THE SUBJECT SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY
*        WARRANTY OF ANY KIND, EITHER EXPRESSED, IMPLIED, OR STATUTORY,
*        INCLUDING, BUT NOT LIMITED TO, ANY WARRANTY THAT THE SUBJECT SOFTWARE
*        WILL CONFORM TO SPECIFICATIONS, ANY IMPLIED WARRANTIES OF
*        MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR FREEDOM FROM
*        INFRINGEMENT, ANY WARRANTY THAT THE SUBJECT SOFTWARE WILL BE ERROR
*        FREE, OR ANY WARRANTY THAT DOCUMENTATION, IF PROVIDED, WILL CONFORM TO
*        THE SUBJECT SOFTWARE. THIS AGREEMENT DOES NOT, IN ANY MANNER,
*        CONSTITUTE AN ENDORSEMENT BY GOVERNMENT AGENCY OR ANY PRIOR RECIPIENT
*        OF ANY RESULTS, RESULTING DESIGNS, HARDWARE, SOFTWARE PRODUCTS OR ANY
*        OTHER APPLICATIONS RESULTING FROM USE OF THE SUBJECT SOFTWARE.
*        FURTHER, GOVERNMENT AGENCY DISCLAIMS ALL WARRANTIES AND LIABILITIES
*        REGARDING THIRD-PARTY SOFTWARE, IF PRESENT IN THE ORIGINAL SOFTWARE,
*        AND DISTRIBUTES IT "AS IS."
*
*        B. Waiver and Indemnity: RECIPIENT AGREES TO WAIVE ANY AND ALL CLAIMS
*        AGAINST THE UNITED STATES GOVERNMENT, ITS CONTRACTORS AND
*        SUBCONTRACTORS, AS WELL AS ANY PRIOR RECIPIENT.  IF RECIPIENT'S USE OF
*        THE SUBJECT SOFTWARE RESULTS IN ANY LIABILITIES, DEMANDS, DAMAGES,
*        EXPENSES OR LOSSES ARISING FROM SUCH USE, INCLUDING ANY DAMAGES FROM
*        PRODUCTS BASED ON, OR RESULTING FROM, RECIPIENT'S USE OF THE SUBJECT
*        SOFTWARE, RECIPIENT SHALL INDEMNIFY AND HOLD HARMLESS THE UNITED
*        STATES GOVERNMENT, ITS CONTRACTORS AND SUBCONTRACTORS, AS WELL AS ANY
*        PRIOR RECIPIENT, TO THE EXTENT PERMITTED BY LAW.  RECIPIENT'S SOLE
*        REMEDY FOR ANY SUCH MATTER SHALL BE THE IMMEDIATE, UNILATERAL
*        TERMINATION OF THIS AGREEMENT.
*
*
*        5. GENERAL TERMS
*
*        A. Termination: This Agreement and the rights granted hereunder will
*        terminate automatically if a Recipient fails to comply with these
*        terms and conditions, and fails to cure such noncompliance within
*        thirty (30) days of becoming aware of such noncompliance.  Upon
*        termination, a Recipient agrees to immediately cease use and
*        distribution of the Subject Software.  All sublicenses to the Subject
*        Software properly granted by the breaching Recipient shall survive any
*        such termination of this Agreement.
*
*        B. Severability: If any provision of this Agreement is invalid or
*        unenforceable under applicable law, it shall not affect the validity
*        or enforceability of the remainder of the terms of this Agreement.
*
*        C. Applicable Law: This Agreement shall be subject to United States
*        federal law only for all purposes, including, but not limited to,
*        determining the validity of this Agreement, the meaning of its
*        provisions and the rights, obligations and remedies of the parties.
*
*        D. Entire Understanding: This Agreement constitutes the entire
*        understanding and agreement of the parties relating to release of the
*        Subject Software and may not be superseded, modified or amended except
*        by further written agreement duly executed by the parties.
*
*        E. Binding Authority: By accepting and using the Subject Software
*        under this Agreement, a Recipient affirms its authority to bind the
*        Recipient to all terms and conditions of this Agreement and that that
*        Recipient hereby agrees to all terms and conditions herein.
*
*        F. Point of Contact: Any Recipient contact with Government Agency is
*        to be directed to the designated representative as follows:
*        Patrick.Hogan@nasa.gov.
*    *//*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016:
 * restored from old nasa sources
 */
package eu.mavinci.desktop.gui.wwext.sunlight;

import com.intel.missioncontrol.PublishSource;
import eu.mavinci.core.obfuscation.IKeepAll;
import gov.nasa.worldwind.geom.Vec4;
import eu.mavinci.core.obfuscation.IKeepAll;

import java.awt.Color;

/**
 * Computes the color of the atmosphere according to the Sun position, the eye position and a direction.
 *
 * <p>Based on code from Sean O'Neil "Real-Time Atmospheric Scattering" Gamedev article and C++ demo.<br>
 * http://www.gamedev.net/reference/articles/article2093.asp<br>
 * http://sponeil.net
 *
 * @author Patrick Murris
 * @version $Id: AtmosphericScatteringComputer.java 10406 2009-04-22 18:28:45Z patrickmurris $
 */

public class AtmosphericScatteringComputer implements IKeepAll {
    private float fInnerRadius;
    private float fOuterRadius;
    private float fScale;

    private int nSamples = 4; // Number of sample rays to use in integral equation
    private float Kr = 0.001f; // 0.0025f; // Rayleigh scattering constant
    private float Kr4PI = Kr * 4.0f * (float)Math.PI;
    private float Km = 0.0015f; // Mie scattering constant
    private float Km4PI = Km * 4.0f * (float)Math.PI;
    private float ESun = 15.0f; // Sun brightness constant
    private float g = -0.85f; // The Mie phase asymmetry factor
    private float fRayleighScaleDepth = 0.25f;
    private float fMieScaleDepth = 0.1f;
    private float[] fWavelength = new float[] {0.650f, 0.570f, 0.475f}; // 650nm red, 570nm green, 475nm blue
    private float[] fWavelength4 = new float[3];

    private float[] fCameraDepth = new float[] {0, 0, 0, 0};
    private float[] fLightDepth = new float[4];
    private float[] fSampleDepth = new float[4];
    private float[] fRayleighSum = new float[] {0, 0, 0};
    private float[] fMieSum = new float[] {0, 0, 0};
    private float[] fAttenuation = new float[3];

    // Optical depth buffer
    private float[] opticalDepthBuffer;
    private float DELTA = 1e-6f;
    private int nChannels = 4;
    private int nBufferWidth = 128;
    private int nBufferHeight = 128;

    private Color TRANSPARENT_COLOR = new Color(0, 0, 0, 0);

    public AtmosphericScatteringComputer(double globeRadius, double thickness) {
        // Init
        fWavelength4[0] = (float)Math.pow(fWavelength[0], 4.0f);
        fWavelength4[1] = (float)Math.pow(fWavelength[1], 4.0f);
        fWavelength4[2] = (float)Math.pow(fWavelength[2], 4.0f);

        fInnerRadius = (float)globeRadius;
        fOuterRadius = (float)(globeRadius + thickness);
        fScale = 1.0f / (fOuterRadius - fInnerRadius);

        // Init optical depth buffer
        this.computeOpticalDepthBuffer();
    }

    public Color getAtmosphereColor(Vec4 lookAtPoint, Vec4 eyePoint, Vec4 lightDirection) {
        // Find out intersection point on world scattering sphere
        Vec4 vRay = lookAtPoint.subtract3(eyePoint);
        vRay = vRay.normalize3();

        // Calculate the closest intersection of the ray with the outer atmosphere
        float B = 2.0f * (float)eyePoint.dot3(vRay);
        float C = (float)(eyePoint.dotSelf3() - fOuterRadius * fOuterRadius);
        float fDet = B * B - 4.0f * C;

        Color color = TRANSPARENT_COLOR;
        if (fDet >= 0) {
            // Camera ray intersect atmosphere
            float fNear1 = 0.5f * (-B - (float)Math.sqrt(fDet));
            float fNear2 = 0.5f * (-B + (float)Math.sqrt(fDet));
            if (fNear1 >= 0 || fNear2 >= 0) {
                // largest distance - not sure why...
                float fNear = Math.max(fNear1, fNear2);
                Vec4 vPos = eyePoint.add3(vRay.multiply3(fNear));
                color = getColorForVertex(vPos, eyePoint, lightDirection);
            }
        }

        return color;
    }

    private Color getColorForVertex(Vec4 vPos, Vec4 vCamera, Vec4 vLightDirection) {
        // Get the ray from the camera to the vertex, and its length (which is the far point of the ray
        // passing through the atmosphere)
        Vec4 vRay = vPos.subtract3(vCamera);
        float fFar = (float)vRay.getLength3();
        vRay = vRay.normalize3();

        // Calculate the closest intersection of the ray with the outer atmosphere (which is the near point
        // of the ray passing through the atmosphere)
        float B = 2.0f * (float)vCamera.dot3(vRay);
        float C = (float)(vCamera.dotSelf3() - fOuterRadius * fOuterRadius);
        float fDet = Math.max(0.0f, B * B - 4.0f * C);
        float fNear = 0.5f * (-B - (float)Math.sqrt(fDet));
        boolean bCameraAbove = true;

        for (int i = 0; i < fCameraDepth.length; i++) {
            fCameraDepth[i] = 0;
        }

        for (int i = 0; i < fLightDepth.length; i++) {
            fLightDepth[i] = 0;
        }

        for (int i = 0; i < fSampleDepth.length; i++) {
            fSampleDepth[i] = 0;
        }

        if (fNear <= 0) {
            // If the near point is behind the camera, it means the camera is inside the atmosphere
            fNear = 0;
            float fCameraHeight = (float)vCamera.getLength3();
            float fCameraAltitude = (fCameraHeight - fInnerRadius) * fScale;
            bCameraAbove = fCameraHeight >= vPos.getLength3();
            float fCameraAngle =
                (float)(bCameraAbove ? vRay.getNegative3().dot3(vCamera) : vRay.dot3(vRay)) / fCameraHeight;
            interpolate(fCameraDepth, fCameraAltitude, 0.5f - fCameraAngle * 0.5f);
        } else {
            // Otherwise, move the camera up to the near intersection point
            vCamera = vCamera.add3(vRay.multiply3(fNear));
            fFar -= fNear;
            fNear = 0;
        }

        // If the distance between the points on the ray is negligible, don't bother to calculate anything
        if (fFar <= DELTA) {
            return TRANSPARENT_COLOR;
        }

        // Initialize a few variables to use inside the loop
        for (int i = 0; i < fRayleighSum.length; i++) {
            fRayleighSum[i] = 0;
        }

        for (int i = 0; i < fMieSum.length; i++) {
            fMieSum[i] = 0;
        }

        float fSampleLength = fFar / nSamples;
        float fScaledLength = fSampleLength * fScale;
        Vec4 vSampleRay = vRay.multiply3(fSampleLength);

        // Start at the center of the first sample ray, and loop through each of the others
        vPos = vCamera.add3(vSampleRay.multiply3(0.5f));
        for (int i = 0; i < nSamples; i++) {
            float fHeight = (float)vPos.getLength3();

            // Start by looking up the optical depth coming from the light source to this point
            float fLightAngle = (float)vLightDirection.dot3(vPos) / fHeight;
            float fAltitude = (fHeight - fInnerRadius) * fScale;
            interpolate(fLightDepth, fAltitude, 0.5f - fLightAngle * 0.5f);

            // If no light light reaches this part of the atmosphere, no light is scattered in at this point
            if (fLightDepth[0] > DELTA) {

                // Get the density at this point, along with the optical depth from the light source to this point
                float fRayleighDensity = fScaledLength * fLightDepth[0];
                float fRayleighDepth = fLightDepth[1];
                float fMieDensity = fScaledLength * fLightDepth[2];
                float fMieDepth = fLightDepth[3];

                // If the camera is above the point we're shading, we calculate the optical depth from the sample point
                // to the camera
                // Otherwise, we calculate the optical depth from the camera to the sample point
                if (bCameraAbove) {
                    float fSampleAngle = (float)vRay.getNegative3().dot3(vPos) / fHeight;
                    interpolate(fSampleDepth, fAltitude, 0.5f - fSampleAngle * 0.5f);
                    fRayleighDepth += fSampleDepth[1] - fCameraDepth[1];
                    fMieDepth += fSampleDepth[3] - fCameraDepth[3];
                } else {
                    float fSampleAngle = (float)vRay.dot3(vPos) / fHeight;
                    interpolate(fSampleDepth, fAltitude, 0.5f - fSampleAngle * 0.5f);
                    fRayleighDepth += fCameraDepth[1] - fSampleDepth[1];
                    fMieDepth += fCameraDepth[3] - fSampleDepth[3];
                }

                // Now multiply the optical depth by the attenuation factor for the sample ray
                fRayleighDepth *= Kr4PI;
                fMieDepth *= Km4PI;

                // Calculate the attenuation factor for the sample ray
                fAttenuation[0] = (float)Math.exp(-fRayleighDepth / fWavelength4[0] - fMieDepth);
                fAttenuation[1] = (float)Math.exp(-fRayleighDepth / fWavelength4[1] - fMieDepth);
                fAttenuation[2] = (float)Math.exp(-fRayleighDepth / fWavelength4[2] - fMieDepth);

                fRayleighSum[0] += fRayleighDensity * fAttenuation[0];
                fRayleighSum[1] += fRayleighDensity * fAttenuation[1];
                fRayleighSum[2] += fRayleighDensity * fAttenuation[2];

                fMieSum[0] += fMieDensity * fAttenuation[0];
                fMieSum[1] += fMieDensity * fAttenuation[1];
                fMieSum[2] += fMieDensity * fAttenuation[2];
            }
            // Move the position to the center of the next sample ray
            vPos = vPos.add3(vSampleRay);
        }

        // Calculate the angle and phase values (this block of code could be handled by a small 1D lookup table,
        // or a 1D texture lookup in a pixel shader)
        float fAngle = (float)vRay.getNegative3().dot3(vLightDirection);
        float[] fPhase = new float[2];
        float fAngle2 = fAngle * fAngle;
        float g2 = g * g;
        fPhase[0] = 0.75f * (1.0f + fAngle2);
        fPhase[1] = 1.5f * ((1 - g2) / (2 + g2)) * (1.0f + fAngle2) / (float)Math.pow(1 + g2 - 2 * g * fAngle, 1.5f);
        fPhase[0] *= Kr * ESun;
        fPhase[1] *= Km * ESun;
        // Calculate the in-scattering color and clamp it to the max color value
        float[] fColor = new float[] {0, 0, 0};
        fColor[0] = fRayleighSum[0] * fPhase[0] / fWavelength4[0] + fMieSum[0] * fPhase[1];
        fColor[1] = fRayleighSum[1] * fPhase[0] / fWavelength4[1] + fMieSum[1] * fPhase[1];
        fColor[2] = fRayleighSum[2] * fPhase[0] / fWavelength4[2] + fMieSum[2] * fPhase[1];
        fColor[0] = Math.min(fColor[0], 1.0f);
        fColor[1] = Math.min(fColor[1], 1.0f);
        fColor[2] = Math.min(fColor[2], 1.0f);

        // Compute alpha transparency (PM 2006-11-19)
        float alpha = (fColor[0] + fColor[1] + fColor[2]) / 3; // Average luminosity
        alpha = (float)Math.min(alpha + 0.50, 1f); // increase opacity

        // Last but not least, return the color
        return new Color(fColor[0], fColor[1], fColor[2], alpha);
    }

    private void interpolate(float[] p, float x, float y) {
        float fX = x * (nBufferWidth - 1);
        float fY = y * (nBufferHeight - 1);
        int nX = Math.min(nBufferWidth - 2, Math.max(0, (int)fX));
        int nY = Math.min(nBufferHeight - 2, Math.max(0, (int)fY));
        float fRatioX = fX - nX;
        float fRatioY = fY - nY;

        int pValueOffset = (nBufferWidth * nY + nX) * 4;

        for (int i = 0; i < nChannels; i++) {
            p[i] =
                opticalDepthBuffer[pValueOffset] * (1 - fRatioX) * (1 - fRatioY)
                    + opticalDepthBuffer[pValueOffset + nChannels] * (fRatioX) * (1 - fRatioY)
                    + opticalDepthBuffer[pValueOffset + nChannels * nBufferWidth] * (1 - fRatioX) * (fRatioY)
                    + opticalDepthBuffer[pValueOffset + nChannels * (nBufferWidth + 1)] * (fRatioX) * (fRatioY);
            pValueOffset++;
        }
    }

    private void computeOpticalDepthBuffer() {
        int nSize = 128;
        int nBufferSamples = 50;

        if (opticalDepthBuffer == null) {
            opticalDepthBuffer = new float[nSize * nSize * 4];
        }

        int nIndex = 0;
        for (int nAngle = 0; nAngle < nSize; nAngle++) {
            // angle goes from 0 to 180 degrees
            float fCos = 1.0f - (nAngle + nAngle) / (float)nSize;
            float fAngle = (float)Math.acos(fCos);

            Vec4 vRay = new Vec4((float)Math.sin(fAngle), (float)Math.cos(fAngle), 0); // Ray pointing to the viewpoint
            for (int nHeight = 0; nHeight < nSize; nHeight++) {
                // from the bottom of the atmosphere to the top
                float fHeight = DELTA + fInnerRadius + ((fOuterRadius - fInnerRadius) * nHeight) / nSize;
                Vec4 vPos = new Vec4(0, fHeight, 0); // The position of the camera

                // If the ray from vPos heading in the vRay direction intersects the inner radius (i.e. the planet),
                // then this spot is not visible from the viewpoint
                float B = 2.0f * (float)vPos.dot3(vRay);
                float Bsq = B * B;
                float Cpart = (float)vPos.dotSelf3();
                float C = Cpart - fInnerRadius * fInnerRadius;
                float fDet = Bsq - 4.0f * C;
                boolean bVisible =
                    (fDet < 0
                        || (0.5f * (-B - (float)Math.sqrt(fDet)) <= 0) && (0.5f * (-B + (float)Math.sqrt(fDet)) <= 0));
                float fRayleighDensityRatio;
                float fMieDensityRatio;
                if (bVisible) {
                    fRayleighDensityRatio = (float)Math.exp(-(fHeight - fInnerRadius) * fScale / fRayleighScaleDepth);
                    fMieDensityRatio = (float)Math.exp(-(fHeight - fInnerRadius) * fScale / fMieScaleDepth);
                } else {
                    // Smooth the transition from light to shadow
                    fRayleighDensityRatio = opticalDepthBuffer[nIndex - nSize * nChannels] * 0.75f;
                    fMieDensityRatio = opticalDepthBuffer[nIndex + 2 - nSize * nChannels] * 0.75f;
                }

                // Determine where the ray intersects the outer radius (the top of the atmosphere)
                // This is the end of our ray for determining the optical depth (vPos is the start)
                C = Cpart - fOuterRadius * fOuterRadius;
                fDet = Bsq - 4.0f * C;
                float fFar = 0.5f * (-B + (float)Math.sqrt(fDet));

                // Next determine the length of each sample, scale the sample ray, and make sure position
                // checks are at the center of a sample ray
                float fSampleLength = fFar / nBufferSamples;
                float fScaledLength = fSampleLength * fScale;
                Vec4 vSampleRay = vRay.multiply3(fSampleLength);
                vPos = vPos.add3(vSampleRay.multiply3(0.5f));

                // Iterate through the samples to sum up the optical depth for the distance the ray travels
                // through the atmosphere
                float fRayleighDepth = 0;
                float fMieDepth = 0;
                for (int i = 0; i < nBufferSamples; i++) {
                    fHeight = (float)vPos.getLength3();
                    float fAltitude = (fHeight - fInnerRadius) * fScale;
                    fAltitude = Math.max(fAltitude, 0.0f);
                    fRayleighDepth += (float)Math.exp(-fAltitude / fRayleighScaleDepth);
                    fMieDepth += (float)Math.exp(-fAltitude / fMieScaleDepth);
                    vPos = vPos.add3(vSampleRay);
                }

                // Multiply the sums by the length the ray traveled
                fRayleighDepth *= fScaledLength;
                fMieDepth *= fScaledLength;

                // Store the results for Rayleigh to the light source, Rayleigh to the camera, Mie to the
                // light source, and Mie to the camera
                opticalDepthBuffer[nIndex++] = fRayleighDensityRatio;
                opticalDepthBuffer[nIndex++] = fRayleighDepth;
                opticalDepthBuffer[nIndex++] = fMieDensityRatio;
                opticalDepthBuffer[nIndex++] = fMieDepth;
            } // height
        } // angle
    }

}
