/*
*        Copyright (c) 2018 Intel Corporation
*        Copyright (c) 2017 Intel Deutschland GmbH
*        Copyright (c) 2016 MAVinci GmbH | A Part of Intel
*
*        This program is free software: you can redistribute it and/or modify
*        it under the terms of the GNU General Public License as published by
*        the Free Software Foundation, either version 3 of the License, or
*        (at your option) any later version.
*
*        Linking this library statically or dynamically with other modules is
*        making a combined work based on this library. Thus, the terms and conditions
*        of the GNU General Public License cover the whole combination.
*
*        As a special exception, the copyright holders of this library give you
*        permission to link this library with independent modules to produce an
*        executable, regardless of the license terms of these independent modules,
*        and to copy and distribute the resulting executable under terms of your
*        choice, provided that you also meet, for each linked independent module,
*        the terms and conditions of the license of that module. An independent module
*        is a module which is not derived from or based on this library. If you modify
*        this library, you may extend this exception to your version of the library,
*        but you are not obliged to do so. If you do not wish to do so, delete this
*        exception statement from your version.
*
*        This program is distributed in the hope that it will be useful,
*        but WITHOUT ANY WARRANTY; without even the implied warranty of
*        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*        GNU General Public License for more details.
*
*        You should have received a copy of the GNU General Public License
*        along with this program.  If not, see https://www.gnu.org/licenses.
*    */

package com.intel.missioncontrol.beans;

import com.google.common.util.concurrent.MoreExecutors;
import com.intel.missioncontrol.PublishSource;
import com.intel.missioncontrol.beans.property.PropertyHelper;
import com.intel.missioncontrol.concurrent.ListenableExecutors;
import java.util.concurrent.Executor;
import javafx.beans.InvalidationListener;
import javafx.beans.Observable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public final class InvalidationListenerWrapper implements InvalidationListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(InvalidationListenerWrapper.class);
    private static final String PACKAGE_NAME;

    static {
        var className = InvalidationListenerWrapper.class.getName();
        PACKAGE_NAME =
            className.substring(0, className.length() - InvalidationListenerWrapper.class.getSimpleName().length() - 1);
    }

    public static InvalidationListener wrap(InvalidationListener listener, Executor executor) {
        if (executor == MoreExecutors.directExecutor() || executor == ListenableExecutors.directExecutor()) {
            return listener;
        }

        return new InvalidationListenerWrapper(listener, executor);
    }

    private final InvalidationListener listener;
    private final Executor executor;
    private int expiredTimeouts;

    public InvalidationListenerWrapper(InvalidationListener listener, Executor executor) {
        this.listener = listener;
        this.executor = executor;
    }

    @Override
    public void invalidated(final Observable observable) {
        executor.execute(
            () -> {
                long startTime = System.nanoTime();
                listener.invalidated(observable);
                long duration = (System.nanoTime() - startTime) / 1000000;
                if (duration > PropertyHelper.getEventHandlerTimeout()) {
                    handleTimeout(PropertyHelper.getEventHandlerTimeout(), duration);
                }
            });
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }

        if (obj instanceof InvalidationListenerWrapper) {
            return listener.equals(((InvalidationListenerWrapper)obj).listener);
        }

        return listener.equals(obj);
    }

    int getExpiredTimeouts() {
        return expiredTimeouts;
    }

    private void handleTimeout(long max, long elapsed) {
        ++expiredTimeouts;

        String stackFrame = null;
        for (var element : new Exception().getStackTrace()) {
            if (!element.getClassName().startsWith(PACKAGE_NAME)) {
                stackFrame = element.toString();
                break;
            }
        }

        LOGGER.warn(
            "InvalidationListener exceeded maximum execution time [max: "
                + max
                + " ms; elapsed: "
                + elapsed
                + " ms]\n\tat "
                + (stackFrame == null ? "<unknown>" : stackFrame));
    }

}
