/*
*        Copyright (c) 2018 Intel Corporation
*        Copyright (c) 2017 Intel Deutschland GmbH
*        Copyright (c) 2016 MAVinci GmbH | A Part of Intel
*
*        This program is free software: you can redistribute it and/or modify
*        it under the terms of the GNU General Public License as published by
*        the Free Software Foundation, either version 3 of the License, or
*        (at your option) any later version.
*
*        Linking this library statically or dynamically with other modules is
*        making a combined work based on this library. Thus, the terms and conditions
*        of the GNU General Public License cover the whole combination.
*
*        As a special exception, the copyright holders of this library give you
*        permission to link this library with independent modules to produce an
*        executable, regardless of the license terms of these independent modules,
*        and to copy and distribute the resulting executable under terms of your
*        choice, provided that you also meet, for each linked independent module,
*        the terms and conditions of the license of that module. An independent module
*        is a module which is not derived from or based on this library. If you modify
*        this library, you may extend this exception to your version of the library,
*        but you are not obliged to do so. If you do not wish to do so, delete this
*        exception statement from your version.
*
*        This program is distributed in the hope that it will be useful,
*        but WITHOUT ANY WARRANTY; without even the implied warranty of
*        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*        GNU General Public License for more details.
*
*        You should have received a copy of the GNU General Public License
*        along with this program.  If not, see https://www.gnu.org/licenses.
*    */

package com.intel.missioncontrol.beans.binding;

import com.intel.missioncontrol.PublishSource;
import com.intel.missioncontrol.measure.Quantity;
import com.intel.missioncontrol.measure.Unit;
import java.nio.file.Path;
import java.nio.file.Paths;


public final class Converters {

    private Converters() {}

    public static <T, U> BidirectionalValueConverter<U, T> invert(BidirectionalValueConverter<T, U> converter) {
        return new BidirectionalValueConverter<U, T>() {
            @Override
            public U convertBack(T value) {
                return converter.convert(value);
            }

            @Override
            public T convert(U value) {
                return converter.convertBack(value);
            }
        };
    }

    public static BidirectionalValueConverter<Boolean, Boolean> not() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Boolean convert(Boolean value) {
                return value != null ? !value : null;
            }

            @Override
            public Boolean convertBack(Boolean value) {
                return value != null ? !value : null;
            }
        };
    }

    public static BidirectionalValueConverter<Integer, Number> intToNumber() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Number convert(Integer value) {
                return value;
            }

            @Override
            public Integer convertBack(Number value) {
                return value != null ? value.intValue() : null;
            }
        };
    }

    public static BidirectionalValueConverter<Number, Integer> numberToInt() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Integer convert(Number value) {
                return value != null ? value.intValue() : null;
            }

            @Override
            public Number convertBack(Integer value) {
                return value;
            }
        };
    }

    public static BidirectionalValueConverter<Long, Number> longToNumber() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Number convert(Long value) {
                return value;
            }

            @Override
            public Long convertBack(Number value) {
                return value != null ? value.longValue() : null;
            }
        };
    }

    public static BidirectionalValueConverter<Number, Long> numberToLong() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Long convert(Number value) {
                return value != null ? value.longValue() : null;
            }

            @Override
            public Number convertBack(Long value) {
                return value;
            }
        };
    }

    public static BidirectionalValueConverter<Double, Number> doubleToNumber() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Number convert(Double value) {
                return value;
            }

            @Override
            public Double convertBack(Number value) {
                return value != null ? value.doubleValue() : null;
            }
        };
    }

    public static BidirectionalValueConverter<Number, Double> numberToDouble() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Double convert(Number value) {
                return value != null ? value.doubleValue() : null;
            }

            @Override
            public Number convertBack(Double value) {
                return value;
            }
        };
    }

    public static BidirectionalValueConverter<Float, Number> floatToNumber() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Number convert(Float value) {
                return value;
            }

            @Override
            public Float convertBack(Number value) {
                return value != null ? value.floatValue() : null;
            }
        };
    }

    public static BidirectionalValueConverter<Number, Float> numberToFloat() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Float convert(Number value) {
                return value != null ? value.floatValue() : null;
            }

            @Override
            public Number convertBack(Float value) {
                return value;
            }
        };
    }

    public static <Q extends Quantity<Q>> BidirectionalValueConverter<Number, Quantity<Q>> numberToQuantity(
            Unit<Q> unit) {
        return new BidirectionalValueConverter<>() {
            @Override
            public Number convertBack(Quantity<Q> value) {
                return value.convertTo(unit).getValue();
            }

            @Override
            public Quantity<Q> convert(Number value) {
                return Quantity.of(value.doubleValue(), unit);
            }
        };
    }

    public static <Q extends Quantity<Q>> BidirectionalValueConverter<Quantity<Q>, Number> quantityToNumber(
            Unit<Q> unit) {
        return new BidirectionalValueConverter<>() {
            @Override
            public Number convert(Quantity<Q> value) {
                return value.convertTo(unit).getValue();
            }

            @Override
            public Quantity<Q> convertBack(Number value) {
                return Quantity.of(value, unit);
            }
        };
    }

    public static BidirectionalValueConverter<Path, String> pathToString() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Path convertBack(String value) {
                return value != null ? Paths.get(value) : null;
            }

            @Override
            public String convert(Path value) {
                return value != null ? value.toString() : null;
            }
        };
    }

    public static BidirectionalValueConverter<String, Path> stringToPath() {
        return new BidirectionalValueConverter<>() {
            @Override
            public Path convert(String value) {
                return value != null ? Paths.get(value) : null;
            }

            @Override
            public String convertBack(Path value) {
                return value != null ? value.toString() : null;
            }
        };
    }

}
