/*
*        Copyright (c) 2018 Intel Corporation
*        Copyright (c) 2017 Intel Deutschland GmbH
*        Copyright (c) 2016 MAVinci GmbH | A Part of Intel
*
*        This program is free software: you can redistribute it and/or modify
*        it under the terms of the GNU General Public License as published by
*        the Free Software Foundation, either version 3 of the License, or
*        (at your option) any later version.
*
*        Linking this library statically or dynamically with other modules is
*        making a combined work based on this library. Thus, the terms and conditions
*        of the GNU General Public License cover the whole combination.
*
*        As a special exception, the copyright holders of this library give you
*        permission to link this library with independent modules to produce an
*        executable, regardless of the license terms of these independent modules,
*        and to copy and distribute the resulting executable under terms of your
*        choice, provided that you also meet, for each linked independent module,
*        the terms and conditions of the license of that module. An independent module
*        is a module which is not derived from or based on this library. If you modify
*        this library, you may extend this exception to your version of the library,
*        but you are not obliged to do so. If you do not wish to do so, delete this
*        exception statement from your version.
*
*        This program is distributed in the hope that it will be useful,
*        but WITHOUT ANY WARRANTY; without even the implied warranty of
*        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*        GNU General Public License for more details.
*
*        You should have received a copy of the GNU General Public License
*        along with this program.  If not, see https://www.gnu.org/licenses.
*    */

package com.intel.missioncontrol.beans.property;

import com.intel.missioncontrol.PublishSource;
import com.intel.missioncontrol.diagnostics.Debugger;
import java.util.concurrent.locks.StampedLock;


public class SimpleAsyncBooleanProperty extends AsyncBooleanPropertyBase {

    private final Object bean;
    private String name;

    public SimpleAsyncBooleanProperty(Object bean) {
        this(bean, new PropertyMetadata.Builder<Boolean>().create());
    }

    public SimpleAsyncBooleanProperty(Object bean, PropertyMetadata<Boolean> metadata) {
        super(metadata);
        this.bean = bean;
        PropertyHelper.checkProperty(bean, this, metadata);
    }

    @Override
    public Object getBean() {
        return bean;
    }

    @Override
    public String getName() {
        if (Debugger.isAttached()) {
            if (name == null) {
                name = PropertyHelper.getPropertyName(bean, this, getMetadataUnsynchronized());
            }

            return name;
        } else {
            StampedLock lock = getValueLock();
            long stamp = 0;
            try {
                String name = null;
                if ((stamp = lock.tryOptimisticRead()) != 0) {
                    name = this.name;
                }

                if (stamp != 0 && !lock.validate(stamp)) {
                    stamp = lock.readLock();
                    name = this.name;
                }

                if (name == null) {
                    if (StampedLock.isReadLockStamp(stamp)) {
                        lock.unlockRead(stamp);
                    }

                    stamp = lock.writeLock();

                    if (this.name == null) {
                        this.name = name = PropertyHelper.getPropertyName(bean, this, getMetadataUnsynchronized());
                    }
                }

                return name;
            } finally {
                if (StampedLock.isReadLockStamp(stamp)) {
                    lock.unlockRead(stamp);
                } else if (StampedLock.isWriteLockStamp(stamp)) {
                    lock.unlockWrite(stamp);
                }
            }
        }
    }

}
