/*
*        Copyright (c) 2010-2018, Oracle and/or its affiliates. All rights reserved.
*        DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
*
*        This code is free software; you can redistribute it and/or modify it
*        under the terms of the GNU General Public License version 2 only, as
*        published by the Free Software Foundation.  Oracle designates this
*        particular file as subject to the "Classpath" exception as provided
*        by Oracle in the LICENSE file that accompanied this code.
*
*        This code is distributed in the hope that it will be useful, but WITHOUT
*        ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
*        FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
*        version 2 for more details (a copy is included in the LICENSE file that
*        accompanied this code).
*
*        You should have received a copy of the GNU General Public License version
*        2 along with this work; if not, write to the Free Software Foundation,
*        Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
*
*        Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
*        or visit www.oracle.com if you need additional information or have any
*        questions.
*    
*
*        Copyright (c) 2018 Intel Corporation
*        Copyright (c) 2017 Intel Deutschland GmbH
*        Copyright (c) 2016 MAVinci GmbH | A Part of Intel
*
*        This program is free software: you can redistribute it and/or modify
*        it under the terms of the GNU General Public License as published by
*        the Free Software Foundation, either version 3 of the License, or
*        (at your option) any later version.
*
*        Linking this library statically or dynamically with other modules is
*        making a combined work based on this library. Thus, the terms and conditions
*        of the GNU General Public License cover the whole combination.
*
*        As a special exception, the copyright holders of this library give you
*        permission to link this library with independent modules to produce an
*        executable, regardless of the license terms of these independent modules,
*        and to copy and distribute the resulting executable under terms of your
*        choice, provided that you also meet, for each linked independent module,
*        the terms and conditions of the license of that module. An independent module
*        is a module which is not derived from or based on this library. If you modify
*        this library, you may extend this exception to your version of the library,
*        but you are not obliged to do so. If you do not wish to do so, delete this
*        exception statement from your version.
*
*        This program is distributed in the hope that it will be useful,
*        but WITHOUT ANY WARRANTY; without even the implied warranty of
*        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*        GNU General Public License for more details.
*
*        You should have received a copy of the GNU General Public License
*        along with this program.  If not, see https://www.gnu.org/licenses.
*    */

package com.intel.missioncontrol.beans.property;

import com.intel.missioncontrol.PublishSource;
import com.intel.missioncontrol.collections.AsyncListChangeListener;
import com.intel.missioncontrol.collections.AsyncObservableList;
import com.intel.missioncontrol.collections.FXAsyncCollections;
import com.intel.missioncontrol.collections.LockedList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;
import javafx.application.Platform;
import javafx.beans.property.ReadOnlyListProperty;
import javafx.beans.property.SimpleListProperty;
import javafx.collections.WeakListChangeListener;


public class UIAsyncListProperty<E> extends SimpleAsyncListProperty<E> {

    private ReadOnlyListProperty<E> readOnlyProperty;

    public UIAsyncListProperty(Object bean) {
        this(bean, new UIPropertyMetadata.Builder<AsyncObservableList<E>>().create());
    }

    public UIAsyncListProperty(Object bean, UIPropertyMetadata<AsyncObservableList<E>> metadata) {
        super(bean, metadata);
    }

    @Override
    public void overrideMetadata(PropertyMetadata<AsyncObservableList<E>> metadata) {
        if (!(metadata instanceof UIPropertyMetadata)) {
            throw new IllegalArgumentException(
                "Metadata can only be overridden with an instance of "
                    + UIPropertyMetadata.class.getSimpleName()
                    + " or its derived classes.");
        }

        super.overrideMetadata(metadata);
    }

    public ReadOnlyListProperty<E> getReadOnlyProperty() {
        StampedLock valueLock = getValueLock();
        long stamp = 0;
        try {
            if ((stamp = valueLock.tryOptimisticRead()) != 0) {
                ReadOnlyListProperty<E> readOnlyProperty = this.readOnlyProperty;
                if (valueLock.validate(stamp) && readOnlyProperty != null) {
                    return readOnlyProperty;
                }
            }

            Object bean = getBean();
            String name = getName();
            ReadOnlyListPropertyImpl<E> property = new ReadOnlyListPropertyImpl<>(bean, name, this);

            stamp = valueLock.writeLock();
            if (readOnlyProperty == null) {
                readOnlyProperty = property;
            }

            return readOnlyProperty;
        } finally {
            if (StampedLock.isWriteLockStamp(stamp)) {
                valueLock.unlockWrite(stamp);
            }
        }
    }

    @Override
    public ListIterator<E> listIterator() {
        return new ListIterator<>() {
            ListIterator<E> it = UIAsyncListProperty.super.listIterator();

            @Override
            public boolean hasNext() {
                return it.hasNext();
            }

            @Override
            public E next() {
                return it.next();
            }

            @Override
            public boolean hasPrevious() {
                return it.hasPrevious();
            }

            @Override
            public E previous() {
                return it.previous();
            }

            @Override
            public int nextIndex() {
                return it.nextIndex();
            }

            @Override
            public int previousIndex() {
                return it.previousIndex();
            }

            @Override
            public void remove() {
                verifyAccess();
                it.remove();
            }

            @Override
            public void set(E e) {
                verifyAccess();
                it.set(e);
            }

            @Override
            public void add(E e) {
                verifyAccess();
                it.add(e);
            }
        };
    }

    @Override
    public ListIterator<E> listIterator(int i) {
        return new ListIterator<>() {
            ListIterator<E> it = UIAsyncListProperty.super.listIterator(i);

            @Override
            public boolean hasNext() {
                return it.hasNext();
            }

            @Override
            public E next() {
                return it.next();
            }

            @Override
            public boolean hasPrevious() {
                return it.hasPrevious();
            }

            @Override
            public E previous() {
                return it.previous();
            }

            @Override
            public int nextIndex() {
                return it.nextIndex();
            }

            @Override
            public int previousIndex() {
                return it.previousIndex();
            }

            @Override
            public void remove() {
                verifyAccess();
                it.remove();
            }

            @Override
            public void set(E e) {
                verifyAccess();
                it.set(e);
            }

            @Override
            public void add(E e) {
                verifyAccess();
                it.add(e);
            }
        };
    }

    @Override
    public E set(int i, E element) {
        verifyAccess();
        return super.set(i, element);
    }

    @Override
    public boolean setAll(E... elements) {
        verifyAccess();
        return super.setAll(elements);
    }

    @Override
    public boolean setAll(Collection<? extends E> elements) {
        verifyAccess();
        return super.setAll(elements);
    }

    @Override
    public void add(int i, E element) {
        verifyAccess();
        super.add(i, element);
    }

    @Override
    public boolean add(E element) {
        verifyAccess();
        return super.add(element);
    }

    @Override
    public boolean addAll(int i, Collection<? extends E> elements) {
        verifyAccess();
        return super.addAll(i, elements);
    }

    @Override
    public boolean addAll(Collection<? extends E> elements) {
        verifyAccess();
        return super.addAll(elements);
    }

    @Override
    public boolean addAll(E... elements) {
        verifyAccess();
        return super.addAll(elements);
    }

    @Override
    public E remove(int i) {
        verifyAccess();
        return super.remove(i);
    }

    @Override
    public boolean remove(Object obj) {
        verifyAccess();
        return super.remove(obj);
    }

    @Override
    public void remove(int from, int to) {
        verifyAccess();
        super.remove(from, to);
    }

    @Override
    public boolean removeAll(Collection<?> objects) {
        verifyAccess();
        return super.removeAll(objects);
    }

    @Override
    public boolean removeAll(E... elements) {
        verifyAccess();
        return super.removeAll(elements);
    }

    @Override
    public boolean removeIf(Predicate<? super E> filter) {
        verifyAccess();
        return super.removeIf(filter);
    }

    @Override
    public boolean retainAll(Collection<?> objects) {
        verifyAccess();
        return super.retainAll(objects);
    }

    @Override
    public boolean retainAll(E... elements) {
        verifyAccess();
        return super.retainAll(elements);
    }

    @Override
    public void replaceAll(UnaryOperator<E> operator) {
        verifyAccess();
        super.replaceAll(operator);
    }

    private void verifyAccess() {
        if (PropertyHelper.isVerifyPropertyAccessEnabled() && !Platform.isFxApplicationThread()) {
            throw new IllegalStateException(
                "Illegal cross-thread access: list can only be modified on the JavaFX application thread.");
        }
    }

    private static class ReadOnlyListPropertyImpl<E> extends SimpleListProperty<E> {

        private final AsyncListChangeListener<E> listener =
            change -> {
                while (change.next()) {
                    final int from = change.getFrom();
                    final int to = change.getTo();

                    if (change.wasPermutated()) {
                        final int[] indexMap = new int[to - from];
                        for (int i = from; i < to; ++i) {
                            indexMap[i - from] = change.getPermutation(i);
                        }

                        if (Platform.isFxApplicationThread()) {
                            List<E> copy = new ArrayList<>(get().subList(from, to));
                            for (int i = 0; i < to - from; ++i) {
                                int newIndex = indexMap[i];
                                set(newIndex, copy.get(i));
                            }
                        } else {
                            Platform.runLater(
                                () -> {
                                    List<E> copy = new ArrayList<>(get().subList(from, to));
                                    for (int i = 0; i < to - from; ++i) {
                                        int newIndex = indexMap[i];
                                        set(newIndex, copy.get(i));
                                    }
                                });
                        }
                    } else if (change.wasUpdated()) {
                        final List<E> updatedSublist = new ArrayList<>(change.getUpdatedSubList());
                        if (Platform.isFxApplicationThread()) {
                            AsyncListChangeListener.AsyncChange<E> updateChange =
                                new AsyncListChangeListener.AsyncChange<>(get()) {
                                    private int cursor = -1;

                                    @Override
                                    public boolean next() {
                                        cursor++;
                                        return cursor == 0;
                                    }

                                    @Override
                                    public void reset() {
                                        cursor = -1;
                                    }

                                    @Override
                                    public int getFrom() {
                                        return from;
                                    }

                                    @Override
                                    public int getTo() {
                                        return to;
                                    }

                                    @Override
                                    public List<E> getRemoved() {
                                        return Collections.emptyList();
                                    }

                                    @Override
                                    protected int[] getPermutation() {
                                        return new int[0];
                                    }

                                    @Override
                                    public List<E> getUpdatedSubList() {
                                        return updatedSublist;
                                    }

                                    @Override
                                    public boolean wasUpdated() {
                                        return true;
                                    }
                                };

                            fireValueChangedEvent(updateChange);
                        } else {
                            Platform.runLater(
                                () -> {
                                    AsyncListChangeListener.AsyncChange<E> updateChange =
                                        new AsyncListChangeListener.AsyncChange<>(get()) {
                                            private int cursor = -1;

                                            @Override
                                            public boolean next() {
                                                cursor++;
                                                return cursor == 0;
                                            }

                                            @Override
                                            public void reset() {
                                                cursor = -1;
                                            }

                                            @Override
                                            public int getFrom() {
                                                return from;
                                            }

                                            @Override
                                            public int getTo() {
                                                return to;
                                            }

                                            @Override
                                            public List<E> getRemoved() {
                                                return Collections.emptyList();
                                            }

                                            @Override
                                            protected int[] getPermutation() {
                                                return new int[0];
                                            }

                                            @Override
                                            public List<E> getUpdatedSubList() {
                                                return updatedSublist;
                                            }

                                            @Override
                                            public boolean wasUpdated() {
                                                return true;
                                            }
                                        };

                                    fireValueChangedEvent(updateChange);
                                });
                        }
                    } else {
                        if (change.wasReplaced()) {
                            final List<E> addedList = new ArrayList<>(change.getAddedSubList());
                            if (Platform.isFxApplicationThread()) {
                                int index = from;
                                int currentSize = size();
                                for (E added : addedList) {
                                    if (index < currentSize) {
                                        set(index++, added);
                                    } else {
                                        add(added);
                                    }
                                }
                            } else {
                                Platform.runLater(
                                    () -> {
                                        int index = from;
                                        int currentSize = size();
                                        for (E added : addedList) {
                                            if (index < currentSize) {
                                                set(index++, added);
                                            } else {
                                                add(added);
                                            }
                                        }
                                    });
                            }
                        } else {
                            if (change.wasRemoved()) {
                                final int toRemoved = from + change.getRemovedSize();
                                if (Platform.isFxApplicationThread()) {
                                    remove(from, toRemoved);
                                } else {
                                    Platform.runLater(() -> remove(from, toRemoved));
                                }
                            }

                            if (change.wasAdded()) {
                                final List<E> addedList = new ArrayList<>(change.getAddedSubList());
                                if (Platform.isFxApplicationThread()) {
                                    addAll(from, addedList);
                                } else {
                                    Platform.runLater(() -> addAll(from, addedList));
                                }
                            }
                        }
                    }
                }
            };

        ReadOnlyListPropertyImpl(Object bean, String name, UIAsyncListProperty<E> source) {
            super(bean, name, FXAsyncCollections.observableArrayList());

            try (LockedList<E> lockedList = source.lock()) {
                setAll(lockedList);
            }

            source.addListener(new WeakListChangeListener<>(listener));
        }

    }

}
