/*
*        Copyright (c) 2010-2018, Oracle and/or its affiliates. All rights reserved.
*        DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
*
*        This code is free software; you can redistribute it and/or modify it
*        under the terms of the GNU General Public License version 2 only, as
*        published by the Free Software Foundation.  Oracle designates this
*        particular file as subject to the "Classpath" exception as provided
*        by Oracle in the LICENSE file that accompanied this code.
*
*        This code is distributed in the hope that it will be useful, but WITHOUT
*        ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
*        FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
*        version 2 for more details (a copy is included in the LICENSE file that
*        accompanied this code).
*
*        You should have received a copy of the GNU General Public License version
*        2 along with this work; if not, write to the Free Software Foundation,
*        Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
*
*        Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
*        or visit www.oracle.com if you need additional information or have any
*        questions.
*    
*
*        Copyright (c) 2018 Intel Corporation
*        Copyright (c) 2017 Intel Deutschland GmbH
*        Copyright (c) 2016 MAVinci GmbH | A Part of Intel
*
*        This program is free software: you can redistribute it and/or modify
*        it under the terms of the GNU General Public License as published by
*        the Free Software Foundation, either version 3 of the License, or
*        (at your option) any later version.
*
*        Linking this library statically or dynamically with other modules is
*        making a combined work based on this library. Thus, the terms and conditions
*        of the GNU General Public License cover the whole combination.
*
*        As a special exception, the copyright holders of this library give you
*        permission to link this library with independent modules to produce an
*        executable, regardless of the license terms of these independent modules,
*        and to copy and distribute the resulting executable under terms of your
*        choice, provided that you also meet, for each linked independent module,
*        the terms and conditions of the license of that module. An independent module
*        is a module which is not derived from or based on this library. If you modify
*        this library, you may extend this exception to your version of the library,
*        but you are not obliged to do so. If you do not wish to do so, delete this
*        exception statement from your version.
*
*        This program is distributed in the hope that it will be useful,
*        but WITHOUT ANY WARRANTY; without even the implied warranty of
*        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*        GNU General Public License for more details.
*
*        You should have received a copy of the GNU General Public License
*        along with this program.  If not, see https://www.gnu.org/licenses.
*    */

package com.intel.missioncontrol.beans.property;

import com.intel.missioncontrol.PublishSource;
import com.intel.missioncontrol.collections.AsyncObservableSet;
import com.intel.missioncontrol.collections.LockedSet;
import java.util.Collection;
import java.util.function.Predicate;
import javafx.application.Platform;
import javafx.beans.InvalidationListener;
import javafx.beans.property.ReadOnlyBooleanProperty;
import javafx.beans.property.ReadOnlyBooleanPropertyBase;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.ReadOnlyIntegerPropertyBase;
import javafx.beans.property.ReadOnlySetProperty;
import javafx.beans.value.ChangeListener;
import javafx.collections.FXCollections;
import javafx.collections.ObservableSet;
import javafx.collections.SetChangeListener;


public class UIAsyncSetProperty<E> extends SimpleAsyncSetProperty<E> {

    private ReadOnlySetProperty<E> readOnlyProperty;

    public UIAsyncSetProperty(Object bean) {
        this(bean, new UIPropertyMetadata.Builder<AsyncObservableSet<E>>().create());
    }

    public UIAsyncSetProperty(Object bean, UIPropertyMetadata<AsyncObservableSet<E>> metadata) {
        super(bean, metadata);
    }

    @Override
    public void overrideMetadata(PropertyMetadata<AsyncObservableSet<E>> metadata) {
        if (!(metadata instanceof UIPropertyMetadata)) {
            throw new IllegalArgumentException(
                "Metadata can only be overridden with an instance of " + UIPropertyMetadata.class.getSimpleName());
        }

        super.overrideMetadata(metadata);
    }

    @Override
    public void set(AsyncObservableSet<E> newValue) {
        verifyAccess();
        super.set(newValue);
    }

    @Override
    public boolean add(E element) {
        verifyAccess();
        return super.add(element);
    }

    @Override
    public boolean addAll(Collection<? extends E> elements) {
        verifyAccess();
        return super.addAll(elements);
    }

    @Override
    public boolean remove(Object obj) {
        verifyAccess();
        return super.remove(obj);
    }

    @Override
    public boolean removeAll(Collection<?> objects) {
        verifyAccess();
        return super.removeAll(objects);
    }

    @Override
    public boolean removeIf(Predicate<? super E> filter) {
        verifyAccess();
        return super.removeIf(filter);
    }

    @Override
    public boolean retainAll(Collection<?> objects) {
        verifyAccess();
        return super.retainAll(objects);
    }

    public ReadOnlySetProperty<E> getReadOnlyProperty() {
        synchronized (mutex) {
            if (readOnlyProperty == null) {
                readOnlyProperty = new ReadOnlySetPropertyImpl();
            }

            return readOnlyProperty;
        }
    }

    private void verifyAccess() {
        if (PropertyHelper.isVerifyPropertyAccessEnabled() && !Platform.isFxApplicationThread()) {
            throw new IllegalStateException(
                "Illegal cross-thread access: list can only be modified on the JavaFX application thread.");
        }
    }

    private class ReadOnlySetPropertyImpl extends ReadOnlySetProperty<E> {
        private class SizeProperty extends ReadOnlyIntegerPropertyBase {
            @Override
            public Object getBean() {
                return ReadOnlySetPropertyImpl.this;
            }

            @Override
            public String getName() {
                return "size";
            }

            @Override
            public int get() {
                return set != null ? set.size() : 0;
            }

            @Override
            protected void fireValueChangedEvent() {
                super.fireValueChangedEvent();
            }
        }

        private class EmptyProperty extends ReadOnlyBooleanPropertyBase {
            @Override
            public Object getBean() {
                return ReadOnlySetPropertyImpl.this;
            }

            @Override
            public String getName() {
                return "empty";
            }

            @Override
            public boolean get() {
                return set == null || set.isEmpty();
            }

            @Override
            protected void fireValueChangedEvent() {
                super.fireValueChangedEvent();
            }
        }

        private ObservableSet<E> set;
        private final SizeProperty size = new SizeProperty();
        private final EmptyProperty empty = new EmptyProperty();

        ReadOnlySetPropertyImpl() {
            AsyncObservableSet<E> asyncList = UIAsyncSetProperty.this.get();
            LockedSet<E> lockedSet = null;
            if (asyncList != null) {
                lockedSet = asyncList.lock();
                set = FXCollections.observableSet(lockedSet);
            } else {
                set = null;
            }

            UIAsyncSetProperty.this.addListener(
                (observable, oldValue, newValue) -> {
                    if (newValue == null) {
                        set = null;
                    } else {
                        try (LockedSet<E> view = newValue.lock()) {
                            set = FXCollections.observableSet(view);
                        }
                    }
                },
                UIAsyncSetProperty.this.getMetadata().getExecutor());

            UIAsyncSetProperty.this.addListener(
                (SetChangeListener<E>)
                    change -> {
                        if (change.wasRemoved()) {
                            set.remove(change.getElementRemoved());
                        }

                        if (change.wasAdded()) {
                            set.add(change.getElementAdded());
                        }
                    },
                UIAsyncSetProperty.this.getMetadata().getExecutor());

            UIAsyncSetProperty.this
                .sizeProperty()
                .addListener(
                    listener -> size.fireValueChangedEvent(), UIAsyncSetProperty.this.getMetadata().getExecutor());

            UIAsyncSetProperty.this
                .emptyProperty()
                .addListener(
                    listener -> empty.fireValueChangedEvent(), UIAsyncSetProperty.this.getMetadata().getExecutor());

            if (lockedSet != null) {
                lockedSet.close();
            }
        }

        @Override
        public ReadOnlyIntegerProperty sizeProperty() {
            return size;
        }

        @Override
        public ReadOnlyBooleanProperty emptyProperty() {
            return empty;
        }

        @Override
        public Object getBean() {
            return UIAsyncSetProperty.this.getBean();
        }

        @Override
        public String getName() {
            return UIAsyncSetProperty.this.getName();
        }

        @Override
        public ObservableSet<E> get() {
            return set;
        }

        @Override
        public void addListener(ChangeListener<? super ObservableSet<E>> listener) {
            UIAsyncSetProperty.this.addListener(listener);
        }

        @Override
        public void removeListener(ChangeListener<? super ObservableSet<E>> listener) {
            UIAsyncSetProperty.this.removeListener(listener);
        }

        @Override
        public void addListener(SetChangeListener<? super E> listener) {
            set.addListener(listener);
        }

        @Override
        public void removeListener(SetChangeListener<? super E> listener) {
            set.removeListener(listener);
        }

        @Override
        public void addListener(InvalidationListener listener) {
            set.addListener(listener);
        }

        @Override
        public void removeListener(InvalidationListener listener) {
            set.removeListener(listener);
        }
    }

}
