/*
*        Copyright (c) 2010-2018, Oracle and/or its affiliates. All rights reserved.
*        DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
*
*        This code is free software; you can redistribute it and/or modify it
*        under the terms of the GNU General Public License version 2 only, as
*        published by the Free Software Foundation.  Oracle designates this
*        particular file as subject to the "Classpath" exception as provided
*        by Oracle in the LICENSE file that accompanied this code.
*
*        This code is distributed in the hope that it will be useful, but WITHOUT
*        ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
*        FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
*        version 2 for more details (a copy is included in the LICENSE file that
*        accompanied this code).
*
*        You should have received a copy of the GNU General Public License version
*        2 along with this work; if not, write to the Free Software Foundation,
*        Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
*
*        Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
*        or visit www.oracle.com if you need additional information or have any
*        questions.
*    
*
*        Copyright (c) 2018 Intel Corporation
*        Copyright (c) 2017 Intel Deutschland GmbH
*        Copyright (c) 2016 MAVinci GmbH | A Part of Intel
*
*        This program is free software: you can redistribute it and/or modify
*        it under the terms of the GNU General Public License as published by
*        the Free Software Foundation, either version 3 of the License, or
*        (at your option) any later version.
*
*        Linking this library statically or dynamically with other modules is
*        making a combined work based on this library. Thus, the terms and conditions
*        of the GNU General Public License cover the whole combination.
*
*        As a special exception, the copyright holders of this library give you
*        permission to link this library with independent modules to produce an
*        executable, regardless of the license terms of these independent modules,
*        and to copy and distribute the resulting executable under terms of your
*        choice, provided that you also meet, for each linked independent module,
*        the terms and conditions of the license of that module. An independent module
*        is a module which is not derived from or based on this library. If you modify
*        this library, you may extend this exception to your version of the library,
*        but you are not obliged to do so. If you do not wish to do so, delete this
*        exception statement from your version.
*
*        This program is distributed in the hope that it will be useful,
*        but WITHOUT ANY WARRANTY; without even the implied warranty of
*        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*        GNU General Public License for more details.
*
*        You should have received a copy of the GNU General Public License
*        along with this program.  If not, see https://www.gnu.org/licenses.
*    */

package com.intel.missioncontrol.collections;

import com.intel.missioncontrol.PublishSource;
import com.intel.missioncontrol.beans.AsyncObservable;
import java.util.IdentityHashMap;
import java.util.concurrent.Executor;
import javafx.beans.InvalidationListener;
import javafx.util.Callback;


final class AsyncElementObserver<E> {

    private static class ElementsMapElement {
        InvalidationListener listener;
        int counter;

        ElementsMapElement(InvalidationListener listener) {
            this.listener = listener;
            this.counter = 1;
        }

        void increment() {
            counter++;
        }

        int decrement() {
            return --counter;
        }

        private InvalidationListener getListener() {
            return listener;
        }
    }

    private Callback<E, AsyncObservable[]> extractor;
    private final Callback<E, InvalidationListener> listenerGenerator;
    private final AsyncObservableListBase<E> list;
    private Executor executor;
    private IdentityHashMap<E, ElementsMapElement> elementsMap = new IdentityHashMap<>();

    AsyncElementObserver(
            Callback<E, AsyncObservable[]> extractor,
            Callback<E, InvalidationListener> listenerGenerator,
            AsyncObservableListBase<E> list) {
        this.extractor = extractor;
        this.listenerGenerator = listenerGenerator;
        this.list = list;
    }

    void setExecutor(Executor executor) {
        this.executor = executor;
    }

    Executor getExecutor() {
        return executor;
    }

    void attachListener(final E e) {
        if (executor == null) {
            throw new IllegalStateException("Executor not initialized.");
        }

        if (elementsMap != null && e != null) {
            if (elementsMap.containsKey(e)) {
                elementsMap.get(e).increment();
            } else {
                InvalidationListener listener = listenerGenerator.call(e);
                for (AsyncObservable o : extractor.call(e)) {
                    o.addListener(listener, executor);
                }

                elementsMap.put(e, new AsyncElementObserver.ElementsMapElement(listener));
            }
        }
    }

    void detachListener(E e) {
        if (elementsMap != null && e != null) {
            AsyncElementObserver.ElementsMapElement el = elementsMap.get(e);
            for (AsyncObservable o : extractor.call(e)) {
                o.removeListener(el.getListener());
            }

            if (el.decrement() == 0) {
                elementsMap.remove(e);
            }
        }
    }

}
