# Copyright © 2018, Intel Corporation.
[CmdletBinding(DefaultParameterSetName='None')]
Param([Parameter(Mandatory=$False)][string]$inputFilePath = "attestationReq.bin")

# Helper functions for building the class
$script:nativeMethods = @();
function Register-NativeMethod([string]$dll, [string]$methodSignature)
{
    $script:nativeMethods += [PSCustomObject]@{ Dll = $dll; Signature = $methodSignature; }
}
function Add-NativeMethods()
{
    $nativeMethodsCode = $script:nativeMethods | % { "
        [DllImport(`"$($_.Dll)`")]
        public static extern $($_.Signature);
    " }

    Add-Type @"
        using System;
        using System.Runtime.InteropServices;
        public static class NativeMethods {
            $nativeMethodsCode
        }
"@
}
$mfaDllPath=(Get-ItemProperty -path 'Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Intel\Intel Authenticate\Engine').ImagePath
if(Test-Path $mfaDllPath){           
                $mfaDllPath = $mfaDllPath -replace '\\','\\'
}else{
                Write-Error "Failed to load dll: $mfaDllPath"
                Exit 1 
}

if(!(Test-Path $inputFilePath)){
                Write-Error "Failed to locate input file: $inputFilePath"
                Exit 1 
}

# Add methods here
Register-NativeMethod $mfaDllPath "uint InitializeMfa(ref IntPtr mfaHandle)"
Register-NativeMethod $mfaDllPath "uint GetSystemAttestationData(IntPtr hMfa, byte[] dataIn, uint sizedataIn, byte[] data, ref uint pDataSize)"
Register-NativeMethod $mfaDllPath "uint FinalizeMfa(IntPtr hMfa)"


# This builds the class and registers them (you can only do this one-per-session, as the type cannot be unloaded?)
Add-NativeMethods


$hMfa=0 -as [System.IntPtr]
$InputBuffer = Get-Content -Path ($inputFilePath) -Encoding Byte -ReadCount 0


$OutBufferSize = 0
$OutBuffer = [System.Byte[]]::CreateInstance([System.Byte],$OutBufferSize)


$result=[NativeMethods]::InitializeMfa([ref] $hMfa)
$resultHex= "{0:x}" -f $result
Write-Verbose "InitializeMfa result is: $resultHex"
if($result -ne 0){
    Write-Error "Failed to initialize Intel Authenticate"
    Exit 1
}

Try
{
	$result=[NativeMethods]::GetSystemAttestationData($hMfa,$InputBuffer,$InputBuffer.Length,$OutBuffer,[ref]$OutBufferSize)
	$resultHex= "{0:x}" -f $result
	Write-Verbose "GetSystemAttestationData result is: $resultHex"
	if($result -ne 2684354614){
		Write-Error "GetSystemAttestationData call failed. Error: $resultHex"
		Exit 1
	}

	$OutBuffer = [System.Byte[]]::CreateInstance([System.Byte],$OutBufferSize)
	$result=[NativeMethods]::GetSystemAttestationData($hMfa,$InputBuffer,$InputBuffer.Length,$OutBuffer,[ref]$OutBufferSize)
	$resultHex= "{0:x}" -f $result
	Write-Verbose "GetSystemAttestationData result is: $resultHex"
	if($result -eq 0){
		$absolutePath = Resolve-Path $inputFilePath
		$dir=([System.IO.FileInfo]"$absolutePath").DirectoryName
		$FilePath = $dir + '\\attestationData.bin'
		$FileStream = [System.IO.File]::OpenWrite($FilePath)
		$FileStream.Write($OutBuffer, 0, $OutBuffer.Length)
		$FileStream.Close()
		Write-Output "Succeeded"
	}
	else {
		Write-Output "GetSystemAttestationData call failed. Error: $resultHex"
	}
}
Finally 
{
	$result=[NativeMethods]::FinalizeMfa($hMfa)
	$resultHex= "{0:x}" -f $result
	Write-Verbose "FinalizeMfa result is: $resultHex"
}

