/*
 * @file       installStoredProcedures.sql
 *
 * @brief      Install stored procedures for Intel Identity Protection Technology with Multi Factor Authentication
 *
 */
/********************************************************************************
*
* INTEL CONFIDENTIAL
*
* Copyright 2015-2018, Intel Corporation. All Rights Reserved.
*
* The source code contained or described herein and all documents related to the
* source code ("Material") are owned by Intel Corporation or its suppliers or
* licensors. Title to the Material remains with Intel Corporation or its
* suppliers and licensors. The Material contains trade secrets and proprietary
* and confidential information of Intel or its suppliers and licensors. The
* Material is protected by worldwide copyright and trade secret laws and treaty
* provisions. No part of the Material may be used, copied, reproduced, modified,
* published, uploaded, posted, transmitted, distributed, or disclosed in any way
* without Intel's prior express written permission.
*
* No license under any patent, copyright, trade secret or other intellectual
* property right is granted to or conferred upon you by disclosure or delivery of
* the Materials, either expressly, by implication, inducement, estoppel or
* otherwise. Any license under such intellectual property rights must be express
* and approved by Intel in writing.
*
********************************************************************************/


IF EXISTS (SELECT * FROM sysobjects WHERE id = object_id(N'[dbo].[MFAB_UpdateUsersAndFactors]') AND OBJECTPROPERTY(id, N'IsProcedure') = 1)
BEGIN
    DROP PROCEDURE [dbo].[MFAB_UpdateUsersAndFactors]
END
GO

CREATE PROCEDURE [dbo].[MFAB_UpdateUsersAndFactors]
(
    @agentGUID uniqueidentifier,
    @xml       xml
)
AS
BEGIN TRY
    -------------------------------------------------------
    -- Procedure: MFAB_UpdateUsersAndFactors
    -- Takes a UserCatalog xml payload and updates the
    -- user and factor information in the MFAbUsers
    -- and MFAbFactors tables.
    --
    -- Note that the available factor types are defined
    -- in the schema for the user catalogue XML. The mappings
    -- between numbers and type values reflects the entries in
    -- the l10n file
    -------------------------------------------------------

    SET NOCOUNT ON

    --EXEC xp_logevent 60000, 'MFAB_UpdateUsersAndFactors function has been called', INFORMATIONAL;

    -- Create temporary users table
    DECLARE @TempUsers TABLE (
       [SID] varchar(184),
       [Username] nvarchar(256),
       [Type] tinyint,
       [IsActive] bit NOT NULL,
       [LastMfabLogin] datetime
    )

    -- create temporary factors table
    DECLARE @TempFactors TABLE(
       [MFAbUsersId] int,
       [Username] nvarchar(256),
       [FactorType] int,
       [EnrollmentState] int,
       [AttemptCount] int
    )

    -- Get system id from the xml
    DECLARE @systemId nvarchar(256)
    SELECT TOP 1 @systemId = SystemId
        FROM MFAbProperties
        WHERE @agentGUID = AgentGuid
    IF @systemId IS NULL
    BEGIN
		DECLARE @@Message nvarchar(1024)      
		SELECT @@Message = 'Cannot find system id via agent guid ' + CAST(@agentGUID as nvarchar(256)) ;
        -- In some cases raiserror does not end processing, so we return here to enforce that.
        EXEC xp_logevent 60000, @@Message , error;
		RAISERROR ('Cannot find system id via agent guid', 11, 1)
        RETURN
	END

    -- Parse the users from the xml into the users temporary table
    -- TODO: IsActive state not sent by client, assuming 1 currently
    -- TODO: LastMfabLogin not sent by client, assuming 1Jan1970 currently
    INSERT INTO @TempUsers ([SID], [Username], [Type], [IsActive], [LastMfabLogin])
        SELECT Users.row.value('@sid[1]', 'varchar(184)') as SID,
               Users.row.value('@username[1]', 'nvarchar(256)') as Username,
               CASE Users.row.value('@userType[1]', 'nvarchar(20)')
                    WHEN 'domain' THEN 2
                    WHEN 'local' THEN 1
               END as Type,
               1 as IsActive,
               '1970-01-01 00:00:00' as LastMfabLogin
        FROM @xml.nodes('/userData/users/user') as Users(row)

    -- Parse the factors from the xml in the factors temporary table
    --- use xpath parent expression to get the username for context
    INSERT INTO @TempFactors ([Username], [FactorType], [EnrollmentState], [AttemptCount])
       SELECT
          Factors.row.value('../../@username[1]', 'nvarchar(256)') as Username,
          case Factors.row.value('@name[1]', 'nvarchar(20)')
                WHEN 'SecureBT' then 1
                WHEN 'PtdFactor_Decl' then 2
                WHEN 'SecureFP' then 3
                WHEN 'AMTLocation' then 4
                WHEN 'SoftFP' then 5
                WHEN 'Face' then 6
				WHEN 'SoftBT' then 7
				WHEN 'PhysicalSmartcard' then 8
                ELSE -1
           end as FactorType,
          case Factors.row.value('@status[1]','nvarchar(20)')
                WHEN 'notenrolled' THEN 1
                WHEN 'enrolled' THEN 2
                WHEN 'expired' THEN 3
                WHEN 'unknown' THEN 4
          end as EnrollmentState,
          Factors.row.value('attempts[1]', 'int') as AttemptCount
       FROM @xml.nodes('/userData/users/user/factors/factor') as Factors(row)

    -- Transaction start
    BEGIN TRANSACTION;

    -- Delete old users (currently present in the users table but not in the new user list)
    DELETE FROM MFAbUsers
        WHERE MFAbUsers.SystemId = @systemId AND
              MFAbUsers.Username NOT IN (SELECT Username FROM @TempUsers)

    -- Update existing users (currently in the users table and also in the new user list)
    -- - systemId and username are constant for the record and are not updated
    UPDATE MFAbUsers
        SET MFAbUsers.SID = [@TempUsers].SID,
            MFAbUsers.Type = [@TempUsers].Type,
            MFAbUsers.IsActive = [@TempUsers].IsActive,
            MFABUsers.LastMfabLogin = [@TempUsers].LastMfabLogin
        FROM @TempUsers
            INNER JOIN MFAbUsers
                ON MFABUsers.systemId = @systemId AND MFABUsers.Username = [@TempUsers].Username

    -- Insert new users (currently not in the users table but present in the new user list)
    INSERT INTO MFAbUsers
        ([SystemId], [SID], [Username], [Type], [IsActive], [LastMfabLogin])
        SELECT @systemId, [SID], [Username], [Type], [IsActive], [LastMfabLogin]
            FROM @TempUsers
            WHERE [@TempUsers].Username NOT IN (SELECT Username FROM MFAbUsers WHERE MFABUsers.systemId = @systemId)

    -- Get the foreign key to the users for each of the factors
    UPDATE @TempFactors
        SET [@TempFactors].MFAbUsersId = MFAbUsers.Id
        FROM MFAbUsers
            INNER JOIN @TempFactors ON MFAbUsers.systemId = @systemId AND
                       MFABUsers.Username = [@TempFactors].Username

    -- Delete old factors (currently present in the table but not in the new factor list)
    DELETE FROM MFAbFactors
        WHERE MFAbFactors.MFAbUsersId IN (SELECT MFAbUsersId FROM @TempFactors)

    -- Insert new factors
    -- TODO: EnrollmentCount not sent by client - assuming 0 currently
    INSERT INTO MFAbFactors
        ([MFAbUsersId], [FactorType], [EnrollmentState], [AttemptCount], [EnrollmentCount])
        SELECT [MFAbUsersId], [FactorType], [EnrollmentState], [AttemptCount], 0
            FROM @TempFactors

    -- transaction end
    COMMIT TRANSACTION;
    --EXEC xp_logevent 60000, 'MFAB_UpdateUsersAndFactors stored procedure completed', INFORMATIONAL;

END TRY
BEGIN CATCH
    EXEC xp_logevent 60000, 'MFAB_UpdateUsersAndFactors stored procedure failed', error;
    SELECT ERROR_PROCEDURE() AS ProcedureName, ERROR_NUMBER() AS ErrorNumber, ERROR_MESSAGE() as ErrorMessage, ERROR_LINE() as ErrorLine;
END CATCH
