/**
 * Copyright (c) 2015, Intel Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Intel Corporation nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.intel.stl.api.management.virtualfabrics;

import com.intel.stl.api.management.WrapperNode;
import com.intel.stl.api.management.XMLConstants;
import com.intel.stl.api.management.qosgroups.QOSGroup;

/**
 * Amount to multiply PktLifeTime by when reported by SM for this Virtual
 * Fabric. This can permit extra time in PathRecords (and hence end to end
 * timeouts) to account for delays in Low Priority Virtual Fabrics which are
 * given low bandwidth allocations. Value will be rounded up to the next power
 * of 2. 0 is invalid, default is 1.
 */
public class PktLifeTimeMult extends WrapperNode<Short> {
    private static final long serialVersionUID = 7657089661617123039L;

    public PktLifeTimeMult() {
        this((short) 1);
    }

    /**
     * Description:
     *
     * @param type
     * @param value
     */
    public PktLifeTimeMult(Short value) {
        super(XMLConstants.PKT_LT_MULT, value);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.intel.stl.api.management.WrapperNode#valueOf(java.lang.String)
     */
    @Override
    protected Short valueOf(String str) {
        Short res = Short.parseShort(str);
        if (res <= 0 || res >= 512) {
            throw new IllegalArgumentException(
                    "Invalid range. Expect [1, 512).");
        }
        return res;
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * com.intel.stl.api.management.WrapperNode#valueString(java.lang.Object)
     */
    @Override
    protected String valueString(Short value) {
        return Short.toString(value);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.intel.stl.api.management.IAttribute#copy()
     */
    @Override
    public PktLifeTimeMult copy() {
        return new PktLifeTimeMult(value);
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "PktLifeTimeMult [type=" + type + ", value=" + value + "]";
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * com.intel.stl.api.management.WrapperNode#installVirtualFabric(com.intel
     * .stl.api.management.virtualfabrics.VirtualFabric)
     */
    @Override
    public void installVirtualFabric(VirtualFabric vf) {
        vf.setPktLifeTimeMult(this);
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * com.intel.stl.api.management.WrapperNode#installQOSGroup(com.intel.stl.
     * api.management.qosgroups.QOSGroup)
     */
    @Override
    public void installQOSGroup(QOSGroup qosgroup) {
        qosgroup.setPktLifeTimeMult(this);
    }

}
