/**
 * Copyright (c) 2018, Intel Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Intel Corporation nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.intel.stl.ui.admin.impl.qosgroups;

import static com.intel.stl.ui.admin.ChangeState.ADD;
import static com.intel.stl.ui.admin.ChangeState.NONE;
import static com.intel.stl.ui.admin.ChangeState.UPDATE;
import static com.intel.stl.ui.common.STLConstants.K2124_NAME_CHECK;
import static com.intel.stl.ui.common.STLConstants.K2173_BASESL_CHECK;
import static com.intel.stl.ui.common.UILabels.STL81001_DUP_NAME;
import static com.intel.stl.ui.common.UILabels.STL81002_DUP_NAME_SUG;
import static com.intel.stl.ui.common.UILabels.STL81012_DUP_BASESL;
import static com.intel.stl.ui.common.UILabels.STL81013_DUP_BASESL_SUG;

import java.util.List;

import com.intel.stl.api.management.IManagementApi;
import com.intel.stl.api.management.qosgroups.QOSGroup;
import com.intel.stl.api.management.virtualfabrics.BaseSL;
import com.intel.stl.ui.admin.Item;
import com.intel.stl.ui.admin.impl.ValidationTask;
import com.intel.stl.ui.admin.view.ValidationDialog;
import com.intel.stl.ui.common.ValidationItem;
import com.intel.stl.ui.common.ValidationModel;

public class QosValidationTask extends ValidationTask<QOSGroup> {
    private final IManagementApi mgmtApi;

    public QosValidationTask(ValidationDialog dialog,
            ValidationModel<QOSGroup> model, List<Item<QOSGroup>> items,
            Item<QOSGroup> toCheck, IManagementApi mgmtApi) {
        super(dialog, model, items, toCheck);
        this.mgmtApi = mgmtApi;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.swing.SwingWorker#doInBackground()
     */
    @Override
    protected Integer doInBackground() throws Exception {
        if (toCheck.getState() == NONE) {
            return 0;
        }

        int count = 0;
        if (toCheck.getState() == UPDATE || toCheck.getState() == ADD) {
            dialog.reportProgress(K2124_NAME_CHECK.getValue() + "...");
            ValidationItem<QOSGroup> vi = uniqueNameCheck(toCheck);
            if (vi != null) {
                publish(vi);
                count += 1;
            }

        }
        if (toCheck.getState() == UPDATE || toCheck.getState() == ADD) {
            dialog.reportProgress(K2173_BASESL_CHECK.getValue() + "...");
            // The same SL value cannot appear in multiple QOSGroups
            ValidationItem<QOSGroup> vi = uniqueBaseSLCheck(toCheck);
            if (vi != null) {
                publish(vi);
                count += 1;
            }

        }

        return count;
    }

    protected ValidationItem<QOSGroup> uniqueNameCheck(Item<QOSGroup> app) {
        long id = app.getId();
        String name = app.getObj().getName();
        for (Item<QOSGroup> item : items) {
            if (item.getId() != id && item.getObj().getName().equals(name)) {
                return new ValidationItem<QOSGroup>(K2124_NAME_CHECK.getValue(),
                        STL81001_DUP_NAME.getDescription(),
                        STL81002_DUP_NAME_SUG.getDescription());
            }
        }
        return null;
    }

    protected ValidationItem<QOSGroup> uniqueBaseSLCheck(Item<QOSGroup> qos) {
        long id = qos.getId();
        BaseSL baseSL = qos.getObj().getBaseSL();
        if (baseSL != null) {
            for (Item<QOSGroup> item : items) {
                if (baseSL.equals(item.getObj().getBaseSL())
                        && item.getId() != id) {
                    return new ValidationItem<QOSGroup>(
                            K2173_BASESL_CHECK.getValue(),
                            STL81012_DUP_BASESL.getDescription(),
                            STL81013_DUP_BASESL_SUG.getDescription());
                }
            }
        }
        return null;
    }

}
