/**
 * Copyright (c) 2018, Intel Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Intel Corporation nor the names of its contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.intel.stl.ui.admin.view.qosgroups;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import org.jdesktop.swingx.VerticalLayout;

import com.intel.stl.api.management.IAttribute;
import com.intel.stl.api.management.qosgroups.QOSGroup;
import com.intel.stl.api.management.virtualfabrics.Enable;
import com.intel.stl.ui.admin.impl.qosgroups.QosGroupRendererModel;
import com.intel.stl.ui.admin.view.AbstractEditorPanel;
import com.intel.stl.ui.admin.view.IAttrRenderer;
import com.intel.stl.ui.common.STLConstants;
import com.intel.stl.ui.common.UIConstants;
import com.intel.stl.ui.common.view.ComponentFactory;

public class QOSGsEditorPanel extends AbstractEditorPanel<QOSGroup> {
    private static final long serialVersionUID = 6842524404375810929L;

    private JPanel mainPanel;

    private JPanel attrsPanel;

    private JCheckBox enableBox;

    private JPanel basicPanel;

    private final List<QosAttrPanel> qosAttrPanels =
            new ArrayList<QosAttrPanel>();

    private final QosGroupRendererModel rendererModel;

    private QosAttrPanel addAttrPanel;

    public QOSGsEditorPanel(QosGroupRendererModel rendererModel) {
        super();
        this.rendererModel = rendererModel;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.intel.stl.ui.admin.view.AbstractEditorPanel#getMainComponent()
     */
    @Override
    protected JComponent getMainComponent() {
        if (mainPanel == null) {
            mainPanel = new JPanel(new BorderLayout());
            mainPanel.setBackground(UIConstants.INTEL_WHITE);
            mainPanel.setBorder(BorderFactory.createTitledBorder(
                    STLConstants.K2112_ATTRIBUTES.getValue()));

            JPanel panel = getBasicPanel();
            mainPanel.add(panel, BorderLayout.NORTH);

            attrsPanel = new JPanel(new VerticalLayout(10));
            attrsPanel.setOpaque(false);
            JScrollPane scrollPane = new JScrollPane(attrsPanel);
            mainPanel.add(scrollPane, BorderLayout.CENTER);
        }
        return mainPanel;
    }

    protected JPanel getBasicPanel() {
        if (basicPanel == null) {
            basicPanel = new JPanel(new FlowLayout(FlowLayout.LEADING, 10, 5));
            basicPanel.setOpaque(false);

            enableBox = ComponentFactory
                    .getIntelCheckBox(STLConstants.K0445_ENABLE.getValue());
            basicPanel.add(enableBox);

        }
        return basicPanel;
    }

    public void setEnabledActionListener(ActionListener listener) {
        enableBox.addActionListener(listener);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.intel.stl.ui.admin.view.AbstractEditorPanel#clear()
     */
    @Override
    public void clear() {
        super.clear();
        attrsPanel.removeAll();
        qosAttrPanels.clear();
        enableBox.setSelected(false);
        enableBox.setEnabled(false);
        rendererModel.setQosNames(new String[0]);
        revalidate();
        repaint();
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * com.intel.stl.ui.admin.view.AbstractEditorPanel#showItemObject(java.lang.
     * Object, java.lang.String[], boolean)
     */
    @Override
    protected void showItemObject(QOSGroup qos, String[] qosNames,
            boolean isEditable) {
        attrsPanel.removeAll();
        qosAttrPanels.clear();
        setBasicAttrs(qos, isEditable);

        List<IAttribute> attrs = getAdvancedAttrs(qos);
        String[] attrNAmes = addAttrs(attrs, isEditable);

        if (isEditable) {
            addAttrPanel = new QosAttrPanel(this, rendererModel);
            addAttrPanel.setDisabledAttrs(attrNAmes);
            attrsPanel.add(addAttrPanel);
        }

        revalidate();
        repaint();

    }

    /*
     * (non-Javadoc)
     *
     * @see
     * com.intel.stl.ui.admin.view.AbstractEditorPanel#itemNameChanged(java.lang
     * .String, java.lang.String)
     */
    @Override
    public void itemNameChanged(String oldName, String newName) {

    }

    protected void setBasicAttrs(QOSGroup qos, boolean isEditable) {
        enableBox.setSelected(qos.getEnable().isSelected());
        enableBox.setEnabled(isEditable);

    }

    protected List<IAttribute> getAdvancedAttrs(QOSGroup qos) {
        List<IAttribute> attrs = new ArrayList<IAttribute>();
        if (qos.getHighPriority() != null) {
            attrs.add(qos.getHighPriority());
        }
        if (qos.getBandwidth() != null) {
            attrs.add(qos.getBandwidth());
        }
        if (qos.getPktLifeTimeMult() != null) {
            attrs.add(qos.getPktLifeTimeMult());
        }
        if (qos.getBaseSL() != null) {
            attrs.add(qos.getBaseSL());
        }
        if (qos.getRespSL() != null) {
            attrs.add(qos.getRespSL());
        }
        if (qos.getFlowControlDisable() != null) {
            attrs.add(qos.getFlowControlDisable());
        }
        if (qos.getPreemptRank() != null) {
            attrs.add(qos.getPreemptRank());
        }
        if (qos.getHoqLife() != null) {
            attrs.add(qos.getHoqLife());
        }
        if (qos.getMulticaseSL() != null) {
            attrs.add(qos.getMulticaseSL());
        }
        return attrs;
    }

    protected String[] addAttrs(List<IAttribute> attrs, boolean isEditable) {
        Set<String> attrNameSet = new HashSet<String>();
        for (IAttribute attr : attrs) {
            QosAttrPanel attrPanel = new QosAttrPanel(this, rendererModel);
            attrPanel.setAttr(attr.getType(), attr, isEditable);
            attrsPanel.add(attrPanel);
            qosAttrPanels.add(attrPanel);
            String rendererName = attrPanel.getRendererName();
            attrNameSet.add(rendererName);

        }

        String[] attrNames = attrNameSet.toArray(new String[0]);
        for (QosAttrPanel vfap : qosAttrPanels) {
            vfap.setDisabledAttrs(attrNames);
        }
        return attrNames;
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * com.intel.stl.ui.admin.view.AbstractEditorPanel#updateItemObject(java.
     * lang.Object)
     */
    @Override
    protected void updateItemObject(QOSGroup obj) {
        obj.setName(getCurrentName());
        obj.setEnable(new Enable(enableBox.isSelected()));
        obj.clear();
        for (QosAttrPanel attrPanel : qosAttrPanels) {
            IAttrRenderer<? extends IAttribute> renderer =
                    attrPanel.getAttrRenderer();
            if (renderer != null) {
                IAttribute attr = renderer.getAttr();
                if (attr != null) {
                    attr.installQOSGroup(obj);
                }
            }
        }

    }

    /**
     * <i>Description:</i>
     *
     */
    public void beginEdit(QosAttrPanel attrPanel) {
        String name = attrPanel.getRendererName();
        Set<String> attrNameSet = new HashSet<String>();
        for (QosAttrPanel vfap : qosAttrPanels) {
            vfap.addDisabledAttr(name);
            attrNameSet.add(vfap.getRendererName());
        }

        qosAttrPanels.add(attrPanel);
        attrNameSet.add(name);
        attrPanel.setDisabledAttrs(attrNameSet.toArray(new String[0]));

        addAttrPanel = new QosAttrPanel(this, rendererModel);
        addAttrPanel.setDisabledAttrs(attrNameSet.toArray(new String[0]));
        attrsPanel.add(addAttrPanel);
    }

    /**
     * <i>Description:</i>
     *
     * @param qosAttrPanel
     */
    public void removeEditor(QosAttrPanel qosAttrPanel) {
        attrsPanel.remove(qosAttrPanel);
        qosAttrPanels.remove(qosAttrPanel);
        revalidate();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.intel.stl.ui.admin.view.AbstractEditorPanel#isEditValid()
     */
    @Override
    public boolean isEditValid() {
        if (!super.isEditValid()) {
            return false;
        }

        for (QosAttrPanel attrPanel : qosAttrPanels) {
            IAttrRenderer<?> renderer = attrPanel.getAttrRenderer();
            if (renderer != null && !renderer.isEditValid()) {
                return false;
            }
        }
        return true;
    }

    /**
     * <i>Description:</i>
     *
     * @param oldRenderer
     * @param newRenderer
     */
    public void changeEditorRenderer(String oldRenderer, String newRenderer) {
        for (QosAttrPanel qgap : qosAttrPanels) {
            qgap.addDisabledAttr(newRenderer);
            qgap.removeDisabledAttr(oldRenderer);
        }
        if (addAttrPanel != null) {
            addAttrPanel.addDisabledAttr(newRenderer);
            addAttrPanel.removeDisabledAttr(oldRenderer);
        }

    }
}
