/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */
/******************************************************************************
 * File Name:   alrm3204.c
 * Description: this file contains alarms related functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsSend
 * Returns: TEstatus
 * Description: Force an alarm in the data transmitted through the specified 
 *              Port. The AlarmSelected parameter specifies the type of alarm 
 *              to be sent
 ******************************************************************************/
TEstatus IXF3204SetAlarmsSend(void *handle, TEword Port, TEword AlarmSelected)
{
    TEword Address;
    TEbyte Data, Mask;

    Address = IXF3204FindPortPLI(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*  validate AlarmSelected parameter */
    switch(AlarmSelected)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_LOS:
            Data = TE_BIT5;
            break;
        case TE_AUXP:
            Data = TE_BIT6;
            break;
        case TE_AIS:
            Data = (TE_BIT6 | TE_BIT5);
            break;
        case TE_AISCI:
            Data = (TE_BIT7 | TE_BIT6 | TE_BIT5);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_LIPATH;
    Mask = (TE_BIT7 | TE_BIT6 |TE_BIT5);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }
    
    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsSend
 * Returns: TEstatus
 * Description: This message returns the force alarm configured.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsSend(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address=IXF3204FindPortPLI(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_LIPATH;
    CHK_STATUS(IXF3204ReadByte(handle,Address, &Data));

    switch(Data & (TE_BIT7 | TE_BIT6 | TE_BIT5))
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT5: 
            ReturnValue[0] = TE_LOS;
            break;
        case TE_BIT6:   
            ReturnValue[0] = TE_AUXP;
            break;
        case (TE_BIT6 | TE_BIT5):  
            ReturnValue[0] = TE_AIS;
            break;
        case (TE_BIT7 | TE_BIT6 | TE_BIT5): 
            ReturnValue[0] = TE_AISCI;
            break;
        default:       
            return TE_STATUS_BADREAD;  
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsSendMFRAI
 * Returns: TEstatus
 * Description: Configures the device to send the multiframe RAI for the 
 *              specified Port. This means that the "Y" bit is set when 
 *              Enabled. For E1 CAS or E1 CRC4 CAS only.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsSendMFRAI(void *handle, TEword Port, TEword State)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT3;

    Address=IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {  
        return TE_STATUS_BAD_PARAMS; 
    }
    /* Validate the State parameter */
    switch(State)
    {
        case TE_ENABLE:
            Data = TE_BIT3;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsSendMFRAI
 * Returns: TEstatus
 * Description: Returns the state of the MFRAI.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsSendMFRAI(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT3;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address=IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address, Mask, &Data));

    switch(Data)
    {
        case TE_BIT3:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default: 
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsSendTS16AIS
 * Returns: TEstatus
 * Description: Sets the sending of the time slot 16 AIS signal for the 
 *              specified Port. This is an "all ones" on time slot 16
 *              of a CAS structure.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsSendTS16AIS(void *handle, TEword Port,
                                     TEword Selection)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT4;

    Address=IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* Validate the Selection parameter */
    switch(Selection)
    {
        case TE_ENABLE:
            Data = TE_BIT4;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsSendTS16AIS
 * Returns: TEstatus
 * Description: Returns the time slot 16 AIS send configuration for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsSendTS16AIS(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT4;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch(Data)
    {
        case TE_BIT4:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsSendREbits
 * Returns: TEstatus
 * Description: Sets the sending of the RE bits associated to CRC 
 *              sub-multiframe I and/or II (RE1 and RE2) for the specified 
 *              Port. This operation is performed only when the device is in
 *              CRC4 mode (RE bits generation enabled). When forced, the 
 *              RE bit is set to zero, which is the asserted condition.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsSendREbits(void *handle, TEword Port,
                                    TEword Selection)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT2 | TE_BIT1);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* Validate the Selection parameter */
    switch(Selection)
    {
        case TE_DISABLE:
            Data = (TE_BIT2 | TE_BIT1);
            break;
        case TE_RE1:
            Data = TE_BIT1;
            break;
        case TE_RE2:
            Data = TE_BIT2;
            break;
        case TE_BOTH:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsSendREbits
 * Returns: TEstatus
 * Description: Returns the state of the forcing of RE bits for the specified 
 *              Port.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsSendREbits(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT2 | TE_BIT1);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch(Data)
    {
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_RE1;
            break;
        case TE_BIT2:
            ReturnValue[0] = TE_RE2;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_BOTH;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsSendRAI
 * Returns: TEstatus
 * Description: Sends the RAI for the specified Port. When J1RAI is selected 
 *              to send, the F-bit in frame 12 of every superframe is 1. 
 *              The Intel IXF3204 must be in J1-12 mode. When RAI is selected,
 *              the RAI bit on the NFAS byte is set to 1 in E1. Also, in T1/J1,
 *              Bit 2 is set to 0 in all time slots.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsSendRAI(void *handle, TEword Port, TEword Selection)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT5 | TE_BIT0);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /* Validate the Selection parameter */
    switch(Selection)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_J1RAI:
            Data = TE_BIT5;
            break;
        case TE_RAI:
            Data = TE_BIT0;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsSendRAI
 * Returns: TEstatus
 * Description: Returns the RAI Send configuration for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsSendRAI(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT5 | TE_BIT0);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_J1RAI;
            break;
        case TE_BIT0:
            ReturnValue[0] = TE_RAI;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsEBitTransparent
 * Returns: TEstatus
 * Description: Configures the source of the E-bits for the specified Port.
 *              When TE_ENABLE is selected, the Intel IXF3204 is in 
 *              transparent mode. The value received from the host is
 *              transmitted. When TE_DISABLE is selected, the E-bits value is
 *              generated as a consequence of a certain condition detected. 
 *              See IXF3204SetAlarmsREbitEnable to set the condition.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsEBitTransparent(void *handle, TEword Port, TEword Mode)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT2 | TE_BIT1);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* Validate the Selection parameter */
    switch(Mode)
    {
        case TE_DISABLE:
            Data = (TE_BIT2 | TE_BIT1);
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXALRM_ECR;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsEBitTransparent
 * Returns: TEstatus
 * Description: Returns the source of the E-bits for the specified Port. 
 *              When TE_ENABLE is returned, the Intel IXF3204 is in
 *              transparent mode. The value received from the host is 
 *              transmitted. When TE_DISABLE is returned, the E-bits value is
 *              generated as a consequence of a certain condition detected. 
 *              See IXF3204SetAlarmsREbitEnable to set the condition.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsEBitTransparent(void *handle, TEword Port,
                                         void *Results)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT2 | TE_BIT1);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXALRM_ECR;

    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch(Data)
    {
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_ENABLE;
            break;
        default:
            return TE_STATUS_BADREAD; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsRAITransparent
 * Returns: TEstatus
 * Description: Configures the source of the RAI for the specified Port.
 *              When TE_ENABLE is selected, the Intel IXF3204 is in
 *              transparent mode. The value received from the host is
 *              transmitted. When TE_DISABLE is selected, the RAI value is
 *              generated as a consequence of a certain condition detected.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsRAITransparent(void *handle, TEword Port, TEword  Mode)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT0;

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /* Validate the Selection parameter */
    switch(Mode)
    {
        case TE_DISABLE:
            Data = TE_BIT0;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXALRM_ECR;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsRAITransparent
 * Returns: TEstatus
 * Description: Returns the RAI Alarm mode for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsRAITransparent(void *handle, TEword Port,
                                         void *Results)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT0;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXALRM_ECR;
    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch(Data)
    {
        case TE_BIT0:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_ENABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsTxStatus
 * Returns: TEstatus
 * Description: Returns the real-time status of the REBE and RAI alarms for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsTxStatus(void *handle, TEword Port, void *Results)
{
    TEbyte Data;
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    Address += IXF3204_TXALRM_SR;
    CHK_STATUS(IXF3204ReadByte(handle,Address,&Data));

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsRxDefect
 * Returns: TEstatus
 * Description: Returns the defect detected in the receive side for the 
 *              specified Port. Each bit is explained below. When the bit is
 *              set to 1, the corresponding defect is present.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsRxDefect(void *handle, TEword Port, void *Results)
{
    TEbyte Data;
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortRxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    Address += IXF3204_RXALRM_DSR;
    CHK_STATUS(IXF3204ReadByte(handle,Address,&Data));

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsRxFailure
 * Returns: TEstatus
 * Description: Returns the failure detected in the receive side for the
 *              specified Port. Each bit is explained below. When the bit is
 *              set to 1, the corresponding alarm failure is present.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsRxFailure(void *handle, TEword Port, void *Results)
{
    TEbyte Data;
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortRxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXALRM_FSR;

    CHK_STATUS(IXF3204ReadByte(handle,Address,&Data));

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsThreshold
 * Returns: TEstatus
 * Description: Configures the set and reset threshold values used to declare 
 *              and clear a failure. When the corresponding defect has occurred
 *              for SetValue number of times, a failure is declared. When the
 *              defect has not occurred for ResetValue number of times, the
 *              failure condition is removed.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsThreshold(void *handle, TEword Defect, TEword SetValue,
                                   TEword ResetValue)
{
    TEbyte LSBSet,LSBReset;
    TEword Address, EndAddress;

    Address = IXF3204_RX_ALARMS_MEM_OFFSET;

    /* validate Defect parameter */
    switch(Defect)
    {
        case TE_E1_OOF:
            LSBSet = IXF3204_RXALRM_TH_SET_E1_OOFL;
            LSBReset = IXF3204_RXALRM_TH_RSET_E1_OOFL;
            break;
        case TE_E1_LOS:    
            LSBSet = IXF3204_RXALRM_TH_SET_E1_LOSL;
            LSBReset = IXF3204_RXALRM_TH_RSET_E1_LOSL;
            break;
        case TE_E1_AIS:
            LSBSet = IXF3204_RXALRM_TH_SET_E1_AISL;
            LSBReset = IXF3204_RXALRM_TH_RSET_E1_AISL;
            break;
        case TE_E1_RAI:
            LSBSet = IXF3204_RXALRM_TH_SET_E1_RAIL;
            LSBReset = IXF3204_RXALRM_TH_RSET_E1_RAIL;
            break;
        case TE_T1_OOF:
            LSBSet = IXF3204_RXALRM_TH_SET_T1_OOFL;
            LSBReset = IXF3204_RXALRM_TH_RSET_T1_OOFL;
            break;
        case TE_T1_LOS:
            LSBSet = IXF3204_RXALRM_TH_SET_T1_LOSL;
            LSBReset = IXF3204_RXALRM_TH_RSET_T1_LOSL;
            break;
        case TE_T1_AIS:
            LSBSet = IXF3204_RXALRM_TH_SET_T1_AISL;
            LSBReset = IXF3204_RXALRM_TH_RSET_T1_AISL;
            break;
        case TE_T1_RAI:
            LSBSet = IXF3204_RXALRM_TH_SET_T1_RAIL;
            LSBReset = IXF3204_RXALRM_TH_RSET_T1_RAIL;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    EndAddress = (TEword)(Address + LSBSet);
    CHK_STATUS(IXF3204WriteWord(handle,EndAddress,SetValue));

    EndAddress = (TEword)(Address + LSBReset);
    CHK_STATUS(IXF3204WriteWord(handle, EndAddress, ResetValue));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsThreshold
 * Returns: TEstatus
 * Description: The configured threshold value used to declare a failure when 
 *              receiving the selected defect
 *              The threshold value used to remove the failure condition when
 *              not receiving the defect.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsThreshold(void *handle, TEword Defect, void *Results)
{
    TEbyte LSBSet,LSBReset;
    TEword Address, EndAddress;
    TEword Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204_RX_ALARMS_MEM_OFFSET;

    switch(Defect)
    {
        case TE_E1_OOF:
            LSBSet = IXF3204_RXALRM_TH_SET_E1_OOFL;
            LSBReset = IXF3204_RXALRM_TH_RSET_E1_OOFL;
            break;
        case TE_E1_LOS:    
            LSBSet = IXF3204_RXALRM_TH_SET_E1_LOSL;
            LSBReset = IXF3204_RXALRM_TH_RSET_E1_LOSL;
            break;
        case TE_E1_AIS:
            LSBSet = IXF3204_RXALRM_TH_SET_E1_AISL;
            LSBReset = IXF3204_RXALRM_TH_RSET_E1_AISL;
            break;
        case TE_E1_RAI:
            LSBSet = IXF3204_RXALRM_TH_SET_E1_RAIL;
            LSBReset = IXF3204_RXALRM_TH_RSET_E1_RAIL;
            break;
        case TE_T1_OOF:
            LSBSet = IXF3204_RXALRM_TH_SET_T1_OOFL;
            LSBReset = IXF3204_RXALRM_TH_RSET_T1_OOFL;
            break;
        case TE_T1_LOS:
            LSBSet = IXF3204_RXALRM_TH_SET_T1_LOSL;
            LSBReset = IXF3204_RXALRM_TH_RSET_T1_LOSL;
            break;
        case TE_T1_AIS:
            LSBSet = IXF3204_RXALRM_TH_SET_T1_AISL;
            LSBReset = IXF3204_RXALRM_TH_RSET_T1_AISL;
            break;
        case TE_T1_RAI:
            LSBSet = IXF3204_RXALRM_TH_SET_T1_RAIL;
            LSBReset = IXF3204_RXALRM_TH_RSET_T1_RAIL;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    EndAddress = (TEword)(Address + LSBSet);
    CHK_STATUS(IXF3204ReadWord(handle,EndAddress,&Data));

    ReturnValue[0] = Data;

    EndAddress = (TEword)(Address + LSBReset);
    CHK_STATUS(IXF3204ReadWord(handle,EndAddress,&Data));

    ReturnValue[1] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsToSystem
 * Returns: TEstatus
 * Description: Sends the specified Pattern to the backplane for the specified
 *              Port. 
 *              When TE_AIS is selected, an all ones pattern is transmitted. 
 *              When TE_AUXP is selected, an alternate pattern is transmitted
 *              (1010101). 
 *              When TE_CUSTOM is selected, the value specified in Pattern is 
 *              transmitted.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsToSystem(void *handle, TEword Port, TEword  Selected,
                                  TEbyte Pattern)
{
    TEword AddressBP;
    TEbyte Data, Mask = (TE_BIT3 | TE_BIT2 | TE_BIT1);

    AddressBP = IXF3204FindPortBackPlane(Port);
    if(AddressBP == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* clear the variable */
    Data = TE_CLEAR_BITS;

    /* Validate the State parameter */
    switch(Selected)
    {
        case TE_CUSTOM:
            Data = (TE_BIT1);
        case TE_DISABLE:
            /* if TE_DISABLE, the Data variable is already cleared */
            AddressBP += IXF3204_BPRPATTERN;
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                CHK_STATUS(IXF3204WriteByteAllPortsRegs(handle,AddressBP,
                                                        Pattern));
            }
            else
            {
                CHK_STATUS(IXF3204WriteByte(handle,AddressBP,Pattern));
            }

            AddressBP -= IXF3204_BPRPATTERN;
            break;
        case TE_AIS:
            Data = (TE_BIT2);
            break;
        case TE_AUXP:
            Data = (TE_BIT3);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    AddressBP += IXF3204_BPRFORCE;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,AddressBP,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,AddressBP,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsToSystem
 * Returns: TEstatus
 * Description: Returns the alarm pattern to be sent to the system side for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsToSystem(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT3 | TE_BIT2 | TE_BIT1);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortBackPlane(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_BPRFORCE;
    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_CUSTOM;
            break;
        case TE_BIT2:
            ReturnValue[0] = TE_AIS;
            break;
        case TE_BIT3:
            ReturnValue[0] = TE_AUXP;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    /*
     * If TE_CUSTOM is selected make a read Pattern                           *
     * ReturnValue[0] has Type and  ReturnValue[1] has Pattern                *
     */
    if (*ReturnValue == TE_CUSTOM)
    {
        Address++;
        CHK_STATUS(IXF3204ReadByte(handle,Address,&Data));

        ReturnValue[1] = Data;
    }
    else 
    {
        ReturnValue[1] = TE_CLEAR_BITS;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsREbitEnable
 * Returns: TEstatus
 * Description: Enables or disables the generation of the E-bit as a 
 *              consequence of an event for the specified Port. When the 
 *              TE_ENABLE is specified for the parameter OOF, then the E-bits
 *              are set to 0 when the OOF condition is detected. The same
 *              criteria applies for LOS and CRCMF. When TE_ENABLE is selected
 *              for the Interworking parameter, then when the CRC inter-working
 *              condition is detected, the E-bits are set to 0.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsREbitEnable(void *handle, TEword Port, TEword crcmf,
                                     TEword los, TEword oof, TEword Interworking)
{
    TEword AddressBP;
    TEbyte Data = TE_CLEAR_BITS;
    TEbyte Mask = (TE_BIT7 | TE_BIT6 | TE_BIT5 | TE_BIT4);

    AddressBP = IXF3204FindPortTxAlarms(Port);
    if(AddressBP == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    switch(los)
    {
        case TE_ENABLE:
            Data |= TE_BIT5;
            break;
        case TE_DISABLE:
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    switch(crcmf)
    {
        case TE_ENABLE:
            Data |= TE_BIT4;
            break;
        case TE_DISABLE:
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    switch(oof)
    {
        case TE_ENABLE:
            Data |= TE_BIT6;
            break;
        case TE_DISABLE:
            break; 
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    switch(Interworking)
    {
        case TE_ENABLE:
            Data |= TE_BIT7;
            break;
        case TE_DISABLE:
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    AddressBP += IXF3204_BPTMODE;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,AddressBP,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,AddressBP,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsREbitEnable
 * Returns: TEstatus
 * Description: Returns the configured options used to generate E-bits alarms 
 *              on OOF and CRC4 Inter-working conditions for the specified Port
 ******************************************************************************/
TEstatus IXF3204GetAlarmsREbitEnable(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT7 | TE_BIT6 | TE_BIT5 | TE_BIT4);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    Address += IXF3204_BPTMODE;
    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    /* CRC Multiframe */
    if(Data & TE_BIT4)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }
    /* LOS */
    if(Data & TE_BIT5)
    {
        ReturnValue[1] = TE_ENABLE;
    }
    else
    {
        ReturnValue[1] = TE_DISABLE;
    }
    /* OOF */
    if(Data & TE_BIT6)
    {
        ReturnValue[2] = TE_ENABLE;
    }
    else
    {
        ReturnValue[2] = TE_DISABLE;
    }
    /* Interworking */
    if(Data & TE_BIT7)
    {
        ReturnValue[3] = TE_ENABLE;
    }
    else
    {
        ReturnValue[3] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsResponseToSys
 * Returns: TEstatus
 * Description: Configures the alarms that is sent to the system when the 
 *              Condition specified is detected on the specified Port. The
 *              allowed alarms are AIS, AUXP, and a custom pattern. 
 ******************************************************************************/
TEstatus IXF3204SetAlarmsResponseToSys(void *handle, TEword Port,
                                       TEword Condition, TEword Alarm)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT7 | TE_BIT6 | TE_BIT5);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* Validate the Alarm parameter */
    switch (Alarm)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;  
        case TE_AIS:
            Data = TE_BIT7;
            break;        
        case TE_AUXP:
            Data = (TE_BIT7 | TE_BIT5);
            break;
        case TE_CUSTOM:
            Data = (TE_BIT7 | TE_BIT6 | TE_BIT5);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Validate the Concition parameter */
    switch(Condition)
    {
        case TE_OOF:
            Address += IXF3204_TXALRM_CAOOF;
            break;
        case TE_LOS:
            Address += IXF3204_TXALRM_CALOS;
            break;
        case TE_AIS:
            Address += IXF3204_TXALRM_CAAIS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsResponseToSys
 * Returns: TEstatus
 * Description: Returns the alarm that is sent when the specified Condition is 
 *              detected for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsResponseToSys(void *handle, TEword Port,
                                       TEword Condition, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT7 | TE_BIT6 | TE_BIT5);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    switch(Condition)
    {
        case TE_OOF:
            Address += IXF3204_TXALRM_CAOOF;
            break;
        case TE_LOS:
            Address += IXF3204_TXALRM_CALOS;
            break;
        case TE_AIS:
            Address += IXF3204_TXALRM_CAAIS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch (Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT7:
            ReturnValue[0] = TE_AIS;
            break;
        case (TE_BIT7 | TE_BIT5):
            ReturnValue[0] = TE_AUXP;
            break;
        case (TE_BIT7 | TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_CUSTOM;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsResponseToLine
 * Returns: TEstatus
 * Description: Configures the alarm that is sent when the specified Condition 
 *              is detected for the specified Port. The allowed alarms are AIS,
 *              AUXP, RAI and Codeword.
 *              The codeword is only for E1 and does not apply for TE_MFLOS
 *              and TE_INTRWORKING.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsResponseToLine(void *handle, TEword Port,
                                        TEword Condition, TEword Alarm)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* Validate the Alarm parameter */
    switch(Alarm)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_AIS:
            Data = TE_BIT2;
            break;
        case TE_AUXP:
            Data = (TE_BIT2 | TE_BIT0);
            break;
        case TE_RAI:
            Data = (TE_BIT2 | TE_BIT1);
            break;
        case TE_CODEWORD:
            Data = (TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /* Validate the Condition parameter */
    switch(Condition)
    {
        case TE_OOF:
            Address += IXF3204_TXALRM_CAOOF;
            break;
        case TE_LOS:
            Address += IXF3204_TXALRM_CALOS;
            break;
        case TE_AIS:
            Address += IXF3204_TXALRM_CAAIS;
            break;
        case TE_MFLOS:
            Data <<= 4;
            Mask <<=4;
            Address += IXF3204_TXALRM_CAINTCRC;
            if(Alarm  == TE_CODEWORD)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_INTERWORKING:
            Address += IXF3204_TXALRM_CAINTCRC;
            if(Alarm  == TE_CODEWORD)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }
    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsResponseToLine
 * Returns: TEstatus
 * Description: Returns the alarm that sent when the specified Condition is
 *              detected for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsResponseToLine(void *handle, TEword Port,
                                        TEword  Condition, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Condition)
    {
        case TE_OOF:
            Address += IXF3204_TXALRM_CAOOF;
            break;
        case TE_LOS:
            Address += IXF3204_TXALRM_CALOS;
            break;
        case TE_AIS:
            Address += IXF3204_TXALRM_CAAIS;
            break;
        case TE_MFLOS:
            Address += IXF3204_TXALRM_CAINTCRC;
            Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4);
            break;
        case TE_INTERWORKING:
            Address += IXF3204_TXALRM_CAINTCRC;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    if((Condition == TE_MFLOS) || (Condition == TE_INTERWORKING))
    {
        switch (Data)
        {
            case TE_CLEAR_BITS:
                ReturnValue[0] = TE_DISABLE;
                break;                           
            case TE_BIT2:
            case TE_BIT6:
                ReturnValue[0] = TE_AIS;
                break;                           
            case (TE_BIT2 | TE_BIT0):
            case (TE_BIT6 | TE_BIT4):
                ReturnValue[0] = TE_AUXP;
                break;
            case (TE_BIT2 | TE_BIT1):
            case (TE_BIT6 | TE_BIT5):
                ReturnValue[0] = TE_RAI;
                break;
            default:
                return TE_STATUS_BADREAD; 
        }
    }
    else
    {
        switch (Data)
        {
            case TE_CLEAR_BITS:
                ReturnValue[0] = TE_DISABLE;
                break;
            case TE_BIT2:
                ReturnValue[0] = TE_AIS;
                break;
            case (TE_BIT2 | TE_BIT0):
                ReturnValue[0] = TE_AUXP;
                break;
            case (TE_BIT2 | TE_BIT1):
                ReturnValue[0] = TE_RAI;
                break;
            case (TE_BIT2 | TE_BIT1 | TE_BIT0):
                ReturnValue[0] = TE_CODEWORD;
                break;
            default:
                return TE_STATUS_BADREAD;
        }
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsCodeword
 * Returns: TEstatus
 * Description: Sets the codeword that is sent when the condition specified in 
 *              the parameter Condition is detected for the specified Port. 
 *              The option specified in the RAI_Codeword parameter enables the
 *              Intel IXF3204 to also transmit the Sa5 codeword with the value
 *              specified (0 or 1) or disables transmission of the Sa5 codeword.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsCodeword(void *handle, TEword Port, TEword Condition,
                                  TEbyte Codeword, TEword RAI_Codeword, 
                                  TEword Sa5)
{
    TEword Address, AddressCode;
    TEbyte Data, MaskCode, Mask = (TE_BIT4 | TE_BIT3);

    MaskCode = TE_BIT0_3;

    AddressCode = IXF3204FindPortTxAlarms(Port);
    if(AddressCode == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    if(Codeword > TE_BIT0_3)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* Validate the RAI_Codeword parameter */
    switch(RAI_Codeword)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;   
        case TE_ENABLE:
            Data = TE_BIT4;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* validate the Sa5 parameter */
    switch(Sa5)
    {
        case TE_ZERO:
            break;   
        case TE_ONE:
            Data |= TE_BIT3;
            break;                           
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Validate the Condition parameter */
    switch(Condition)
    {
        case TE_OOF:
            Address = (TEword)(AddressCode + IXF3204_TXALRM_CAOOF);
            AddressCode += IXF3204_TXALRM_CWOOFLOS;
            break;
        case TE_LOS:
            Address = (TEword)(AddressCode + IXF3204_TXALRM_CALOS);
            AddressCode += IXF3204_TXALRM_CWOOFLOS;
            /* move to upper nibble */
            Codeword <<= 4;
            MaskCode <<= 4;
            break;
        case TE_AIS:
            Address = (TEword)(AddressCode + IXF3204_TXALRM_CAAIS);
            AddressCode += IXF3204_TXALRM_CWAIS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,AddressCode,Codeword,
                                               MaskCode));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    CHK_STATUS(IXF3204WriteBit(handle,AddressCode,Codeword,MaskCode));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsCodeword
 * Returns: TEstatus
 * Description: Returns the codeword that is sent when the condition specified 
 *              in the parameter Condition is detected for the specified Port.
 *              Also returns the Sa5 codeword configuration.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsCodeword(void *handle, TEword Port, TEword Condition,
                                  void *Results)
{
    TEword Address,Address1;
    TEbyte Data, Mask = (TE_BIT4 | TE_BIT3);
    TEbyte Mask1 = (TE_BIT3 | TE_BIT2 | TE_BIT1 | TE_BIT0);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    Address1 = Address;

    /* Validate the Condition parameter */
    switch(Condition)
    {
        case TE_OOF:
            Address += IXF3204_TXALRM_CAOOF;
            Address1 += IXF3204_TXALRM_CWOOFLOS;
            break;
        case TE_LOS:
            Address += IXF3204_TXALRM_CALOS;
            Address1 += IXF3204_TXALRM_CWOOFLOS;
            Mask1 = TE_BIT4_7;
            break;
        case TE_AIS:
            Address += IXF3204_TXALRM_CAAIS;
            Address1 += IXF3204_TXALRM_CWAIS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch (Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[1] = TE_DISABLE;
            ReturnValue[2] = TE_ZERO;
            break;
        case TE_BIT3:
            ReturnValue[1] = TE_DISABLE;
            ReturnValue[2] = TE_ONE;
            break;
        case TE_BIT4:
            ReturnValue[1] = TE_ENABLE;
            ReturnValue[2] = TE_ZERO;
            break;
        case (TE_BIT4 | TE_BIT3):
            ReturnValue[1] = TE_ENABLE;
            ReturnValue[2] = TE_ONE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address1,Mask1,&Data));

    if(Condition == TE_LOS)
    {
        Data >>= IXF3204_NIBBLE;
    }
    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsMFRAITransparent
 * Returns: TEstatus
 * Description: Configures the Multiframe RAI mode for the specified Port. 
 *              When TE_ENABLE is selected, the Intel IXF3204 is in 
 *              transparent mode. The value received from the host is
 *              transmitted. When TE_DISABLE is selected the RAI value is
 *              generated as a consequence of a certain condition detected.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsMFRAITransparent(void *handle, TEword Port,
                                          TEword Mode)
{
    TEword Address;
    TEbyte Data, Mask;

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    /* Validate the Selection parameter */
    switch(Mode)
    {
        case TE_DISABLE:
            Data = TE_BIT3;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXALRM_ECR;
    Mask = TE_BIT3;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsMFRAITransparent
 * Returns: TEstatus
 * Description: Returns the configured Multiframe RAI mode for the specified 
 *              Port. When TE_ENABLE is returned, the Intel IXF3204 is in
 *              transparent mode. The value received from the host is being 
 *              transmitted. When TE_DISABLE is returned the RAI value is
 *              generated as a consequence of a certain condition detected.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsMFRAITransparent(void *handle, TEword Port,
                                           void *Results)
{
    TEword Address;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXALRM_ECR;
    Mask = TE_BIT3;
    CHK_STATUS(IXF3204ReadBit(handle,Address, Mask, &Data));

    switch(Data)
    {
        case TE_BIT3:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_ENABLE;
            break;
        default:
            return TE_STATUS_BADREAD; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsMFAISTransparent
 * Returns: TEstatus
 * Description: Configures the Multiframe AIS (time slot 16 AIS) mode for the 
 *              specified Port. When TE_ENABLE is selected, the Intel IXF3204
 *              is in transparent mode. The value received from the host is
 *              transmitted. When TE_DISABLE is selected the AIS value is
 *              generated as a consequence of a certain condition detected.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsMFAISTransparent(void *handle, TEword Port,
                                          TEword  Mode)
{
    TEword Address;
    TEbyte Data, Mask;

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    /* Validate the Selection parameter */
    switch(Mode)
    {
        case TE_DISABLE:
            Data = TE_BIT4;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXALRM_ECR;
    Mask = TE_BIT4;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Data,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsMFAISTransparent
 * Returns: TEstatus
 * Description: Returns the configured Multiframe AIS (time slot 16 AIS) mode 
 *              for the specified Port. When TE_ENABLE is returned, the 
 *              Intel IXF3204 is in transparent mode. The value received from
 *              the host is being transmitted. When TE_DISABLE is returned the
 *              AIS value is generated as a consequence of a certain condition
 *              detected.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsMFAISTransparent(void *handle, TEword Port,
                                           void *Results)
{
    TEword Address;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXALRM_ECR;
    Mask = TE_BIT4;
    CHK_STATUS(IXF3204ReadBit(handle,Address,Mask,&Data));

    switch(Data)
    {
        case TE_BIT4:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_ENABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsE1CASResponseToLine
 * Returns: TEstatus
 * Description: This function configures the consequent action taken when 
 *              detected the condition selected in the parameter Condition 
 *              transmitting the alarm selected in the parameter Alarm
 *
 *              If TE_LOCASMF is selected in Condition the only valid option 
 *              in Alarm are TE_MFAIS, TE_RAI and TE_MFRAI.
 *              If TE_LOS is selected in Condition the only valid option in 
 *              Alarm are TE_MFAIS and TE_MFRAI.
 *              If TE_OOF is selected in Condition the only valid option in 
 *              Alarm is TE_MFAIS.
 *              If TE_MFRAI is selected in Condition the only valid option in 
 *              Alarm is TE_MFRAI.
 ******************************************************************************/
TEstatus IXF3204SetAlarmsE1CASResponseToLine(void *handle, TEword Port, 
                                             TEword condition, TEword alarm,
                                             TEword mode)
{
    TEword Address;
    TEbyte data, mask;

    mask = TE_BIT0;

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* add the register offset */
    Address += IXF3204_TXALRM_CACAS;

    /* validate mode parameter */
    switch(mode)
    {
        case TE_ENABLE:
            data = TE_BIT0;
            break;
        case TE_DISABLE:
            data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* validate parameters */
    switch(condition)
    {
        case TE_LOCASMF:
            switch(alarm)
            {
                case TE_MFAIS:
                    data <<= 6;
                    mask <<= 6;
                    break;
                case TE_RAI:
                    data <<= 3;
                    mask <<= 3;
                    break;
                case TE_MFRAI:
                    data <<= 1;
                    mask <<= 1;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_LOS:
            switch(alarm)
            {
                case TE_MFAIS:
                    data <<= 5;
                    mask <<= 5;
                    break;
                case TE_MFRAI:
                    data <<= 2;
                    mask <<= 2;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_OOF:
            /* if TE_OFF only accept TE_MFAIS */
            if(alarm == TE_MFAIS)
            {
                data <<= 4;
                mask <<= 4;
            }
            else
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_MFRAI:
            /* if TE_MFRAI only accept TE_MFRAI */
            if(alarm != TE_MFRAI)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,data,mask));

        return TE_STATUS_SUCCESS;
     }

    CHK_STATUS(IXF3204WriteBit(handle,Address,data,mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsE1CASResponseToLine
 * Returns: TEstatus
 * Description: This function reads if the selected consequent action in the 
 *              parameter Alarm due the condition detected specified in 
 *              Condition is enable or disable.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsE1CASResponseToLine(void *handle, TEword Port,
                                             TEword condition, TEword alarm,
                                             void *Results)
{
    TEword Address;
    TEbyte data, mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    mask = TE_BIT0;

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* validate parameters */
    switch(condition)
    {
        case TE_LOCASMF:
            switch(alarm)
            {
                case TE_MFAIS:
                    mask <<= 6;
                    break;
                case TE_RAI:
                    mask <<= 3;
                    break;
                case TE_MFRAI:
                    mask <<= 1;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_LOS:
            switch(alarm)
            {
                case TE_MFAIS:
                    mask <<= 5;
                    break;
                case TE_MFRAI:
                    mask <<= 2;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_OOF:
            /* if TE_OFF only accept TE_MFAIS */
            if(alarm == TE_MFAIS)
            {
                mask <<= 4;
            }
            else
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_MFRAI:
            /* if TE_MFRAI only accept TE_MFRAI */
            if(alarm != TE_MFRAI)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* add the register offset */
    Address += IXF3204_TXALRM_CACAS;
    /* read from the register */
    CHK_STATUS(IXF3204ReadBit(handle,Address,mask,&data));

    if(data)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetAlarmsResponseUponFailure
 * Returns: TEstatus
 * Description: This message configures if the consequent action will be taken
 *              upon failure or defect condition, if TE_ENABLE is selected a
 *              failure will generate the action. 
 ******************************************************************************/
TEstatus IXF3204SetAlarmsResponseUponFailure(void *handle, TEword Port, 
                                             TEword oof, TEword ais,
                                             TEword los)
{
    TEword Address;
    TEbyte data, mask;

    mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* add the register offset */
    Address += IXF3204_TXALRM_FAILDEFSEL;

    /* validate OOF parameter */
    switch(oof)
    {
        case TE_ENABLE:
            data = TE_BIT2;
            break;
        case TE_DISABLE:
            data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* validate AIS parameter */
    switch(ais)
    {
        case TE_ENABLE:
            data |= TE_BIT1;
            break;
        case TE_DISABLE:
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* validate LOS parameter */
    switch(los)
    {
        case TE_ENABLE:
            data |= TE_BIT0;
            break;
        case TE_DISABLE:
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* if Backplane shift data and mask */
    if((TE_BP0 <= Port) && (Port <= TE_BP3))
    { 
        data <<= 4;
        mask <<= 4;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,data,mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetAlarmsResponseUponFailure
 * Returns: TEstatus
 * Description: This message returns if the consequent action will be taken 
 *              upon failure or defect condition.
 ******************************************************************************/
TEstatus IXF3204GetAlarmsResponseUponFailure(void *handle, TEword Port,
                                             void *Results)
{
    TEword Address;
    TEbyte data, mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);

    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* add the register offset */
    Address += IXF3204_TXALRM_FAILDEFSEL;

    /* if Backplane shift data and mask */
    if((TE_BP0 <= Port) && (Port <= TE_BP3))
    { 
        mask <<= 4;
    }

    /* read from the register */
    CHK_STATUS(IXF3204ReadBit(handle,Address,mask,&data));

    /* check OOF */
    if(data & TE_BIT2)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    /* check AIS */
    if(data & TE_BIT1)
    {
        ReturnValue[1] = TE_ENABLE;
    }
    else
    {
        ReturnValue[1] = TE_DISABLE;
    }

    /* check LOS */
    if(data & TE_BIT0)
    {
        ReturnValue[2] = TE_ENABLE;
    }
    else
    {
        ReturnValue[2] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

