/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */
/******************************************************************************
 * File Name:   backp3204.c
 * Description: this file contains Backplane functions
 *
 ******************************************************************************/
#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneConcentration
 * Returns: TEstatus
 * Description: Configures the concentration mode and rate to transmit or 
 *              receive in the backplane module for the specified Port and 
 *              Direction. When TE_DISABLE is selected, Rate is ignored.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneConcentration(void *handle, TEword Port,
                                          TEword Direction, TEword Value, 
                                          TEword Rate)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT5 | TE_BIT4);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {  
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch (Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(PartialAddress);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTMODE);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }


    switch(Value)
    {
        case TE_ENABLE:
            /*validate Rate parameter*/
            switch (Rate)
            {
                case TE_4X:
                    Data = TE_BIT5;
                    break;
                case TE_8X:
                    Data = (TE_BIT5 | TE_BIT4);
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS; 
            }
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_BPTMODE;
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,(TEword)(Address + IXF3204_BPTMODE),
                                     Data,Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}


/*******************************************************************************
 * Function Name: IXF3204GetBackplaneConcentration
 * Returns: TEstatus
 * Description: Returns the configured concentration and rate for the specified
 *              direction for the specified Port and Direction. 
 ******************************************************************************/
TEstatus IXF3204GetBackplaneConcentration(void *handle, TEword Port,
                                          TEword Direction, void *Results)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT5 | TE_BIT4);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch (Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTMODE);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;        
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            ReturnValue[1] = TE_CLEAR_BITS;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_ENABLE;
            ReturnValue[1] = TE_4X;
            break;
        case (TE_BIT5 | TE_BIT4):
            ReturnValue[0] = TE_ENABLE;
            ReturnValue[1] = TE_8X;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneDataRate
 * Returns: TEstatus
 * Description: Configures the rate used to move data for the specified Port 
 *              and Direction.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneDataRate(void *handle, TEword Port ,
                                     TEword Direction, TEword Rate)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT3 | TE_BIT2);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch (Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(PartialAddress);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTMODE);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Rate parameter*/
    switch (Rate)
    {
        case TE_R1536:
            Data = TE_CLEAR_BITS;
            break;
        case TE_R1544:
            Data = TE_BIT2;
            break;
        case TE_R2048:
            Data = TE_BIT3;
            break;
        default:	 
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_BPTMODE;
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,(TEword)(Address + IXF3204_BPTMODE),
                                     Data,Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneDataRate
 * Returns: TEstatus
 * Description: Returns the configured data rate for the specified Port and 
 *              Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneDataRate(void *handle, TEword Port ,
                                     TEword Direction, void *Results)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT3 | TE_BIT2);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch (Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTMODE);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data parameter*/
    switch (Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_R1536;
            break;
        case TE_BIT2:
            ReturnValue[0] = TE_R1544;
            break;
        case TE_BIT3:
            ReturnValue[0] = TE_R2048;
            break;
        default:  
            return TE_STATUS_BADREAD; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneMasterSlaveMode
 * Returns: TEstatus
 * Description: Configures the backplane to use the receive clock, sync pulse
 *              and multiframe sync pulse as a master or slave for the 
 *              specified Port and Direction. Also the transmit side can be
 *              configured to use the clock and sync pulse as a master or slave.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneMasterSlaveMode(void *handle, TEword Port, 
                                            TEword Direction, TEword Clk, 
                                            TEword Sync, TEword Msync)
{
    TEbyte Data;
    TEword Address;
    TEword PartialAddress;
    TEbyte Mask;
    TEstatus Status;

    Data = TE_CLEAR_BITS;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Sync parameter*/
    switch (Sync)
    {
        case TE_MASTER: 
            Data |= TE_BIT1;
            break;
        case TE_SLAVE: 
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Clk parameter*/
    switch (Clk)
    {
        case TE_MASTER: 
            break;
        case TE_SLAVE: 
            Data |= TE_BIT2;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch (Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRMASTER);
            switch (Msync)
            {
                case TE_MASTER: 
                    Data |= TE_BIT0;
                    break;
                case TE_SLAVE: 
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }

            Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTMASTER);
            /*validate Msync parameter*/
            switch (Msync)
            {
                case TE_NA: 
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;	
            }
            Mask = (TE_BIT2 | TE_BIT1);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_ADDR_OFFSET10;
            Mask = (TE_BIT2 | TE_BIT1);
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,(TEword)
                                     (Address + IXF3204_ADDR_OFFSET10),Data,
                                     (TE_BIT2 | TE_BIT1))) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneMasterSlaveMode
 * Returns: TEstatus
 * Description: Returns the configured mode for the clock, sync pulse and 
 *              multiframe sync pulse for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneMasterSlaveMode(void *handle, TEword Port, 
                                            TEword Direction, void *Results)
{
    TEbyte Data;
    TEbyte Mask1 = (TE_BIT2 | TE_BIT1 | TE_BIT0);
    TEbyte Mask2 = (TE_BIT2 | TE_BIT1);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch (Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRMASTER);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask1,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            /* MF Sync */
            if(Data & TE_BIT0)
            {
                ReturnValue[2] = TE_MASTER;
            }
            else
            {
                ReturnValue[2] = TE_SLAVE;
            }
            /* Sync */
            if(Data & TE_BIT1)
            {
                ReturnValue[1] = TE_MASTER;
            }
            else
            {
                ReturnValue[1] = TE_SLAVE;
            }
            /* Clock */
            if(Data & TE_BIT2)
            {
                ReturnValue[0] = TE_SLAVE;
            }
            else
            {
                ReturnValue[0] = TE_MASTER;
            }
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTMASTER);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask2,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            /* Sync */
            if(Data & TE_BIT1)
            {
                ReturnValue[1] = TE_MASTER;
            }
            else
            {
                ReturnValue[1] = TE_SLAVE;
            }
            /* Clock */
            if(Data & TE_BIT2)
            {
                ReturnValue[0] = TE_SLAVE;
            }
            else
            {
                ReturnValue[0] = TE_MASTER;
            }
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneMapping
 * Returns: TEstatus
 * Description: Configures the mapping style used in the backplane for the
 *              specified Port and Direction. TE_DISABLE must be selected when
 *              both the line and the backplane are in the same mode (E1-E1 or
 *              T1-T1). The other options are valid when the line side is in T1
 *              mode and the backplane is in E1 mode. The other options are
 *              TE_CHTOCH and TE_G802. Backplane Mapping cannot be performed 
 *              when the line side is in E1 mode and the backplane is in E1
 *              mode.
 *              When TE_CHTOCH is selected, time slots 1, 2, and 3 of the
 *              T1 side are mapped to time slot 1, 2, and 3 of the E1 side. 
 *              Time slot 4 in E1 is not used. Time slots 4, 5, and 6 of the T1
 *              side are mapped to time slots 5, 6, and 7 of the E1 side and so
 *              on. 
 *              This mapping style is know also as 1 in 4. T1 has 24 times 
 *              slots. E1 has 32 time slots.
 *              When TE_G802 is selected then the receive time slots 1 to 15 
 *              of T1 are mapped to time slots 1 to 15 of E1, and time slots 16
 *              to 24 of T1 are mapped to time slots 17 to 25 of E1 and vice 
 *              versa when TE_TX is selected.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneMapping(void *handle, TEword Port, TEword Direction,
                                    TEword Mapping)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT1 | TE_BIT0);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(PartialAddress);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTMODE);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Mapping parameter*/
    switch(Mapping)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_CHTOCH:
            Data = TE_BIT0;
            break;
        case TE_G802:
            Data = TE_BIT1;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_BPTMODE;
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,(TEword)(Address + IXF3204_BPTMODE),
                                     Data,Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneMapping
 * Returns: TEstatus
 * Description: Return the configured mapping style for the specified Port and
 *              Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneMapping(void *handle, TEword Port, TEword Direction,
                                    void *Results)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT1 | TE_BIT0);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTMODE);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data parameter*/
    switch (Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT0:
            ReturnValue[0] = TE_CHTOCH;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_G802;
            break;
        default:  
            return TE_STATUS_BADREAD; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplanePayloadSettings
 * Returns: TEstatus
 * Description: Configures the backplane bit rate and the replication position 
 *              of signaling and data for the specified Port and Direction.
 *              When TE_1X is selected, only 0x00 is allowed in PosSig and 
 *              PosData.
 *              When TE_2X is selected, 0x00 and 0x01 are allowed.
 *              When TE_4X is selected, 0x00 to 0x03 are allowed.
 *              When TE_8X is selected, 0x00 to 0x07 are allowed.
 ******************************************************************************/
TEstatus IXF3204SetBackplanePayloadSettings(void *handle, TEword Port, 
                                            TEword Direction, TEword ByteRep, 
                                            TEword PosSig, TEword PosData)
{
    TEbyte Data;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRREPLIC);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTREPLIC);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    switch(ByteRep)
    {
        case TE_1X:
            Data = TE_CLEAR_BITS;
            /* The posistion should be only one, TE_POS0 */
            if((PosSig != TE_POS0) || (PosData != TE_POS0)) 
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_2X:
            Data = TE_BIT6;
            /* The position should be only TE_POS0 or TE_POS1 */
            if((PosSig < TE_POS0) || (PosSig > TE_POS1) ||
               (PosData < TE_POS0) || (PosData > TE_POS1))
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_4X:
            Data = TE_BIT7;
            /* The position should be only betwen TE_POS0 and TE_POS3 */
            if((PosSig < TE_POS0) || (PosSig > TE_POS3) ||
               (PosData < TE_POS0) || (PosData > TE_POS3))
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_8X:
            Data = (TE_BIT7 | TE_BIT6);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /* 
     * Set the proper bit(s) depending of the selected 
     * position of the data 
     */
    switch(PosData)
    {
        case TE_POS0:
            break;
        case TE_POS1:
            Data |= TE_BIT0;
            break;
        case TE_POS2:
            Data |= TE_BIT1;
            break;
        case TE_POS3:
            Data |= (TE_BIT1 | TE_BIT0);
            break;
        case TE_POS4:
            Data |= TE_BIT2;
            break;
        case TE_POS5:
            Data |= (TE_BIT2 | TE_BIT0);
            break;
        case TE_POS6:
            Data |= (TE_BIT2 | TE_BIT1);
            break;
        case TE_POS7:
            Data |= (TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* 
     * Set the proper bit(s) depending of the selected 
     * position of the signaling 
     */
    switch(PosData)
    {
        case TE_POS0:
            break;
        case TE_POS1:
            Data |= TE_BIT3;
            break;
        case TE_POS2:
            Data |= TE_BIT4;
            break;
        case TE_POS3:
            Data |= (TE_BIT4 | TE_BIT3);
            break;
        case TE_POS4:
            Data |= TE_BIT5;
            break;
        case TE_POS5:
            Data |= (TE_BIT5 | TE_BIT3);
            break;
        case TE_POS6:
            Data |= (TE_BIT5 | TE_BIT4);
            break;
        case TE_POS7:
            Data |= (TE_BIT5 | TE_BIT4 | TE_BIT3);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Data))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_ADDR_OFFSET10;
            if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Data))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteByte(handle,
                                      (TEword)(Address + IXF3204_ADDR_OFFSET10),
                                      Data)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /*perform set operation*/
    return IXF3204WriteByte(handle,Address,Data);
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplanePayloadSettings
 * Returns: TEstatus
 * Description: Returns the configured backplane bit rate and the replication
 *              position of signaling and data for the specified Port and
 *              Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplanePayloadSettings(void *handle, TEword Port,
                                            TEword Direction, void *Results)
{
    TEbyte Data;
    TEbyte DataTemp;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRREPLIC);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTREPLIC);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    DataTemp = (TEbyte)(Data & (TE_BIT7 | TE_BIT6));
    switch(DataTemp)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_1X;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_2X;
            break;
        case TE_BIT7:
            ReturnValue[0] = TE_4X;
            break;
        case (TE_BIT7 | TE_BIT6):
            ReturnValue[0] = TE_8X;
            break;
        default:	 
            return TE_STATUS_BADREAD; 
    }

    /* Get the Signaling Position */
    switch(Data & (TE_BIT5 | TE_BIT3 | TE_BIT3))
    {
        case TE_CLEAR_BITS:
            ReturnValue[1] = TE_POS0;
            break;
        case TE_BIT3:
            ReturnValue[1] = TE_POS1;
            break;
        case TE_BIT4:
            ReturnValue[1] = TE_POS2;
            break;
        case (TE_BIT4 | TE_BIT3):
            ReturnValue[1] = TE_POS3;
            break;
        case TE_BIT5:
            ReturnValue[1] = TE_POS4;
            break;
        case (TE_BIT5 | TE_BIT3):
            ReturnValue[1] = TE_POS5;
            break;
        case (TE_BIT5 | TE_BIT4):
            ReturnValue[1] = TE_POS6;
            break;
        case (TE_BIT5 | TE_BIT4 | TE_BIT3):
            ReturnValue[1] = TE_POS7;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    /* Get the Data Position */
    switch(Data & (TE_BIT2 | TE_BIT1 | TE_BIT0))
    {
        case TE_CLEAR_BITS:
            ReturnValue[2] = TE_POS0;
            break;
        case TE_BIT0:
            ReturnValue[2] = TE_POS1;
            break;
        case TE_BIT1:
            ReturnValue[2] = TE_POS2;
            break;
        case (TE_BIT1 | TE_BIT0):
            ReturnValue[2] = TE_POS3;
            break;
        case TE_BIT2:
            ReturnValue[2] = TE_POS4;
            break;
        case (TE_BIT2 | TE_BIT0):
            ReturnValue[2] = TE_POS5;
            break;
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[2] = TE_POS6;
            break;
        case (TE_BIT2 | TE_BIT1 | TE_BIT0):
            ReturnValue[2] = TE_POS7;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneTSTriState
 * Returns: TEstatus
 * Description: Sets each time slot as output or tri-state for the specified 
 *              Port. Each bit of the TimeSlot parameter corresponds to one 
 *              time slot. The LSB of TimeSlot corresponds to the first time
 *              slot (time slot 0 for E1 or J1 mode; time slot 1 for T1 mode). 
 *              The time slot is set as output when the corresponding bit is 0.
 *              The time slot is tri-stated when the corresponding bit is 1.
 *              There are 32 bits to be set or cleared for this function.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneTSTriState(void *handle, TEword Port,
                                       TEdword TimeSlot)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress + IXF3204_BPRTSDIS0_7);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WritedWordAllPortsRegs(handle,Address,TimeSlot))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return(IXF3204WritedWord(handle,Address,TimeSlot));

}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneTSTriState
 * Returns: TEstatus
 * Description: Return a value of 32 bits where each bit corresponds to a time
 *              slot for the specified Port. The LSB corresponds to the first
 *              time slot (time slot 0 for E1 or J1 mode; time slot 1 for T1
 *              mode). When the bit is 0 the corresponding time slot is
 *              in tri-state. When the bit is 1 means that the corresponding 
 *              time slot is configured as output.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneTSTriState(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress + IXF3204_BPRTSDIS0_7);
    /*perform get operation*/
    return(IXF3204ReaddWord(handle,Address,ReturnValue));
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneSettings
 * Returns: TEstatus
 * Description: Sets some settings related to data, clock, sync and multi-sync
 *              pulse for the specified Port and Direction. Edge, Polarity and 
 *              Width are related to the sync and multi-sync pulse. The Edge
 *              parameter sets the edge used to sample the sync and multi-sync
 *              pulses. Polarity sets the active level for the sync and
 *              multi-sync pulse. The Width parameter sets the width of the
 *              sync and multi-sync pulses in clock cycles.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneSettings(void *handle, TEword Port, 
                                     TEword Direction, TEword Edge, 
                                     TEword Polarity, TEword Width, 
                                     TEword DataEdge, TEword ClkRate)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT1 | TE_BIT0);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRWAVE);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTWAVE);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate ClkRate parameter*/
    switch(ClkRate)
    {
        case TE_1X:
            Data = TE_CLEAR_BITS;
            break;
        case TE_2X:
            Data = TE_BIT0;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate DataEdge parameter*/
    switch(DataEdge)
    {
        case TE_RISING:
            break;
        case TE_FALLING:
            Data |= TE_BIT1;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Width parameter*/
    switch(Width) 
    {
        case TE_1X:
            break;
        case TE_2X:
            Data |= TE_BIT4;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Polarity parameter*/
    switch(Polarity)
    {
        case TE_HIGH:
            break;
        case TE_LOW:
            Data |= TE_BIT5;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Edge parameter*/
    switch(Edge)
    {
        case TE_RISING:
            break;
        case TE_FALLING:
            Data |= TE_BIT6;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_ADDR_OFFSET10;
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,
                                     (TEword)(Address + IXF3204_ADDR_OFFSET10),
                                     Data, Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneSettings
 * Returns: TEstatus
 * Description: Returns the configured values of the clock, data, sync and 
 *              multiframe sync pulses for the Port and Direction specified.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneSettings(void *handle, TEword Port, 
                                     TEword Direction,void *Results)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT1 | TE_BIT0);
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRWAVE);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTWAVE);
            break;
        default:	 
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* Edge */
    if(Data & TE_BIT6)
    {
        ReturnValue[0] = TE_FALLING;
    }
    else
    {
        ReturnValue[0] = TE_RISING;
    }
    /* Polarity */
    if(Data & TE_BIT5)
    {
        ReturnValue[1] = TE_LOW;
    }
    else
    {
        ReturnValue[1] = TE_HIGH;
    }
    /* Width */
    if(Data & TE_BIT4)
    {
        ReturnValue[2] = TE_2X;
    }
    else
    {
        ReturnValue[2] = TE_1X;
    }
    /* DataEdge */
    if(Data & TE_BIT1)
    {
        ReturnValue[3] = TE_FALLING;
    }
    else
    {
        ReturnValue[3] = TE_RISING;
    }
    /* Clock Rae */
    if(Data & TE_BIT0)
    {
        ReturnValue[4] = TE_2X;
    }
    else
    {
        ReturnValue[4] = TE_1X;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneFbitGapped
 * Returns: TEstatus
 * Description: Configures whether or not the F-bit is gapped for the specified
 *              Port and Direction. When Enable is selected, the F-bit is 
 *              gapped.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneFbitGapped(void *handle, TEword Port, 
                                       TEword Direction, TEword Selection)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT7;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRWAVE);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTWAVE);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Selection parameter*/
    switch(Selection)
    {
        case TE_ENABLE:
            Data = TE_BIT7;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_ADDR_OFFSET10;
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,
                                     (TEword)(Address + IXF3204_ADDR_OFFSET10),
                                     Data, Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneFbitGapped
 * Returns: TEstatus
 * Description: Returns the configured value that indicates whether or not the
 *              F-bit is gapped for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneFbitGapped(void *handle, TEword Port, 
                                       TEword Direction, void *Results)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT7;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRWAVE);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTWAVE);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT7)
    {
        ReturnValue[0] = TE_ENABLE;
    }else{
        ReturnValue[0] = TE_DISABLE;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneTSClkGap
 * Returns: TEstatus
 * Description: Enables or disables the clock gapping for the specified time 
 *              slot (TS) (or all time slots when TE_ALL is selected) for the
 *              specified Port and Direction. The clock gapping must be done 
 *              only when the selected backplane is configured as master. 
 ******************************************************************************/
TEstatus IXF3204SetBackplaneTSClkGap(void *handle, TEword Port,
                                     TEword Direction, TEword ts, TEword Mode)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT7;
    TEword Address;
    TEstatus Status;
    TEword Backplane_RAM_offset, TempAddress = TE_CLEAR_BITS;
    int LoopTS;

    /*validate TS parameter*/
    if(ts != TE_ALL)
    {
        if((ts < TE_TS0) || (ts > TE_TS31))
        {
            return TE_STATUS_BAD_PARAMS;
        }
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
            /* get the Tx address to configure both directions */
            TempAddress = IXF3204FindPortRAMTx(Port);
            if(TempAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
        case TE_RX:
            /*Get offsets to perform the set operations*/
            Backplane_RAM_offset = IXF3204FindPortRAMRx(Port);
            if(Backplane_RAM_offset == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_TX:
            Backplane_RAM_offset = IXF3204FindPortRAMTx(Port);
            if(Backplane_RAM_offset == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /* validate Mode parameter */
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT7;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default :
            return TE_STATUS_BAD_PARAMS;
    }

    if(ts == TE_ALL)
    {
        for(LoopTS = 0;LoopTS < IXF3204_MAX_TS;LoopTS++)
        {
            Address = (TEword)(Backplane_RAM_offset + LoopTS);
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,
                                                        Mask))!= 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                if(Direction == TE_BOTH)
                {
                    Address = (TEword)(TempAddress + LoopTS);
                    if((Status = IXF3204WriteBitAllPortsMem(handle,Address,
                                                            Data,Mask))!= 
                       TE_STATUS_SUCCESS)
                    {
                        return Status;
                    }
                }
                continue;
            }

            /*perform set operation*/
            if((Status = IXF3204WriteBit(handle,Address,Data,Mask))!=
                TE_STATUS_SUCCESS)
            {
                return Status;
            }
            if(Direction == TE_BOTH)
            {
                Address = (TEword)(TempAddress + LoopTS);
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
        }
        return TE_STATUS_SUCCESS;
    }
    else
    {
        /* Set all the ports with the same configuration */
        if(Port == TE_ALL)
        {
            Address = (TEword)(Backplane_RAM_offset + IXF3204GetTimeSlot(ts));
            if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
            if(Direction == TE_BOTH)
            {
                Address = (TEword)(TempAddress + IXF3204GetTimeSlot(ts));
                if((Status = IXF3204WriteBitAllPortsMem(handle,Address,
                                                        Data,Mask))!= 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        }

        if(Direction == TE_BOTH)
        {
            Address = (TEword)(TempAddress + IXF3204GetTimeSlot(ts));
            /*perform set operation*/
            if((Status = IXF3204WriteBit(handle,Address,Data,TE_BIT7))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        Address = (TEword)(Backplane_RAM_offset + IXF3204GetTimeSlot(ts));
        /*perform set operation*/
        return IXF3204WriteBit(handle,Address,Data,Mask);
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneTSClkGap
 * Returns: TEstatus
 * Description: Returns the gapping configuration of the clock for the 
 *              specified time slot (TS), Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneTSClkGap(void *handle, TEword Port, 
                                     TEword Direction, TEword ts, void *Results)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT7;
    TEword Address;
    TEword Backplane_RAM_offset;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /* validate Direction parameter */
    switch(Direction)
    {
        case TE_RX:
            /*Get offsets to perform the get operations*/
            Backplane_RAM_offset = IXF3204FindPortRAMRx(Port);
            if(Backplane_RAM_offset == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_TX:
            Backplane_RAM_offset = IXF3204FindPortRAMTx(Port);
            if (Backplane_RAM_offset == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate TS parameter*/
    if((ts < TE_TS0) || (ts > TE_TS31))
    {
        return TE_STATUS_BAD_PARAMS;
    }
    else
    {
        Address = (TEword)(Backplane_RAM_offset + IXF3204GetTimeSlot(ts));
        /*perform get operation*/
        if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Data & TE_BIT7)
        {
            ReturnValue[0] = TE_ENABLE;
        }
        else
        {
            ReturnValue[0] = TE_DISABLE;
        }
        return TE_STATUS_SUCCESS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneInterleaving
 * Returns: TEstatus
 * Description: Configures the module to perform internal data-signal 
 *              interleaving for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneInterleaving(void *handle, TEword Port,
                                         TEword Direction, TEword Mode)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT2;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRWAVE);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTWAVE);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT2;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default :	
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_ADDR_OFFSET10;
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,
                                     (TEword)(Address + IXF3204_ADDR_OFFSET10),
                                     Data, Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneInterleaving
 * Returns: TEstatus
 * Description: Returns module internal data-signal interleaving configuration 
 *              for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneInterleaving(void *handle, TEword Port,
                                         TEword Direction, void *Results)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT2;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_BPRWAVE);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_BPTWAVE);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT2)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneSyncOffset
 * Returns: TEstatus
 * Description: Sets the offset (in clock cycles) used to delay the sync pulse
 *              from the beginning of the frame for the specified Port and
 *              Direction.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneSyncOffset(void *handle, TEword Port, 
                                       TEword Direction, TEword Offset)
{
    TEword Address;
    TEword PartialAddress;
    TEbyte Data;
    TEstatus Status;
    TEbyte Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (TEword)(IXF3204_BPROFFSETA + PartialAddress);
            break;
        case TE_TX:
            Address = (TEword)(IXF3204_BPTOFFSETA + PartialAddress);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Data = (TEbyte)(Offset);
 
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Data))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        Address++;
        Data = (TEbyte)(Offset >> 8);
        /* write most significant bits */
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        /* write to the other direction */
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_ADDR_OFFSETF;
            Data = (TEbyte)(Offset);
            if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Data))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
            Address++;
            Data = (TEbyte)(Offset >> 8);
            /* write most significant bits */
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        if ((Status = IXF3204WriteByte(handle,
                                       (TEword)(Address + IXF3204_ADDR_OFFSET10),
                                       Data)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    /*perform set operation*/
    if ((Status = IXF3204WriteByte(handle,Address,Data)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }
    
    Address++;
    Data = (TEbyte)(Offset >> 8);
    if(Direction == TE_BOTH)
    {
        /* write most significant bits */
        if((Status = IXF3204WriteBit(handle,
                                     (TEword)(Address + IXF3204_ADDR_OFFSET10),
                                     Data,Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneSyncOffset
 * Returns: TEstatus
 * Description: Returns the offset used to delay the sync pulse from the
 *              beginning of the frame for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneSyncOffset(void *handle, TEword Port,
                                       TEword Direction, void *Results)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);
    TEbyte Data2;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Direction)
    {
        case TE_RX:        
            Address = (TEword)(IXF3204_BPROFFSETA + PartialAddress);
            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            Address = (TEword)(IXF3204_BPROFFSETB + PartialAddress);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data2)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            break; 
        case TE_TX:
            Address = (TEword)(IXF3204_BPTOFFSETA + PartialAddress);
            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            Address = (TEword)(IXF3204_BPTOFFSETB + PartialAddress);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,
                                        (TE_BIT2 | TE_BIT1 | TE_BIT0),&Data2)) 
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
            break; 
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    ReturnValue[0] = (Data | (TEword) (Data2 << 8));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBackplaneDataInversionMask
 * Returns: TEstatus
 * Description: Sets the mask for the receive or transmit data for the 
 *              specified Port and Direction. When a bit is set to 1 in the
 *              Mask, the data in that bit position is inverted for all 
 *              time slots.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneDataInversionMask(void * handle, TEword Port,
                                              TEword Direction,
                                              TEbyte Mask)
{
    TEword Address;
    TEstatus Status;

    Address = IXF3204FindPortBackPlane(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address += IXF3204_BPRXINVMSK;
            break;
        case TE_TX:
            Address += IXF3204_BPTXINVMSK0;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            Address += IXF3204_ADDR_OFFSET4;
            if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteByte(handle,
                                      (TEword)(Address + IXF3204_ADDR_OFFSET4),
                                      Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /* preforms the set operation */
    return(IXF3204WriteByte(handle,Address,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneDataInversionMask
 * Returns: TEstatus
 * Description: Returns the configured data inversion mask in the specified
 *              direction for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneDataInversionMask(void * handle, TEword Port, 
                                              TEword Direction, void * Results)
{
    TEword Address;
    TEbyte Value;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortBackPlane(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Direction)
    {
        case TE_RX:
            Address += IXF3204_BPRXINVMSK;
            break;
        case TE_TX:
            Address += IXF3204_BPTXINVMSK0;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /* performs the get operation */
    if((Status = IXF3204ReadByte(handle,Address, &Value))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Value;

    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204SetBackplaneSignalingInversionMask
 * Returns: TEstatus
 * Description: Sets the mask for the receive or transmit signaling for the
 *              specified backplane signaling Port and Direction. When a bit is
 *              set to 1 in the Mask, the signaling in that bit position is
 *              inverted for all time slots.
 ******************************************************************************/
TEstatus IXF3204SetBackplaneSignalingInversionMask(void * handle, TEword Port, 
                                                   TEword Direction,TEbyte Mask)
{
    TEword Address, AddressBoth;
    TEbyte InMask, WriteMask;
    TEstatus Status;

    AddressBoth = TE_CLEAR_BITS;
    /* Validate that mask is between 0x00 and 0x0F */
    if(Mask > TE_BIT0_3)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    InMask = Mask;
    WriteMask = TE_BIT0_3;

    /*validate Direction*/
    switch(Direction)
    {
        case TE_BOTH:
            AddressBoth = IXF3204FindPortTxSysSigIF_CDR(Port);
            if(AddressBoth == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            InMask <<= 1;
            WriteMask <<= 1;
        case TE_RX:
            Address = IXF3204FindPortRxSysSigIF_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address += IXF3204_RXSYS_CR1;
            break;
        case TE_TX:
            Address = IXF3204FindPortTxSysSigIF_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            InMask <<= 1;
            WriteMask <<= 1;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if(Direction == TE_BOTH)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressBoth,InMask,
                                                     WriteMask))!= 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            InMask >>= 1;
            WriteMask >>= 1;
        }
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,InMask,
                                                 WriteMask))!= TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        if ((Status = IXF3204WriteBit(handle,AddressBoth,InMask,WriteMask))
            != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        InMask >>= 1;
        WriteMask >>= 1;
    }

    return(IXF3204WriteBit(handle,Address,InMask, WriteMask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBackplaneSignalingInversionMask
 * Returns: TEstatus
 * Description: Returns the configured signaling inversion mask in the 
 *              specified backplane signaling Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetBackplaneSignalingInversionMask(void * handle, TEword Port, 
                                                   TEword Direction,
                                                   void * Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;
    TEbyte Value, Mask;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortBackPlane(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Direction)
    {
        case TE_RX:
            Address = IXF3204FindPortRxSysSigIF_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address += IXF3204_RXSYS_CR1;
            Mask = (TE_BIT3 | TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        case TE_TX:
            Address = IXF3204FindPortTxSysSigIF_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Mask = (TE_BIT4 | TE_BIT3 | TE_BIT2 | TE_BIT1);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }

    if((Status = IXF3204ReadBit(handle,Address, Mask, &Value))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* if Tx, a shift is needed */
    if(Direction == TE_TX) 
    {
        Value >>= 1;
    }

    ReturnValue[0] = Value;

    return TE_STATUS_SUCCESS;
}

