/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   bert3204.c
 * Description: This file contains BERT functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetBERTPattern
 * Returns: TEstatus
 * Description: Configures the pattern that the specified BERT transmits and
 *              receives. See the table in the reference manual for the values 
 *              needed to generate the desired pattern.
 ******************************************************************************/
TEstatus IXF3204SetBERTPattern(void *handle, TEword bert, TEbyte ps0, 
                               TEbyte ps1,TEbyte ps2, TEbyte ps3, TEbyte lr, 
                               TEbyte tr, TEbyte ctr)
{
    TEword Data, Address;
    TEword PartialAddress;
    TEstatus Status;

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* validate parameter lr and tr */
    if((lr | tr) > (TE_BIT4 | TE_BIT0_3))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* validate parameter ctr */
    if(ctr > TE_BIT0_3)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Data = (TEword)((ps1 << 8) | ps0);
    Address = (TEword)(PartialAddress);

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteWordAllPortsMem(handle,Address,Data)) !=
            TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteWord(handle,Address,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
        
    Address = (TEword)(PartialAddress + IXF3204_BERT_PATTERN_CFG_2);
    Data = (TEword)((ps3 << 8) | ps2);

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteWordAllPortsMem(handle,Address,Data)) !=
            TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteWord(handle,Address,Data))!= 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    Address = (TEword)(PartialAddress + IXF3204_BERT_PATTERN_LENGTH);
    Data = (TEword)((tr << 8) | lr);

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteWordAllPortsMem(handle,Address,Data)) !=
            TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteWord(handle,Address,Data))!= 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    Address = (TEword)(PartialAddress + IXF3204_BERT_MODE_CFG);

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsMem(handle,Address,ctr)) !=
            TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    else
    {
        /*Perform the Set operation*/
        return IXF3204WriteByte(handle,Address,ctr);
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTPattern
 * Returns: TEstatus
 * Description: Returns the configuration pattern of the specified BERT.
 ******************************************************************************/
TEstatus IXF3204GetBERTPattern(void *handle, TEword bert, void *Results)
{
    TEword Data;
    TEbyte Data1;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress);
    /*Perform the get operation*/
    if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /* ps0 */
    ReturnValue[0] = (TEbyte)Data;
    /* ps1 */
    ReturnValue[1] = (Data >> 8);

    Address = (TEword)(IXF3204_BERT_PATTERN_CFG_2 + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /* ps2 */
    ReturnValue[2] = (TEbyte)Data;
    /* ps3 */
    ReturnValue[3] = (Data >> 8);

    Address = (TEword)(IXF3204_BERT_PATTERN_LENGTH + PartialAddress);
    /*Perform the get operation*/
    if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /* lr */
    ReturnValue[4] = (TEbyte)Data;
    /* tr */
    ReturnValue[5] = (Data >> 8);

    Address = (TEword)(IXF3204_BERT_MODE_CFG + PartialAddress);
    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data1))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /* ctr */
    ReturnValue[6] = Data1;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBERTRxSyncCriteria
 * Returns: TEstatus
 * Description: Sets the synchronization criteria used to declare
 *              synchronization and lost synchronization of the pattern 
 *              received. The sync state is declared when there are at least
 *              InSyncWindow number of consecutive bits without error. Lost 
 *              sync is declared when there are more than ErrorNumber errors 
 *              in a window of OutSyncWindow size (bits). This function affects
 *              all the BERTs.
 ******************************************************************************/
TEstatus IXF3204SetBERTRxSyncCriteria(void *handle, TEword InSyncWindow,
                                      TEword OutSyncWindow, TEword ErrorNumber)
{
    TEstatus Status;

    /*Perform the Set operation*/
    if((Status = IXF3204WriteWord(handle,IXF3204_BERT_INSYNC_W0,InSyncWindow))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Perform the Set operation*/    
    if((Status = IXF3204WriteWord(handle,IXF3204_BERT_OUTSYNC_W0,
                                  OutSyncWindow)) != TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /*Perform the Set operation*/
    return IXF3204WriteWord(handle,IXF3204_BERT_OUTSYNC_T0,ErrorNumber);
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTRxSyncCriteria
 * Returns: TEstatus
 * Description: Returns the configured values in the BERT module to declare 
 *              InSync and OutSync. This function affects all the BERTs.
 ******************************************************************************/
TEstatus IXF3204GetBERTRxSyncCriteria(void *handle, void *Results)
{
    TEword Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Perform the get operation*/
    if((Status = IXF3204ReadWord(handle,IXF3204_BERT_INSYNC_W0,&Data))!= 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    /*Perform the get operation*/
    if((Status = IXF3204ReadWord(handle,IXF3204_BERT_OUTSYNC_W0,&Data))!= 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[1] = Data;

    /*Perform the get operation*/
    if((Status = IXF3204ReadWord(handle,IXF3204_BERT_OUTSYNC_T0,&Data))!= 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[2] = Data;

    return TE_STATUS_SUCCESS;

}

/*******************************************************************************
 * Function Name: IXF3204SetBERTControlSync
 * Returns: TEstatus
 * Description: Configures the synchronization mode (Automatic/Manual). When 
 *              Auto is selected, the specified BERT looks for synchronization
 *              automatically when sync is lost. When Manual is selected, the 
 *              BERT does not look for synchronization until the instruction
 *              to do so is received.
 ******************************************************************************/
TEstatus IXF3204SetBERTControlSync(void *handle, TEword bert, TEword Selection)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT1;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);

    switch(Selection)
    {
        case TE_MANUAL:
            Data = TE_CLEAR_BITS;
            break;
        case TE_AUTO:
            Data = TE_BIT1;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask)) !=
            TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTControlSync
 * Returns: TEstatus
 * Description: Returns the synchronization mode configured in the BERT 
 *              specified.
 ******************************************************************************/
TEstatus IXF3204GetBERTControlSync(void *handle, TEword bert, void *Results)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT1;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS){
        return Status;
    }

    /*validate Data*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_MANUAL;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_AUTO;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name:  IXF3204SetBERTReSync
 * Returns: TEstatus
 * Description: Forces the BERT to look for synchronization even when it is
 *              already synchronized.
 ******************************************************************************/
TEstatus IXF3204SetBERTReSync(void *handle, TEword bert)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT3;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);

    Data = TE_BIT3;

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask)) !=
            TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}


/*******************************************************************************
 * Function Name: IXF3204SetBERTControlEnable
 * Returns: TEstatus
 * Description: Enables or disables the specified BERT to analyze the incoming 
 *              data. Also enables or disables BERT to generate the configured 
 *              pattern.
 ******************************************************************************/
TEstatus IXF3204SetBERTControlEnable(void *handle, TEword bert, 
                                     TEword Selection)
{
    TEbyte Data, Both;
    TEbyte Mask = TE_BIT0;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if (PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Both = TE_FALSE;
    switch(Selection)
    {
        case TE_ENABLE:
            Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);
            Data = TE_BIT0;
            Both = TE_TRUE;
            break;
        case TE_DISABLE:
            Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);
            Data = TE_CLEAR_BITS;
            Both = TE_TRUE;
            break;
        case TE_ENABLERX:
            Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);
            Data = TE_BIT0;
            break;
         case TE_ENABLETX:
            Address = (TEword)(IXF3204_BERT_TXCTRL + PartialAddress);
            Data = TE_BIT0;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if(Both)
        {
            /*Perform the Set operation*/
            if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
            Address = (TEword)(IXF3204_ADDR_OFFSETF + PartialAddress);
        }
        if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    
    if(Both)
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        Address = (TEword)(IXF3204_ADDR_OFFSETF + PartialAddress);
    }

    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTControlEnable
 * Returns: TEstatus
 * Description: Returns the current status of the BERT enable bits (receive, 
 *              transmit or both). When the returned value is Disable, the
 *              generator and the analyzer are disabled.
 ******************************************************************************/
TEstatus IXF3204GetBERTControlEnable(void *handle, TEword bert, void *Results)
{
    TEbyte ValueRx, ValueTx;
    TEbyte Mask = TE_BIT0;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);
    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&ValueRx))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Address = (TEword)(IXF3204_ADDR_OFFSETF + PartialAddress);
    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&ValueTx)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(ValueRx | (ValueTx << 1))
    {
        case TE_BIT0:
            ReturnValue[0] = TE_ENABLERX;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_ENABLETX;
            break;
        case (TE_BIT1 | TE_BIT0):
            ReturnValue[0] = TE_ENABLE;
            break;
        default:
            ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBERTTSBitmask
 * Returns: TEstatus
 * Description: Defines which bits in the time slot are used to generate or 
 *              analyze data for the specified Port. The Mask applies to ALL
 *              time slots (for the specified Port). The Mask applies to
 *              receive and transmit operations. Bit 7 is the MSB and is the
 *              first bit received (transmitted).
 ******************************************************************************/
TEstatus IXF3204SetBERTTSBitmask(void *handle, TEword Port, TEbyte Mask)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindPortTxSignaling_CDR(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_TXSIG_BERTM + PartialAddress);

    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteByte(handle,Address,Mask))!=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindPortRxBusIF(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXBUSIF_CR2 + PartialAddress);

    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteByte(handle,Address,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTTSBitmask
 * Returns: TEstatus
 * Description: Returns the bit mask used in the generation or analysis 
 *              process for the specified Port. The bit mask defines which bits
 *              in the time slot are used to generate or analyze BERT data.
 *              The Mask applies to ALL the time slots (for the specified 
 *              Port). The Mask applies to receive and transmit operations.
 ******************************************************************************/
TEstatus IXF3204GetBERTTSBitmask(void *handle, TEword Port, void *Results)
{
    TEbyte Data;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindPortTxSignaling_CDR(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_TXSIG_BERTM + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBERTTSBitmaskToBP
 * Returns: TEstatus
 * Description: Defines which bits in the time slot are used to generate or
 *              analyze BERT data for the specified Backplane. The Mask applies
 *              to ALL the time slots (for the specified Backplane). The Mask
 *              applies to receive and transmit operations. Bit 8 provides the
 *              mask for the F-bit. 
 ******************************************************************************/
TEstatus IXF3204SetBERTTSBitmaskToBP(void *handle, TEword Port, TEword Mask)
{
    TEbyte Data, Mask1, Mask2;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    if( Mask > (TE_BIT8 | TE_BIT0_7))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address = (TEword)(IXF3204_BPRMASKB + PartialAddress);

    Mask1 = (TEbyte)(Mask & TE_BIT0_7);

    /* write the mask for Rx */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,(TEbyte)Mask1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        if ((Status = IXF3204WriteByte(handle,Address,(TEbyte)(Mask1))) !=
            TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    
    Address = (TEword)(IXF3204_BPRMASK8 + PartialAddress);

    Mask1 = (TEbyte)((Mask >> 7) & TE_BIT1);
    Data = (TEbyte) Mask1;

    Mask2 = TE_BIT1;

    /* write the mask for the Rx F-bit */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask2))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        Status = IXF3204WriteBit(handle, Address, Data, Mask2);
        if(Status != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    Address = (TEword)(IXF3204_BPTMASKB + PartialAddress);
    Mask1 = (TEbyte)(Mask & TE_BIT0_7);

    /* write the mask for Tx */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,(TEbyte)Mask1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteByte(handle,Address,(TEbyte)(Mask1)))!=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    Address = (TEword)(IXF3204_BPTMASK8 + PartialAddress);
    
    /* write the mask for the Tx F-bit */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask2))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle, Address, Data, Mask2);
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTTSBitmaskToBP
 * Returns: TEstatus
 * Description: Returns the bit mask used in the generation or analysis process.
 *              Defines which bits in the time slot are used to generate or
 *              analyze BERT data for the specified Backplane. The Mask applies
 *              to ALL the time slots (for the specified Backplane). The Mask 
 *              applies to receive and transmit operations.
 ******************************************************************************/
TEstatus IXF3204GetBERTTSBitmaskToBP(void *handle, TEword Port, void *Results)
{
    TEbyte Data;
    TEbyte Data2;
    TEbyte Mask = TE_BIT1;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operation*/
    PartialAddress = IXF3204FindPortBackPlane(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BPRMASKB + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Address = (TEword)(IXF3204_BPRMASK8 + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data2))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* move the F-bit mask and put it in the upper byte */
    ReturnValue[0] = (Data | ((Data2 & TE_BIT1) << 7));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBERTControlLoadReg
 * Returns: TEstatus
 * Description: Configures the way that the Error counter and total counter 
 *              registers are refreshed. When Auto is selected, these registers
 *              are refreshed every second. When Manual is selected, the
 *              automatic load is disabled and the counters are loaded at that
 *              time only. In order to load the counters again this function 
 *              must be used again.
 ******************************************************************************/
TEstatus IXF3204SetBERTControlLoadReg(void *handle, TEword bert, TEword Mode)
{
    TEbyte Data;
    TEbyte Mask = (TE_BIT4 | TE_BIT2);
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);

    switch(Mode)
    {
        case TE_AUTO:
            Data = TE_BIT2;
            break;
        case TE_MANUAL:
            Data = TE_BIT4;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTControlLoadReg
 * Returns: TEstatus
 * Description: Returns the configured mode in the specified BERT to load the
 *              counter registers.
 ******************************************************************************/
TEstatus IXF3204GetBERTControlLoadReg(void *handle, TEword bert, void *Results)
{
    TEbyte Data;
    TEbyte Mask = TE_BIT2;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_RXCTRL + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data)
    {
        ReturnValue[0] = TE_AUTO;
    }
    else
    {
        ReturnValue[0] = TE_NONAUTO;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTTotalCount
 * Returns: TEstatus
 * Description: Returns the bit count register value.
 ******************************************************************************/
TEstatus IXF3204GetBERTTotalCount(void *handle, TEword bert, void *Results)
{
    TEword WValue;
    TEbyte BValue;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_TOTAL_CNT0 + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadWord(handle,Address,&WValue)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Address = (TEword)(IXF3204_BERT_TOTAL_CNT2 + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&BValue)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = ((BValue << 16) | WValue);
    return TE_STATUS_SUCCESS;

}

/*******************************************************************************
 * Function Name: IXF3204GetBERTErrorCount
 * Returns: TEstatus
 * Description: Returns the error count value.
 ******************************************************************************/
TEstatus IXF3204GetBERTErrorCount(void *handle, TEword bert, void *Results)
{
    TEword WValue;
    TEbyte BValue;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_ERROR_CNT0 + PartialAddress);
    /*Perform the get operation*/
    if((Status = IXF3204ReadWord(handle,Address,&WValue)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Address = (TEword)(IXF3204_BERT_ERROR_CNT2 + PartialAddress);
    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&BValue))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = ((BValue << 16) | WValue);

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBERTErrorInsert
 * Returns: TEstatus
 * Description: Defines the bit error rate to be inserted in the pattern
 *              generated by the BERT. Rates are 10-1, 10-2, 10-3, 10-4, 10-5,
 *              10-6, 10-7. 
 ******************************************************************************/
TEstatus IXF3204SetBERTErrorInsert(void *handle, TEword bert, TEword Rate)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    /*get offsets to perform the set operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_TXCTRL + PartialAddress);
    if(Rate == TE_SINGLEERROR)
    {
        Data = TE_BIT4;
        Mask = TE_BIT4;
    }
    else
    {
        switch(Rate)
        {
            case TE_DISABLE:
                Data = TE_CLEAR_BITS;
                break;
            case TE_RATE1:
                Data = TE_BIT1;
                break;
            case TE_RATE2:
                Data = TE_BIT2;
                break;
            case TE_RATE3:
                Data = (TE_BIT2 | TE_BIT1);
                break;
            case TE_RATE4:
                Data = TE_BIT3;
                break;
            case TE_RATE5:
                Data = (TE_BIT3 | TE_BIT1);
                break;
            case TE_RATE6:
                Data = (TE_BIT3 | TE_BIT2);
                break;
            case TE_RATE7:
                Data = (TE_BIT3 | TE_BIT2 | TE_BIT1);
                break;
            default:   
                return TE_STATUS_BAD_PARAMS; 
        }
        Mask = (TE_BIT3 | TE_BIT2 | TE_BIT1);
    }

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTErrorInsert
 * Returns: TEstatus
 * Description: Returns the current error insertion rate of the specified BERT.
 ******************************************************************************/
TEstatus IXF3204GetBERTErrorInsert(void *handle, TEword bert, void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Mask = (TE_BIT3 | TE_BIT2 | TE_BIT1);

    /*get offsets to perform the get operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_TXCTRL + PartialAddress);
    /*Perform the get operation*/
    if((Status=IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate read data*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_RATE1;
            break;
        case TE_BIT2:
            ReturnValue[0] = TE_RATE2;
            break;
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_RATE3;
            break;
        case TE_BIT3:
            ReturnValue[0] = TE_RATE4;
            break;
        case (TE_BIT3 | TE_BIT1):
            ReturnValue[0] = TE_RATE5;
            break;
        case (TE_BIT3 | TE_BIT2):
            ReturnValue[0] = TE_RATE6;
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_RATE7;
            break;
        default:    
            return TE_STATUS_BADREAD; 
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetBERTStatus
 * Returns: TEstatus
 * Description: Returns the current status of the specified BERT. 
 ******************************************************************************/
TEstatus IXF3204GetBERTStatus(void *handle, TEword bert, void *Results)
{
    TEbyte Data;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operation*/
    PartialAddress = IXF3204FindBERT(bert);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_BERT_RXSTATUS + PartialAddress);
    /*Perform the get operation*/
    if((Status=IXF3204ReadByte(handle,Address,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    *ReturnValue = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetDRSDMPattern
 * Returns: TEstatus
 * Description: Configures the DRS and DMW generator used to generate the u-Law
 *              pattern or A-Law pattern.
 ******************************************************************************/
TEstatus IXF3204SetDRSDMPattern(void *handle, TEword bert, TEword DRSType,
                                TEword DMWType)
{
    TEword Address;
    TEbyte Data, Mask = (TE_BIT1 | TE_BIT0);
    TEstatus Status;

    Address = IXF3204FindBERT(bert);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Mode parameter*/
    switch(DRSType)
    {
        case TE_ULAW:
            Data = TE_BIT0;
            break;
        case TE_ALAW:
            Data = TE_CLEAR_BITS;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate DMWType parameter*/
    switch(DMWType)
    {
        case TE_ULAW:
            Data |= TE_BIT1;
            break;
        case TE_ALAW:
            Data |= TE_CLEAR_BITS;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_BERT_DRS_DMW_CNT;

    /* if TE_ALL write to all ports */
    if(bert == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    return (IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetDRSDMPattern
 * Returns: TEstatus
 * Description: Returns the configured law in the DRS and DMW generator.
 ******************************************************************************/
TEstatus IXF3204GetDRSDMPattern(void *handle, TEword bert, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;
    TEbyte Mask = (TE_BIT1 | TE_BIT0);

    CHK_NULL(ReturnValue);

    Address=IXF3204FindBERT(bert);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    Address += IXF3204_BERT_DRS_DMW_CNT;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0]=TE_ALAW;
            ReturnValue[1]=TE_ALAW;
            break;
        case TE_BIT0:
            ReturnValue[0]=TE_ULAW;
            ReturnValue[1]=TE_ALAW;
            break;
        case TE_BIT1:
            ReturnValue[0]=TE_ALAW;
            ReturnValue[1]=TE_ULAW;
            break;
        case (TE_BIT1 | TE_BIT0):
            ReturnValue[0]=TE_ULAW;
            ReturnValue[1]=TE_ULAW;
            break;
        default:
            return TE_STATUS_BADREAD;
    }
    return TE_STATUS_SUCCESS;
}

