/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   bop3204.c
 * Description: this file contains BOP functions
 *
 ******************************************************************************/


#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetBOPMode
 * Returns: TEstatus
 * Description: Enables or disables the BOPs reception for the specified 
 *              Port.
 ******************************************************************************/
TEstatus IXF3204SetBOPMode(void *handle, TEword Port, TEword Mode)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT5;
    TEstatus Status;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    /*validate Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT5;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBOPMode
 * Returns: TEstatus
 * Description: Returns BOP reception configuration for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetBOPMode(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Mask = TE_BIT5;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data))!= 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(Data)
    {
        case TE_BIT5:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] =TE_DISABLE;
            break;
        default:   
            return TE_STATUS_BADREAD;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBOPCounterMode
 * Returns: TEstatus
 * Description: Configures the way that the Intel IXF3204 counts the BOP for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetBOPCounterMode(void *handle, TEword Port,TEword Mode)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT1;
    TEstatus Status;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Mode parameter*/
    switch(Mode)
    {
        case TE_CONTINUOUS:
            Data = TE_CLEAR_BITS;
            break;
        case TE_HYSTERESIS:
            Data = TE_BIT1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXFDL_BOPCR;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBOPCounterMode
 * Returns: TEstatus
 * Description: Returns the configured mode to count the received BOPs for the
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetBOPCounterMode(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Mask = TE_BIT1;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXFDL_BOPCR;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_CONTINUOUS;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_HYSTERESIS;
            break;
        default:
            return TE_STATUS_BADREAD;        
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBOPRxRAIEnable
 * Returns: TEstatus
 * Description: Enables or disables RAI detection in the BOP for the specified 
 *              Port.
 ******************************************************************************/
TEstatus IXF3204SetBOPRxRAIEnable(void *handle, TEword Port, TEword Mode)
{
    TEword Address;
    TEbyte Data;
    TEbyte Mask = TE_BIT6;
    TEstatus Status;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT6;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    Address += IXF3204_RXFDL_BOPCR;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBOPRxRAIEnable
 * Returns: TEstatus
 * Description: Returns the configured option for detection of RAI in the BOP 
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetBOPRxRAIEnable(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword	*ReturnValue = (TEdword *)Results;
    TEbyte Mask = TE_BIT6;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXFDL_BOPCR;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_ENABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBOPRxThreshold
 * Returns: TEstatus
 * Description: Configures the number of BOPs that must be received to declare 
 *              a valid BOP for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetBOPRxThreshold(void *handle, TEword Port, TEbyte Value)
{
    TEword Address;
    TEbyte Mask = (TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT2);
    TEstatus Status;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Value parameter  [0x00..0x0F]*/
    if (Value > TE_BIT0_3)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXFDL_BOPCR;
    Value <<= 2;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Value,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    return(IXF3204WriteBit(handle,Address,Value,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBOPRxThreshold
 * Returns: TEstatus
 * Description: Returns the configured number of BOPs needed to declare a 
 *              valid BOP for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetBOPRxThreshold(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEstatus Status;
    TEbyte Mask = (TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT2);
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);

    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXFDL_BOPCR;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = (Data >> 2);
    return Status;
}

/*******************************************************************************
 * Function Name: IXF3204SetBOPClearCounter
 * Returns: TEstatus
 * Description: Clears the BOP bit counter for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetBOPClearCounter(void *handle, TEword Port)
{
    TEword Address;
    TEbyte Data = TE_BIT0;
    TEbyte Mask = TE_BIT0;
    TEstatus Status;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXFDL_BOPCR;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBOPRxStatus
 * Returns: TEstatus
 * Description: Returns the current status of the BOP reception for the
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetBOPRxStatus(void *handle,TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data, Mask = TE_BIT0;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXFDL_BOPSR;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return Status;
}

/*******************************************************************************
 * Function Name: IXF3204GetBOPRxCode
 * Returns: TEstatus
 * Description: Return the BOP code received for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetBOPRxCode(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEbyte Mask = (TE_BIT7 | TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT2);

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXFDL_BOPSR;

    /*Perform the get operation*/
    Status = IXF3204ReadBit(handle,Address, Mask, &Data);
    if(Status != TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = (Data >> 2);
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBOPTxMode
 * Returns: TEstatus
 * Description: Configures the BOP transmit mode for the specified Port. When 
 *              TE_TIMES is selected, the BOP is transmitted the number of
 *              times specified in the parameter Times. When TE_CONTINUOUS is 
 *              selected, the BOP is transmitted continuously. 
 ******************************************************************************/
TEstatus IXF3204SetBOPTxMode(void *handle, TEword Port, TEword Mode,
                             TEword Times)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask = (TE_BIT1 | TE_BIT0);

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortTxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXFDL_BOPCR;
    /*validate Mode parameter*/
    switch(Mode)
    {
        case TE_CONTINUOUS:
            Data = TE_BIT0;
            break;
        case TE_TIMES:
            Data = TE_BIT1;
            /* write number of Times */
            Address++;
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllPortsRegs(handle,Address,Times))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteWord(handle,Address,Times)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            Address--;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    return (IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBOPTxMode
 * Returns: TEstatus
 * Description: Returns the configured mode to transmit BOPs and the number
 *              of times the BOP has been sent when applicable for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetBOPTxMode(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEword DataWord;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;
    TEbyte Data, Mask = (TE_BIT1 | TE_BIT0);

    CHK_NULL(ReturnValue);

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortTxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    
    Address += IXF3204_TXFDL_BOPCR;

    /* perform the get operation */
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch (Data)
    {
        case TE_CLEAR_BITS: 
            ReturnValue[0] = TE_NONCONTINUOUS;
            break;
        case TE_BIT0: 
            ReturnValue[0] = TE_CONTINUOUS;
            break;
        default:  
            return TE_STATUS_BADREAD; 
    }

    Address += IXF3204_TXFDL_HCR;

    /* perform the get operation */
    if((Status = IXF3204ReadWord(handle, Address, &DataWord)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /* number of times configured to transmit the BOP */
    ReturnValue[1] = DataWord;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetBOPTxCode
 * Returns: TEstatus
 * Description: Sets the BOP code to be transmitted for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetBOPTxCode(void *handle, TEword Port, TEbyte Code)
{
    TEword Address;
    TEbyte Mask;
    TEstatus Status;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortTxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* validate code parameter */
    if (Code > (TE_BIT5 | TE_BIT4 | TE_BIT0_3))
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXFDL_BOPCR;
    Code <<= 2;
    Mask = (TE_BIT7 | TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT2);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Code,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /* perform the set operation */
    return(IXF3204WriteBit(handle,Address,Code,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetBOPTxCode
 * Returns: TEstatus
 * Description: Returns the BOP code being transmitted for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetBOPTxCode(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;
    TEbyte Data, Mask;

    CHK_NULL(ReturnValue);

    /* get offsets to perform the set operations */
    Address = IXF3204FindPortTxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXFDL_BOPCR;
    Mask = (TE_BIT7 | TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT2);

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /*Note that the BOP code must be returned in the least significant bits*/
    ReturnValue[0] = (Data >> 2);

    return TE_STATUS_SUCCESS;
}

