/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   clks3204.c
 * Description: this file contains clock selector functions
 *
 ******************************************************************************/

#include "ixf3204.h"


/*******************************************************************************
 * Function Name: IXF3204SetClkSelOneSecondConfig
 * Returns: TEstatus
 * Description: Configures the parameters for the global clock module. These
 *              parameters are the source clock rate and the source for the
 *              one second generation.
 ******************************************************************************/
TEstatus IXF3204SetClkSelOneSecondConfig(void *handle, TEword Rate,
                                         TEword ClkSource)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr0);

    Data = TE_CLEAR_BITS;
    /*validate Rate parameter*/
    switch(Rate)
    {
        case TE_R1536:  
            break;
        case TE_R1544:
            Data |= TE_BIT5;
            break;
        case TE_R2048:
            Data |= TE_BIT6;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    /*validate ClkSource parameter*/
    switch(ClkSource)
    {
        case TE_BTCLK0:
            Data |= TE_BIT0;
        case TE_RCLK0:  
            break;
        case TE_BTCLK1:
            Data |= TE_BIT0;
        case TE_RCLK1:
            Data |= TE_BIT2;
            break;
        case TE_BTCLK2:
            Data |= TE_BIT0;
        case TE_RCLK2:
            Data |= TE_BIT3;
            break;
        case TE_BTCLK3:
            Data |= TE_BIT0;
        case TE_RCLK3:
            Data |= (TE_BIT3 | TE_BIT2);
            break;
        case TE_BTCLK4:
            Data |= TE_BIT0;
        case TE_RCLK4:
            Data |= TE_BIT4;
            break;
        case TE_BTCLK5:
            Data |= TE_BIT0;
        case TE_RCLK5:
            Data |= (TE_BIT4 | TE_BIT2);
            break;
        case TE_BTCLK6:
            Data |= TE_BIT0;
        case TE_RCLK6:
            Data |= (TE_BIT4 | TE_BIT3);
            break;
        case TE_BTCLK7:
            Data |= TE_BIT0;
        case TE_RCLK7:
            Data |= (TE_BIT4 | TE_BIT3 | TE_BIT2);
            break;
        case TE_EXTT1CLK:
            Data |= TE_BIT1;
            break;
        case TE_EXTE1CLK:
            Data |= (TE_BIT1 | TE_BIT0);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3);

  /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelOneSecondConfig
 * Returns: TEstatus
 * Description: Returns the configured global parameters for the one second
 *              generation.
 ******************************************************************************/
TEstatus IXF3204GetClkSelOneSecondConfig(void *handle, void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEstatus StatusValue;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr0;
    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3);

    /*perform get operation*/
    if((StatusValue = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return StatusValue;
    }

    if(!(Data & (TE_BIT6 | TE_BIT5)))
    {
        ReturnValue[0] = TE_R1536;
    }
    else
    {
        if(Data & TE_BIT6)
        {
            ReturnValue[0] = TE_R2048;
        }

        if(Data & TE_BIT5)
        {
            ReturnValue[0] = TE_R1544;
        }
    }

    if(Data & TE_BIT0)
    {
        switch(Data & (TE_BIT4 | TE_BIT3 | TE_BIT2))
        {
            case TE_CLEAR_BITS:
                ReturnValue[1] = TE_BTCLK0;
                break;
            case TE_BIT2:
                ReturnValue[1] = TE_BTCLK1;
                break;
            case TE_BIT3:
                ReturnValue[1] = TE_BTCLK2;
                break;
            case (TE_BIT3 | TE_BIT2):
                ReturnValue[1] = TE_BTCLK3;
                break;
            case TE_BIT4:
                ReturnValue[1] = TE_BTCLK4;
                break;
            case (TE_BIT4 | TE_BIT2):
                ReturnValue[1] = TE_BTCLK5;
                break;
            case (TE_BIT4 | TE_BIT3):
                ReturnValue[1] = TE_BTCLK6;
                break;
            case (TE_BIT4 | TE_BIT3 | TE_BIT2):
                ReturnValue[1] = TE_BTCLK7;
                break;
            default:  
                return TE_STATUS_BAD_PARAMS; 
        }
    }
    else
    {
        switch(Data & (TE_BIT4 | TE_BIT3 | TE_BIT2))
        {
            case TE_CLEAR_BITS:
                ReturnValue[1] = TE_RCLK0;
                break;
            case TE_BIT2:
                ReturnValue[1] = TE_RCLK1;
                break;
            case TE_BIT3:
                ReturnValue[1] = TE_RCLK2;
                break;
            case (TE_BIT3 | TE_BIT2):
                ReturnValue[1] = TE_RCLK3;
                break;
            case TE_BIT4:
                ReturnValue[1] = TE_RCLK4;
                break;
            case (TE_BIT4 | TE_BIT2):
                ReturnValue[1] = TE_RCLK5;
                break;
            case (TE_BIT4 | TE_BIT3):
                ReturnValue[1] = TE_RCLK6;
                break;
            case (TE_BIT4 | TE_BIT3 | TE_BIT2):
                ReturnValue[1] = TE_RCLK7;
                break;
            default:  
                return TE_STATUS_BAD_PARAMS;
        }
    }

    if(Data & TE_BIT1)
    {
        if(Data & TE_BIT0)
        {
            ReturnValue[1] = TE_EXTE1CLK;
        }
        else
        {
            ReturnValue[1] = TE_EXTT1CLK;
        }
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelReferenceClkSource
 * Returns: TEstatus
 * Description: Selects the clock to use for reference to generate an internal
 *              one second timing signal.
 ******************************************************************************/
TEstatus IXF3204SetClkSelReferenceClkSource(void *handle, TEword Reference)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr1);

    Data = TE_CLEAR_BITS;
    /*validate Reference parameter*/
    switch(Reference)
    {
        case TE_RCLK0:
            break;
        case TE_RCLK1:
            Data |= TE_BIT4;
            break;
        case TE_RCLK2:
            Data |= TE_BIT5;
            break;
        case TE_RCLK3:
            Data |= (TE_BIT5 | TE_BIT4);
            break;
        case TE_RCLK4:
            Data |= TE_BIT6;
            break;
        case TE_RCLK5:
            Data |= (TE_BIT6 | TE_BIT4);
            break;
        case TE_RCLK6:
            Data |= (TE_BIT6 | TE_BIT5);
            break;
        case TE_RCLK7:
            Data |= (TE_BIT6 | TE_BIT5 | TE_BIT4);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4);

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelReferenceClkSource
 * Returns: TEstatus
 * Description: Returns the clock being used for reference to generate an 
 *              internal one second timing signal.
 ******************************************************************************/
TEstatus IXF3204GetClkSelReferenceClkSource(void *handle, void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr1);
    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(Data & (TE_BIT6 | TE_BIT5 | TE_BIT4))
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_RCLK0;
            break;
        case TE_BIT4:
            ReturnValue[0] = TE_RCLK1;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_RCLK2;
            break;
        case (TE_BIT5 | TE_BIT4):
            ReturnValue[0] = TE_RCLK3;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_RCLK4;
            break;
        case (TE_BIT6 | TE_BIT4):
            ReturnValue[0] = TE_RCLK5;
            break;
        case (TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_RCLK6;
            break;
        case (TE_BIT6 | TE_BIT5 | TE_BIT4):
            ReturnValue[0] = TE_RCLK7;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelPrescaleRateConf
 * Returns: TEstatus
 * Description: Configures the prescale type and rate used in the one second 
 *              pulse generation.
 ******************************************************************************/
TEstatus IXF3204SetClkSelPrescaleRateConf(void *handle, TEword BaseClk,
                                          TEword rate)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr1);

    /* validate BaseClk parameter */
    switch(BaseClk)
    {
        case TE_E1:
            /* validate rate parameter */
            switch(rate)
            {
                case TE_1X:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_8X:
                    Data = TE_BIT2;
                    break;
                case TE_16X:
                    Data = TE_BIT3;
                    break;
                case TE_24X:
                    Data = (TE_BIT3 | TE_BIT2);
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }

            Mask = (TE_BIT3 | TE_BIT2);
            /*perform set operation*/
            return IXF3204WriteBit(handle,Address,Data,Mask);
        case TE_T1:
            switch (rate)
            {
                case TE_1X:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_8X:
                    Data = TE_BIT0;
                    break;
                case TE_16X:
                    Data = TE_BIT1;
                    break;
                case TE_24X:
                    Data = (TE_BIT1 | TE_BIT0);
                    break;
                default:   
                    return TE_STATUS_BAD_PARAMS;
            }

            Mask = (TE_BIT1 | TE_BIT0);
            /*perform set operation*/
            return IXF3204WriteBit(handle,Address,Data,Mask);
        default:   
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelPrescaleRateConf
 * Returns: TEstatus
 * Description: Returns the prescale rate for the specified type.
 ******************************************************************************/
TEstatus IXF3204GetClkSelPrescaleRateConf(void *handle, TEword BaseClk,
                                          void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr1);

    /*validate BaseClk parameter*/
    switch(BaseClk)
    {
        case TE_E1:
            Mask = (TE_BIT3 | TE_BIT2);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            switch(Data & (TE_BIT3 | TE_BIT2))
            {
                case TE_CLEAR_BITS:
                    ReturnValue[0] = TE_1X;
                    break;
                case TE_BIT2:
                    ReturnValue[0] = TE_8X;
                    break;
                case TE_BIT3:
                    ReturnValue[0] = TE_16X;
                    break;
                case (TE_BIT3 | TE_BIT2):
                    ReturnValue[0] = TE_24X;
                    break;
                default:
                    return TE_STATUS_BADREAD;
            }
            break;
        case TE_T1:
            Mask = (TE_BIT1 | TE_BIT0);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            switch(Data & (TE_BIT1 | TE_BIT0))
            {
                case TE_CLEAR_BITS:
                    ReturnValue[0] = TE_1X;
                    break;  
                case TE_BIT0: 
                    ReturnValue[0] = TE_8X;
                    break;  
                case TE_BIT1:
                    ReturnValue[0] = TE_16X;
                    break;
                case (TE_BIT1 | TE_BIT0): 
                    ReturnValue[0] = TE_24X;
                    break;
                default:
                    return TE_STATUS_BADREAD;
            }
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelADPLLSource
 * Returns: TEstatus
 * Description: Configures the source port for the ADPLL reference clock for
 *              the specified Direction.
 ******************************************************************************/
TEstatus IXF3204SetClkSelADPLLSource (void *handle, TEword Direction,
                                      TEword Source)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
        case TE_RX:
            Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr2);
            break;
        case TE_TX:
            Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr3);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Source parameter*/
    switch(Source)
    {
        case TE_CLK0:
            Data = TE_CLEAR_BITS;
            break;
        case TE_CLK1:
            Data = TE_BIT5;
            break;
        case TE_CLK2:
            Data = TE_BIT6;
            break;
        case TE_CLK3:
            Data = (TE_BIT6 | TE_BIT5);
            break;
        case TE_CLK4:
            Data = TE_BIT7;
            break; 
        case TE_CLK5:
            Data = (TE_BIT7 | TE_BIT5);
            break; 
        case TE_CLK6:
            Data = (TE_BIT7 | TE_BIT6);
            break; 
        case TE_CLK7:
            Data = (TE_BIT7 | TE_BIT6 | TE_BIT5);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT7 |TE_BIT6 | TE_BIT5);

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,
                                     (TEword)(Address + IXF3204_ADDR_OFFSET1),
                                     Data, Mask)) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /* perform set operation */
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelADPLLSource
 * Returns: TEstatus
 * Description: Returns the configured source port for the ADPLL reference 
 *              clock for the specified Direction.
 ******************************************************************************/
TEstatus IXF3204GetClkSelADPLLSource(void *handle, TEword Direction,
                                     void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr2);
            break;
        case TE_TX:
            Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr3);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    Mask = (TE_BIT7 |TE_BIT6 | TE_BIT5);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_CLK0;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_CLK1;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_CLK2;
            break;
        case (TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_CLK3;
            break;
        case TE_BIT7:
            ReturnValue[0] = TE_CLK4;
            break;
        case (TE_BIT7 | TE_BIT5):
            ReturnValue[0] = TE_CLK5;
            break;
        case (TE_BIT7 | TE_BIT6):
            ReturnValue[0] = TE_CLK6;
            break;
        case (TE_BIT7 | TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_CLK7;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelIntDistributionSource
 * Returns: TEstatus
 * Description: Configures the port and clock source for the internal clock
 *              generation. 
 ******************************************************************************/
TEstatus IXF3204SetClkSelIntDistributionSource(void *handle, TEword Clk,
                                               TEword Source)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;

    /*validate Clk parameter*/
    switch(Clk)
    {
        case TE_CLKA:
            Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr2);
            break;
        case TE_CLKB:
            Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr3);
            break;
        default:     
            return TE_STATUS_BAD_PARAMS;  
    }

    /*validate Source parameter*/
    switch(Source)
    {
        case TE_RCLK0:
            Data = TE_CLEAR_BITS;
            break;
        case TE_RCLK1:
            Data = TE_BIT0;
            break;
        case TE_RCLK2:
            Data = TE_BIT1;
            break;
        case TE_RCLK3:
            Data = (TE_BIT1 | TE_BIT0);
            break;
        case TE_RCLK4:
            Data = TE_BIT2;
            break;
        case TE_RCLK5:
            Data = (TE_BIT2 | TE_BIT0);
            break;
        case TE_RCLK6:
            Data = (TE_BIT2 | TE_BIT1);
            break;
        case TE_RCLK7:
            Data = (TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        case TE_BTCLK0:
            Data = TE_BIT3;
            break;
        case TE_BTCLK1:
            Data = (TE_BIT3 | TE_BIT0);
            break;
        case TE_BTCLK2:
            Data = (TE_BIT3 | TE_BIT1);
            break;
        case TE_BTCLK3:
            Data = (TE_BIT3 | TE_BIT1 | TE_BIT0);
            break;
        case TE_BTCLK4:
            Data = (TE_BIT3 | TE_BIT2);
            break;
        case TE_BTCLK5:
            Data = (TE_BIT3 | TE_BIT2 | TE_BIT0);
            break; 
        case TE_BTCLK6:
            Data = (TE_BIT3 | TE_BIT2 | TE_BIT1);
            break;
        case TE_BTCLK7:
            Data = TE_BIT0_3;
            break;
        case TE_EXTT1CLK:
            Data = TE_BIT4;
            break;
        case TE_EXTE1CLK:
            Data = (TE_BIT4 | TE_BIT3);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    Mask = (TE_BIT4 | TE_BIT0_3);

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelIntDistributionSource
 * Returns: TEstatus
 * Description: Returns the configured port and source for the internal clock
 *              generation.
 ******************************************************************************/
TEstatus IXF3204GetClkSelIntDistributionSource(void *handle, TEword Clk,
                                               void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*validate Clk parameter*/
    switch(Clk)
    {
        case TE_CLKA:
            Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr2);
            break;
        case TE_CLKB:
            Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr3);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT4 | TE_BIT0_3);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Fvalue parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_RCLK0;
            break;
        case TE_BIT0:
            ReturnValue[0] = TE_RCLK1;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_RCLK2;
            break;
        case (TE_BIT1 | TE_BIT0):
            ReturnValue[0] = TE_RCLK3;
            break;
        case TE_BIT2:
            ReturnValue[0] = TE_RCLK4;
            break;
        case (TE_BIT2 | TE_BIT0):
            ReturnValue[0] = TE_RCLK5;
            break;
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_RCLK6;
            break;
        case (TE_BIT2 | TE_BIT1 | TE_BIT0):
            ReturnValue[0] = TE_RCLK7;
            break;
        case TE_BIT3:
            ReturnValue[0] = TE_BTCLK0;
            break;
        case (TE_BIT3 | TE_BIT0):
            ReturnValue[0] = TE_BTCLK1;
            break;
        case (TE_BIT3 | TE_BIT1):
            ReturnValue[0] = TE_BTCLK2;
            break;
        case (TE_BIT3 | TE_BIT1 | TE_BIT0):
            ReturnValue[0] = TE_BTCLK3;
            break;
        case (TE_BIT3 | TE_BIT2):
            ReturnValue[0] = TE_BTCLK4;
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT0):
            ReturnValue[0] = TE_BTCLK5;
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_BTCLK6;
            break;
        case TE_BIT0_3:
            ReturnValue[0] = TE_BTCLK7;
            break;
        case TE_BIT4:
            ReturnValue[0] = TE_EXTT1CLK;
            break;
        case (TE_BIT4 | TE_BIT3):
            ReturnValue[0] = TE_EXTE1CLK;
            break;
        default :  
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelClkInputRate
 * Returns: TEstatus
 * Description: Configures the ADPLL to know the rate at which the clock is 
 *              being received for the specified Direction- backplane (transmit
 *              only) or line (receive only). This is the rate of the clock
 *              going in to the ADPLL.
 ******************************************************************************/
TEstatus IXF3204SetClkSelClkInputRate(void *handle, TEword Direction,
                                      TEword Rate)
{
    TEbyte Data;
    TEword Address;
    TEbyte Mask;

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr4);

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            /*validate Rate parameter*/
            switch(Rate)
            {
                case TE_R1544:
                    Data = TE_CLEAR_BITS; 
                    break;
                case TE_R2048:
                    Data = TE_BIT2; 
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS;
            }
            Mask = TE_BIT2;
            break;
        case TE_TX:
            switch(Rate)
            {
                case TE_R1536:
                    Data = TE_CLEAR_BITS; 
                    break;
                case TE_R1544:
                    Data = TE_BIT6; 
                    break;
                case TE_R2048:
                    Data = TE_BIT7; 
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS;
            }
            Mask = (TE_BIT7 | TE_BIT6);
            break;
        case TE_BOTH:
            Mask = (TE_BIT7 | TE_BIT6 | TE_BIT2);
            switch(Rate)
            {
                case TE_R1536:
                    Data = TE_CLEAR_BITS;
                    Mask &= ~TE_BIT2;
                    break;
                case TE_R1544:
                    Data = TE_BIT6;
                    break;
                case TE_R2048:
                    Data = (TE_BIT7 | TE_BIT2);
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelClkInputRate
 * Returns: TEstatus
 * Description: Returns the configured clock rate received by the ADPLL from 
 *              the line or backplane depending on the Direction specified.
 ******************************************************************************/
TEstatus IXF3204GetClkSelClkInputRate(void *handle, TEword Direction,
                                      void *Results)
{
    TEbyte Data, Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr4);

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Mask = TE_BIT2;
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            /*validate Data parameter*/
            switch(Data)
            {
                case TE_CLEAR_BITS:
                    ReturnValue[0] = TE_R1544;
                    break;
                case TE_BIT2:
                    ReturnValue[0] = TE_R2048;
                    break;
                default: 
                    return TE_STATUS_BADREAD; 
            }
            break;
        case TE_TX:
            Mask = (TE_BIT7 | TE_BIT6);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            switch(Data)
            {
                case TE_CLEAR_BITS:
                    ReturnValue[0] = TE_R1536;
                    break;
                case TE_BIT6:
                    ReturnValue[0] = TE_R1544;
                    break;
                case TE_BIT7:
                    ReturnValue[0] = TE_R2048;
                    break;
                default: return TE_STATUS_BADREAD;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelClkOutputRate
 * Returns: TEstatus
 * Description: Configures the desired clock rate for the Direction specified.
 ******************************************************************************/
TEstatus IXF3204SetClkSelClkOutputRate(void *handle, TEword Direction,
                                       TEword Rate)
{
    TEbyte Data;
    TEword Address;
    TEbyte Mask;

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr4);

    /* validate the Rate parameter */
    switch(Rate){
        case TE_R1536:
            Data = TE_CLEAR_BITS; 
            break;
        case TE_R1544:
            Data = TE_BIT0;
            break;
        case TE_R2048:
            Data = TE_BIT1; 
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }
    Mask = (TE_BIT1 | TE_BIT0);

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            break;
        case TE_TX:
            Data <<= 4;
            Mask <<= 4;
            break;
        case TE_BOTH:
            Data |= (Data << 4);
            Mask |= (Mask << 4);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelClkOutputRate
 * Returns: TEstatus
 * Description: Returns the configured clock rate configured for the specified 
 *              Direction.
 ******************************************************************************/
TEstatus IXF3204GetClkSelClkOutputRate(void *handle, TEword Direction,
                                       void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = (IXF3204_ClockSelectorOffset + IXF3204_ClkSelectorAddr4);

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Mask = (TE_BIT1 | TE_BIT0);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            /*validate Data parameter*/
            switch(Data)
            {
                case TE_CLEAR_BITS:
                    ReturnValue[0] = TE_R1536;
                    break;
                case TE_BIT0:
                    ReturnValue[0] = TE_R1544;
                    break;
                case TE_BIT1:
                    ReturnValue[0] = TE_R2048;
                    break;
                default: 
                    return TE_STATUS_BADREAD;  
            }
            break;
        case TE_TX:
            Mask = (TE_BIT5 | TE_BIT4);
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            switch(Data)
            {
                case TE_CLEAR_BITS:
                    ReturnValue[0] = TE_R1536;
                    break;
                case TE_BIT4:
                    ReturnValue[0] = TE_R1544;
                    break;
                case TE_BIT5:
                    ReturnValue[0] = TE_R2048;
                    break;
                default:
                    return TE_STATUS_BADREAD; 
            }
            break;
        
        default:   
            return TE_STATUS_BAD_PARAMS;  
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelPortClkSource
 * Returns: TEstatus
 * Description: Selects the clock source for the specified Port and Direction. 
 *              Three clock sources can be selected for each port. One for
 *              transmit to the backplane, one for receive from the backplane 
 *              and one for transmit to the line. Each port and direction can
 *              have a different clock source. This command must be sent for
 *              each port and direction to select the clock source. To
 *              configure all ports and directions, this function must be used
 *              24 times.
 ******************************************************************************/
TEstatus IXF3204SetClkSelPortClkSource(void *handle, TEword Port,
                                       TEword Direction, TEword Source)
{
    TEbyte Data, Data2;
    TEbyte Mask, Mask2;
    TEword Address;
    TEword PerPortRegister;
    TEstatus Status;

    /*Get offsets to perform the get operations*/
    PerPortRegister = IXF3204FindPortClockSelector(Port);
    if(PerPortRegister == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_ClkSelectorAddr0 + PerPortRegister);

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RXBP:
            /*validate Source parameter*/
            switch(Source)
            {
                case TE_RCLK:
                    Data2 = TE_CLEAR_BITS;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_BRCLK:
                    Data2 = TE_BIT2;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_EXTCLKINTA:
                    Data2 = TE_BIT3;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_EXTCLKINTB:
                    Data2 = (TE_BIT3 | TE_BIT2);
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_ADPLL:
                    Data2 = TE_CLEAR_BITS;
                    Data = TE_BIT2;
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS;
            }
            Mask = TE_BIT2;
            Mask2 = (TE_BIT2 | TE_BIT3);
            break;
        case TE_TXBP:
            switch(Source)
            {
                case TE_RCLK:
                    Data2 = TE_CLEAR_BITS;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_BTCLK:
                    Data2 = TE_BIT4;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_EXTCLKINTA:
                    Data2 = TE_BIT5;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_EXTCLKINTB:
                    Data2 = (TE_BIT5 | TE_BIT4);
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_ADPLL:
                    Data2 = TE_CLEAR_BITS;
                    Data = TE_BIT3;
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS;
            }
            Mask = TE_BIT3;
            Mask2 = (TE_BIT5 | TE_BIT4);
            break; 
        case TE_TXLINE:
            switch(Source)
            {
                case TE_RCLK:
                    Data2 = TE_CLEAR_BITS;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_BTCLK:
                    Data2 = TE_BIT0;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_EXTCLKINTA:
                    Data2 = TE_BIT1;
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_EXTCLKINTB:
                    Data2 = (TE_BIT1 | TE_BIT0);
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_ADPLL:
                    Data = TE_BIT1;
                    Data2 = TE_CLEAR_BITS;
                    break;
                default:  
                    return TE_STATUS_BAD_PARAMS;  
            }
            Mask = TE_BIT1;
            Mask2 = (TE_BIT1 | TE_BIT0);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*perform set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /* if TE_ADPLL doesn't need to continue */
    if(Source == TE_ADPLL)
    {
        return TE_STATUS_SUCCESS;
    }

    Address = (TEword)(IXF3204_ClkSelectorAddr1 + PerPortRegister);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data2,Mask2))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data2,Mask2);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelPortClkSource
 * Returns: TEstatus
 * Description: Returns the configured source clock for the specified Port and
 *              Direction.
 ******************************************************************************/
TEstatus IXF3204GetClkSelPortClkSource(void *handle, TEword Port,
                                       TEword Direction, void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEword PerPortRegister;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PerPortRegister = IXF3204FindPortClockSelector(Port);
    if(PerPortRegister == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_ClkSelectorAddr0 + PerPortRegister);

    /*validate Direction parameter*/
    switch (Direction)
    {
        case TE_RXBP:
            Mask = TE_BIT2;
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Data == TE_BIT2)
            {
                ReturnValue[0] = TE_ADPLL;
            }
            else
            {
                Address = (TEword)(IXF3204_ClkSelectorAddr1 + PerPortRegister);
                Mask = (TE_BIT3 | TE_BIT2);
                /*perform get operation*/
                if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                /*validate Data parameter*/
                switch(Data)
                {
                    case TE_CLEAR_BITS:
                        ReturnValue[0] = TE_RCLK;
                        break;
                    case TE_BIT2:
                        ReturnValue[0] = TE_BRCLK;
                        break;
                    case TE_BIT3:
                        ReturnValue[0] = TE_EXTCLKINTA;
                        break;
                    case (TE_BIT3 | TE_BIT2):
                        ReturnValue[0] = TE_EXTCLKINTB;
                        break;
                    default:   
                        return TE_STATUS_BAD_PARAMS;  
                }
            }
            break;   
        case TE_TXBP:
            Mask = TE_BIT3;
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Data == TE_BIT3)
            {
                ReturnValue[0] = TE_ADPLL;
            }
            else
            {
                Address = (TEword)(IXF3204_ClkSelectorAddr1 + PerPortRegister);
                Mask = (TE_BIT5 | TE_BIT4);
                /*perform get operation*/
                if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                /*validate Data parameter*/
                switch(Data)
                {
                    case TE_CLEAR_BITS:
                        ReturnValue[0] = TE_RCLK;
                        break;
                    case TE_BIT4:
                        ReturnValue[0] = TE_BTCLK;
                        break;
                    case TE_BIT5:
                        ReturnValue[0] = TE_EXTCLKINTA;
                        break;
                    case (TE_BIT5 | TE_BIT4):
                        ReturnValue[0] = TE_EXTCLKINTB;
                        break;
                    default:  
                        return TE_STATUS_BAD_PARAMS; 
                }
            }
            break;
        case TE_TXLINE:
            Mask = TE_BIT1;
            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Data == TE_BIT1)
            {
                ReturnValue[0] = TE_ADPLL;
            }
            else
            {
                Address = (TEword)(IXF3204_ClkSelectorAddr1 + PerPortRegister);
                Mask = (TE_BIT1 | TE_BIT0);
                /*perform get operation*/
                if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                /*validate Data parameter*/
                switch(Data)
                {
                    case TE_CLEAR_BITS:
                        ReturnValue[0] = TE_RCLK;
                        break;
                    case TE_BIT0:
                        ReturnValue[0] = TE_BTCLK;
                        break;
                    case TE_BIT1:
                        ReturnValue[0] = TE_EXTCLKINTA;
                        break;
                    case (TE_BIT1 | TE_BIT0):
                        ReturnValue[0] = TE_EXTCLKINTB;
                        break;
                    default:  
                        return TE_STATUS_BAD_PARAMS; 
                }
            }
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;  
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelTxSyncPulseConf
 * Returns: TEstatus
 * Description: Configures the sync source, the rate of the sync and multiframe
 *              sync rate for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetClkSelTxSyncPulseConf(void *handle, TEword Port, 
                                         TEword SyncSource, TEword SyncRate,
                                         TEword MFSyncRate)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;
    TEword PerPortRegister;

    /*Get offsets to perform the get operations*/
    PerPortRegister = IXF3204FindPortClockSelector(Port);
    if(PerPortRegister == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_ClkSelectorAddr2 + PerPortRegister);

    /*validate SyncSource parameter*/
    switch(SyncSource)
    {
        case TE_RXFRAMER:
            Data = TE_CLEAR_BITS;
            break;
        case TE_TXBP:
            Data = TE_BIT0;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    /*validate SyncRate parameter*/
    switch(SyncRate)
    {
        case TE_193:
            break;
        case TE_256:
            Data |= TE_BIT1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    /*validate MFSSyncRate parameter*/
    switch(MFSyncRate)
    {
        case TE_12FRAMES:
            break;
        case TE_16FRAMES:
            Data |= TE_BIT2;
            break;
        case TE_24FRAMES:
            Data |= TE_BIT3;
            break;
        case TE_72FRAMES:
            Data |= (TE_BIT3 | TE_BIT2);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }

    Mask = (TE_BIT0_3);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelTxSyncPulseConf
 * Returns: TEstatus
 * Description: Returns the sync source, the rate of the sync and multiframe 
 *              sync rate for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetClkSelTxSyncPulseConf(void *handle, TEword Port,
                                         void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEword PerPortRegister;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PerPortRegister = IXF3204FindPortClockSelector(Port);
    if(PerPortRegister == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_ClkSelectorAddr2 + PerPortRegister);
    Mask = (TE_BIT0_3);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* Sync Source */
    if (Data & TE_BIT0)
    {
        ReturnValue[0] = TE_TXBP;
    }
    else
    {
        ReturnValue[0] = TE_RXFRAMER;
    }
    /* SyncRate */
    if (Data & TE_BIT1)
    {
        ReturnValue[1] = TE_256;
    }
    else
    {
        ReturnValue[1] = TE_193;
    }

    /* MultiFrame Sync Rate */
    if(Data & TE_BIT3)
    {
        if(Data & TE_BIT2)
        {
            ReturnValue[2] = TE_72FRAMES;
        }
        else
        {
            ReturnValue[2] = TE_24FRAMES;
        }
    } 
    else
    {
        if(Data & TE_BIT2)
        {
            ReturnValue[2] = TE_16FRAMES;
        }else{
            ReturnValue[2] = TE_12FRAMES;
        }
    }
    
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelSyncPulseFreeRun
 * Returns: TEstatus
 * Description: Configures the framing signals to be synchronized to their 
 *              sources or generated from free-run counters for the specified
 *              Port.
 ******************************************************************************/
TEstatus IXF3204SetClkSelSyncPulseFreeRun(void *handle, TEword Port,
                                          TEword BackplaneTx, 
                                          TEword BackplaneRx, TEword TxSync, 
                                          TEword TxMSync)
{
    TEbyte Mask, Data;
    TEstatus Status;
    TEword Address;
    TEword PerPortRegister;

    /*Get offsets to perform the get operations*/
    PerPortRegister = IXF3204FindPortClockSelector(Port);
    if(PerPortRegister == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_ClkSelectorAddr3 + PerPortRegister);

    /*validate BackplaneTx parameter*/
    switch(BackplaneTx)
    {
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_DISABLE:
            Data = TE_BIT3;
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    /*validate BackplaneRx parameter*/
    switch(BackplaneRx)
    {
        case TE_ENABLE:
            break;
        case TE_DISABLE:
            Data |= TE_BIT2;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }

    /*validate TxSync parameter*/
    switch(TxSync)
    {
        case TE_ENABLE:
            break;
        case TE_DISABLE:
            Data |= TE_BIT0;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }

    /*validate TxMSync parameter*/
    switch(TxMSync)
    {
        case TE_ENABLE:
            break;
        case TE_DISABLE:
            Data |= TE_BIT1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    Mask = TE_BIT0_3;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetClkSelSyncPulseFreeRun
 * Returns: TEstatus
 * Description: Returns the framing signal synchronization configuration for 
 *              the specified Port. Framing signals can be synchronized to 
 *              their sources or generated from free-run counters. 
 ******************************************************************************/
TEstatus IXF3204GetClkSelSyncPulseFreeRun(void *handle, TEword Port,
                                          void *Results)
{
    TEbyte Mask, Data;
    TEword Address;
    TEword PerPortRegister;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PerPortRegister = IXF3204FindPortClockSelector(Port);
    if(PerPortRegister == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0_3;
    Address = (TEword)(IXF3204_ClkSelectorAddr3 + PerPortRegister);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /* BackplaneTx */
    if(Data & TE_BIT3)
    {
        ReturnValue[0] = TE_DISABLE;
    }
    else
    {
        ReturnValue[0] = TE_ENABLE;
    }
    /* BackplaneRx */
    if(Data & TE_BIT2)
    {
        ReturnValue[1] = TE_DISABLE;
    }
    else
    {
        ReturnValue[1] = TE_ENABLE;
    }
    /* TxSync */
    if(Data & TE_BIT0)
    {
        ReturnValue[2] = TE_DISABLE;
    }
    else
    {
        ReturnValue[2] = TE_ENABLE;
    }
    /* TxMSync */
    if(Data & TE_BIT1)
    {
        ReturnValue[3] = TE_DISABLE;
    }
    else
    {
        ReturnValue[3] = TE_ENABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetClkSelSyncPulseFreeRunRate
 * Returns: TEstatus
 * Description: Configures the rate for the backplane sync pulse for both 
 *              directions for the specified Port. Sends a sync pulse to the
 *              backplane at the specified transmit rate. Configures the 
 *              backplane to send a sync pulse at the specified receive rate.
 ******************************************************************************/
TEstatus IXF3204SetClkSelSyncPulseFreeRunRate(void *handle, TEword Port,
                                              TEword BackplaneRx, 
                                              TEword BackplaneTx)
{
    TEbyte Mask, Data;
    TEstatus Status;
    TEword Address;
    TEword PerPortRegister;

    /*Get offsets to perform the get operations*/
    PerPortRegister = IXF3204FindPortClockSelector(Port);
    if(PerPortRegister == TE_STATUS_BAD_PARAMS)
    {  
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_ClkSelectorAddr2 + PerPortRegister);

    /*validate BackplaneRx parameter*/
    switch(BackplaneRx)
    {
        case TE_192:
            Data = TE_CLEAR_BITS; 
            break;
        case TE_193:
            Data = TE_BIT4;
            break;
        case TE_256:
            Data = TE_BIT5;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate BackplaneTx parameter*/
    switch(BackplaneTx)
    {
        case TE_192:
            break; 
        case TE_193:
            Data |= TE_BIT6;
            break;
        case TE_256:
            Data |= TE_BIT7;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;  
    }

    Mask = TE_BIT4_7;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}


/*******************************************************************************
 * Function Name: IXF3204GetClkSelSyncPulseFreeRunRate
 * Returns: TEstatus
 * Description: Returns the configured rate for the backplane sync pulse for
 *              both directions for the specified Port. 
 ******************************************************************************/
TEstatus IXF3204GetClkSelSyncPulseFreeRunRate(void *handle, TEword Port,
                                              void *Results)
{
    TEbyte Data, Mask;
    TEword Address;
    TEword PerPortRegister;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PerPortRegister = IXF3204FindPortClockSelector(Port);
    if(PerPortRegister == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_ClkSelectorAddr2 + PerPortRegister);
    Mask = TE_BIT4_7;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT5)
    {
        ReturnValue[0] = TE_256;
    }
    else
    {
        if(Data & TE_BIT4)
        {
            ReturnValue[0] = TE_193;
        }
        else
        {
            ReturnValue[0] = TE_192;
        }
    }

    if(Data & TE_BIT7)
    {
        ReturnValue[1] = TE_256;
    }
    else
    {
        if(Data & TE_BIT6)
        {
            ReturnValue[1] = TE_193;
        }
        else
        {
            ReturnValue[1] = TE_192;
        }
    }
    return TE_STATUS_SUCCESS;
}

