/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   codw3204.c
 * Description: this file contains framer functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetCodewordRxMode
 * Returns: TEstatus
 * Description: Sets the codeword receive mode configuration for the specified
 *              Port. Codeword detection can be enabled or disabled for the
 *              Intel IXF3204. When TE_DISABLE is selected, only Sa6 counters
 *              and loop signals are disabled.
 ******************************************************************************/
TEstatus IXF3204SetCodewordRxMode(void *handle, TEword Port, TEword Mode)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT0;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return (IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetCodewordRxMode
 * Returns: TEstatus
 * Description: Returns the codeword receive mode configuration (enabled or 
 *              disabled) for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetCodewordRxMode(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Validate the Data parameter*/
    switch(Data)
    {
        case TE_BIT0:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default :
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetCodewordRxValue
 * Returns: TEstatus
 * Description: Returns the pattern received in the last multiframe in the 
 *              NationalBit specified for the specified Port. The lower nibble
 *              holds the pattern received in the submultiframe I and the
 *              upper nibble holds the pattern received in the submultiframe II.
 ******************************************************************************/
TEstatus IXF3204GetCodewordRxValue(void *handle, TEword Port, TEword Selection,
                                   void *Results)
{
    TEword Address;
    TEbyte AddrOffset, Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Selection parameter*/
    switch(Selection)
    {
        case TE_SINFAS:
            AddrOffset = IXF3204_NATBITS_SINFR;
            break;
        case TE_SIFAS:
            AddrOffset = IXF3204_NATBITS_SIFR;
            break;
        case TE_SA4:
            AddrOffset = IXF3204_NATBITS_SA4R;
            break;
        case TE_SA5:
            AddrOffset = IXF3204_NATBITS_SA5R;
            break;
        case TE_SA6:
            AddrOffset = IXF3204_NATBITS_SA6R;
            break;
        case TE_SA7:
            AddrOffset = IXF3204_NATBITS_SA7R;
            break;
        case TE_SA8:
            AddrOffset = IXF3204_NATBITS_SA8R;
            break;
        default:
            return  TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(Address + AddrOffset);

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle, Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return Status;
}

/*******************************************************************************
 * Function Name: IXF3204SetCodewordRxCounterMode
 * Returns: TEstatus
 * Description: Enables or disables the CRC counter in Sa6 or the E-bit counter
 *              in Sa6 for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetCodewordRxCounterMode(void *handle, TEword Port, 
                                         TEword Counter, TEword Mode)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Validate the parameter*/
    switch(Mode) 
    {
        case TE_ENABLE:
            Data = TE_BIT1;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Counter parameter*/
    switch(Counter) 
    {
        case TE_CRC:
            Data <<= 1;
            Mask = TE_BIT2;
            break;
        case TE_EBIT:
            Mask = TE_BIT1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return (IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetCodewordRxCounterMode
 * Returns: TEstatus
 * Description: Returns the mode of the Counter specified (enabled or disabled)
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetCodewordRxCounterMode(void *handle, TEword Port, 
                                         TEword Counter,void *Results)
{
    TEword Address;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Counter parameter*/
    switch(Counter)
    {
        case TE_CRC:
            Mask = TE_BIT2;
            break;
        case TE_EBIT:
            Mask = TE_BIT1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Validate the Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT1:
        case TE_BIT2:
            ReturnValue[0] = TE_ENABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetCodewordCounter
 * Returns: TEstatus
 * Description: Returns the value of the specified Counter and Port.
 ******************************************************************************/
TEstatus IXF3204GetCodewordCounter(void *handle, TEword Port,
                                   TEword Counter, void *Results)
{
    TEword Address;
    TEword AddrOffset, Data;
    TEbyte Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Counter parameter*/
    switch(Counter)
    {
        case TE_CRC:
            AddrOffset = IXF3204_NATBITS_CRCCR1;
            break;
        case TE_EBIT:
            AddrOffset = IXF3204_NATBITS_ECR1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(Address + AddrOffset);

    /* perform get operation */
    if((Status = IXF3204ReadWord(handle, Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetCodewordTxMode
 * Returns: TEstatus
 * Description: Enables or disables the Intel IXF3204 to transmit the 
 *              specified Sa/Si bit for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetCodewordTxMode(void *handle, TEword Port, TEword Selection,
                                  TEword Mode)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;

    /* Get offsets to perform the set operations */
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* Validate the Mode parameter */
    switch(Selection)
    {
        case TE_SA4:
            Data = TE_BIT3;
            Mask = TE_BIT3;
            break;
        case TE_SA5:
            Data = TE_BIT4;
            Mask = TE_BIT4;
            break;
        case TE_SA6:
            Data = TE_BIT5;
            Mask = TE_BIT5;
            break;
        case TE_SA7:
            Data = TE_BIT6;
            Mask = TE_BIT6;
            break;
        case TE_SA8:
            Data = TE_BIT7;
            Mask = TE_BIT7;
            break;
        case TE_SIFAS:
            Data = TE_BIT0;
            Mask = TE_BIT0;
            break;
        case TE_SINFAS:
            Data = TE_BIT1;
            Mask = TE_BIT1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Validate the Mode parameter */
    switch(Mode)
    {
        case TE_ENABLE:
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_NATBITS_TXENABLE;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /* perform set operation */
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetCodewordTxMode
 * Returns: TEstatus
 * Description: Returns the Sa/Si bit transmission mode for the S-bit 
 *              specified for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetCodewordTxMode(void *handle, TEword Port, 
                                  TEword Selection, void *Results)
{
    TEword Address;
    TEbyte Data, Mask; 
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /* Get offsets to perform the get operations */
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* Validate the Selection parameter */
    switch(Selection)
    {
        case TE_SA4:
            Mask = TE_BIT3;
            break;
        case TE_SA5:
            Mask = TE_BIT4;
            break;
        case TE_SA6:
            Mask = TE_BIT5;
            break;
        case TE_SA7:
            Mask = TE_BIT6;
            break;
        case TE_SA8:
            Mask = TE_BIT7;
            break;
        case TE_SIFAS:
            Mask = TE_BIT0;
            break;
        case TE_SINFAS:
            Mask = TE_BIT1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_NATBITS_TXENABLE;
    /* perform get operation */
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Validate the Data parameter*/
    if(Data)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetCodewordTxValue
 * Returns: TEstatus
 * Description: Sets the pattern to be transmitted in the specified Sa/Si bit 
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetCodewordTxValue(void *handle, TEword Port, TEword Selection,
                                   TEbyte Value)
{
    TEword Address;
    TEbyte AddrOffset;
    TEstatus Status;

    /* validate Selection parameter */
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* Validate the Selection parameter */
    switch(Selection)
    {
        case TE_SA4:
            AddrOffset = IXF3204_NATBITS_TXSA4;
            break;
        case TE_SA5:
            AddrOffset = IXF3204_NATBITS_TXSA5;
            break;
        case TE_SA6:
            AddrOffset = IXF3204_NATBITS_TXSA6;
            break;
        case TE_SA7:
            AddrOffset = IXF3204_NATBITS_TXSA7;
            break;
        case TE_SA8:
            AddrOffset = IXF3204_NATBITS_TXSA8;
            break;
        case TE_SIFAS:
            AddrOffset = IXF3204_NATBITS_TXSIFAS;
            break;
        case TE_SINFAS:
            AddrOffset = IXF3204_NATBITS_TXSINFAS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    Address = (TEword)(Address + AddrOffset);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Value))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /* perform set operation */
    return(IXF3204WriteByte(handle,Address, Value));
}

/*******************************************************************************
 * Function Name: IXF3204GetCodewordTxValue
 * Returns: TEstatus
 * Description: Returns the pattern being transmitted in the specified Sa/Si 
 *              bit for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetCodewordTxValue(void *handle, TEword Port,
                                   TEword Selection, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte AddrOffset, Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /* Get offsets to perform the get operations */
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /* Validate the Selection parameter */
    switch(Selection)
    {
        case TE_SA4:
            AddrOffset = IXF3204_NATBITS_TXSA4;
            break;
        case TE_SA5:
            AddrOffset = IXF3204_NATBITS_TXSA5;
            break;
        case TE_SA6:
            AddrOffset = IXF3204_NATBITS_TXSA6;
            break;
        case TE_SA7:
            AddrOffset = IXF3204_NATBITS_TXSA7;
            break;
        case TE_SA8:
            AddrOffset = IXF3204_NATBITS_TXSA8;
            break;
        case TE_SIFAS:
            AddrOffset = IXF3204_NATBITS_TXSIFAS;
            break;
        case TE_SINFAS:
            AddrOffset = IXF3204_NATBITS_TXSINFAS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(Address + AddrOffset);

    /* perform get operation */
    if((Status = IXF3204ReadByte(handle, Address, &Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetCodewordStatus
 * Returns: TEstatus
 * Description: Returns the action taken for the last codeword received for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetCodewordStatus(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /* Get offsets to perform the get operations */
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);
    Address += IXF3204_NATBITS_SR;

    /* perform get operation */
    if((Status = IXF3204ReadBit(handle, Address, Mask, &Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

