/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/*******************************************************************************
 * File Name:   ddl3204.c
 * Description: this file contains DDL functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetDDLTxMessage
 * Returns: TEstatus
 * Description: Sets the values to be transmitted in the data link channel 
 *              when the port is configured in T1 SLC-96 mode for the specified
 *              Port. The values are as described below.
 ******************************************************************************/
TEstatus IXF3204SetDDLTxMessage(void *handle, TEword Port, TEbyte Byte1,
                                TEbyte Byte2, TEbyte Byte3)
{
    TEword Address;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXFDL_DDLSR1;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Byte1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        Address++;
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Byte2))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        Address++;
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Byte3))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    if((Status = IXF3204WriteByte(handle,Address,Byte1)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Address++;
    /*perform set operation*/
    if((Status = IXF3204WriteByte(handle,Address,Byte2)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Address++;
    /*perform set operation*/
    if((Status = IXF3204WriteByte(handle,Address,Byte3)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetDDLTxMessage
 * Returns: TEstatus
 * Description: Returns the values being transmitted in the data link channel 
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetDDLTxMessage(void *handle,TEword Port, void *Results)
{
    TEword AddressTx;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    
    AddressTx += IXF3204_TXFDL_DDLSR1;
    /* perform get operation */
    if((Status = IXF3204ReadByte(handle,AddressTx, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    AddressTx++;
    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,AddressTx, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[1] = Data;
    AddressTx++;
    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,AddressTx, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[2] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetDDLRxValid
 * Returns: TEstatus
 * Description: Returns the status of the received message for the specified 
 *              Port.
 ******************************************************************************/
TEstatus IXF3204GetDDLRxValid(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data, Mask = TE_BIT6;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXFDL_DDLSR1;
    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetDDLRxMessage
 * Returns: TEstatus
 * Description: Returns the current message received in the data link channel 
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetDDLRxMessage(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;
    TEbyte Mask = (TE_BIT5 | TE_BIT4 | TE_BIT0_3);

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXFDL_DDLSR1;
    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    Address++;
    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[1] = Data;
    Address++;
    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[2] = Data;

    return TE_STATUS_SUCCESS;
}

