/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File name:   erro3204.c
 * Description: this file contains error insertion functions
 *
 ******************************************************************************/


#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetErrorRate
 * Returns: TEstatus
 * Description: Configures the error rate used when transmitting errors to the 
 *              line for the specified Port.
 *              Rate specifies the amount of time slots to wait before trying
 *              to insert the next error
 ******************************************************************************/
TEstatus IXF3204SetErrorRate(void *handle, TEword Port, TEword Rate)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;

    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Rate parameter*/
    switch(Rate)
    {
        case TE_EXP0:
            Data = TE_CLEAR_BITS;
            break;
        case TE_EXP_1:
            Data = TE_BIT0;
            break;
        case TE_EXP_2:
            Data = TE_BIT1;
            break;
        case TE_EXP_3:
            Data = (TE_BIT1 | TE_BIT0);
            break;
        case TE_EXP_4:
            Data = TE_BIT2;
            break;
        case TE_EXP_5:
            Data = (TE_BIT2 | TE_BIT0);
            break;
        case TE_EXP_6:
            Data = (TE_BIT2 | TE_BIT1);
            break;
        case TE_EXP_0:
            Data = TE_BIT3;
            break;
        case TE_R1:
            Data = (TE_BIT3 | TE_BIT0);
            break;
        case TE_R1_2:
            Data = (TE_BIT3 | TE_BIT1);
            break;
        case TE_R1_3:
            Data = (TE_BIT3 | TE_BIT1 | TE_BIT0);
            break;
        case TE_R1_4:
            Data = (TE_BIT3 | TE_BIT2);
            break;
        case TE_R1_5:
            Data = (TE_BIT3 | TE_BIT2 | TE_BIT0);
            break;
        case TE_R1_6:
            Data = (TE_BIT3 | TE_BIT2 | TE_BIT1);
            break;
        case TE_R1_7:
            Data = TE_BIT0_3;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_LIEI;
    Mask = TE_BIT0_3;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetErrorRate
 * Returns: TEstatus
 * Description: Returns the error rate configured for the specified Port. 
 ******************************************************************************/
TEstatus IXF3204GetErrorRate(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    Address += IXF3204_LIEI;

    Mask = TE_BIT0_3;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_EXP0;
            break;
        case TE_BIT0:
            ReturnValue[0] = TE_EXP_1;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_EXP_2;
            break;
        case (TE_BIT1 | TE_BIT0):
            ReturnValue[0] = TE_EXP_3;
            break;
        case TE_BIT2:
            ReturnValue[0] = TE_EXP_4;
            break;
        case (TE_BIT2 | TE_BIT0):
            ReturnValue[0] = TE_EXP_5;
            break;
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_EXP_6;
            break;
        case TE_BIT3:
            ReturnValue[0]=TE_EXP_0;
            break;
        case (TE_BIT3 | TE_BIT0):
            ReturnValue[0] = TE_R1;
            break;
        case (TE_BIT3 | TE_BIT1):
            ReturnValue[0] = TE_R1_2;
            break;
        case (TE_BIT3 | TE_BIT1 | TE_BIT0):
            ReturnValue[0] = TE_R1_3;
            break;
        case (TE_BIT3 | TE_BIT2):
            ReturnValue[0] = TE_R1_4;
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT0):
            ReturnValue[0] = TE_R1_5;
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_R1_6;
            break;
        case TE_BIT0_3:
            ReturnValue[0] = TE_R1_7;
            break;
        default:  
            return TE_STATUS_BADREAD;  
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetErrorSend
 * Returns: TEstatus
 * Description: Configures the type of error to be sent for the specified Port.
 *              The Error parameter specifies the type of error.
 ******************************************************************************/
TEstatus IXF3204SetErrorSend(void *handle, TEword Port, TEword Error)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;

    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Rate parameter*/
    switch(Error)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_BPV:
            Data = TE_BIT7;
            break;
        case TE_HDB3CV:
            Data = (TE_BIT7 | TE_BIT4);
            break;
        case TE_FAS:   /* or TE_FTFE */
            Data = (TE_BIT7 | TE_BIT5);
            break;
        case TE_NFAS:  /* or TE_FSCRC */
            Data = (TE_BIT7 | TE_BIT5 | TE_BIT4);
            break;
        case TE_MFASCAS:
            Data = (TE_BIT7 | TE_BIT6);
            break;
        case TE_MFASCRC4:
            Data = (TE_BIT7 | TE_BIT6 | TE_BIT4);
            break;
        case TE_SMFICRC4:
            Data = (TE_BIT7 | TE_BIT6 | TE_BIT5);
            break;
        case TE_SMFIICRC4:
            Data = TE_BIT4_7;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }

    Address += IXF3204_LIERR;
    Mask = TE_BIT4_7;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetErrorSend
 * Returns: TEstatus
 * Description: Returns the type of error transmitted for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetErrorSend(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask; 
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offset to do get operation*/
    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    Address += IXF3204_LIERR;
    Mask = TE_BIT4_7;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT7:
            ReturnValue[0] = TE_BPV;
            break;
        case (TE_BIT7 | TE_BIT4):
            ReturnValue[0] = TE_HDB3CV;
            break;
        case (TE_BIT7 | TE_BIT5):
            ReturnValue[0] = TE_FAS;
            break;
        case (TE_BIT7 | TE_BIT5 | TE_BIT4):
            ReturnValue[0] = TE_NFAS;
            break;
        case (TE_BIT7 | TE_BIT6):
            ReturnValue[0] = TE_MFASCAS;
            break;
        case (TE_BIT7 | TE_BIT6 | TE_BIT4):
            ReturnValue[0] = TE_MFASCRC4;
            break;
        case (TE_BIT7 | TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_SMFICRC4;
            break;
        case TE_BIT4_7:
            ReturnValue[0] = TE_SMFIICRC4;
            break;
        default:
            return TE_STATUS_BADREAD;  
    }

    return TE_STATUS_SUCCESS;
}


