/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   FDL3204.c
 * Description: this file contains FDL functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetFDLResetFIFO
 * Returns: TEstatus
 * Description: Clears and reset the FIFO and sets the FDL module in its 
 *              initial state for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204SetFDLResetFIFO(void *handle, TEword Port, TEword Direction)
{
    TEbyte Mask;
    TEword Address;
    TEstatus Status;

    switch(Direction)
    {
        case TE_RX:
            /*get offsets to perform the set operations*/
            Address = IXF3204FindPortRxFDL(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_TX:
            /*Get offsets to perform the set operations*/
            Address = IXF3204FindPortTxFDL(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }
    Mask = TE_BIT1;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,TE_BIT1,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,TE_BIT1,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLDataInv
 * Returns: TEstatus
 * Description: Enables or disables the inversion or the data to be transmitted
 *              and received for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204SetFDLDataInv(void *handle, TEword Port, TEword Direction,
                              TEword Mode)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;
    TEword AddressBoth = TE_CLEAR_BITS;

    /*Validate the Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT4;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
            AddressBoth = IXF3204FindPortTxFDL(Port);
            if(AddressBoth == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
        case TE_RX:
            Address = IXF3204FindPortRxFDL(Port);
            break;        
        case TE_TX:
            /*Get offsets to perform the write operations*/
            Address = IXF3204FindPortTxFDL(Port);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT4;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressBoth,Data,
                                                     Mask)) != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        if((Status = IXF3204WriteBit(handle,AddressBoth,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLDataInv
 * Returns: TEstatus
 * Description: Returns the configured options of the FDL data inversion for 
 *              the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetFDLDataInv(void *handle, TEword Port, TEword Direction,
                              void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Validate the Direction parameter*/    
    switch(Direction)
    {
        case TE_RX:
            /*Get offsets to perform the write operations*/
            Address = IXF3204FindPortRxFDL(Port);

            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_TX:
            Address = IXF3204FindPortTxFDL(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT4;
    
    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Validate the Direction parameter*/
    switch(Data)
    {
        case TE_BIT4:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204SetFDLE1BitMask
 * Returns: TEstatus
 * Description: Specifies the national bits mask of time slot 0 on an E1 frame 
 *              that carries FDL for the specified Port and Direction. 
 ******************************************************************************/
TEstatus IXF3204SetFDLE1BitMask(void *handle, TEword Port, TEword Direction,
                                TEbyte SaMask)
{
    TEbyte Mask;
    TEstatus Status;
    TEword Address, AddressBoth = TE_CLEAR_BITS;
    TEword PartialAddress;

    /*Validate the Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
            PartialAddress = IXF3204FindPortTxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            AddressBoth = (TEword)(IXF3204_TXFDL_DLCR + PartialAddress);
        case TE_RX:
            PartialAddress = IXF3204FindPortRxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_TXFDL_HCR + PartialAddress);
            break;
        case TE_TX:
            PartialAddress = IXF3204FindPortTxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_TXFDL_DLCR + PartialAddress);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    if (SaMask > (TE_BIT4 | TE_BIT0_3))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT4 | TE_BIT0_3);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,SaMask,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressBoth,SaMask,
                                                     Mask)) != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        if((Status = IXF3204WriteBit(handle,AddressBoth,SaMask,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
     
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,SaMask,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLE1BitMask
 * Returns: TEstatus
 * Description: Returns the bit(s) that is (are) used to carry FDL data for 
 *              the specified Port and Direction. A 1 indicates that the bit
 *              is being used to carry FDL data.
 ******************************************************************************/
TEstatus IXF3204GetFDLE1BitMask(void *handle, TEword Port, TEword Direction,
                                void *Results)
{
    TEbyte Data ;
    TEbyte Mask;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Mask = TE_BIT4 | TE_BIT3 | TE_BIT2 | TE_BIT1 | TE_BIT0;

    /*Validate the Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            PartialAddress = IXF3204FindPortRxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_TXFDL_HCR + PartialAddress);
            break;
        case TE_TX:
            PartialAddress = IXF3204FindPortTxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_TXFDL_DLCR + PartialAddress);
            break;
        default:       
            return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLT1Rate
 * Returns: TEstatus
 * Description: Configures the rate of the FDL in a T1 frame for the specified
 *              Port and Direction.
 ******************************************************************************/
TEstatus IXF3204SetFDLT1Rate(void *handle, TEword Port, TEword Direction,
                             TEword Rate)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address, AddressBoth = TE_CLEAR_BITS;
    TEword PartialAddress;

    /*Validate the Direction parameter*/
    switch(Direction)
    {
        case TE_BOTH:
            PartialAddress = IXF3204FindPortTxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            AddressBoth = (TEword)(IXF3204_TXFDL_DLCR + PartialAddress);
        case TE_RX:
            PartialAddress = IXF3204FindPortRxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_RXFDL_DLCR + PartialAddress);
            break;
        case TE_TX:
            PartialAddress = IXF3204FindPortTxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_TXFDL_DLCR + PartialAddress);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    /*Validate the Direction parameter*/
    switch(Rate)
    {
        case TE_4K:
            Data = TE_CLEAR_BITS;
            break;
        case TE_2KODD:
            Data = TE_BIT5;
            break;
        case TE_2KEVEN:
            Data = TE_BIT6;
            break;   
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    Mask = (TE_BIT6 | TE_BIT5);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressBoth,Data,
                                                     Mask)) != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        if((Status = IXF3204WriteBit(handle,AddressBoth,Data,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLT1Rate
 * Returns: TEstatus
 * Description: Returns the configured rate of the FDL in a T1 frame for the 
 *              specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetFDLT1Rate(void *handle, TEword Port, TEword Direction,
                             void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Mask = (TE_BIT6 | TE_BIT5);

    /*Validate the Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            PartialAddress = IXF3204FindPortRxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_RXFDL_DLCR + PartialAddress);
            break;
        case TE_TX:
            PartialAddress = IXF3204FindPortTxFDL(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_TXFDL_DLCR + PartialAddress);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }


    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Validate the Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_4K;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_2KODD;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_2KEVEN;
            break;
        default: 
            return TE_STATUS_BADREAD; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLTxStatus
 * Returns: TEstatus
 * Description: Returns the status of the transmitting MOP for the specified 
 *              Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLTxStatus(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte  Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the write operations*/
    PartialAddress = IXF3204FindPortTxFDL(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Perform the get operation*/
    Address = (TEword)(IXF3204_TXFDL_IR + PartialAddress);
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLLinkIDV52Enable
 * Returns: TEstatus
 * Description: Enables or disables the detection of the link ID V5.2 protocol 
 *              for the specified Port. This protocol works only if the device 
 *              is in E1 mode.
 ******************************************************************************/
TEstatus IXF3204SetFDLLinkIDV52Enable(void * handle, TEword Port, TEword Mode)
{ 
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    switch(Mode) 
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_ENABLE:
            Data = TE_BIT3;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }

    Mask = TE_BIT3;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLLinkIDV52Enable
 * Returns: TEstatus
 * Description: Returns configured mode for detection of link ID V5.2 protocol.
 *              This protocol works only if the device is in E1 mode.
 ******************************************************************************/
TEstatus IXF3204GetFDLLinkIDV52Enable(void * handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortNational(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    Mask = TE_BIT3;
    Status = IXF3204ReadBit(handle,Address, Mask, &Data);
    if(Status!= TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data) 
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

