/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   fram3204.c
 * Description: this file contains framer functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetFramerType
 * Returns: TEstatus
 * Description: Configures the Framer Byte Format and the Framer Mode for the 
 *              specified Port.
 *              Also configures the slip buffer according to the format 
 *              specified. 
 ******************************************************************************/
TEstatus IXF3204SetFramerType(void *handle, TEword Port, TEword Format,
                              TEword Mode)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;

    /* configures the Rx Framer with the selected mode */
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Format)
    {
        case TE_E1:
            /* validate Mode parameter*/
            switch(Mode)
            {
                case TE_FAS:
                    Data = TE_BIT3;
                    break;
                case TE_CRC4:
                    Data = (TE_BIT5 | TE_BIT3);
                    break;
                case TE_CAS:
                    Data = (TE_BIT6 | TE_BIT3);
                    break;
                case TE_CRC4CAS:
                    Data = (TE_BIT6 | TE_BIT5 | TE_BIT3);
                    break;
                case TE_TRANS:
                    Data = (TE_BIT7 | TE_BIT3);
                    break;
                case TE_UNFRAMED:
                    Data = (TE_BIT7 | TE_BIT3);
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_T1:
            /* validate Mode parameter*/
            switch(Mode)
            {
                case TE_D4:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_ESF:
                    Data = TE_BIT5;
                    break;
                case TE_SLC96:
                    Data = TE_BIT6;
                    break;
                case TE_TRANS:
                    Data = TE_BIT7;
                    break;
                case TE_UNFRAMED:
                    Data = TE_BIT7;
                    break;
                default:  
                    return TE_STATUS_BAD_PARAMS;   
            }
            break;
        case TE_J1:
            /* validate Mode parameter*/
            switch(Mode)
            {
                case TE_J12:
                    Data = TE_BIT4;
                    break;
                case TE_J24:
                    Data = (TE_BIT5 | TE_BIT4);
                    break;
                case TE_TRANS:
                    Data = (TE_BIT7 | TE_BIT4);
                    break;
                case TE_UNFRAMED:
                    Data = (TE_BIT7 | TE_BIT4);
                    break;
                default:  
                    return TE_STATUS_BAD_PARAMS;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    
    Mask = (TE_BIT4_7 | TE_BIT3);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*perform set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /* configures the Tx Framer with the selected mode */
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Format)
    {
        case TE_E1:
            /* validate Mode parameter*/
            switch(Mode)
            {
                case TE_FAS:
                    Data = TE_BIT5;
                    break;
                case TE_CRC4:
                    Data = (TE_BIT5 | TE_BIT2);
                    break;
                case TE_CAS:
                    Data = (TE_BIT5 | TE_BIT3);
                    break;
                case TE_CRC4CAS:
                    Data = (TE_BIT5 | TE_BIT3 | TE_BIT2);
                    break;
                case TE_TRANS:
                    Data = (TE_BIT5 | TE_BIT4);
                    break;
                case TE_UNFRAMED:
                    Data = (TE_BIT5 | TE_BIT4);
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_T1:
            /* validate Mode parameter*/
            switch(Mode)
            {
                case TE_D4:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_ESF:
                    Data = TE_BIT2;
                    break;
                case TE_SLC96:
                    Data = TE_BIT3;
                    break;
                case TE_TRANS:
                    Data = TE_BIT4;
                    break;
                case TE_UNFRAMED:
                    Data = TE_BIT4;
                    break;
                default:  
                    return TE_STATUS_BAD_PARAMS;  
            }
            break;
        case TE_J1:
            /* validate Mode parameter*/
            switch(Mode)
            {
                case TE_J12:
                    Data = TE_BIT6;
                    break;
                case TE_J24:
                    Data = (TE_BIT6 | TE_BIT2);
                    break;
                case TE_TRANS:
                    Data = (TE_BIT6 | TE_BIT4);
                    break;
                case TE_UNFRAMED:
                    Data = (TE_BIT6 | TE_BIT4);
                    break;
                default:  return TE_STATUS_BAD_PARAMS;   
            }
            break;
        default:
            ;
    }

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT2);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*perform set operation*/
        if((Status =IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /* configures the slip buffer with the selected mode */
    Address = IXF3204FindPortSlipBuffer(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* validate Format parameter*/
    switch(Format)
    {
        case TE_T1:
            Data = TE_CLEAR_BITS;
            break;
        case TE_J1:
            Data = TE_CLEAR_BITS;
            break;
        case TE_E1:
            Data = TE_BIT0;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;   
    }

    Mask = TE_BIT0;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        Address += IXF3204_SB_SBUSET;
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*perform set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        Address += IXF3204_SB_SBUSET;
        
        /*perform set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }


    /* configures the module to discard the F bit when Mode is different than
     * TE_UNFRAME
     */
    Address = IXF3204FindPortRxBusIF(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    if(Mode == TE_UNFRAMED)
    {
        Data = TE_BIT7;
    }
    else
    {
        Data = TE_CLEAR_BITS;
    }

    Mask = TE_BIT7;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*perform set operation*/
        if((Status =IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /* configures the module to discard the F bit when Mode is different than
     * TE_UNFRAME
     */
    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Mask = TE_BIT7;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerType
 * Returns: TEstatus
 * Description: Returns the framer byte format and the framer mode for the
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerType(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Mask;
    TEbyte Data;
    TEword verify_mode, verify_type;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Mask = (TE_BIT4_7 | TE_BIT3);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* validate Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_T1;
            ReturnValue[1] = TE_D4;
            break;
        case TE_BIT3:
            ReturnValue[0] = TE_E1;
            ReturnValue[1] = TE_FAS;
            break;
        case TE_BIT4:
            ReturnValue[0] = TE_J1;
            ReturnValue[1] = TE_J12;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_T1;
            ReturnValue[1] = TE_ESF;
            break;
        case (TE_BIT5 | TE_BIT3):
            ReturnValue[0] = TE_E1;
            ReturnValue[1] = TE_CRC4;
            break;
        case (TE_BIT5 | TE_BIT4):
            ReturnValue[0] = TE_J1;
            ReturnValue[1] = TE_J24;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_T1;
            ReturnValue[1] = TE_SLC96;
            break;
        case (TE_BIT6 | TE_BIT3):
            ReturnValue[0] = TE_E1;
            ReturnValue[1] = TE_CAS;
            break;
        case (TE_BIT6 | TE_BIT5 | TE_BIT3):
            ReturnValue[0] = TE_E1;
            ReturnValue[1] = TE_CRC4CAS;
            break;
        case TE_BIT7:
            ReturnValue[0] = TE_T1;
            ReturnValue[1] = TE_TRANS;
            break;
        case (TE_BIT7 | TE_BIT3):
            ReturnValue[0] = TE_E1;
            ReturnValue[1] = TE_TRANS;
            break;
        case (TE_BIT7 | TE_BIT4):
            ReturnValue[0] = TE_J1;
            ReturnValue[1] = TE_TRANS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }

    /* Check Tx Framer */
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT2);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* validate Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            verify_mode = (TEword)(ReturnValue[0] ^ TE_T1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_D4);
            break;
        case TE_BIT2:
            verify_mode = (TEword)(ReturnValue[0] ^ TE_T1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_ESF);
            break;
        case TE_BIT3:
            verify_mode = (TEword)(ReturnValue[0] ^ TE_T1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_SLC96);
            break;
        case TE_BIT4:
            verify_mode = (TEword)(ReturnValue[0] ^ TE_T1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_TRANS);
            break;
        case TE_BIT5:
            verify_mode = (TEword)(ReturnValue[0] ^ TE_E1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_FAS);
            break;
        case (TE_BIT5 | TE_BIT2):
            verify_mode = (TEword)(ReturnValue[0] ^ TE_E1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_CRC4);
            break;
        case (TE_BIT5 | TE_BIT3):
            verify_mode = (TEword)(ReturnValue[0] ^ TE_E1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_CAS);
            break;
        case (TE_BIT5 | TE_BIT3 | TE_BIT2):
            verify_mode = (TEword)(ReturnValue[0] ^ TE_E1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_CRC4CAS);
            break;
        case (TE_BIT5 | TE_BIT4):
            verify_mode = (TEword)(ReturnValue[0] ^ TE_E1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_TRANS);
            break;
        case TE_BIT6:
            verify_mode = (TEword)(ReturnValue[0] ^ TE_J1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_J12);
            break;
        case (TE_BIT6 | TE_BIT2):
            verify_mode = (TEword)(ReturnValue[0] ^ TE_J1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_J24);
            break;
        case (TE_BIT6 | TE_BIT4):
            verify_mode = (TEword)(ReturnValue[0] ^ TE_J1);
            verify_type = (TEword)(ReturnValue[1] ^ TE_TRANS);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }

    /* verify that Rx framer an Tx framer are configured in the same mode */
    if (verify_mode || verify_type)
    {
        return TE_STATUS_BAD_CONFIG;
    }

    /* get the slip buffer address */
    Address = IXF3204FindPortSlipBuffer(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Mask = TE_BIT0;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }


    switch(Data)
    {
        case TE_CLEAR_BITS:
            verify_mode = (TEword)((ReturnValue[0] ^ TE_T1) 
                                   && (ReturnValue[0] ^ TE_J1));
            break;
        case TE_BIT0:
            verify_mode = (TEword)(ReturnValue[0] ^ TE_E1);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;  
    }

    /* verify that rx slip buffer is configured in the same mode */
    if (verify_mode)
    {
        return TE_STATUS_BAD_CONFIG;
    }

    Address += IXF3204_SB_SBUSET;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(Data)
    {
        case TE_CLEAR_BITS:
            verify_mode = (TEword)((ReturnValue[0] ^ TE_T1) 
                                   && (ReturnValue[0] ^ TE_J1));
            break;
        case TE_BIT0:
            verify_mode = (TEword)(ReturnValue[0] ^ TE_E1);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;  
    }

    /* verify that tx slip buffer is configured in the same mode */
    if (verify_mode)
    {
        return TE_STATUS_BAD_CONFIG;
    }

    /* verify if configured unframed */
    Address = IXF3204FindPortRxBusIF(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT7)
    {
        if(ReturnValue[1] == TE_TRANS)
        {
            ReturnValue[1] = TE_UNFRAMED;
        }
        else
        {
            return TE_STATUS_BAD_CONFIG;
        }
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerTxCRCTransparent
 * Returns: TEstatus
 * Description: Sets the source for the CRC (4 or 6) framing information for 
 *              the specified Port. The CRC (4 or 6) framing information can
 *              be generated by the framer or it can come from the backplane.
 *              When Value is TE_ENABLE the framer will not generates the CRC 
 *              information
 ******************************************************************************/
TEstatus IXF3204SetFramerTxCRCTransparent(void *handle, TEword Port,
                                          TEword Value)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* validate Value parameter*/
    switch(Value)
    {
        case TE_ENABLE:
            Data = TE_BIT1;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    Mask = TE_BIT1;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}


/*******************************************************************************
 * Function Name: IXF3204GetFramerTxCRCTransparent
 * Returns: TEstatus
 * Description: Returns the source for the CRC (4 or 6) framing information
 *              framer or backplane, if TE_ENABLE means that the CRC 
 *              information is not generated by the framer
 ******************************************************************************/
TEstatus IXF3204GetFramerTxCRCTransparent(void *handle,TEword Port,
                                          void *Results) 
{
    TEstatus Status;
    TEword Address;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT1;
    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if (Data & TE_BIT1)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }
    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204SetFramerTxCASAlignment
 * Returns: TEstatus
 * Description: Sets the multiframe structure to be aligned to FAS or NFAS for
 *              the specified Port. This only applies when CAS is enabled. See
 *              IXF3204SetFramerType for enabling CAS.
 ******************************************************************************/
TEstatus IXF3204SetFramerTxCASAlignment(void *handle,TEword Port, TEword Value)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* validate Value parameter*/
    switch(Value)
    {
        case TE_FAS:
            Data = TE_CLEAR_BITS;
            break;
        case TE_NFAS:
            Data = TE_BIT0;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }

    Mask = TE_BIT0;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerTxCASAlignment
 * Returns: TEstatus
 * Description: Returns the transmit framer alignment (FAS or NFAS) for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerTxCASAlignment(void *handle, TEword Port,
                                        void *Results)
{
    TEstatus Status;
    TEword Address;
    TEbyte Data;
    TEbyte Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Mask = TE_BIT0;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT0)
    {
        ReturnValue[0] = TE_NFAS;
    }
    else
    {
        ReturnValue[0] = TE_FAS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerTxMFPulseSource
 * Returns: TEstatus
 * Description: Configures the source of the multiframe pulse for the specified
 *              Port. The source can be the line or the backplane (system).
 ******************************************************************************/
TEstatus IXF3204SetFramerTxMFPulseSource(void *handle, TEword Port,
                                         TEword Source)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(Address + IXF3204_TXFRAMER_MP);

    /* validate Source parameter*/
    switch(Source)
    {
        case TE_LINE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_SYS:
            Data = TE_BIT7;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT7;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerTxMFPulseSource
 * Returns: TEstatus
 * Description: Returns source of the multiframe pulse for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerTxMFPulseSource(void *handle, TEword Port,
                                         void *Results)
{
    TEstatus Status;
    TEword Address;
    TEbyte Data;
    TEbyte Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(Address + IXF3204_TXFRAMER_MP);
    Mask = TE_BIT7;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT7)
    {
        ReturnValue[0] = TE_SYS;
    }
    else
    {
        ReturnValue[0] = TE_LINE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerRxMFPulseAlignment
 * Returns: TEstatus
 * Description: Configures the alignment for the system multiframe pulse for 
 *              the specified Port and Direction when the frame format is
 *              CRC4 or CAS.
 ******************************************************************************/
TEstatus IXF3204SetFramerRxMFPulseAlignment(void *handle, TEword Port,
                                            TEword Direction, TEword Value)
{
    TEbyte Data, Data2 = TE_CLEAR_BITS;
    TEstatus Status;
    TEbyte Mask, Mask2 = TE_CLEAR_BITS;
    TEword Address;
    TEword Address2 = TE_CLEAR_BITS;

    /* validate Direction parameter*/
    switch(Direction)
    {
        case TE_TX:
            /*Get offsets to perform the set operations*/
            Address = IXF3204TxFindPort(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;  
            }
            Address = (TEword)(Address + IXF3204_TXFRAMER_MP);

            /* validate Value parameter*/
            switch(Value)
            {
                case TE_CRC4:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_CAS:
                    Data = TE_BIT6;
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS; 
            }

            Mask = TE_BIT6;

            break;
        case TE_RX:
            /*Get offsets to perform the set operations*/
            Address = IXF3204RxFindPort(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

            /* validate Value parameter*/
            switch(Value)
            {
                case TE_CAS:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_CRC4:
                    Data = TE_BIT2;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;                
            }

            Mask = TE_BIT2;

            break;
        case TE_BOTH:
            /*Get offsets to perform the set operations*/
            Address = IXF3204TxFindPort(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(Address + IXF3204_TXFRAMER_MP);

            /* validate Value parameter*/
            switch(Value)
            {
                case TE_CRC4:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_CAS:
                    Data = TE_BIT6;
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS; 
            }
            Mask = TE_BIT6;

            /*Get offsets to perform the set operations*/
            Address2 = IXF3204RxFindPort(Port);
            if(Address2 == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address2 = (TEword)(IXF3204_RXFRAMER_CR3 + Address2);

            /* validate Value parameter*/
            switch(Value)
            {
                case TE_CAS:
                    Data2 = TE_CLEAR_BITS;
                    break;
                case TE_CRC4:
                    Data2 = TE_BIT2;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            Mask2 = TE_BIT2;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;   
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if(Direction == TE_BOTH)
        {
            /* perform set operation */
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address2,Data2,
                                                     Mask2)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }  
        return TE_STATUS_SUCCESS;
    }

    if(Direction == TE_BOTH)
    {
        /* perform set operation */
        if((Status = IXF3204WriteBit(handle,(TEword)Address2,Data2,Mask2)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }  
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerRxMFPulseAlignment
 * Returns: TEstatus
 * Description: Returns the current configuration of the system multiframe
 *              pulse alignment for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetFramerRxMFPulseAlignment(void *handle, TEword Port,
                                            TEword Direction, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte Data;
    TEbyte Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /* validate Direction parameter*/
    switch(Direction)
    {
        case TE_TX:
            /*Get offsets to perform the set operations*/
            Address = IXF3204TxFindPort(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;  
            }
            Address++;
            Mask = TE_BIT6;

            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if (Data & TE_BIT6)
            {
                ReturnValue[0] = TE_CAS;
            }
            else
            {
                ReturnValue[0] = TE_CRC4;
            }
            break;
        case TE_RX:
            /*Get offsets to perform the set operations*/
            PartialAddress = IXF3204RxFindPort(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(IXF3204_RXFRAMER_CR3 + PartialAddress);
            Mask = TE_BIT2;

            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if (Data & TE_BIT2)
            {
                ReturnValue[0] = TE_CRC4;
            }
            else
            {
                ReturnValue[0] = TE_CAS;
            }
            break;
        case TE_BOTH:
            /*Get offsets to perform the set operations*/
            Address = IXF3204TxFindPort(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            Address++;

            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,TE_BIT6,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if (Data & TE_BIT6)
            {
                ReturnValue[0] = TE_CAS;
            }
            else
            {
                ReturnValue[0] = TE_CRC4;
            }

            /*Get offsets to perform the set operations*/
            PartialAddress = IXF3204RxFindPort(Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(IXF3204_RXFRAMER_CR3 + PartialAddress);

            /*perform get operation*/
            if((Status = IXF3204ReadBit(handle,Address,TE_BIT2,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Data & TE_BIT2)
            {
                ReturnValue[1] = TE_CRC4;
            }
            else
            {
                ReturnValue[1] = TE_CAS;
            }
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerAutoResync
 * Returns: TEstatus
 * Description: Configures framer resynchronization for the specified Port 
 *              when it is out of frame. When Mode is TE_ENABLE, the framer 
 *              tries to resynchronize to the incoming data stream whenever it 
 *              is out of frame.
 ******************************************************************************/
TEstatus IXF3204SetFramerAutoResync(void *handle, TEword Port,TEword Mode)
{
    TEbyte Data;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* validate Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT2;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,TE_BIT2))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,TE_BIT2);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerAutoResync
 * Returns: TEstatus
 * Description: Returns status of framer auto-resynchronization when OOF for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerAutoResync(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT2,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT2)
    {
        ReturnValue[0]= TE_ENABLE;
    }
    else
    {
        ReturnValue[0]= TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerRxForceOOF
 * Returns: TEstatus
 * Description: Forces the framer to go to OOF for the specified Port. When
 *              Auto resync is enabled, the framer searches for new frame
 *              synchronization.
 ******************************************************************************/
TEstatus IXF3204SetFramerRxForceOOF(void *handle, TEword Port)
{
    TEbyte Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR1 + Address);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,TE_BIT6,TE_BIT6))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    if((Status = IXF3204WriteBit(handle,Address,TE_BIT6,TE_BIT6)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerRxCRC
 * Returns: TEstatus
 * Description: Configures the receive framer to validate the incoming data 
 *              with CRC for the specified Port. When CRC validation is
 *              selected, the third parameter configures whether a CRC error
 *              can cause OOF. When the CRC parameter is Disable, CRCOOF must
 *              be set to TE_NA.
 ******************************************************************************/
TEstatus IXF3204SetFramerRxCRC(void *handle, TEword Port, TEword crc,
                               TEword crcoof)
{
    TEbyte Data, Data2;
    TEword Address;
    TEword Address2;
    TEstatus Status;

    Data2 = TE_CLEAR_BITS;

    /*Get offsets to perform the set operations*/
    Address2 = IXF3204RxFindPort(Port);
    if(Address2 == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* validate the CRC parameter */
    switch(crc)
    {
        case TE_DISABLE:
            Address = (TEword)(Address2);
            Data = TE_CLEAR_BITS;
            /*perform set operation*/
            break;
        case TE_ENABLE:
            Address = (TEword)(Address2);
            Address2 = (TEword)(IXF3204_RXFRAMER_CR2 + Address2);
            Data = TE_BIT1;
            /* validate the CRCOOF parameter */
            switch(crcoof)
            {
                case TE_DISABLE:
                    Data2 = TE_CLEAR_BITS;
                    break;
                case TE_ENABLE:
                    Data2 = TE_BIT0;
                    break;
                case TE_NA: /* if CRC was enable, CRCOOF must be defined */
                default:
                    return TE_STATUS_BAD_PARAMS;  
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;  
    }


    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,TE_BIT1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        /* if CRC is Enable write CRCOOF */
        if(crc == TE_ENABLE)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address2,
                                                     Data2,TE_BIT0)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        return TE_STATUS_SUCCESS;
    }
    /* if CRC is Enable write CRCOOF */
    if(crc == TE_ENABLE)
    {
        if((Status = IXF3204WriteBit(handle,Address2,Data2,TE_BIT0)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    return IXF3204WriteBit(handle,Address,Data,TE_BIT1);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerRxCRC
 * Returns: TEstatus
 * Description: Returns the framer CRC configuration for incoming data
 *              validation and OOF error generation based on CRC for the
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerRxCRC(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT1,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(!Data)
    {
        ReturnValue[0] = TE_DISABLE;
        ReturnValue[1] = TE_NA;
        return TE_STATUS_SUCCESS;
    }

    /* CRC is enable */
    ReturnValue[0] = TE_ENABLE;

    Address = (TEword)(IXF3204_RXFRAMER_CR2 + Address);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT0,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT0)
    {
        ReturnValue[1] = TE_ENABLE;
    }
    else
    {
        ReturnValue[1] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerRxFbitSync
 * Returns: TEstatus
 * Description: Configures the receive framer synchronization for the 
 *              specified Port to look for Ft and Fs bits or only the Ft bit. 
 *              This function also configures the number of F-bits needed to
 *              declare sync. 
 ******************************************************************************/
TEstatus IXF3204SetFramerRxFbitSync(void *handle, TEword Port, TEword Value,
                                    TEword FtSelected)
{
    TEbyte Data, Data1;
    TEstatus Status;
    TEword Address, Address1;

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* validate FtSelected parameter*/
    switch(FtSelected)
    {
        case TE_NA:
        case TE_FTFS:
            Data = TE_CLEAR_BITS;
            break;
        case TE_FT:
            Data = TE_BIT0;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;   
    }

    Address1 = (TEword)(IXF3204_RXFRAMER_CR2 + Address);

    /* validate Value parameter*/
    switch(Value)
    {
        case TE_F10:
            Data1 = TE_CLEAR_BITS;
            break;
        case TE_F24:
            Data1 = TE_BIT1;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if(FtSelected != TE_NA)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,
                                                     Data,TE_BIT0)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address1,Data1,TE_BIT1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    if(FtSelected != TE_NA )
    {
        /*perform set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,TE_BIT0)) !=
            TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address1,Data1,TE_BIT1);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerRxFbitSync
 * Returns: TEstatus
 * Description: Returns the receive framer F-bit synchronization configuration
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerRxFbitSync(void *handle, TEword Port, void *Results)
{
    TEword Address, PartialAddress;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204RxFindPort(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR2 + PartialAddress);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT1,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT1)
    {
        ReturnValue[0] = TE_F24;
    }
    else
    {
        ReturnValue[0] = TE_F10;
    }

    Address = (TEword)(PartialAddress);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT0,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT0)
    {
        ReturnValue[1] = TE_FT;
    }
    else
    {
        ReturnValue[1] = TE_FTFS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerT1OOFCriteria
 * Returns: TEstatus
 * Description: Sets the criteria to declare Loss of Frame alignment for the
 *              specified Port. When "Errors" F-bits have been found in error
 *              in a window of "Window" frames, the framer declares LOF. It is
 *              important to note that "Window" must be greather than "Errors".
 ******************************************************************************/
TEstatus IXF3204SetFramerT1OOFCriteria(void *handle, TEword Port, TEword Window,
                                       TEword Errors)
{
    TEbyte Mask, Data;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR2 + Address);
    /* Window must be between TE_WIN1 and TE_WIN7 */
    switch(Window)
    {
        case TE_WIN1:
            Data = TE_BIT5;
            break;
        case TE_WIN2:
            Data = TE_BIT6;
            break;
        case TE_WIN3:
            Data = (TE_BIT6 |TE_BIT5);
            break;
        case TE_WIN4:
            Data = TE_BIT7;
            break;
        case TE_WIN5:
            Data = (TE_BIT7 |TE_BIT5);
            break;
        case TE_WIN6:
            Data = (TE_BIT7 |TE_BIT6);
            break;
        case TE_WIN7:
            Data = (TE_BIT7 | TE_BIT6 |TE_BIT5);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Validate the Error parameter */
    switch(Errors)
    {
        case TE_ERR1:
            Data |= TE_BIT2;
            break;
        case TE_ERR2:
            Data |= TE_BIT3;
            break;
        case TE_ERR3:
            Data |= (TE_BIT3 |TE_BIT2);
            break;
        case TE_ERR4:
            Data |= TE_BIT4;
            break;
        case TE_ERR5:
            Data |= (TE_BIT4 |TE_BIT2);
            break;
        case TE_ERR6:
            Data |= (TE_BIT4 |TE_BIT3);
            break;
        case TE_ERR7:
            Data |= (TE_BIT4 | TE_BIT3 |TE_BIT2);
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT4_7 | TE_BIT3 | TE_BIT2);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle, Address, Data, Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerT1OOFCriteria
 * Returns: TEstatus
 * Description: Returns the configured criteria to declare LOF for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerT1OOFCriteria(void *handle, TEword Port, void *Results)
{
    TEbyte Mask;
    TEstatus Status;
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR2 + Address);
    Mask = (TE_BIT4_7 | TE_BIT3 | TE_BIT2);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* Find the Window size */
    switch(Data & (TE_BIT7 | TE_BIT6 | TE_BIT5))
    {
        case TE_BIT5:
            ReturnValue[0] = TE_WIN1;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_WIN2;
            break;
        case (TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_WIN3;
            break;
        case TE_BIT7:
            ReturnValue[0] = TE_WIN4;
            break;
        case (TE_BIT7 | TE_BIT5):
            ReturnValue[0] = TE_WIN5;
            break;
        case (TE_BIT7 | TE_BIT6):
            ReturnValue[0] = TE_WIN6;
            break;
        case (TE_BIT7 | TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_WIN7;
            break;
        default:  
            return TE_STATUS_BADREAD;  
    }

    /* validate TempValue parameter*/
    switch(Data & (TE_BIT4 | TE_BIT3 | TE_BIT2))
    {
        case TE_BIT2:
            ReturnValue[1] = TE_ERR1;
            break;
        case TE_BIT3:
            ReturnValue[1] = TE_ERR2;
            break;
        case (TE_BIT3 | TE_BIT2):
            ReturnValue[1] = TE_ERR3;
            break;
        case TE_BIT4:
            ReturnValue[1] = TE_ERR4;
            break;
        case (TE_BIT4 | TE_BIT2):
            ReturnValue[1] = TE_ERR5;
            break;
        case (TE_BIT4 | TE_BIT3):
            ReturnValue[1] = TE_ERR6;
            break;
        case (TE_BIT4 | TE_BIT3 | TE_BIT2):
            ReturnValue[1] = TE_ERR7;
            break;
        default:  
            return TE_STATUS_BADREAD;  
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerE1OOFCriteria
 * Returns: TEstatus
 * Description: Sets the criteria to declare Out of Frame in E1 mode for the 
 *              specified Port.
 *              When TE_FAS is selected and three consecutive FAS words contain
 *              at least one error, the framer declares OOF. 
 *              When TE_FASNFAS is selected, the framer declares OOF when it
 *              receives three consecutive FAS words with at least one error
 *              or three consecutive NFAS words with bit two set to zero.
 ******************************************************************************/
TEstatus IXF3204SetFramerE1OOFCriteria(void *handle, TEword Port, TEword Value)
{
    TEbyte Data;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /* validate Value parameter*/
    switch(Value)
    {
        case TE_FAS:
            Data = TE_CLEAR_BITS;
            break;
        case TE_FASNFAS:
            Data = TE_BIT6;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,TE_BIT6))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,TE_BIT6);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerE1OOFCriteria
 * Returns: TEstatus
 * Description: Returns the configuration criteria to declare OOF for the
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerE1OOFCriteria(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT6,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT6)
    {
        ReturnValue[0] = TE_FASNFAS;
    }
    else
    {
        ReturnValue[0] = TE_FAS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerE1CRC4Interworking
 * Returns: TEstatus
 * Description: Configures CRC4 Inter-working for the specified Port. When
 *              Inter-working is enabled and the Intel IXF3204 is 
 *              communicating with a device that doesn't support CRC, the
 *              Intel IXF3204 disables CRC4 internally. When Inter-working is 
 *              disabled, the Intel IXF3204 will detect CRC errors and will not
 *              sincronize to CRC4 multi-frame, when it is not supported on a 
 *              device the Intel IXF3204 is communicating with.
 *              The parameter G706 enables or disables the Intel IXF3204
 *              to report OOF when looking for Interworking. If TE_ENABLE is 
 *              selected OOF will not be reported.
 ******************************************************************************/
TEstatus IXF3204SetFramerE1CRC4Interworking(void *handle, TEword Port,
                                            TEword Criteria, TEword g706)
{
    TEbyte Data, Mask;
    TEstatus Status;
    TEword Address;

    Mask = (TE_BIT7 | TE_BIT1);

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /* validate Criteria parameter*/
    switch(Criteria)
    {
        case TE_ENABLE:
            Data = TE_BIT7;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    /* validate g706 parameter */
    switch(g706)
    {
        case TE_ENABLE:
            Data |= TE_BIT1;
            break;
        case TE_DISABLE:
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerE1CRC4Interworking
 * Returns: TEstatus
 * Description: Returns the status of CRC4 Inter-working for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerE1CRC4Interworking(void *handle, TEword Port,
                                            void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Mask = (TE_BIT7 | TE_BIT1);

    /*Get offsets to perform the get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* read the criteria */
    if(Data & TE_BIT7)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0]= TE_DISABLE;
    }

    /* read the g706 compliance */
    if(Data & TE_BIT1)
    {
        ReturnValue[1] = TE_ENABLE;
    }
    else
    {
        ReturnValue[1] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerE1OOFonCRCErrorsCriteria
 * Returns: TEstatus
 * Description: Sets the criteria to declare Loss of Multiframe (LMF) alignment
 *              for the specified Port. When TE_NOCRC4 is selected, the CRC
 *              error does not cause loss of muliframe alignment. When TE_CRC
 *              is selected, a LMF is declared when the CRC error rate is
 *              greater than or equal to 915 errors per second.
 ******************************************************************************/
TEstatus IXF3204SetFramerE1OOFonCRCErrorsCriteria(void *handle, TEword Port, 
                                                  TEword Value)
{
    TEbyte Data;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /* validate Value parameter*/
    switch(Value)
    {
        case TE_NOCRC4:
            Data = TE_CLEAR_BITS;
            break;
        case TE_CRC4:
            Data = TE_BIT5;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,TE_BIT5))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,TE_BIT5);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerE1OOFonCRCErrorsCriteria
 * Returns: TEstatus
 * Description: Returns the criteria to declare Loss of Multiframe (LMF) 
 *              alignment for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerE1OOFonCRCErrorsCriteria(void *handle, TEword Port, 
                                                  void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT5,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT5)
    {
        ReturnValue[0] = TE_CRC4;
    }
    else
    {
        ReturnValue[0] = TE_NOCRC4;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerE1CASLossCriteria
 * Returns: TEstatus
 * Description: Configures the framer to declare or not declare LMF (time slot
 *              16 = 0) for the specified Port. When TE_NOCAS is selected, the
 *              value in time slot 16 does not cause LMF alignment. When TE_CAS
 *              is selected, the framer declares LMF when the value in time 
 *              slot 16 is equal to zero for a complete multiframe.
 ******************************************************************************/
TEstatus IXF3204SetFramerE1CASLossCriteria(void *handle, TEword Port, 
                                           TEword Value)
{
    TEbyte Data;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }
    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /* validate Value parameter*/
    switch(Value)
    {
        case TE_NOCAS:
            Data = TE_CLEAR_BITS;
            break;
        case TE_CAS:
            Data = TE_BIT4;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,TE_BIT4))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,TE_BIT4);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerE1CASLossCriteria
 * Returns: TEstatus
 * Description: Returns the criteria to declare LMF based on the time slot 16 
 *              value for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerE1CASLossCriteria(void *handle, TEword Port,
                                           void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT4,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT4)
    {
        ReturnValue[0] = TE_CAS;
    }
    else
    {
        ReturnValue[0] = TE_NOCAS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFramerCRC4LossCriteria
 * Returns: TEstatus
 * Description: Sets the criteria to declare Loss of CRC4 Alignment for the
 *              specified Port. When TE_NOCRC4 is selected, errors in CRC4 
 *              alignment words do not cause loss of CRC4 alignment. When
 *              TE_CRC4 is selected then a loss of CRC4 alignment is declared
 *              when two consecutive CRC4 MFA word are received with errors in
 *              them. 
 ******************************************************************************/
TEstatus IXF3204SetFramerCRC4LossCriteria(void *handle, TEword Port,
                                          TEword Value)
{
    TEbyte Data;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /* validate Value parameter*/
    switch(Value)
    {
        case TE_NOCRC4:
            Data = TE_CLEAR_BITS;
            break;
        case TE_CRC4:
            Data = TE_BIT3;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;   
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,TE_BIT3))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,TE_BIT3);
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerCRC4LossCriteria
 * Returns: TEstatus
 * Description: Returns the configured criteria to declare loss of CRC4 
 *              multiframe alignment for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFramerCRC4LossCriteria(void *handle, TEword Port,
                                          void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXFRAMER_CR3 + Address);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT3,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT3)
    {
        ReturnValue[0] = TE_CRC4;
    }
    else
    {
        ReturnValue[0] = TE_NOCRC4;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerStatus
 * Returns: TEstatus
 * Description: Returns the current status of the receive framer for the 
 *              specified Port. The returned value is 16 bits wide. This
 *              function should not be used if the firmware is running. 
 *              Instead use IXF3204GetInterruptFramer.
 ******************************************************************************/
TEstatus IXF3204GetFramerStatus(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEbyte Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte MSB, LSB;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204RxFindPort(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXFRAMER_SR0 + PartialAddress);

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    LSB = Data;
    Address = (TEword)(IXF3204_RXFRAMER_SR1 + PartialAddress);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT7,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    MSB = (TEbyte)(Data >> 7);
    ReturnValue[0] = ((MSB << 8) | LSB);

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerFBitErrorCounter
 * Returns: TEstatus
 * Description: Returns the current value of the specified Counter for the
 *              specified Port. The counters are updated every second with the
 *              last second value. The counter contains the value accumulated 
 *              in the previous second only. It resets every second whether
 *              read or not. The value returned is 13 bits wide.
 ******************************************************************************/
TEstatus IXF3204GetFramerFBitErrorCounter(void *handle, TEword Port,
                                          TEword Selection, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte Data;
    TEbyte MSB, LSB;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204RxFindPort(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;  
    }

    /* validate Selection parameter*/
    switch(Selection)
    {
        case TE_FT:
        case TE_FE:
        case TE_FAS:
            Address = (TEword)(IXF3204_RXFRAMER_SR2 + PartialAddress);
            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            LSB = Data;
            Address = (TEword)(IXF3204_RXFRAMER_SR3 + PartialAddress);

            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            MSB = Data;
            ReturnValue[0] = ((MSB << 8) | LSB);
            break;
        case TE_FTFS:
        case TE_FASNFAS:
            Address = (TEword)(IXF3204_RXFRAMER_SR4 + PartialAddress);
            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            LSB = Data;
            Address = (TEword)(IXF3204_RXFRAMER_SR5 + PartialAddress);

            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            MSB = Data;
            ReturnValue[0] = ((MSB << 8) | LSB);
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;           
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerOOFCounter
 * Returns: TEstatus
 * Description: Returns the number of OOF errors that occurred in the last
 *              second interval for the specified Port. The returned value is 
 *              8 bits wide.
 ******************************************************************************/
TEstatus IXF3204GetFramerOOFCounter(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204RxFindPort(Port);
    if(PartialAddress==TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_SR6 + PartialAddress);

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFramerCRCErrorCounter
 * Returns: TEstatus
 * Description: Returns the number of CRC errors that occurred in the last
 *              second interval for the specified Port. The returned value is
 *              10 bits wide.
 ******************************************************************************/
TEstatus IXF3204GetFramerCRCErrorCounter(void *handle, TEword Port,
                                         void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte Data, Mask;
    TEbyte MSB, LSB;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204RxFindPort(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(IXF3204_RXFRAMER_SR7 + PartialAddress);

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    LSB = Data;

    Mask = (TE_BIT1 | TE_BIT0);

    Address = (TEword)(IXF3204_RXFRAMER_SR8 + PartialAddress);
    if((Status = IXF3204ReadBit(handle,Address,Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    MSB = Data;
    ReturnValue[0] = ((MSB << 8) | LSB);

    return TE_STATUS_SUCCESS;
}


