/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   gctr3204.c
 * Description: this file contains general control functions
 *
 ******************************************************************************/

#include "ixf3204.h"
#include "ini3204.h"
#include "fw.h"
#include "malloc.h"

/*******************************************************************************
 * Function Name: IXF3204SetSoftReset
 * Returns: TEstatus
 * Description: Resets all the state machines. All the configuration registers
 *              are set to their default value. If TE_ENABLE is selected the
 *              device will remain inactive.
 ******************************************************************************/
TEstatus IXF3204SetSoftReset(void *handle, TEword Mode)
{
    TEbyte Value;
    TEword Address = IXF3204_RESET_DEVICE_REG;
    TEstatus Status;
    
    switch(Mode) 
    {
        case TE_DISABLE:
            if( (Status = IXF3204WriteByte(handle,Address, TE_CLEAR_BITS)) != 
                TE_STATUS_SUCCESS)
            {
                return Status; 
            }
            break;
        case TE_ENABLE:
            Value = (TE_BIT7 | TE_BIT5 | TE_BIT0);
            /*Perform a Reset*/ 
            if((Status = IXF3204WriteByte(handle,Address, Value)) != 
               TE_STATUS_SUCCESS)
            {
                return Status; 
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetSoftReset
 * Returns: TEstatus
 * Description: Returns the current reset status of the device
 ******************************************************************************/
TEstatus IXF3204GetSoftReset(void *handle, void * Results)
{
    TEbyte Value;
    TEword Address;
    TEstatus Status;
    TEdword	*ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /* initialize address */
    Address = IXF3204_RESET_DEVICE_REG;
    if((Status = IXF3204ReadByte(handle,Address, &Value)) != 
       TE_STATUS_SUCCESS)
    {
        return Status; 
    }

    switch(Value)
    {
        case (TE_BIT7 | TE_BIT5 | TE_BIT0):
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetCPUReset
 * Returns: TEstatus
 * Description: Resets the internal CPU so that the instruction pointer points
 *              to the initial address. If TE_ENABLE is selected the CPU will
 *              remain inactive.
 ******************************************************************************/
TEstatus IXF3204SetCPUReset(void *handle, TEword Mode)
{
    TEword Address;
    TEstatus Status;

    Address = IXF3204_RESET_CPU_REG;

    switch(Mode) 
    {
        case TE_ENABLE:
            /* Enables the Reset, the CPU will not work */
            if((Status = IXF3204WriteByte(handle,Address,TE_BIT0)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            break;
        case TE_DISABLE:
            /* Disables the Reset, the CPU will work */
            if((Status = IXF3204WriteByte(handle,Address,TE_CLEAR_BITS)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetCPUReset
 * Returns: TEstatus
 * Description: Returns the current reset status of the CPU
 ******************************************************************************/
TEstatus IXF3204GetCPUReset(void *handle, void * Results)
{
    TEbyte Value;
    TEword Address;
    TEstatus Status;
    TEdword	*ReturnValue = (TEdword *)Results;
    
    CHK_NULL(ReturnValue);

    Address = IXF3204_RESET_CPU_REG;

    if((Status = IXF3204ReadByte(handle,Address, &Value)) != 
       TE_STATUS_SUCCESS)
    {
        return Status; 
    }

    switch(Value)
    {
        case TE_BIT0:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetPortEnable
 * Returns: TEstatus
 * Description: Disables or enables the selected port keeping the configuration
 *              parameters unchanged.
 *              When disabled, in the receive side, no information is processed
 *              from the line. The rx-backplane bus drives all 0s.  
 *              In the transmit side, either all  0s or all 1s (depending on
 *              the last bit sent before disabling it) are sent continuously. 
 *              The clock is transmitted to line and backplane as defined by
 *              the configuration, that is, the clock is not shutdown.
 ******************************************************************************/
TEstatus IXF3204SetPortEnable(void *handle, TEword Port, TEword Mode)
{
    TEword Address;
    TEbyte PortBP, Data, Mask;
    TEstatus Status;

    /*Reset PLI*/
    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Mode)
    {
        case TE_DISABLE:
            Data = TE_BIT3;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT3;
    Address += IXF3204_LIERST;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Reset TxFramer*/
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Mode) 
    {
        case TE_DISABLE:
            Data = TE_BIT7;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT7;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Reset RxFramer*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address++;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Reset BackPlane*/
    switch(Port)
    {
        case TE_PORT0:
            PortBP = TE_BP0;
            break;
        case TE_PORT1:
            PortBP = TE_BP1;
            break;
        case TE_PORT2:
            PortBP = TE_BP2;
            break;
        case TE_PORT3:
            PortBP = TE_BP3;
            break;
        case TE_PORT4:
            PortBP = TE_BP4;
            break;
        case TE_PORT5: 
            PortBP = TE_BP5;
            break;
        case TE_PORT6:
            PortBP = TE_BP6;
            break;
        case TE_PORT7:
            PortBP = TE_BP7;
            break;
        default:
            /* This will never happen */
            PortBP = (TEbyte)NULL;
    }

    Address = IXF3204FindPortBackPlane(PortBP);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Mode) 
    {
        case TE_DISABLE:
            Data = TE_BIT0;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;
    Address += IXF3204_BPUSE;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Reset ClockSelector*/
    Address = IXF3204FindPortClockSelector(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Mode) 
    {
        case TE_DISABLE:
            Data = TE_BIT0;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Reset Rx SlipBuffer*/
    Address = IXF3204FindPortSlipBuffer(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Mode) 
    {
        case TE_DISABLE:
            Data = TE_BIT1;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT1;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Reset Tx SlipBuffer*/
    switch(Mode)
    {
        case TE_DISABLE:
            Data =TE_BIT1;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT1;
    Address += IXF3204_SB_SBUSET;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Reset TxSignalingCDR*/
    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    switch(Mode) 
    {
        case TE_DISABLE:
            Data = TE_BIT0;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Reset RxBusIF*/
    Address = IXF3204FindPortRxBusIF(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Mode) 
    {
        case TE_DISABLE:
            Data = TE_BIT0;
            break;
        case TE_ENABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetPortEnable
 * Returns: TEstatus
 * Description: Returns the current reset status of all the modules for the 
 *              specified Port. The value returned is 8 bits wide, where each
 *              bit represents a module. When the bit value is 1, the
 *              corresponding module is reset.
 ******************************************************************************/
TEstatus IXF3204GetPortEnable(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte PortBP, Data, Mask; 
    TEdword	*ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Mask = TE_BIT3;

    /*IXF3204Read PLI*/
    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_LIERST;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data) 
    {
        ReturnValue[0] = TE_BIT0;
    }
    else 
    {
        ReturnValue[0] = TE_CLEAR_BITS;
    }

    /*IXF3204Read TxFramer*/
    Address = IXF3204TxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT7;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data) 
    {
        ReturnValue[0] |= TE_BIT1;
    }

    /*Reset RxFramer*/
    Address = IXF3204RxFindPort(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT7;
    Address++;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    if(Data)
    {
        ReturnValue[0] |= TE_BIT2;
    }

    /*Reset BackPlane*/
    switch(Port)
    {
        case TE_PORT0:
            PortBP = TE_BP0;
            break;
        case TE_PORT1:
            PortBP = TE_BP1;
            break;
        case TE_PORT2:
            PortBP = TE_BP2;
            break;
        case TE_PORT3:
            PortBP = TE_BP3;
            break;
        case TE_PORT4:
            PortBP = TE_BP4;
            break;
        case TE_PORT5:
            PortBP = TE_BP5;
            break;
        case TE_PORT6:
            PortBP = TE_BP6;
            break;
        case TE_PORT7:
            PortBP = TE_BP7;
            break;
        default:
            /* This will never happen */
            PortBP = (TEbyte)NULL;
    }

    Address = IXF3204FindPortBackPlane(PortBP);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;
    Address += IXF3204_BPUSE;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data) 
    {
        ReturnValue[0] |= TE_BIT3;
    }

    /*Reset ClockSelector*/
    Address = IXF3204FindPortClockSelector(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data) 
    {
        ReturnValue[0] |= TE_BIT4;
    }

    /*Reset SlipBuffer*/
    Address = IXF3204FindPortSlipBuffer(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT1;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return  Status;
    }

    if(Data) 
    {
        ReturnValue[0] |= TE_BIT5;
    }

    /*Reset TxSignalingCDR*/
    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data) 
    {
        ReturnValue[0] |= TE_BIT6;
    }

    /*Reset RxBusIF*/
    Address = IXF3204FindPortRxBusIF(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data) 
    {
        ReturnValue[0] |= TE_BIT7;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetPortFunctionReset
 * Returns: TEstatus
 * Description: Enables or disables the specified Function for the specified 
 *              Port. 
 *              When TE_BERT is specified in Function, Port must be TE_BERT0 
 *              to TE_BERT7.
 *              In the same way, when TE_HDLC is specified in Function, Port 
 *              must be TE_HDLC0 to TE_HDLC23.
 *              When TE_SIGNALING, TE_ALARMS or TE_FDL is specified in 
 *              Function, Port may be TE_PORT0 to TE_PORT 7.
 *              If TE_ENABLE, the selected module will be in reset mode, so it 
 *              will inactive
 ******************************************************************************/
TEstatus IXF3204SetPortFunctionReset(void *handle, TEword Port,TEword Function,
                                     TEword Mode)
{
    TEword Address;
    TEbyte Data, Data1, Mask = TE_BIT7;
    TEstatus Status;

    Data = TE_BIT7;

    switch(Function)
    {
        case TE_BERT:
            Address = IXF3204FindBERT(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /*validate Mode parameter*/
            switch(Mode)
            {
                case TE_ENABLE:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_DISABLE:
                    Data = TE_BIT0;
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS; 
            }

            Mask = TE_BIT0;
            Address += IXF3204_BERT_RXCTRL;

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                /* Resetting Rx BERT */
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                /* Resetting Tx BERT */
                /* add 0x08 to have an offset of 0x0f */
                /* 0x07 was added before              */
                Address += IXF3204_ADDR_OFFSET8;
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                return TE_STATUS_SUCCESS;
            }

            /* Resetting Rx BERT */
            if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            /* Resetting Tx BERT */
            /* add 0x08 to have an offset of 0x0f */
            /* 0x07 was added before              */
            Address += IXF3204_ADDR_OFFSET8;  
            if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            
            return TE_STATUS_SUCCESS;
        case TE_SIGNALING:
            /*validate Mode parameter*/
            switch(Mode)
            {
                case TE_ENABLE:
                    Data = TE_BIT0;
                    Data1 = TE_BIT1;
                    break;
                case TE_DISABLE:
                    Data = TE_CLEAR_BITS;
                    Data1 = TE_CLEAR_BITS;
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS;
            }

            Mask = TE_BIT0;
            /*Get offsets to perform the set operations*/
            Address = IXF3204FindPortRxSignaling_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Mask = TE_BIT1;
            /*Get offsets to perform the set operations*/
            Address = IXF3204FindPortTxSignaling_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data1,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data1,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Mask = TE_BIT0;

            /************************************************************
             *Next switch performs a mapping of ports TE_PORT? to TE_BP?*
             *for functions                                             *
             *IXF3204FindPortRxSysSigIF_CDR(Port)                       *
             *IXF3204FindPortTxSysSigIF_CDR(Port)                       *
             ************************************************************/
            switch(Port)
            {
                case TE_PORT0:
                    Port = TE_BP0; 
                    break;
                case TE_PORT1:
                    Port = TE_BP1;
                    break;
                case TE_PORT2:
                    Port = TE_BP2;
                    break;
                case TE_PORT3: 
                    Port = TE_BP3; 
                    break;
                case TE_PORT4: 
                    Port = TE_BP4;
                    break;
                case TE_PORT5: 
                    Port = TE_BP5; 
                    break;
                case TE_PORT6: 
                    Port = TE_BP6; 
                    break;
                case TE_PORT7: 
                    Port = TE_BP7; 
                    break;
                default: 
                    break;
            }

            Address = IXF3204FindPortRxSysSigIF_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = IXF3204FindPortTxSysSigIF_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        case TE_HDLC:
            /*validate Mode parameter*/
            switch(Mode)
            {
                case TE_ENABLE:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_DISABLE:
                    Data = TE_BIT0;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS; 
            }

            Mask = TE_BIT0;
            Address = IXF3204FindPortRxHDLC(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllHDLCMem(handle,Address,Data,
                                                       Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = IXF3204FindPortTxHDLC(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllHDLCMem(handle,Address,Data,
                                                       Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        case TE_ALARMS:
            Address = IXF3204FindPortRxAlarms(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /*validate Mode parameter*/
            switch(Mode)
            {
                case TE_ENABLE:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_DISABLE:
                    Data = TE_BIT0;
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS;  
            }

            Mask = TE_BIT0;

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                return TE_STATUS_SUCCESS;
            }
            /*perform set operation*/
            if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            return TE_STATUS_SUCCESS;
        case TE_FDL :
            /*validate Mode parameter*/
            switch(Mode)
            {
                case TE_ENABLE:
                    Data = TE_CLEAR_BITS;
                    break;
                case TE_DISABLE:
                    Data = TE_BIT0;
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS;
            }

            Mask = TE_BIT0;
            Address = IXF3204FindPortRxFDL(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = IXF3204FindPortTxFDL(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetPortFunctionReset
 * Returns: TEstatus
 * Description: Returns the current reset status of the selected module
 ******************************************************************************/
TEstatus IXF3204GetPortFunctionReset(void *handle, TEword Port,
                                     TEword Function, void * Results)
{
    TEword Address;
    TEbyte Mask, Value;
    TEstatus Status;
    TEdword	*ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    switch(Function)
    {
        case TE_BERT:
            Address = IXF3204FindBERT(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Mask = TE_BIT0;
            Address += IXF3204_BERT_RXCTRL;
            /* Reading only Rx BERT */
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Value)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Value) 
            {
                ReturnValue[0] = TE_DISABLE;
            } 
            else 
            {
                ReturnValue[0] = TE_ENABLE;
            }
            break;
        case TE_SIGNALING:
            /*validate Mode parameter*/
            Mask = TE_BIT0;

            /*Get offsets to perform the set operations*/
            Address = IXF3204FindPortRxSignaling_CDR(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* reading only from Rx signaling*/
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Value)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Value) 
            {
                ReturnValue[0] = TE_ENABLE;
            } 
            else 
            {
                ReturnValue[0] = TE_DISABLE;
            }
            break;
        case TE_HDLC:
            Mask = TE_BIT0;
            Address = IXF3204FindPortRxHDLC(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Reading only Rx HDLC */
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Value)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Value) 
            {
                ReturnValue[0] = TE_DISABLE;
            }
            else
            {
                ReturnValue[0] = TE_ENABLE;
            }
            break;
        case TE_ALARMS:
            Address = IXF3204FindPortRxAlarms(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Mask = TE_BIT0;
            /* Read value */
            if((Status = IXF3204ReadBit(handle,Address,Mask,&Value)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Value) 
            {
                ReturnValue[0] = TE_DISABLE;
            }
            else 
            {
                ReturnValue[0] = TE_ENABLE;
            }
            break;
        case TE_FDL :
            Mask = TE_BIT0;
            Address = IXF3204FindPortRxFDL(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            /* Read only Rx FDL */
            if((Status = IXF3204ReadBit(handle, Address, Mask, &Value)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            if(Value) 
            {
                ReturnValue[0] = TE_DISABLE;
            }
            else
            {
                ReturnValue[0] = TE_ENABLE;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }
    
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetDeviceRevisionID
 * Returns: TEstatus
 * Description: Returns the Intel IXF3204 revision ID value. The first byte
 *              corresponds to the hardware revision. The second byte
 *              corresponds to the firmware revision.
 ******************************************************************************/
TEstatus IXF3204GetDeviceRevisionID(void *handle, void *Results)
{
    TEword Address = IXF3204_ADDR_HW_ID;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /* get the hardware ID */
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    Address = IXF3204_ADDR_FW_ID;
    /* get the Firmware ID */
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[1] = Data;

    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204Initialize
 * Returns: TEstatus
 * Description: Links the device structure to the handle and performs any 
 *              initializing routines required to get the Intel IXF3204 into
 *              a default state.
 ******************************************************************************/
TEstatus IXF3204Initialize( void **pphandle, TEbindstruct *p_bind_struct)
{

    IXF3204struct *pIXF3204 = (IXF3204struct *)NULL;

    /*
     *  valid parameters
     */

    if(p_bind_struct == NULL) 
    {
        return TE_STATUS_BAD_PARAMS;
    }

    if(pphandle == NULL) 
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*
     *  allocate memory for IXF3204 object
     */
    pIXF3204 = (IXF3204struct *) TE_OS_ALLOC_MEM( sizeof(IXF3204struct) );

    if (pIXF3204 == NULL)
    {
        return TE_STATUS_MEM_ALLOC_FAILURE;
    }

    /*
     *  fill object
     */

    pIXF3204->base_address = p_bind_struct->base_address;
    pIXF3204->long_base_address = p_bind_struct->long_base_address;
    pIXF3204->client_handle = p_bind_struct->client_handle;

    /* TO DO:  Fill more elements here if added to structure definition in
     *         ixf3204.h
     */

    /*
     *  return object
     */
    *pphandle = (void *)pIXF3204;


    if(!pIXF3204->client_handle)
		{
        return TE_STATUS_BADREAD;
		}

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204Shutdown
 * Returns: TEstatus
 * Description: Terminates the binding between the device driver and the 
 *              Intel IXF3204.
 ******************************************************************************/
TEstatus IXF3204Shutdown( void *handle )
{
    /*
     *  valid parameters
     */
    if((int)handle == (int)NULL) 
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*
     *  free object before terminating 
     */
    TE_OS_DEALLOC_MEM( handle );

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: fwload
 * Returns: TEstatus
 * Description: This functions load the firmware in the IXF3204 
 ******************************************************************************/
TEstatus IXF3204ControlLoadFirmware(void * handle)
{
    TEbyte Data;
    TEword Index;
    TEword size;
    TEaddress Address;

    size = sizeof(Firmware);

    /* disable the CPU */
    IXF3204SetCPUReset(handle,TE_ENABLE);

    Address = IXF3204_CPU_ROM_ADDRESS;

    /* write the firmware */
    for(Index = 0; Index < size; Index++, Address++)
    {
        IXF3204WriteByte(handle,Address,Firmware[Index]);
    }

    Address = IXF3204_CPU_ROM_ADDRESS;
    /* verify if was copied correctly */
    for(Index = 0; Index < size; Index++, Address++)
    {
        IXF3204ReadByte(handle,Address,&Data);
        if(Data != Firmware[Index])
        {
            return TE_STATUS_ERROR_LOAD_VERIFY;
        }
    }

    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204SetPortSoftReset
 * Returns: TEstatus
 * Description: Reset the port selected into its default value and the mapped
 *              module (HDLC or BERT), also clears all the interrupt mask
 *              related to the port and HDLC/BERT (if mapped)
 ******************************************************************************/
TEstatus IXF3204SetPortSoftReset(void *handle, TEword Port)
{

    TEword Address, base_addr;
    TEbyte loop;
    TEdword result;

    /* find the base address of the LIU */
    base_addr = IXF3204FindPortPLI(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore LIU to its default values */
    for(loop = 0; loop < IXF3204_LIU_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_LIU_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,IXF3204_LIU_def_value[loop]));
    }

    /* find the base address of the Rx Framer */
    base_addr = IXF3204RxFindPort(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Rx Framer to its default values */
    for(loop = 0; loop < IXF3204_RXFRAMER_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_RXFRAMER_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_RXFRAMER_def_value[loop]));
    }

    /* find the base address of the Tx Framer */
    base_addr = IXF3204TxFindPort(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Tx Framer to its default values */
    for(loop = 0; loop < IXF3204_TXFRAMER_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_TXFRAMER_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_TXFRAMER_def_value[loop]));
    }

    /* find the base address of the Rx FDL */
    base_addr = IXF3204FindPortRxFDL(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Rx FDL to its default values */
    for(loop = 0; loop < IXF3204_RXFDL_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_RXFDL_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_RXFDL_def_value[loop]));
    }

    /* find the base address of the Tx FDL */
    base_addr = IXF3204FindPortTxFDL(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Tx FDL to its default values */
    for(loop = 0; loop < IXF3204_TXFDL_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_TXFDL_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_TXFDL_def_value[loop]));
    }

    /* find the base address of the Rx Mapping */
    base_addr = IXF3204FindPortRxBusIF(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Rx Mapping to its default values */
    for(loop = 0; loop < IXF3204_RXMAP_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_RXMAP_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_RXMAP_def_value[loop]));
    }

    /* find the base address of the Rx Signaling */
    base_addr = IXF3204FindPortRxSignaling_CDR(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Rx Signaling to its default values */
    for(loop = 0; loop < IXF3204_RXSIG_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_RXSIG_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_RXSIG_def_value[loop]));
    }

    /* find the base address of the Tx Signaling */
    base_addr = IXF3204FindPortTxSignaling_CDR(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Tx Signaling to its default values */
    for(loop = 0; loop < IXF3204_TXSIG_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_TXSIG_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_TXSIG_def_value[loop]));
    }

    /* find the base address of the Rx Alarms */
    base_addr = IXF3204FindPortRxAlarms(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Rx Alarms to its default values */
    for(loop = 0; loop < IXF3204_RXALRM_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_RXALRM_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_RXALRM_def_value[loop]));
    }

    /* find the base address of the Tx Alarms */
    base_addr = IXF3204FindPortTxAlarms(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Tx Alarms to its default values */
    for(loop = 0; loop < IXF3204_TXALRM_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_TXALRM_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_TXALRM_def_value[loop]));
    }

    /* find the base address of the Slip buffer */
    base_addr = IXF3204FindPortSlipBuffer(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Slip buffer to its default values */
    for(loop = 0; loop < IXF3204_SLIP_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_SLIP_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_SLIP_def_value[loop]));
    }

    /* find the base address of the Backplane */
    base_addr = IXF3204FindPortBackPlane(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Backplane to its default values */
    for(loop = 0; loop < IXF3204_BP_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_BP_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_BP_def_value[loop]));
    }

    /* find the base address of the Rx Backplane Signaling*/
    base_addr = IXF3204FindPortRxSysSigIF_CDR(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Rx Backplane Signaling to its default values */
    for(loop = 0; loop < IXF3204_RXBPSIG_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_RXBPSIG_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_RXBPSIG_def_value[loop]));
    }

    /* find the base address of the Tx Backplane Signaling*/
    base_addr = IXF3204FindPortTxSysSigIF_CDR(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Tx Backplane Signaling to its default values */
    for(loop = 0; loop < IXF3204_TXBPSIG_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_TXBPSIG_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_TXBPSIG_def_value[loop]));
    }

    /* find the base address of the Synchronization */
    base_addr = IXF3204FindPortClockSelector(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore Synchronization to its default values */
    for(loop = 0; loop < IXF3204_SYNC_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_SYNC_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_SYNC_def_value[loop]));
    }

    /* find the base address of the National bits */
    base_addr = IXF3204FindPortNational(Port);
    if(base_addr == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* restore National bitsto its default values */
    for(loop = 0; loop < IXF3204_NAT_num_addr; loop++)
    {
        Address = (TEword)(base_addr + IXF3204_NAT_address_offset[loop]);
        CHK_STATUS(IXF3204WriteByte(handle,Address,
                                    IXF3204_NAT_def_value[loop]));
    }

    /* clean the perfomance database */
    CHK_STATUS(IXF3204SetPEClearCounter(handle,Port,TE_NEAREND));
    CHK_STATUS(IXF3204SetPEClearCounter(handle,Port,TE_FAREND));
    CHK_STATUS(IXF3204SetPEClearCounter(handle,Port,TE_GENERIC));

    /* check if there are mapping */
    CHK_STATUS(IXF3204GetMapPortSelect(handle,Port,&result));

    if(result != TE_NORMAL)
    {
    /* check if the mapping is with HDLC */
        base_addr = IXF3204FindPortRxHDLC(Port);
        if(base_addr == TE_STATUS_BAD_PARAMS)
        {
            /* if not HDLC, check for BERT */
            /* find the base address of the BERT */
             base_addr = IXF3204FindBERT(Port);
             if(base_addr == TE_STATUS_BAD_PARAMS)
             {
                 return TE_STATUS_BAD_PARAMS;
             }

             /* restore BERT to its default values */
             for(loop = 0; loop < IXF3204_BERT_num_addr; loop++)
             {
                 Address = (TEword)(base_addr 
                                    + IXF3204_BERT_address_offset[loop]);
                 CHK_STATUS(IXF3204WriteByte(handle,Address,
                                             IXF3204_BERT_def_value[loop]));
             }
        }

        /* restore Rx HDLC to its default values */
        for(loop = 0; loop < IXF3204_RXHDLC_num_addr; loop++)
        {
            Address = (TEword)(base_addr + IXF3204_RXHDLC_address_offset[loop]);
            CHK_STATUS(IXF3204WriteByte(handle,Address,
                                        IXF3204_RXHDLC_def_value[loop]));
        }

        base_addr = IXF3204FindPortTxHDLC(Port);
        if(base_addr == TE_STATUS_BAD_PARAMS)
        {
            return TE_STATUS_BAD_PARAMS;
        }

        /* restore Tx HDLC to its default values */
        for(loop = 0; loop < IXF3204_TXHDLC_num_addr; loop++)
        {
            Address = (TEword)(base_addr + IXF3204_TXHDLC_address_offset[loop]);
            CHK_STATUS(IXF3204WriteByte(handle,Address,
                                        IXF3204_TXHDLC_def_value[loop]));
        }
    }

    return TE_STATUS_SUCCESS;
}

