/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   hdlc3204.c
 * Description: this file contains HDLC functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetHDLCMode
 * Returns: TEstatus
 * Description: Configures the HDLC mode.
 ******************************************************************************/
TEstatus IXF3204SetHDLCMode(void *handle, TEword hdlc, TEword Value)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /*validate parameter Value*/
    switch(Value)
    {
        case TE_LAPB: 
            Data = TE_CLEAR_BITS;
            break;
        case TE_LAPD: 
            Data = TE_BIT5;
            break;
        case TE_LAPV5: 
            Data = TE_BIT6;
            break;
        case TE_TRANSPARENT: 
            Data = (TE_BIT6 | TE_BIT5);
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    Mask = (TE_BIT6 | TE_BIT5);

    /* Set all the ports with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllHDLCMem(handle,Address,Data,
                                               Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /* perform set operation*/
    return (IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCMode
 * Returns: TEstatus
 * Description: Returns the mode configured in the specified HDLC.
 ******************************************************************************/
TEstatus IXF3204GetHDLCMode(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEbyte Mask, Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    Mask = (TE_BIT6 | TE_BIT5);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_LAPB;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_LAPD;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_LAPV5;
            break;
        case (TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_TRANSPARENT;
            break;
        default: 
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCAddress
 * Returns: TEstatus
 * Description: Configures the two addresses used to filter messages in the 
 *              receive direction. The mask field is specified as a 16-bit 
 *              parameter where a 1 means a bit to compare in the addresses.
 ******************************************************************************/
TEstatus IXF3204SetHDLCAddress(void *handle, TEword hdlc, TEword Address1,
                               TEword Address2, TEword Mask)
{
    TEword Address;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS) 
    {
        return TE_STATUS_BAD_PARAMS;
    }
    /*IXF3204Write Mask High */
    Address += IXF3204_RXHDLC_HAMR; 

    /* Set all the ports with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllHDLCMem(handle,Address,(TEbyte)(Mask 
                                                                        >> 8)))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteByte(handle,Address,(TEbyte)(Mask >> 8))) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*IXF3204Write Mask Low   */
    Address++; 
    /* Set all the ports with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllHDLCMem(handle,Address,(TEbyte)Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteByte(handle,Address,(TEbyte)Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*IXF3204Write Address1 high */
    Address++; 
    /* Set all the ports with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllHDLCMem(handle,Address,(TEbyte)(Address1
                                                >> 8))) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteByte(handle,Address,(TEbyte)(Address1 >> 8))) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*IXF3204Write Address2 high */
    Address++;  
    /* Set all the ports with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllHDLCMem(handle,Address,(TEbyte)(Address2 
                                                >> 8))) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteByte(handle,Address,(TEbyte)(Address2 >> 8))) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*IXF3204Write Address1 low  */
    Address++;  
    /* Set all the ports with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllHDLCMem(handle,Address,(TEbyte)Address1)
            ) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteByte(handle,Address,(TEbyte)(Address1))) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*IXF3204Write Address2 low  */
    Address++;

    /* Set all the HDLCs with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllHDLCMem(handle,Address,(TEbyte)Address2)
            ) != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    if((Status = IXF3204WriteByte(handle,Address,(TEbyte)(Address2))) !=
       TE_STATUS_SUCCESS) 
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCAddress
 * Returns: TEstatus
 * Description: Returns the two addresses used to filter the messages in the 
 *              receive direction and the mask used.
 ******************************************************************************/
TEstatus IXF3204GetHDLCAddress(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEword TempValue, TempValue2;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    
    /* Reading Mask */
    Address += IXF3204_RXHDLC_HAMR;
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempValue = (TEword)(Data << 8);

    Address++;
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempValue |= (TEword)Data;

    ReturnValue[2] = TempValue;

    /* Reading Addresses High */
    Address++; 
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempValue = (TEword)(Data << 8);

    Address++; 
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempValue2 = (TEword)(Data << 8);

    /* Reading Addresses Low */
    Address++; 
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempValue |= (TEword)Data;

    Address++; 
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempValue2 |= (TEword)Data;

    /* Address 1 */
    ReturnValue[0] = TempValue;
    /* Address 2 */
    ReturnValue[1] = TempValue2;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCLength
 * Returns: TEstatus
 * Description: Sets the Length threshold for the message being received or 
 *              transmitted for the specified Direction. When the message 
 *              exceeds the limit, it is aborted by the HDLC module. The 
 *              maximum receive threshold is 65, 535 bytes. When 0x0000 is 
 *              selected, no length is checked.
 *              In transmit only the 7 LSBits are used where the maximum 
 *              transmit length threshold is 64 bytes. When bit 7 is 1 the 
 *              length threshold is fixed to 64 bytes regardless of the other 
 *              bits.
 ******************************************************************************/
TEstatus IXF3204SetHDLCLength(void *handle, TEword hdlc, TEword Direction,
                              TEword Length)
{
    TEword AddressRx, AddressTx;
    TEbyte Data;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    AddressRx = IXF3204FindPortRxHDLC(hdlc);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressTx = IXF3204FindPortTxHDLC(hdlc);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            /*write LSB */
            Data = (TEbyte)(TE_BIT0_7 & Length);
            AddressRx += IXF3204_RXHDLC_MLR1;

            /* Set all the ports with the same configuration */
            if(hdlc == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllHDLCMem(handle,AddressRx,
                                                        Length))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /* perform set operation*/
                if((Status = IXF3204WriteWord( handle,AddressRx, Length)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        case TE_TX:
            Data = (TEbyte)((TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3) & Length);
            if(Length >= IXF3204_BUFFER_LENGTH)
            {
                Data = IXF3204_BUFFER_LENGTH;
            }
            /* 7 LSB */
            AddressTx += IXF3204_TXHDLC_HLR;
            /* Set all the ports with the same configuration */
            if(hdlc == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllHDLCMem(handle,AddressTx,Data))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteByte( handle,AddressTx,Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCLength
 * Returns: TEstatus
 * Description: Returns the length threshold in bytes for the message being 
 *              received from the line. The value returned is 0 to 65,535.
 ******************************************************************************/
TEstatus IXF3204GetHDLCLength(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEword Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /*IXF3204Read of LSB*/
    Address += IXF3204_RXHDLC_MLR1;
    if((Status = IXF3204ReadWord(handle,Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCConfig
 * Returns: TEstatus
 * Description: Configures the CRC validations and the data inversion for the 
 *              specified HDLC and Direction.
 ******************************************************************************/
TEstatus IXF3204SetHDLCConfig(void *handle, TEword hdlc, TEword Direction,
                              TEword DataInv, TEword CRCEnable)
{
    TEword AddressRx, AddressTx;
    TEbyte Data, Mask = (TE_BIT4 | TE_BIT2);
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    AddressRx = IXF3204FindPortRxHDLC(hdlc);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressTx = IXF3204FindPortTxHDLC(hdlc);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* validate parameters DataInv */
    switch(DataInv)
    {
        case TE_ENABLE:
            Data = TE_BIT4;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* validate parameters CRCEnable */
    switch(CRCEnable)
    {
        case TE_ENABLE:
            Data |= TE_BIT2;
            break;
        case TE_DISABLE:
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* validate parameters Direction */
    switch(Direction)
    {
        /* write to Rx */
        case TE_RX:
            /* Set all the ports with the same configuration */
            if(hdlc == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressRx,Data,
                                                       Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteBit(handle,AddressRx,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            break;
        case TE_TX:
            /* Set all the ports with the same configuration */
            if(hdlc == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressTx,Data,
                                                       Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /* write to Tx */
                if((Status = IXF3204WriteBit(handle,AddressTx,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            break;
        case TE_BOTH:
            /* Set all the ports with the same configuration */
            if(hdlc == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressRx,Data,
                                                       Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressTx,Data,
                                                       Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /* write to Rx */
                if((Status = IXF3204WriteBit(handle,AddressRx,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                /* write to Tx */
                if((Status = IXF3204WriteBit(handle,AddressTx,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCConfig
 * Returns: TEstatus
 * Description: Returns the configured mode for CRC checking and data inversion
 *              of the specified HDLC and Direction.
 ******************************************************************************/
TEstatus IXF3204GetHDLCConfig(void *handle, TEword hdlc, TEword Direction,
                              void *Results)
{
    TEword Address;
    TEbyte Mask = (TE_BIT4 | TE_BIT2);
    TEbyte Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            /*Get offsets to perform the get operations*/
            Address = IXF3204FindPortRxHDLC(hdlc);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_TX:
            Address = IXF3204FindPortTxHDLC(hdlc);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE; 
            ReturnValue[1] = TE_DISABLE;
            break;
        case TE_BIT2:
            ReturnValue[0] = TE_DISABLE;
            ReturnValue[1] = TE_ENABLE; 
            break;
        case TE_BIT4:
            ReturnValue[0] = TE_ENABLE;
            ReturnValue[1] = TE_DISABLE;
            break;
        case (TE_BIT4 | TE_BIT2):
            ReturnValue[0] = TE_ENABLE;
            ReturnValue[1] = TE_ENABLE;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCTSBitmask
 * Returns: TEstatus
 * Description: Defines which bits in the time slot are used to generate or 
 *              analyze HDLC data for the specified Port. The Mask applies to 
 *              ALL the time slots (for the specified Port). The mask applies 
 *              to receive and transmit operations. 
 ******************************************************************************/
TEstatus IXF3204SetHDLCTSBitmask(void *handle, TEword Port, TEbyte Mask) 
{
    TEstatus Status;
    TEword AddressTx, AddressRx;

    /*Get offsets to perform the set operations*/
    AddressTx = IXF3204FindPortTxSignaling_CDR(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressRx = IXF3204FindPortRxBusIF(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*IXF3204Write for TxSignalingCDR*/
    AddressTx += IXF3204_TXSIG_HDLCM;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressTx,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteByte(handle,AddressTx,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*Write for RxBusIF*/
    AddressRx += IXF3204_RXBUSIF_CR3;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressRx,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    if((Status = IXF3204WriteByte(handle,AddressRx, Mask)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCTSBitmask
 * Returns: TEstatus
 * Description: Returns the bit mask used in the generation or analysis 
 *              process for the specified Port. The bit mask defines which 
 *              bits in the time slot are used to generate or analyze HDLC data.
 ******************************************************************************/
TEstatus IXF3204GetHDLCTSBitmask(void *handle, TEword Port, void *Results)
{

    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXSIG_HDLCM;

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    Address = IXF3204FindPortRxBusIF(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXBUSIF_CR3;

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data != (TEbyte)ReturnValue[0])
    {
        return TE_STATUS_BAD_CONFIG;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCTSBitmaskToBP
 * Returns: TEstatus
 * Description: Defines which bits in the time slot are used to generate or 
 *              analyze HDLC data for the specified Backplane. The Mask applies
 *              to ALL the time slots (for the specified Backplane). The mask 
 *              applies to receive and transmit operations. Bit 8 provides the
 *              mask for the F-bit.
 ******************************************************************************/
TEstatus IXF3204SetHDLCTSBitmaskToBP(void *handle, TEword Port, TEword Mask)
{
    TEword Address, BackPlaneOffset;
    TEstatus Status;
    TEbyte Data;

    /*Get offsets to perform the Set operations*/
    BackPlaneOffset = IXF3204FindPortBackPlane(Port);
    if(BackPlaneOffset == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Write of the 8 Least significant bits*/
    Address = (TEword)(IXF3204_BPRMASKH + BackPlaneOffset);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,(TEbyte)Mask)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteByte(handle,Address,(TEbyte)Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
        
    Address = (TEword)(IXF3204_BPTMASKH + BackPlaneOffset);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,(TEbyte)Mask)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteByte(handle,Address,(TEbyte)Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }    

    /*Write, Most significant bit (in bit 0)*/
    Data = (TEbyte)(Mask >> 8);
    Address = (TEword)(IXF3204_BPRMASK8 + BackPlaneOffset);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,TE_BIT0)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,TE_BIT0)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    Address = (TEword)(IXF3204_BPTMASK8 + BackPlaneOffset);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,TE_BIT0)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        if((Status = IXF3204WriteBit(handle,Address,Data,TE_BIT0)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCTSBitmaskToBP
 * Returns: TEstatus
 * Description: Returns the bit mask used in the generation or analysis 
 *              process for the specified Backplane. The bit mask defines 
 *              which bits in the time slot are used to generate or analyze 
 *              data.
 ******************************************************************************/
TEstatus IXF3204GetHDLCTSBitmaskToBP(void *handle, TEword Port, void *Results)
{
    TEword BackPlaneOffset, Address;
    TEbyte Data;
    TEbyte Data1;
    TEdword *ReturnValue = (TEdword *)Results;
    TEdword verify;
    TEstatus Status;
    
    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    BackPlaneOffset = IXF3204FindPortBackPlane(Port);
    if(BackPlaneOffset == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Read of the 8 Least significant bits*/
    Address = (TEword)(BackPlaneOffset + IXF3204_BPRMASKH);

    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Read, Most significant bit (in bit 0)*/
    Address++;
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT0,&Data1)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = (TEdword)((Data1 << 8) | Data);

    Address = (TEword)(BackPlaneOffset + IXF3204_BPTMASKH);

    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Read, Most significant bit (in bit 0)*/
    Address++;
    if((Status = IXF3204ReadBit(handle,Address,TE_BIT0,&Data1)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    verify = (TEdword)((Data1 << 8) | Data);
    if(verify != (TEdword) ReturnValue[0])
    {
        return TE_STATUS_BAD_CONFIG;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCRxFIFODone
 * Returns: TEstatus
 * Description: The host indicates to the specified HDLC that the host has 
 *              completed the read of the currently available FIFO.
 ******************************************************************************/
TEstatus IXF3204SetHDLCRxFIFODone(void *handle, TEword hdlc)
{
    TEword Address;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXHDLC_HCR;

    /* perform set operation*/
    if((Status = IXF3204WriteBit(handle,Address,TE_BIT0,TE_BIT0)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCTxFIFOSend
 * Returns: TEstatus
 * Description: Signals the specified HDLC module that the host has completed 
 *              the filling of a buffer (TE_FIFO) or the message is complete
 *              (TE_EOM), so the HDLC can start sending it. This function can 
 *              also abort the sending of the current message (TE_ABORT).
 ******************************************************************************/
TEstatus IXF3204SetHDLCTxFIFOSend(void *handle, TEword hdlc, TEword Value)
{
    TEword Address;
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Validate Value parameter*/
    switch(Value)
    {
        case TE_EOM:
            Data = TE_BIT0;
            break;
        case TE_FIFO:
            Data = TE_BIT1;
            break;
        case TE_ABORT:
            Data = TE_BIT2;
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    Address++;
    Mask = (TE_BIT2 |TE_BIT1 | TE_BIT0);

    /*perform set operation*/
    if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCRxMessageStatus
 * Returns: TEstatus
 * Description: Returns the status of the message received or being received 
 *              by the specified HDLC.
 ******************************************************************************/
TEstatus IXF3204GetHDLCRxMessageStatus(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    Address += IXF3204_RXHDLC_SR;
    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCRxMessageLength
 * Returns: TEstatus
 * Description: Returns the length of the message received.
 ******************************************************************************/
TEstatus IXF3204GetHDLCRxMessageLength(void *handle, TEword hdlc, void *Results)
{
    TEword Address,Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Read, of the low byte*/
    Address += IXF3204_RXHDLC_RMLR1;

    /*perform get operation*/
    if ((Status = IXF3204ReadWord(handle,Address, &Data)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCRxMessageAddressCompare
 * Returns: TEstatus
 * Description: Returns the comparison of the address of the message received 
 *              for the specified HDLC with the address specified in 
 *              IXF3204SetHDLCAddress.
 ******************************************************************************/
TEstatus IXF3204GetHDLCRxMessageAddressCompare(void *handle, TEword hdlc,
                                               void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXHDLC_AMSR;

    /*perform get operation*/
    if ((Status = IXF3204ReadByte(handle,Address,&Data)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCRxFIFOStatus
 * Returns: TEstatus
 * Description: Returns the current state of the receive FIFO.
 ******************************************************************************/
TEstatus IXF3204GetHDLCRxFIFOStatus(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXHDLC_IR;

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address, &Data))!=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCTxFIFOStatus
 * Returns: TEstatus
 * Description: Returns the current state of the transmit FIFO.
 ******************************************************************************/
TEstatus IXF3204GetHDLCTxFIFOStatus(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortTxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }
    Address += IXF3204_TXHDLC_IR;

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address, &Data)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCRxMessageData
 * Returns: TEstatus
 * Description: Returns the contents of the receive FIFO buffer for the 
 *              specified HDLC. The buffer size is one byte.
 ******************************************************************************/
TEstatus IXF3204GetHDLCRxMessageData(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the Get operations*/
    Address = IXF3204FindPortRxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_RXHDLC_FIFORD;

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address, &Data)) !=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCTxMessageData
 * Returns: TEstatus
 * Description: Writes a byte to the data transmission FIFO.
 ******************************************************************************/
TEstatus IXF3204SetHDLCTxMessageData(void *handle, TEword hdlc, TEbyte Data)
{
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }
    Address += IXF3204_TXHDLC_FIFOWR;
    /*perform set operation*/
    return(IXF3204WriteByte(handle,Address,Data));
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCReset
 * Returns: TEstatus
 * Description: Resets the FIFO and sets the specified HDLC to initial state 
 *              for the specified Direction.
 ******************************************************************************/
TEstatus IXF3204SetHDLCReset(void *handle, TEword hdlc, TEword Direction)
{
    TEword AddressRx, AddressTx;
    TEbyte Mask = TE_BIT1;
    TEbyte Data = TE_BIT1;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    AddressTx = IXF3204FindPortTxHDLC(hdlc);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressRx = IXF3204FindPortRxHDLC(hdlc);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate Direction parameter and perform corresponding write*/
    switch(Direction)
    {
        case TE_RX:
            /* Set all the ports with the same configuration */
            if(hdlc == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressRx,
                                                         Data,Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,AddressRx,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            break;
        case TE_TX:
            /* Set all the ports with the same configuration */
            if(hdlc == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressTx,
                                                         Data,Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteBit(handle,AddressTx,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            break;
        case TE_BOTH:
            /* Set all the ports with the same configuration */
            if(hdlc == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressRx,
                                                         Data,Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressTx,
                                                         Data,Mask)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteBit(handle,AddressRx,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                
                if((Status = IXF3204WriteBit(handle,AddressTx,Data,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCTxInterframe
 * Returns: TEstatus
 * Description: Configures the sequence that must be sent between frames. The 
 *              allowed sequence is ones and flags (01111110). 
 ******************************************************************************/
TEstatus IXF3204SetHDLCTxInterframe(void *handle, TEword hdlc, TEword Value)
{
    TEword Address;
    TEbyte Mask = TE_BIT5;
    TEbyte Data;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Value parameter*/
    switch(Value)
    {
        case TE_FLAGS:
            Data = TE_BIT5;
            break;
        case TE_ONES:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllHDLCMem(handle,Address,Data,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    if((Status = IXF3204WriteBit(handle,Address,Data, Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetHDLCTxInterframe
 * Returns: TEstatus
 * Description: Returns the configured sequence that is sent between frames. 
 *              The allowed sequence is ones and flags (01111110). 
 ******************************************************************************/
TEstatus IXF3204GetHDLCTxInterframe(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEbyte Mask = TE_BIT5, Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortTxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data parameter*/
    switch(Data)
    {
        case TE_BIT5:
            ReturnValue[0] = TE_FLAGS;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_ONES;
            break;
        default:
            return TE_STATUS_BADREAD; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetHDLCOperationMode
 * Returns: TEstatus
 * Description: Configures the HDLC in both directions to analyze and generate 
 *              flags. When Normal mode is selected, FCS is generated and 
 *              detected. When Transparent mode is selected, generation and 
 *              detection of headers is not done. 
 ******************************************************************************/
TEstatus IXF3204SetHDLCOperationMode(void *handle, TEword hdlc, TEword Mode)
{
    TEword AddressRx, AddressTx;
    TEbyte Mask = TE_BIT3;
    TEbyte Data;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    AddressRx = IXF3204FindPortRxHDLC(hdlc);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    AddressTx = IXF3204FindPortTxHDLC(hdlc);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Validate Mode parameter*/
    switch(Mode)
    {
        case TE_TRANSPARENT:
            Data = TE_BIT3;
            break;
        case TE_NORMAL:  
            Data = TE_CLEAR_BITS;
            break;
        default:             
            return TE_STATUS_BAD_PARAMS; 
    }

    /* Set all the ports with the same configuration */
    if(hdlc == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressTx,Data,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if((Status = IXF3204WriteBitAllHDLCMem(handle,AddressRx,Data,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    if((Status = IXF3204WriteBit(handle,AddressTx,Data, Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if((Status = IXF3204WriteBit(handle,AddressRx,Data, Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}


