/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   intr3204.c
 * Description: this file contains interrupt functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetInterruptPortMask
 * Returns: TEstatus
 * Description: Masks the interrupts generated by each port. Each bit 
 *              corresponds to a port. Bit 0 corresponds to port 1. Bit 1 
 *              corresponds to port 2, and so on. Setting the bit enables the
 *              interrupt.
 ******************************************************************************/
TEstatus IXF3204SetInterruptPortMask(void *handle, TEbyte Mask)
{
    TEword Address = IXF3204_IMASKA;

    /*Perform the Set operation*/
    return(IXF3204WriteByte(handle,Address, Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptPortMask
 * Returns: TEstatus
 * Description: Returns the configured mask. Each bit corresponds to a port. 
 *              Bit 0 corresponds to the port 1. Bit 1 corresponds to port 2,
 *              and so on.
 ******************************************************************************/
TEstatus IXF3204GetInterruptPortMask(void *handle, void *Results)
{
    TEword Address = IXF3204_IMASKA;
    TEbyte Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptPort
 * Returns: TEstatus
 * Description: Returns the state of the interrupts for all ports.
 ******************************************************************************/
TEstatus IXF3204GetInterruptPort(void *handle, void *Results)
{
    TEword Address = IXF3204_ISTA;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptHDLCBERTMask
 * Returns: TEstatus
 * Description: Masks the interrupts generated by the BERT and HDLC 
 *              controllers. Setting the bit enables the interrupt.
 ******************************************************************************/
TEstatus IXF3204SetInterruptHDLCBERTMask(void *handle, TEbyte Mask)
{
    TEbyte intMask;

    TEword Address = IXF3204_IMASKB;

    intMask = TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3;

    if(intMask > (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Perform the Set operation*/
    return(IXF3204WriteBit(handle,Address,Mask,intMask));
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptHDLCBERTMask
 * Returns: TEstatus
 * Description: Returns the configured interrupt mask.
 ******************************************************************************/
TEstatus IXF3204GetInterruptHDLCBERTMask(void *handle, void *Results)
{
    TEbyte intMask;

    TEword Address = IXF3204_IMASKB;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    intMask = TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle, Address, intMask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptHDLCBERT
 * Returns: TEstatus
 * Description: Returns the state of the line from the HDLC's and BERT's.
 ******************************************************************************/
TEstatus IXF3204GetInterruptHDLCBERT(void *handle, void *Results)
{
    TEbyte intMask;

    TEword Address = IXF3204_ISTB;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    intMask = TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle, Address, intMask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptSourceMask
 * Returns: TEstatus
 * Description: Sets the mask in the interrupt source register for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204SetInterruptSourceMask(void *handle, TEword Port, TEbyte Mask)
{
    TEstatus Status;
    TEword Address;

    Address = IXF3204FindPortMISRC(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port ==TE_ALL)
    {
        if((Status = IXF3204WriteByteAllInt(handle,Address,Mask,
                                            IXF3204_SOURCE_INT)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perform the Set operation*/
    return(IXF3204WriteByte(handle,Address, Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptSourceMask
 * Returns: TEstatus
 * Description: Returns the mask of the interrupt source register for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptSourceMask(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortMISRC(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptSource
 * Returns: TEstatus
 * Description: Returns the value of the interrupt source register for the 
 *              specified Port. This function returns the source of the 
 *              interrupt.
 ******************************************************************************/
TEstatus IXF3204GetInterruptSource(void *handle,TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortISRC(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptAlarmsMask
 * Returns: TEstatus
 * Description: Sets the mask for the alarm interrupt register for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204SetInterruptAlarmsMask(void  *handle, TEword Port,
                                       TEbyte RxAlarms1, TEbyte RxAlarms2,
                                       TEbyte TxAlarms1)
{

    TEstatus Status;
    TEword Address1, Address2, Address3;

    Address1 = IXF3204FindPortMR1ALRMI(Port);
    if(Address1 == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address2 = IXF3204FindPortMR2ALRMI(Port);
    if(Address2 == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address3 = IXF3204FindPortMTXALRM(Port);
    if(Address3 == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port ==TE_ALL)
    {
        if((Status = IXF3204WriteByteAllInt(handle,Address1,RxAlarms1,
                                            IXF3204_MODULE_INT)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if((Status = IXF3204WriteByteAllInt(handle,Address2,RxAlarms2,
                                            IXF3204_MODULE_INT)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if((Status = IXF3204WriteByteAllInt(handle,Address3,TxAlarms1,
                                            IXF3204_MODULE_INT)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,Address1, RxAlarms1)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,Address2, RxAlarms2)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Perform the Set operation*/
    return(IXF3204WriteByte(handle,Address3, TxAlarms1));
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptAlarmsMask
 * Returns: TEstatus
 * Description: Returns the configured mask for the alarms interrupts for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptAlarmsMask(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortMR1ALRMI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    Address = IXF3204FindPortMR2ALRMI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[1] = Data;

    Address = IXF3204FindPortMTXALRM(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[2] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptAlarms
 * Returns: TEstatus
 * Description: Returns the value of the alarms interrupt registers for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptAlarms(void *handle, TEword Port, TEword Alarm, 
                                   void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /* read the selectd alarm */
    switch(Alarm)
    {
        case TE_RXALARM1:
            Address = IXF3204FindPortR1ALRMI(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;  
            }

            /*Perform the get operation*/
            if((Status = IXF3204ReadByte(handle, Address, &Data)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            ReturnValue[1] = Data;

            Address += IXF3204_ADDR_OFFSET8;
            /*Perform the get operation*/
            if((Status = IXF3204ReadByte(handle, Address, &Data)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            ReturnValue[0] = Data;
            break;
        case TE_RXALARM2:
            Address = IXF3204FindPortR2ALRMI(Port);
            /*Perform the get operation*/
            if((Status =IXF3204ReadByte(handle, Address, &Data)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            ReturnValue[1] = Data;
            Address += IXF3204_ADDR_OFFSET8;
            /*Perform the get operation*/
            if((Status = IXF3204ReadByte(handle, Address, &Data)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            ReturnValue[0] = Data;
            break;
        case TE_TXALARM:
            Address = IXF3204FindPortTXALRM(Port);
            /*Perform the get operation*/
            if((Status = IXF3204ReadByte(handle, Address, &Data)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            ReturnValue[1] = Data;
            Address += IXF3204_ADDR_OFFSET8;
            /*Perform the get operation*/
            if((Status = IXF3204ReadByte(handle, Address, &Data)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            ReturnValue[0] = Data;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptFramerMask
 * Returns: TEstatus
 * Description: Sets the mask for the framer interrupt register for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204SetInterruptFramerMask(void *handle, TEword Port, TEbyte Mask)
{
    TEstatus Status;
    TEword Address;

    Address = IXF3204FindPortMFRAMEI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port ==TE_ALL)
    {
        if((Status = IXF3204WriteByteAllInt(handle,Address,Mask,
                                            IXF3204_MODULE_INT)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return(IXF3204WriteByte(handle,Address, Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptFramerMask
 * Returns: TEstatus
 * Description: Returns the configured mask for the framer interrupts for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptFramerMask(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortMFRAMEI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptFramer
 * Returns: TEstatus
 * Description: Returns the value of the framer interrupt register for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptFramer(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortFRAMEI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[1] = Data;

    Address += IXF3204_ADDR_OFFSET8;
    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptFDLMask
 * Returns: TEstatus
 * Description: Sets the mask for the FDL interrupt register for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204SetInterruptFDLMask(void *handle, TEword Port, TEbyte Mask)
{
    TEstatus Status;
    TEword Address;

    Address = IXF3204FindPortMFDLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port ==TE_ALL)
    {
        if((Status = IXF3204WriteByteAllInt(handle,Address,Mask,
                                            IXF3204_MODULE_INT)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perform the Set operation*/
    return(IXF3204WriteByte(handle,Address, Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptFDLMask
 * Returns: TEstatus
 * Description: Returns the configured mask for the FDL interrupts for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptFDLMask(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortMFDLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }
    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptFDL
 * Returns: TEstatus
 * Description: Returns the value of the FDL interrupt register for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptFDL(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortFDLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptEventsMask
 * Returns: TEstatus
 * Description: Sets the mask for the interrupt register related to events 
 *              such as: slip, signaling change, code violation, multiframe 
 *              tick, etc. for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetInterruptEventsMask(void *handle, TEword Port, TEbyte Mask1,
                                       TEbyte Mask2)
{
    TEword Address1, Address2;
    TEstatus Status;

    Address1 = IXF3204FindPortMEVENTS1I( Port);
    if(Address1 == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    Address2 = IXF3204FindPortMEVENTS2I(Port);
    if(Address2 == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /* Set all the ports with the same configuration */
    if(Port ==TE_ALL)
    {
        if((Status = IXF3204WriteByteAllInt(handle,Address1,Mask1,
                                            IXF3204_MODULE_INT)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        if((Status = IXF3204WriteByteAllInt(handle,Address2,Mask2,
                                            IXF3204_MODULE_INT)) 
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,Address1, Mask1)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return(IXF3204WriteByte(handle,Address2, Mask2));
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptEventsMask
 * Returns: TEstatus
 * Description: Returns the configured mask for the interrupt events for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptEventsMask(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortMEVENTS1I(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    Address = IXF3204FindPortMEVENTS2I(Port);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[1] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptEvent
 * Returns: TEstatus
 * Description: Returns the value of the Event interrupt registers specified 
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetInterruptEvent(void *handle, TEword Port, TEword Event,
                                  void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    switch(Event)
    {
        case TE_EVENT1:
            Address = IXF3204FindPortEVENTS1I(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        case TE_EVENT2:
            Address = IXF3204FindPortEVENTS2I(Port);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            break;
        default:     
            return TE_STATUS_BAD_PARAMS; 
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptHDLCMask
 * Returns: TEstatus
 * Description: Sets the mask for the transmit and receive interrupt register 
 *              related to HDLC for the specified Port. TxMask1 sets the mask 
 *              for transmit HDLC 0 to 7. TxMask2 sets the mask for transmit
 *              HDLC 8 to 15, and so on.
 ******************************************************************************/
TEstatus IXF3204SetInterruptHDLCMask(void *handle, TEbyte TxMask1, 
                                     TEbyte TxMask2,TEbyte TxMask3, 
                                     TEbyte RxMask4, TEbyte RxMask5,
                                     TEbyte RxMask6)
{
    TEstatus Status;
  
    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,IXF3204_MTXHDLC0, TxMask1)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,IXF3204_MTXHDLC1, TxMask2)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,IXF3204_MTXHDLC2, TxMask3)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,IXF3204_MRXHDLC0, RxMask4)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,IXF3204_MRXHDLC1, RxMask5)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /*Perform the Set operation*/
    if((Status = IXF3204WriteByte(handle,IXF3204_MRXHDLC2, RxMask6)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptHDLCMask
 * Returns: TEstatus
 * Description: Returns the configured mask for the HDLC interrupt.
 ******************************************************************************/
TEstatus IXF3204GetInterruptHDLCMask(void *handle, void *Results)
{
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, IXF3204_MTXHDLC0, &Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, IXF3204_MTXHDLC1, &Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[1] = Data;

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, IXF3204_MTXHDLC2, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[2] = Data;

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, IXF3204_MRXHDLC0, &Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[3] = Data;

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, IXF3204_MRXHDLC1, &Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[4] = Data;

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, IXF3204_MRXHDLC2, &Data)) != 
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[5] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptHDLC
 * Returns: TEstatus
 * Description: Returns the value of the HDLC interrupt register specified.
 ******************************************************************************/
TEstatus IXF3204GetInterruptHDLC(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    switch(hdlc)
    {
        case TE_RXHDLC1:
            Address = IXF3204_RXHDLC0;
            break;
        case TE_RXHDLC2:
            Address = IXF3204_RXHDLC1;
            break;
        case TE_RXHDLC3:
            Address = IXF3204_RXHDLC2;
            break ;
        case TE_TXHDLC1:
            Address = IXF3204_TXHDLC0;
            break;
        case TE_TXHDLC2:
            Address = IXF3204_TXHDLC1;
            break;
        case TE_TXHDLC3:
            Address = IXF3204_TXHDLC2;
            break;
        default:    
            return TE_STATUS_BAD_PARAMS;
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptBERTMask
 * Returns: TEstatus
 * Description: Sets the mask for the BERT interrupt register.
 ******************************************************************************/
TEstatus IXF3204SetInterruptBERTMask(void *handle, TEbyte Mask)
{
    /*Perform the Set operation*/
    return(IXF3204WriteByte(handle,IXF3204_MBERT, Mask));

}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptBERTMask
 * Returns: TEstatus
 * Description: Returns the configured mask for the BERT interrupts.
 ******************************************************************************/
TEstatus IXF3204GetInterruptBERTMask(void *handle, void *Results)
{
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, IXF3204_MBERT, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptBERT
 * Returns: TEstatus
 * Description: Returns the value of the BERT interrupt register.
 ******************************************************************************/
TEstatus IXF3204GetInterruptBERT(void *handle, void *Results)
{
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Perform the get operation*/
    if((Status = IXF3204ReadByte(handle, IXF3204_BERT, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptFirmwareMask
 * Returns: TEstatus
 * Description: Sets the mask for the Firmware interrupt register.
 ******************************************************************************/
TEstatus IXF3204SetInterruptFirmwareMask(void *handle, TEword Mask)
{
    TEbyte intMask, value;

    intMask = TE_BIT7;

    /* if any mask wants to enable, set the bit in the highest int register */
    if(Mask != 0x0000)
    {
        value = TE_BIT7;
    }
    else
    {
      value = TE_CLEAR_BITS;
    }

    /* enable/disable firmware interrupt in the upper level of the tree */
    CHK_STATUS(IXF3204WriteBit(handle,IXF3204_IMASKB,value,intMask));

    /* enable/disable firmware interrupt in the lower level of the tree */
    return (IXF3204WriteWord(handle,IXF3204_FW1MASK,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptFirmwareMask
 * Returns: TEstatus
 * Description: Gets the mask for the Firmware interrupt register.
 ******************************************************************************/
TEstatus IXF3204GetInterruptFirmwareMask(void *handle, void * Results)
{
    TEword data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /* reads the mask registers */
    IXF3204ReadWord(handle,IXF3204_FW1MASK,&data);

    ReturnValue[0] = data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetInterruptFirmware
 * Returns: TEstatus
 * Description: Return the value of the Firmware interrupt register.
 ******************************************************************************/
TEstatus IXF3204GetInterruptFirmware(void *handle, void * Results)
{
    TEword data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /* reads the mask registers */
    IXF3204ReadWord(handle,IXF3204_FW1,&data);

    ReturnValue[0] = data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptTxHDLC
 * Returns: TEstatus
 * Description: Enables or disables the interrupt generation when transmit EOM
 ******************************************************************************/
TEstatus IXF3204SetInterruptTxHDLC(void *handle, TEword hdlc, TEword mode)
{
    TEword Address;
    TEbyte data, mask;

    mask = TE_BIT7;

    Address = IXF3204FindPortTxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* validate mode parameter */
    switch(mode)
    {
        case TE_ENABLE:
            data = TE_BIT7;
            break;
        case TE_DISABLE:
            data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    if(hdlc == TE_ALL)
    {
        /* apply to all hdlc's */
        CHK_STATUS(IXF3204WriteBitAllHDLCMem(handle,Address,data,mask));
    }
    else
    {
        /* apply only to one HDLC */
        CHK_STATUS(IXF3204WriteBit(handle,Address,data,mask));
    }

    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204GetInterruptTxHDLC
 * Returns: TEstatus
 * Description: This message returns if the transmission of EOM generate 
 *              interrupt.
 ******************************************************************************/
TEstatus IXF3204GetInterruptTxHDLC(void *handle, TEword hdlc, void *Results)
{
    TEword Address;
    TEbyte data, mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    mask = TE_BIT7;

    Address = IXF3204FindPortTxHDLC(hdlc);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,mask,&data));

    /* validate parameter read */
    if(data)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptTxFDL
 * Returns: TEstatus
 * Description: Enables or disables the interrupt generation when transmit EOM
 ******************************************************************************/
TEstatus IXF3204SetInterruptTxFDL(void *handle, TEword port, TEword mode)
{
    TEword Address;
    TEbyte data, mask;

    mask = TE_BIT7;

    Address = IXF3204FindPortTxFDL(port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* validate mode parameter */
    switch(mode)
    {
        case TE_ENABLE:
            data = TE_BIT7;
            break;
        case TE_DISABLE:
            data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    if(port == TE_ALL)
    {
        /* apply to all ports */
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,data,mask));
    }
    else
    {
        /* apply only to one FDL */
        CHK_STATUS(IXF3204WriteBit(handle,Address,data,mask));
    }

    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204GetInterruptTxFDL
 * Returns: TEstatus
 * Description: This message returns if the transmission of EOM generate 
 *              interrupt.
 ******************************************************************************/
TEstatus IXF3204GetInterruptTxFDL(void *handle, TEword port, void *Results)
{
    TEword Address;
    TEbyte data, mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    mask = TE_BIT7;

    Address = IXF3204FindPortTxFDL(port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,mask,&data));

    /* validate parameter read */
    if(data)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetInterruptRxCodeword
 * Returns: TEstatus
 * Description: Enables or disables the interrupt generation when receiving
 *              changes in the RAI or Sa5 bit of the codeword
 ******************************************************************************/
TEstatus IXF3204SetInterruptRxCodeword(void *handle, TEword port, TEword sa5,
                                       TEword rai)
{
    TEword Address;
    TEbyte data, mask;

    mask = TE_BIT5 | TE_BIT4;

    Address = IXF3204FindPortNational(port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* validate sa5 parameter */
    switch(sa5)
    {
        case TE_ENABLE:
            data = TE_BIT5;
            break;
        case TE_DISABLE:
            data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* validate rai parameter */
    switch(rai)
    {
        case TE_ENABLE:
            data |= TE_BIT4;
            break;
        case TE_DISABLE:
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    if(port == TE_ALL)
    {
        /* apply to all ports */
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,data,mask));
    }
    else
    {
        /* apply only to one FDL */
        CHK_STATUS(IXF3204WriteBit(handle,Address,data,mask));
    }

    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204GetInterruptRxCodeword
 * Returns: TEstatus
 * Description: This message returns if changes in the reception of Sa5 or RAI 
 *              bit generate Interrupt.
 ******************************************************************************/
TEstatus IXF3204GetInterruptRxCodeword(void *handle, TEword port, void *Results)
{
    TEword Address;
    TEbyte data, mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    mask = TE_BIT5 | TE_BIT4;

    Address = IXF3204FindPortNational(port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    CHK_STATUS(IXF3204ReadBit(handle,Address,mask,&data));

    /* check for Sa5 bit */
    if(data & TE_BIT5)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    /* check for RAI bit */
    if(data & TE_BIT4)
    {
        ReturnValue[1] = TE_ENABLE;
    }
    else
    {
        ReturnValue[1] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

