/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY  OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   io3204.c
 * Description: This file has the Encapsulated Access Messges defined in
 *              the API
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204WriteByte
 * Returns: TEstatus
 * Description: Writes a byte to the address specified in Address. This 
 *              function writes the entire byte to the register address 
 *              overwriting the previous register contents.
 ******************************************************************************/
TEstatus IXF3204WriteByte( void *handle, TEword Address, TEbyte Value )
{

    CHK_STATUS(IXF3204OSWriteByte(handle, Address, Value));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204ReadByte
 * Returns: TEstatus
 * Description: Reads the value from the specified address. This function 
 *              returns the entire byte value from the register address.
 ******************************************************************************/
TEstatus IXF3204ReadByte(void *handle, TEword Address, TEbyte *Results)
{

    CHK_STATUS(IXF3204OSReadByte(handle, Address, Results));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteBit
 * Returns: TEstatus
 * Description: Writes a bit-masked value to the specified address. The mask 
 *              ensures that only specified bits within the register are 
 *              changed.
 ******************************************************************************/
TEstatus IXF3204WriteBit( void *handle, TEword Address, TEbyte Value, 
                          TEbyte Mask )
{
    TEbyte Data;

     /*first read the Address to be write masked*/
    CHK_STATUS(IXF3204ReadByte(handle, Address, &Data));
     
    /*apply the value using the mask to the read value*/
    Value =  (TEbyte)((Data & ~Mask) | (Mask & Value));

    /*write the Value masked to Address*/
    return IXF3204WriteByte(handle, Address,Value);
}

/*******************************************************************************
 * Function Name: IXF3204ReadBit
 * Returns: TEstatus
 * Description: Reads the value from the specified address. Applies the mask 
 *              specified in the Mask parameter to return only desired bit 
 *              values. The other bits are 0. A 1 in the mask indicates that 
 *              the bit is to be read
 ******************************************************************************/
TEstatus IXF3204ReadBit( void *handle, TEword Address, TEbyte Mask, 
                         TEbyte *Results )
{
    TEbyte Value;

    CHK_STATUS(IXF3204ReadByte(handle, Address, &Value));
    Results[0] = (TEbyte)(Mask & Value);

    return TE_STATUS_SUCCESS;	
}

/*******************************************************************************
 * Function Name: IXF3204WriteWord
 * Returns: TEstatus
 * Description: Writes a 16 bits value, writed the lowest byte to the address 
 *              specified in Address, and the upper byte to the address 
 *              specified in Address + 1. This function writes the entire bytes 
 *              to the register addresses overwriting the previous registers 
 *              contents
 ******************************************************************************/
TEstatus IXF3204WriteWord( void *handle, TEword Address, TEword Value )
{
    TEbyte Data;
        
    /* write the LSB */
    Data = (TEbyte) Value;
    CHK_STATUS(IXF3204WriteByte(handle, Address, Data));

    /* write the MSB */
    Address++;
    Data = (TEbyte) (Value >> 8);
    return IXF3204WriteByte(handle, Address, Data);
}

/*******************************************************************************
 * Function Name: IXF3204ReadWord
 * Returns: TEstatus
 * Description: Reads the value from the specified address (the lower byte) 
 *              and the consecutive address (the upper byte). This function 
 *              returns the entire 16 bits value from the register address.
 ******************************************************************************/
TEstatus IXF3204ReadWord(void *handle, TEword Address, TEword *Results )
{
    TEbyte Value = 0;
    TEword Data = 0;

    /*read the LSB*/ 
    CHK_STATUS(IXF3204ReadByte(handle, Address, &Value));
    Results[0] = Value;

    /*read the MSB*/ 
    Address++;
    CHK_STATUS(IXF3204ReadByte(handle, Address, &Value));
    Data = Value;
    Results[0] = (TEword)(Results[0] | (Data << 8));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WritedWord
 * Returns: TEstatus
 * Description: Writes a 32 bits value, writed the lowest byte to the address 
 *              specified in Address, and the follow byte to the address 
 *              specified in Address + 1, and so on. This function writes the 
 *              entire bytes to the register addresses overwriting the previous
 *              registers contents.
 ******************************************************************************/
TEstatus IXF3204WritedWord(void *handle, TEword Address, TEdword Value )
{
    TEbyte Data;
        
    /* write the LSB */
    Data = (TEbyte)Value;
    CHK_STATUS(IXF3204WriteByte(handle, Address, Data));

    /* write Byte */
    Address++;
    Data = (TEbyte)(Value >> 8);
    CHK_STATUS(IXF3204WriteByte(handle, Address, Data));

    /* write B */
    Address++;
    Data = (TEbyte)(Value >> 16);
    CHK_STATUS(IXF3204WriteByte(handle, Address, Data));

    /*write the MSB*/
    Address++;
    Data = (TEbyte)(Value >> 24);
    return IXF3204WriteByte(handle, Address, Data);
}

/*******************************************************************************
 * Function Name: IXF3204ReaddWord
 * Returns: TEstatus
 * Description: Reads the value from the specified address (the lower byte) and 
 *              the consecutive address (the second byte) and so on. This 
 *              function returns the entire 32 bits value from the register 
 *              address.
 ******************************************************************************/
TEstatus IXF3204ReaddWord( void *handle, TEword Address, TEdword *Results )
{
    TEbyte Value;
    TEdword Data;

    /* read the LSB */ 
    CHK_STATUS(IXF3204ReadByte(handle, Address, &Value));
    Results[0] = Value;

    Address++;
    CHK_STATUS(IXF3204ReadByte(handle, Address, &Value));
    Data = Value;
    Results[0] = Results[0] | (Data << 8);

    Address++;
    CHK_STATUS(IXF3204ReadByte(handle, Address, &Value));
    Data = Value;
    Results[0] = Results[0] | (Data << 16);

    /*read the MSB*/ 
    Address++;
    CHK_STATUS(IXF3204ReadByte(handle, Address, &Value));
    Data = Value;
    Results[0] = Results[0] | (Data << 24);

    return TE_STATUS_SUCCESS;
}


