/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   loop3204.c
 * Description: this file contains loop back functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetLoopbackMode
 * Returns: TEstatus
 * Description: Configures the LIU loopback mode for the specified Port. 
 *              When TE_ENABLE is selected, the loopback specified by loop is
 *              enabled. 
 *              When TE_LINE is selected, the data coming from the line is
 *              looped immediately. 
 *              When TE_PAYLOAD is selected, the data coming from the line is
 *              looped after the framer. 
 *              When TE_LOCAL is selected, the data coming from the backplane 
 *              is looped on the line side.
 ******************************************************************************/
TEstatus IXF3204SetLoopbackMode(void *handle, TEword Port, TEword Loop,
                                TEword Mode)
{
    TEword AddressPLI, AddressSig, Address;
    TEbyte Data, Mask;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    AddressPLI = IXF3204FindPortPLI(Port);
    if(AddressPLI == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Get offsets to perform the set operations*/
    AddressSig = IXF3204FindPortTxSignaling_CDR(Port);
    if(AddressSig == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Loop parameter*/
    switch(Loop)
    {
        case TE_LINE:
            /*Validate the Mode parameter*/
            switch(Mode)
            {
                case TE_ENABLE:
                    Data = TE_BIT0;
                    break;
                case TE_DISABLE:
                    Data = TE_CLEAR_BITS;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(AddressPLI + IXF3204_LIPATH);
            Mask = TE_BIT0;
            break;
        case TE_LOCAL:
            /*Validate the Mode parameter*/
            switch(Mode)
            {
                case TE_ENABLE:
                    Data = TE_BIT4;
                    break;
                case TE_DISABLE:
                    Data = TE_CLEAR_BITS;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(AddressPLI + IXF3204_LIPATH);
            Mask = TE_BIT4;
            break;
        case TE_PAYLOAD:
            /*Validate the Mode parameter*/
            switch(Mode)
            {
                case TE_ENABLE:
                    Data = TE_BIT5;
                    break;
                case TE_DISABLE:
                    Data = TE_CLEAR_BITS;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
            Address = (TEword)(AddressSig + IXF3204_TXSIG_MAP);
            Mask = (TE_BIT6 | TE_BIT5);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }
    
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetLoopbackMode
 * Returns: TEstatus
 * Description: Returns the device loopback mode for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLoopbackMode(void *handle, TEword Port, TEword Loop,
                                 void *Results)
{
    TEword AddressPLI, AddressSig, Address;
    TEbyte Data, Mask = TE_BIT5;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    AddressPLI = IXF3204FindPortPLI(Port);
    if(AddressPLI == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*Get offsets to perform the get operations*/
    AddressSig = IXF3204FindPortTxSignaling_CDR(Port);
    if(AddressSig == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Loop parameter*/
    switch(Loop)
    {
        case TE_LINE:
            Address = (TEword)(AddressPLI + IXF3204_LIPATH);
            Mask = TE_BIT0;
            break;
        case TE_LOCAL:
            Address = (TEword)(AddressPLI + IXF3204_LIPATH);
            Mask = TE_BIT4;
            break;
        case TE_PAYLOAD:
            Address = (TEword)(AddressSig + IXF3204_TXSIG_MAP);
            Mask = (TE_BIT6 | TE_BIT5);
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    
    /*Validate the Data parameter*/
    switch(Data)
    {
        case TE_BIT4:
        case TE_BIT5:
        case TE_BIT0:
            ReturnValue[0] = TE_ENABLE;
            break;
        case (TE_BIT6 | TE_BIT5):
        case TE_BIT6:
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default: 
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLoopbackPayload
 * Returns: TEstatus
 * Description: Sets the specified time slot (or all time slots when 
 *              TE_ALL is selected) for the specified Port in payload mode 
 *              when TE_ENABLE is selected. The Mask parameter specifies the 
 *              bits that are looped back. A 1 enables the bit to be looped. 
 *              When Mask is 0xFF, all bits are looped.
 ******************************************************************************/
TEstatus IXF3204SetLoopbackPayload(void *handle, TEword Port, TEword TimeSlot,
                                   TEword Mode, TEbyte Mask)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask1, Loop;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxSignalingLine_CDM(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    if(((TimeSlot < TE_TS0) || (TimeSlot > TE_TS31)) && (TimeSlot != TE_ALL))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT5;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    Mask1 = (TE_BIT6 | TE_BIT5);
    /* if TE_ALL start from TS0 */
    if(TimeSlot == TE_ALL)
    {
        Address = (TEword)(Address + IXF3204GetTimeSlot(TE_TS0));
    }
    else
    {
        Address = (TEword)(Address + IXF3204GetTimeSlot(TimeSlot));
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if(TimeSlot != TE_ALL)
        {
            Loop = (TEbyte)(IXF3204_MAX_TS - 1);
        }
        else
        {
            /* if TS == TE_ALL start from TS0 */
            Loop = 0;
        }
        for(; Loop < IXF3204_MAX_TS; Loop++)
        {
            if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Data,Mask1))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
            Address++;
        }
    }
    else
    {
        if(TimeSlot != TE_ALL)
        {
            Loop = (TEbyte)(IXF3204_MAX_TS - 1);
        }
        else
        {
            /* if TS == TE_ALL start from TS0 */
            Loop = 0;
        }
        for(; Loop < IXF3204_MAX_TS; Loop++)
        {
            /*perform set operation*/
            if((Status = IXF3204WriteBit(handle,Address,Data,Mask1)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
            Address++;
        }
        
    }

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXSIG_LOOPM;


    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return(IXF3204WriteByte(handle,Address, Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetLoopbackPayload
 * Returns: TEstatus
 * Description: Returns the payload loopback status of the specified time slot
 *              and the configured mask for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLoopbackPayload(void *handle, TEword Port, TEword TimeSlot,
                                   void *Results)
{
    TEword Address;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortTxSignalingLine_CDM(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    if((TimeSlot < TE_TS0) || (TimeSlot > TE_TS31))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*IXF3204Reads payload loopback and Mask*/
    Address = (TEword)(Address + IXF3204GetTimeSlot(TimeSlot));
    Mask = (TE_BIT6 | TE_BIT5);
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data;

    /*Validate the Data parameter*/
    switch(Data)
    {
        case TE_BIT5:
            ReturnValue[0] = TE_ENABLE;
            break;
        default: 
            /*
              if different than 0x20 must be declared as disable
               because the same register affects mapping
            */
            ReturnValue[0] = TE_DISABLE;
            break;
    }

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXSIG_LOOPM;

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[1] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLoopbackDetection
 * Returns: TEstatus
 * Description: Configures the in-band loop codes detection for the specified 
 *              Port.
 ******************************************************************************/
TEstatus IXF3204SetLoopbackDetection(void *handle, TEword Port, TEword Mode)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask1 = TE_BIT2;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT2;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;    
    }

    Address += IXF3204_TXALRM_CACR;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask1));
}

/*******************************************************************************
 * Function Name: IXF3204GetLoopbackDetection
 * Returns: TEstatus
 * Description: Returns the in-band loop codes detection configuration for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLoopbackDetection(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT2;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXALRM_CACR;

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Validate the Data parameter*/
    switch(Data)
    {
        case TE_BIT2:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default: 
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLoopbackReAction
 * Returns: TEstatus
 * Description: Configures loopback reaction for the specified Port. Enables 
 *              or disables the Intel IXF3204 to execute the loop when 
 *              detected in-band or out-band.
 ******************************************************************************/
TEstatus IXF3204SetLoopbackReAction(void *handle, TEword Port, TEword Mode)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask1 = TE_BIT3;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT3;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS; 
    }

    Address += IXF3204_TXALRM_CACR;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*perform set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask1));
}

/*******************************************************************************
 * Function Name: IXF3204GetLoopbackReAction
 * Returns: TEstatus
 * Description: Returns the configuration to allow the Intel IXF3204 to 
 *              create a loop when detected in-band and out-band for the
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLoopbackReAction(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT3;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXALRM_CACR;
    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Validate the Data parameter*/
    switch(Data) 
    {
        case TE_BIT3:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default: 
            return TE_STATUS_BADREAD;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLoopbackTxInBand
 * Returns: TEstatus
 * Description: Enables or disables the transmission of the in-band loop code 
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetLoopbackTxInBand(void *handle, TEword Port, TEword Code)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask1 = (TE_BIT1 | TE_BIT0);

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Code parameter*/
    switch(Code)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_LOOPUP:
            Data = (TE_BIT1 | TE_BIT0);
            break;
        case TE_LOOPDOWN:
            Data = TE_BIT1;
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    Address += IXF3204_TXALRM_CACR;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask1));
}

/*******************************************************************************
 * Function Name: IXF3204GetLoopbackTxInBand
 * Returns: TEstatus
 * Description: Returns the in-band code currently transmitted for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLoopbackTxInBand(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data;
    TEbyte Mask = (TE_BIT1 | TE_BIT0);
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortTxAlarms(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXALRM_CACR;
    /*perform get operation*/
    Status = IXF3204ReadBit(handle,Address, Mask,&Data);
    if(Status != TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Validate the Data parameter*/
    switch(Data)
    {
        case (TE_BIT1 | TE_BIT0):
            ReturnValue[0] = TE_LOOPUP;
            break;
        case TE_BIT1:
            ReturnValue[0] = TE_LOOPDOWN;
            break;
        default:
            ReturnValue[0] = TE_DISABLE;
            break;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetLoopcodeRxInBand
 * Returns: TEstatus
 * Description: Return the status of the receive loop code for the specified 
 *              Port. 
 ******************************************************************************/
TEstatus IXF3204GetLoopcodeRxInBand(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Mask, Data;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortTxAlarms(Port);

    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXALRM_CASR;
    Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetLoopbackStatus
 * Returns: TEstatus
 * Description: Return the loopback status- activated local or remotely for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLoopbackStatus(void *handle, TEword Port, void *Results )
{
    TEword AddressPLI, AddressSig;
    TEbyte Mask, ValueLine, ValuePayload;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    AddressPLI = IXF3204FindPortPLI(Port);
    if(AddressPLI == TE_STATUS_BAD_PARAMS) 
    {
        return TE_STATUS_BAD_PARAMS;
    }
    AddressPLI += IXF3204_LILOOP;

    /*Get offsets to perform the get operations*/
    AddressSig = IXF3204FindPortTxSignaling_CDR(Port);
    if(AddressSig == TE_STATUS_BAD_PARAMS) 
    {
        return TE_STATUS_BAD_PARAMS;
    }
    AddressSig += IXF3204_TXSIG_LOOPST;
    Mask = TE_BIT0;

    /* Read the line loopback status bit */
    Status = IXF3204ReadBit(handle, AddressPLI, Mask, &ValueLine);
    if(Status != TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* Read the payload loopback status bit */
    Status = IXF3204ReadBit(handle, AddressSig, Mask, &ValuePayload);
    if(Status != TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = (ValueLine | (ValuePayload << 1));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLoopbackMFSource
 * Returns: TEstatus
 * Description: Configures the multiframe bit source for a payload loopback 
 *              configuration for the specified Port. The possible sources 
 *              are transmit System and receive Line.
 ******************************************************************************/
TEstatus IXF3204SetLoopbackMFSource(void *handle, TEword Port, TEword Source)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT6;

    /*Validate the Source parameter*/
    switch(Source) 
    {
        case TE_SYS:
            Data = TE_CLEAR_BITS;
            break;
        case TE_LINE:
            Data = TE_BIT6;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return(IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetLoopbackMFSource
 * Returns: TEstatus
 * Description: Returns the configured source of the multiframe bit in payload 
 *              loopback for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLoopbackMFSource(void *handle, TEword Port, void *Results )
{
    TEword Address;
    TEbyte Mask, Value;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT6;

    /*perform get operation*/
    Status = IXF3204ReadBit(handle, Address, Mask, &Value);
    if(Status != TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Value) 
    { 
        ReturnValue[0] = TE_LINE;
    }
    else {
        ReturnValue[0] = TE_SYS;
    }

    return TE_STATUS_SUCCESS;
}

