/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */
/******************************************************************************
 * File Name:   misc3204.c
 * Description: This file contains miscellaneous functions
 *
 ******************************************************************************/

#include "ixf3204.h"


/*******************************************************************************
 * Function Name: IXF3204SetMapPortSelect
 * Returns: TEstatus
 * Description: Maps the Port specified with the specified BERT or HDLC. When 
 *              TE_NORMAL is selected, the current map is removed. This 
 *              function configures transmit and receive.
 *              This function can affect or be affected when payload loopback
 *              is configured. When the loopback is configured, the mapping is
 *              over written and vice versa.
 ******************************************************************************/
TEstatus IXF3204SetMapPortSelect(void *handle, TEword Port, TEword Source)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;

    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Source parameter*/
    if(((Source >= TE_BERT0) && (Source <= TE_BERT7)) || 
        ((Source >= TE_DRS0) && (Source <= TE_DRS7)) ||
        ((Source >= TE_DMW0) && (Source <= TE_DMW7)) ) 
    {
        Data = TE_BIT6;
    }
    else
    {
        if((Source >= TE_HDLC0) && (Source <= TE_HDLC23))
        {
            Data = (TE_BIT6 | TE_BIT5);
        }
        else
        {
            if(Source == TE_NORMAL)
            {
                Data = TE_CLEAR_BITS;
            }
            else
            {
                return TE_STATUS_BAD_PARAMS;
            }
        }
    }
    
    /*validate Source parameter assign value to the Data to be written*/
    switch(Source)
    {
        case TE_BERT0:
        case TE_HDLC0:
            Data |= TE_CLEAR_BITS;
            break;
        case TE_BERT1:
        case TE_HDLC1:
            Data |= TE_BIT0;
            break;
        case TE_BERT2:
        case TE_HDLC2:
            Data |= TE_BIT1;
            break;
        case TE_BERT3:
        case TE_HDLC3:
            Data |= (TE_BIT1 | TE_BIT0);
            break;
        case TE_BERT4:
        case TE_HDLC4:
            Data |= TE_BIT2;
            break;
        case TE_BERT5:
        case TE_HDLC5:
            Data |= (TE_BIT2 | TE_BIT0);
            break;
        case TE_BERT6:
        case TE_HDLC6:
            Data |= (TE_BIT2 | TE_BIT1);
            break;
        case TE_BERT7:
        case TE_HDLC7:
            Data |= (TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        case TE_DMW0:
        case TE_HDLC8:
            Data |= TE_BIT3;
            break;
        case TE_DMW1:
        case TE_HDLC9:
            Data |= (TE_BIT3 | TE_BIT0);
            break;
        case TE_DMW2:
        case TE_HDLC10:
            Data |= (TE_BIT3 | TE_BIT1);
            break;
        case TE_DMW3:
        case TE_HDLC11:
            Data |= (TE_BIT3 | TE_BIT1 | TE_BIT0);
            break;
        case TE_DMW4:
        case TE_HDLC12:
            Data |= (TE_BIT3 | TE_BIT2);
            break;
        case TE_DMW5:
        case TE_HDLC13:
            Data |= (TE_BIT3 | TE_BIT2 | TE_BIT0);
            break;
        case TE_DMW6:
        case TE_HDLC14:
            Data |= (TE_BIT3 | TE_BIT2 | TE_BIT1);
            break;
        case TE_DMW7:
        case TE_HDLC15:
            Data |= TE_BIT0_3;
            break;
        case TE_DRS0:
        case TE_HDLC16:
            Data |= TE_BIT4;
            break;
        case TE_DRS1:
        case TE_HDLC17:
            Data |= (TE_BIT4 | TE_BIT0);
            break;
        case TE_DRS2:
        case TE_HDLC18:
            Data |= (TE_BIT4 | TE_BIT1);
            break;
        case TE_DRS3:
        case TE_HDLC19:
            Data |= (TE_BIT4 | TE_BIT1 | TE_BIT0);
            break;
        case TE_DRS4:
        case TE_HDLC20:
            Data |= (TE_BIT4 | TE_BIT2);
            break;
        case TE_DRS5:
        case TE_HDLC21:
            Data |= (TE_BIT4 | TE_BIT2 | TE_BIT0);
            break;
        case TE_DRS6:
        case TE_HDLC22:
            Data |= (TE_BIT4 | TE_BIT2 | TE_BIT1);
            break;
        case TE_DRS7:
        case TE_HDLC23:
            Data |= (TE_BIT4 | TE_BIT2 |TE_BIT1 | TE_BIT0);
            break;
        case TE_NORMAL:
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXSIG_MAP;
    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3);
    if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Address = IXF3204FindPortRxBusIF(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXBUSIF_CR1;
    return (IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetMapPortSelect
 * Returns: TEstatus
 * Description: Returns the mapping mode of the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetMapPortSelect(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Source;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Mask, Value;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }
    Address += IXF3204_TXSIG_MAP;

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3);

    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(Data & (TE_BIT6 | TE_BIT5))
    {
        case (TE_BIT6 | TE_BIT5):
            Value = TE_HDLC;
            break;
        case TE_BIT6:
            Value = TE_BERT;
            break;
        case TE_CLEAR_BITS:
        case TE_BIT5:      /* this bit can be affected by payload loopback */
            ReturnValue[0] = TE_NORMAL;
            return TE_STATUS_SUCCESS;
        default:
            return TE_STATUS_BADREAD;
    }

    Source = (TEbyte)(Data & (TE_BIT4 | TE_BIT0_3));

    switch(Source)
    {
        case TE_CLEAR_BITS:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT0;
            }
            else
            {
                ReturnValue[0] = TE_HDLC0;
            }
            break;
        case TE_BIT0:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT1;
            }
            else
            {
                ReturnValue[0] = TE_HDLC1;
            }
            break;
        case TE_BIT1:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT2;
            }
            else
            {
                ReturnValue[0] = TE_HDLC2;
            }
            break;
        case (TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT3;
            }
            else
            {
                ReturnValue[0] = TE_HDLC3;
            }
            break;
        case TE_BIT2:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT4;
            }
            else
            {
                ReturnValue[0] = TE_HDLC4;
            }
            break;
        case (TE_BIT2 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT5;
            }
            else
            {
                ReturnValue[0] = TE_HDLC5;
            }

            break;
        case (TE_BIT2 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT6;
            }
            else
            {
                ReturnValue[0] = TE_HDLC6;
            }
            break;
        case (TE_BIT2 | TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT7;
            }
            else
            {
                ReturnValue[0] = TE_HDLC7;
            }
            break;
        case TE_BIT3:
            if(Value == TE_BERT) 
            {
                ReturnValue[0] = TE_DMW0;
            }
            else
            {
                ReturnValue[0] = TE_HDLC8;
            }
            break;
        case (TE_BIT3 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW1;
            }
            else
            {
                ReturnValue[0] = TE_HDLC9;
            }
            break;
        case (TE_BIT3 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW2;
            }
            else
            {
                ReturnValue[0] = TE_HDLC10;
            }
            break;
        case (TE_BIT3 | TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW3;
            }
            else
            {
                ReturnValue[0] = TE_HDLC11;
            }
            break;
        case (TE_BIT3 | TE_BIT2):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW4;
            } 
            else
            {
                ReturnValue[0] = TE_HDLC12;
            }
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW5;
            }
            else
            {
                ReturnValue[0] = TE_HDLC13;
            }
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW6;
            } 
            else
            {
                ReturnValue[0] = TE_HDLC14;
            }
            break;
        case TE_BIT0_3:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW7;
            }
            else
            {
                ReturnValue[0] = TE_HDLC15;
            }
            break;
        case TE_BIT4:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS0;
            }
            else
            {
                ReturnValue[0] = TE_HDLC16;
            }
            break;
        case (TE_BIT4 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS1;
            }
            else
            {
                ReturnValue[0] = TE_HDLC17;
            }
            break;
        case (TE_BIT4 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS2;
            }
            else
            {
                ReturnValue[0] = TE_HDLC18;
            }
            break;
        case (TE_BIT4 | TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS3;
            }
            else
            {
                ReturnValue[0] = TE_HDLC19;
            }
            break;
        case (TE_BIT4 | TE_BIT2):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS4;
            }
            else
            {
                ReturnValue[0] = TE_HDLC20;
            }
            break;
        case (TE_BIT4 | TE_BIT2 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS5;
            }
            else
            {
                ReturnValue[0] = TE_HDLC21;
            }
            break;
        case (TE_BIT4 | TE_BIT2 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS6;
            }
            else
            {
                ReturnValue[0] = TE_HDLC22;
            }
            break;
        case (TE_BIT4 | TE_BIT2 | TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS7;
            }
            else
            {
                ReturnValue[0] = TE_HDLC23;
            }
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetMapTSSelect
 * Returns: TEstatus
 * Description: Maps the specified time slot (TS) (or all time slots when 
 *              TE_ALL is selected) of the Port specified with the selected
 *              BERT or HDLC. When TE_NORMAL is selected, the current map is
 *              removed. This function configures transmit and receive.
 *              This function can affect or be affected when payload loopback 
 *              is configured. When the loopback is configured, the mapping is
 *              over written and vice versa.
 ******************************************************************************/
TEstatus IXF3204SetMapTSSelect(void *handle, TEword Port,TEword  Source, 
                               TEword TimeSlot)
{

    TEword Address, AddressTx, AddressRx;
    TEbyte Data, Mask;
    TEstatus Status;

    if((TimeSlot < TE_TS0) || (TimeSlot > TE_TS31))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3);
    /*******************************************************************
     *validate Source parameter                                        *
     *******************************************************************/
    if(((Source >= TE_BERT0) && (Source <= TE_BERT7)) || 
       ((Source >= TE_DRS0) && (Source <= TE_DRS7)) ||
       ((Source >= TE_DMW0 && Source <= TE_DMW7)))
    {
        Data = TE_BIT6;
    }
    else if((Source >= TE_HDLC0) && (Source <= TE_HDLC23))
    {
        Data = (TE_BIT6 | TE_BIT5);
    }
    else if(Source == TE_NORMAL)
    {
        Data = TE_CLEAR_BITS;
    }
    else
    {
        return TE_STATUS_BAD_PARAMS;
    }
    
    /********************************************************************
     * validate Source parameter assign value to the Data to be written *
     ********************************************************************/
    switch(Source)
    {
        case TE_BERT0:
        case TE_HDLC0:
            Data |= TE_CLEAR_BITS;
            break;
        case TE_BERT1:
        case TE_HDLC1:
            Data |= TE_BIT0;
            break;
        case TE_BERT2:
        case TE_HDLC2:
            Data |= TE_BIT1;
            break;
        case TE_BERT3:
        case TE_HDLC3:
            Data |= (TE_BIT1 | TE_BIT0);
            break;
        case TE_BERT4:
        case TE_HDLC4:
            Data |= TE_BIT2;
            break;
        case TE_BERT5:
        case TE_HDLC5:
            Data |= (TE_BIT2 | TE_BIT0);
            break;
        case TE_BERT6:
        case TE_HDLC6:
            Data |= (TE_BIT2 | TE_BIT1);
            break;
        case TE_BERT7:
        case TE_HDLC7:
            Data |= (TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        case TE_DMW0:
        case TE_HDLC8:
            Data |= TE_BIT3;
            break;
        case TE_DMW1:
        case TE_HDLC9:
            Data |= (TE_BIT3 | TE_BIT0);
            break;
        case TE_DMW2:
        case TE_HDLC10:
            Data |= (TE_BIT3 | TE_BIT1);
            break;
        case TE_DMW3:
        case TE_HDLC11:
            Data |= (TE_BIT3 | TE_BIT1 | TE_BIT0);
            break;
        case TE_DMW4:
        case TE_HDLC12:
            Data |= (TE_BIT3 | TE_BIT2);
            break;
        case TE_DMW5:
        case TE_HDLC13:
            Data |= (TE_BIT3 | TE_BIT2 | TE_BIT0);
            break;
        case TE_DMW6:
        case TE_HDLC14:
            Data |= (TE_BIT3 | TE_BIT2 | TE_BIT1);
            break;
        case TE_DMW7:
        case TE_HDLC15:
            Data |= TE_BIT0_3;
            break;
        case TE_DRS0:
        case TE_HDLC16:
            Data |= TE_BIT4;
            break;
        case TE_DRS1:
        case TE_HDLC17:
            Data |= (TE_BIT4 | TE_BIT0);
            break;
        case TE_DRS2:
        case TE_HDLC18:
            Data |= (TE_BIT4 | TE_BIT1);
            break;
        case TE_DRS3:
        case TE_HDLC19:
            Data |= (TE_BIT4 | TE_BIT1 | TE_BIT0);
            break;
        case TE_DRS4:
        case TE_HDLC20:
            Data |= (TE_BIT4 | TE_BIT2);
            break;
        case TE_DRS5:
        case TE_HDLC21:
            Data |= (TE_BIT4 | TE_BIT2 | TE_BIT0);
            break;
        case TE_DRS6:
        case TE_HDLC22:
            Data |= (TE_BIT4 | TE_BIT2 | TE_BIT1);
            break;
        case TE_DRS7:
        case TE_HDLC23:
            Data |= (TE_BIT4 | TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        case TE_NORMAL:
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Address = IXF3204FindPortTxSignalingLine_CDM(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        /* if it was not Port, lets check for BP */
        Address = IXF3204FindPortRAMTx(Port);

        if(Address == TE_STATUS_BAD_PARAMS)
        { 
            return TE_STATUS_BAD_PARAMS;
        }
        AddressTx = Address;
        AddressRx = IXF3204FindPortRAMRx(Port);

        if(AddressRx == TE_STATUS_BAD_PARAMS)
        { 
            return TE_STATUS_BAD_PARAMS;
        }
    } 
    else
    {
        AddressTx = Address;
        AddressRx = IXF3204FindPortRxBusIF_CDM(Port);

        if(AddressRx == TE_STATUS_BAD_PARAMS)
        { 
            return TE_STATUS_BAD_PARAMS;
        }
    }
    AddressTx = (TEword)(AddressTx + IXF3204GetTimeSlot(TimeSlot));
    if((Status = IXF3204WriteBit(handle,AddressTx,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    AddressRx = (TEword)(AddressRx + IXF3204GetTimeSlot(TimeSlot));
    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3);

    return(IXF3204WriteBit(handle,AddressRx,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetMapTSSelect
 * Returns: TEstatus
 * Description: Returns the mapping mode of the specified time slot (TS) and 
 *              Port.
 ******************************************************************************/
TEstatus IXF3204GetMapTSSelect(void *handle, TEword Port, TEword TimeSlot,
                               void *Results)
{
    TEword Address;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;
    TEbyte Mask, Value;

    CHK_NULL(ReturnValue);

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT0_3);

    Address = IXF3204FindPortTxSignalingLine_CDM(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        /* if it was not Port, lets check for BP */
        Address = IXF3204FindPortRAMTx(Port);

        if(Address == TE_STATUS_BAD_PARAMS)
        { 
            return TE_STATUS_BAD_PARAMS;
        }
    }

    if((TimeSlot < TE_TS0) || (TimeSlot > TE_TS31))
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(Address + IXF3204GetTimeSlot(TimeSlot));
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    
    switch(Data & (TE_BIT6 | TE_BIT5))
    {
        case (TE_BIT6 | TE_BIT5):
            Value = TE_HDLC;
            break;
        case TE_BIT6:
            Value = TE_BERT;
            break;
        case TE_CLEAR_BITS:
        case TE_BIT5:      /* this bit can be affected by payload loopback */
            ReturnValue[0] = TE_NORMAL;
            return TE_STATUS_SUCCESS;
        default:
            return TE_STATUS_BADREAD;
    }

    switch(Data & (TE_BIT4 | TE_BIT0_3))
    {
        case TE_CLEAR_BITS:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT0;
            }
            else
            {
                ReturnValue[0] = TE_HDLC0;
            }
            break;
        case TE_BIT0:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT1;
            }
            else
            {
                ReturnValue[0] = TE_HDLC1;
            }
            break;
        case TE_BIT1:
            if(Value == TE_BERT) {
                ReturnValue[0] = TE_BERT2;
            } 
            else
            {
                ReturnValue[0] = TE_HDLC2;
            }
            break;
        case (TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT3;
            }
            else
            {
                ReturnValue[0] = TE_HDLC3;
            }
            break;
        case TE_BIT2:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT4;
            }
            else
            {
                ReturnValue[0] = TE_HDLC4;
            }
            break;
        case (TE_BIT2 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT5;
            }
            else
            {
                ReturnValue[0] = TE_HDLC5;
            }
            break;
        case (TE_BIT2 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT6;
            }
            else
            {
                ReturnValue[0] = TE_HDLC6;
            }
            break;
        case (TE_BIT2 | TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_BERT7;
            }
            else
            {
                ReturnValue[0] = TE_HDLC7;
            }
            break;
        case TE_BIT3:
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW0;
            }
            else
            {
                ReturnValue[0] = TE_HDLC8;
            }
            break;
        case (TE_BIT3 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW1;
            }
            else
            {
                ReturnValue[0] = TE_HDLC9;
            }
            break;
        case (TE_BIT3 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW2;
            }
            else
            {
                ReturnValue[0] = TE_HDLC10;
            }
            break;
        case (TE_BIT3 | TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW3;
            }else{
                ReturnValue[0] = TE_HDLC11;
            }
            break;
        case (TE_BIT3 | TE_BIT2):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW4;
            }
            else
            {
                ReturnValue[0] = TE_HDLC12;
            }
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW5;
            }
            else
            {
                ReturnValue[0] = TE_HDLC13;
            }
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DMW6;
            }
            else
            {
                ReturnValue[0] = TE_HDLC14;
            }
            break;
        case TE_BIT0_3:
            if(Value == TE_BERT) 
            {
                ReturnValue[0] = TE_DMW7;
            }
            else
            {
                ReturnValue[0] = TE_HDLC15;
            }
            break;
        case TE_BIT4:
            if(Value == TE_BERT) 
            {
                ReturnValue[0] = TE_DRS0;
            }
            else
            {
                ReturnValue[0] = TE_HDLC16;
            }
            break;
        case (TE_BIT4 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS1;
            }
            else
            {
                ReturnValue[0] = TE_HDLC17;
            }
            break;
        case (TE_BIT4 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS2;
            }
            else
            {
                ReturnValue[0] = TE_HDLC18;
            }
            break;
        case (TE_BIT4 | TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS3;
            }
            else
            {
                ReturnValue[0] = TE_HDLC19;
            }
            break;
        case (TE_BIT4 | TE_BIT2):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS4;
            }
            else
            {
                ReturnValue[0] = TE_HDLC20;
            }
            break;
        case (TE_BIT4 | TE_BIT2 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS5;
            }
            else
            {
                ReturnValue[0] = TE_HDLC21;
            }
            break;
        case (TE_BIT4 | TE_BIT2 | TE_BIT1):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS6;
            }
            else
            {
                ReturnValue[0] = TE_HDLC22;
            }
            break;
        case (TE_BIT4 | TE_BIT2 | TE_BIT1 | TE_BIT0):
            if(Value == TE_BERT)
            {
                ReturnValue[0] = TE_DRS7;
            }
            else
            {
                ReturnValue[0] = TE_HDLC23;
            }
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFullPortStatus
 * Returns: TEstatus
 * Description: Returns the current status of the specified Port. The value 
 *              returned is 32 bits length.
 ******************************************************************************/
TEstatus IXF3204GetFullPortStatus(void * handle, TEword Port, void *Results)
{
    TEbyte Value;
    TEword Address;
    TEdword TempResult, *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /* Find the address of the LSB */
    Address =  IXF3204FindPortStatus(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* Read bits 0 - 7 */
    if((Status = IXF3204ReadByte(handle,Address, &Value)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempResult = (TEdword)Value;

    /* Read bits 8 - 15 */
    Address++;
    if((Status = IXF3204ReadByte(handle,Address, &Value)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempResult |= (TEdword)(Value << 8);

    /* Read bits 16 - 23 */
    Address++;
    if((Status = IXF3204ReadByte(handle,Address, &Value)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempResult |= (TEdword)(Value << 16);

    /* Read bits 24 - 31 */
    Address++;
    if((Status = IXF3204ReadByte(handle,Address, &Value)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    TempResult |= (TEdword)(Value << 24);

    ReturnValue[0] = TempResult;

    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204SetE1TxCASSpareBits
 * Returns: TEstatus
 * Description: This message sets the spare bits to be transmitted in the time
 *              slot 16.
 *              Note that the bit 1 will be cleared, the only valid bits are
 *              bits 0, 2 and 3
 ******************************************************************************/
TEstatus IXF3204SetE1TxCASSpareBits(void *handle, TEword Port, TEbyte Spare)
{
    TEword Address;
    TEbyte Mask = TE_BIT3 | TE_BIT2 | TE_BIT0;

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    {  
        return TE_STATUS_BAD_PARAMS; 
    }

    Address += IXF3204_TXALRM_SPARE;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,Spare,Mask));

        return TE_STATUS_SUCCESS;
    }

    CHK_STATUS(IXF3204WriteBit(handle,Address,Spare,Mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204E1GetTxCASSpareBits
 * Returns: TEstatus
 * Description: This message returns the spare bits being transmitted in time 
 *              slot 16.
 ******************************************************************************/
TEstatus IXF3204GetE1TxCASSpareBits(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT3 | TE_BIT2 | TE_BIT0;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortTxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_TXALRM_SPARE;

    CHK_STATUS(IXF3204ReadBit(handle,Address, Mask, &Data));

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetE1RxCASSpareBits
 * Returns: TEstatus
 * Description: This message returns the Spare received in the timeslot 16
 ******************************************************************************/
TEstatus IXF3204GetE1RxCASSpareBits(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT0_3;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Address = IXF3204FindPortRxAlarms(Port);
    if (Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Address += IXF3204_RXALRM_SPARE;

    CHK_STATUS(IXF3204ReadBit(handle,Address, Mask, &Data));

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

