/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   mop3204.c
 * Description: this file contains MOP functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPSource
 * Returns: TEstatus
 * Description: Configures the operation mode of the MOP module for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPSource(void *handle, TEword Port, TEword Selection)
{
    TEword AddressTx,AddressRx;
    TEstatus Status;
    TEbyte Data, Mask = TE_BIT3;

    /*Get offsets to perform the write operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Selection parameter*/
    switch(Selection)
    {
        case TE_NORMAL:
            Data = TE_CLEAR_BITS;
            break;
        case TE_TRANSPARENT:
            Data = TE_BIT3;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressTx,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

        if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressRx,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    if((Status = IXF3204WriteBit(handle,AddressTx,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if((Status = IXF3204WriteBit(handle,AddressRx,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPSource
 * Returns: TEstatus
 * Description: Returns the configured source of the MOP message for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPSource(void *handle, TEword Port, void *Results)
{
    TEword AddressRx, AddressTx;
    TEbyte Data, verify,Mask = TE_BIT3;
    TEstatus Status;
    TEdword  *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the read operations*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    
    /*Get offsets to perform the read operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*Perform the Get operation*/
    if((Status = IXF3204ReadBit(handle,AddressRx,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Perform the Get operation*/
    if((Status = IXF3204ReadBit(handle,AddressTx,Mask,&verify)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* validate that tx and Rx are configured in the same way */
    if(verify != Data)
    {
        return TE_STATUS_BAD_CONFIG;
    }

    /*Validate Data read parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_NORMAL;
            break;
        case TE_BIT3:
            ReturnValue[0] = TE_TRANSPARENT;
            break;
        default: 
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPCRCEnable
 * Returns: TEstatus
 * Description: Configures the MOP module to perform CRC calculation for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPCRCEnable(void *handle, TEword Port,TEword Selection)
{
    TEword AddressTx, AddressRx;
    TEstatus Status;
    TEbyte Data, Mask = TE_BIT2;

    /*Get offsets to perform the write operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Selection parameter*/
    switch(Selection)
    {
        case TE_ENABLE:
            Data = TE_BIT2;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressTx,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

        if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressRx,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perform the Get operation*/
    if((Status = IXF3204WriteBit(handle,AddressTx,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if((Status = IXF3204WriteBit(handle,AddressRx,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPCRCEnable
 * Returns: TEstatus
 * Description: Returns configured mode for CRC calculation in the MOP for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPCRCEnable(void *handle, TEword Port, void *Results)
{
    TEword AddressRx, AddressTx;
    TEbyte Data, verify, Mask = TE_BIT2; 
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform read operation*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Get offsets to perform read operation*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Perform the Get operation*/
    if((Status = IXF3204ReadBit(handle,AddressRx, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Perform the Get operation*/
    if((Status = IXF3204ReadBit(handle,AddressTx, Mask, &verify)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* validate that tx and Rx are configured in the same way */
    if(verify != Data)
    {
        return TE_STATUS_BAD_CONFIG;
    }

    /*Validate the Data parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        case TE_BIT2:
            ReturnValue[0] = TE_ENABLE;
            break;
        default: 
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPTransportChannel
 * Returns: TEstatus
 * Description: Configures which channel the MOP module uses to send the MOP 
 *              data for the specified Port. The F-bit is used as the channel 
 *              when TE_FBIT is selected. Common channel signaling is used
 *              when TE_CCS is selected. When TE_CCS is selected, the time
 *              slot and bit mask must be configured.
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPTransportChannel(void *handle, TEword Port,
                                          TEword Channel)
{
    TEword AddressTx, AddressRx;
    TEstatus Status;
    TEbyte Data, Mask = TE_BIT6;

    /*Get offsets to perform the set operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;  
    }

    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Validate the Channel parameter*/
    switch(Channel)
    {
        case TE_FBIT:
            Data = TE_CLEAR_BITS;
            break;
        case TE_CCS:
            Data = TE_BIT6;
            break;
        default :   
            return TE_STATUS_BAD_PARAMS; 
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressTx,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

        if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressRx,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perform the set operation*/
    if((Status = IXF3204WriteBit(handle,AddressTx,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return(IXF3204WriteBit(handle,AddressRx,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPTransportChannel
 * Returns: TEstatus
 * Description: Returns the configured channel to carry the MOP message for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPTransportChannel(void *handle, TEword Port, 
                                          void *Results)
{
    TEword AddressRx, AddressTx;
    TEbyte Data, verify, Mask = TE_BIT6;
    TEstatus Status;
    TEdword  *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the read operations*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Get offsets to perform the read operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Perform the Get operation*/
    if((Status = IXF3204ReadBit(handle,AddressRx, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*Perform the Get operation*/
    if((Status = IXF3204ReadBit(handle,AddressTx, Mask, &verify)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* validate that tx and Rx are configured in the same way */
    if(verify != Data)
    {
        return TE_STATUS_BAD_CONFIG;
    }

    /*Validate Data read parameter*/
    switch(Data)
    {
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_FBIT;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_CCS;
            break;
        default:
            return TE_STATUS_BADREAD; 
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPCCSMask
 * Returns: TEstatus
 * Description: Configures the time slots and bits that are used to carry the
 *              MOP message for the specified Port. The TimeSlotMask parameter 
 *              is a 32-bit value where each bit corresponds to a time slot. 
 *              The LSB corresponds to time slot zero and so on. A 1 in the 
 *              TimeSlotMask means that the corresponding time slot carries
 *              the MOP message.
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPCCSMask(void *handle, TEword Port, 
                                 TEdword TimeSlotMask,TEbyte BitMask)
{
    TEword AddressTx, AddressRx;
    TEstatus Status;
    TEword Data;

    /*Get offsets to perform the read operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* Perform Write for TimeSlotMask[MSB][B][B][LSB] and BitMask 
     * forTxFDL and RxFDL
     */

    Data = (TEword) TimeSlotMask;
    AddressRx += IXF3204_RXFDL_CCSTSR1;
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteWordAllPortsRegs(handle,AddressRx,Data))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

    }
    else
    {
        if((Status = IXF3204WriteWord(handle,AddressRx,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    Data = (TEword) (TimeSlotMask>>16);
    AddressRx += IXF3204_ADDR_OFFSET2;
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteWordAllPortsRegs(handle,AddressRx,Data))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

    }
    else
    {
        if((Status = IXF3204WriteWord(handle,AddressRx,Data)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    AddressRx += IXF3204_ADDR_OFFSET2;
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressRx,BitMask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

    }
    else
    {
        if((Status = IXF3204WriteByte(handle,AddressRx,BitMask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    AddressTx += IXF3204_TXFDL_CCSTSR1;
    Data = (TEword) TimeSlotMask;
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteWordAllPortsRegs(handle,AddressTx,Data))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

    }
    else
    {
        if((Status = IXF3204WriteWord(handle,AddressTx,Data)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    AddressTx += IXF3204_ADDR_OFFSET2;
    Data = (TEword) (TimeSlotMask>>16);
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteWordAllPortsRegs(handle,AddressTx,Data))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

    }
    else
    {
        if((Status = IXF3204WriteWord(handle,AddressTx,Data)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    AddressTx += IXF3204_ADDR_OFFSET2;
    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressRx,BitMask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

    }
    else
    {
        if((Status = IXF3204WriteByte(handle,AddressTx,BitMask)) !=
           TE_STATUS_SUCCESS){
            return Status;
        }
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPCCSMask
 * Returns: TEstatus
 * Description: Returns the time slots and bits used to carry the MOP message 
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPCCSMask(void *handle, TEword Port, void *Results)
{
    TEword AddressRx, AddressTx;
    TEstatus Status;
    TEbyte Data, verify;
    TEdword Data1, verify1;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Get offsets to perform the read operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*ReturnValue[0] has the Time Slot Mask*/
    AddressRx += IXF3204_RXFDL_CCSTSR1;
    AddressTx += IXF3204_TXFDL_CCSTSR1;
    if((Status = IXF3204ReaddWord(handle, AddressRx, &Data1))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    ReturnValue[0] = Data1;

    if((Status = IXF3204ReaddWord(handle, AddressTx, &verify1))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* ReturnValue[1] has the Mask*/
    AddressRx += IXF3204_ADDR_OFFSET4;
    AddressTx += IXF3204_ADDR_OFFSET4;
    if((Status = IXF3204ReadByte(handle, AddressRx, &Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status; 
    }

    ReturnValue[1] = Data;

    if((Status = IXF3204ReadByte(handle, AddressTx, &verify))!=
       TE_STATUS_SUCCESS)
    {
        return Status; 
    }

    /* validate that tx and Rx are configured in the same way */
    if((verify != Data) || (verify1 != Data1))
    {
        return TE_STATUS_BAD_CONFIG;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPRxMaxLength
 * Returns: TEstatus
 * Description: Configures the maximum length of the MOP message allowed to be 
 *              received for the specified Port. 
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPRxMaxLength(void *handle, TEword Port, TEword Length)
{
    TEword AddressRx;
    TEstatus Status;

    /*Get offsets to perform the set operation*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Perform IXF3204Writebyte for Length[MSB][LSB]*/
    AddressRx += IXF3204_RXFDL_MMLR1;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteWordAllPortsRegs(handle,AddressRx,Length))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    if((Status = IXF3204WriteWord(handle,AddressRx,Length)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPRxMaxLength
 * Returns: TEstatus
 * Description: Returns the maximum length MOP message allowed to be received 
 *              for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPRxMaxLength(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEword Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /*ReturnValue[0] has the maximum length*/
    Address += IXF3204_RXFDL_MMLR1;
    if((Status = IXF3204ReadWord(handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPDone
 * Returns: TEstatus
 * Description: Informs the MOP module that the process of the current MOP has 
 *              been completed for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPDone(void *handle, TEword Port)
{
    TEword AddressRx;
    TEstatus Status;
    TEbyte Data, Mask;

    Data = Mask = TE_BIT0;

    /*Get offsets to perform the set operations*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    AddressRx += IXF3204_RXFDL_HCR;
    if((Status = IXF3204WriteBit(handle,AddressRx,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPRxStatus
 * Returns: TEstatus
 * Description: Returns the current status of the MOP reception for the 
 *              specified Port. This function returns a 16-bit value.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPRxStatus(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask;
    TEdword  *ReturnValue = (TEdword *)Results;
    TEstatus Status;
    TEdword DwResult;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operation*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    Address += IXF3204_RXFDL_SR;
    Mask = (TE_BIT5 | TE_BIT4 |TE_BIT0_3);
    if((Status = IXF3204ReadBit(handle,Address,0x3F,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    Address -= IXF3204_ADDR_OFFSET3;
    Mask = TE_BIT0_3;
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    DwResult = Data;
    ReturnValue[0] |= (DwResult << 8);

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPRxLength
 * Returns: TEstatus
 * Description: Returns the length of the latest MOP received for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPRxLength(void *handle, TEword Port, void *Results)
{
    TEword AddressRx;
    TEstatus Status;
    TEword Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*MOP   ReturnValue[0]*/
    AddressRx += IXF3204_RXFDL_RLR1;
    if((Status = IXF3204ReadWord(handle,AddressRx,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPRxData
 * Returns: TEstatus
 * Description: Returns the byte read from the MOP receive FIFO for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPRxData(void *handle, TEword Port, void *Results)
{
    TEword AddressRx;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Perform get operation*/
    AddressRx += IXF3204_RXFDL_FIFORD;
    if((Status = IXF3204ReadByte(handle,AddressRx,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPTxInterframe
 * Returns: TEstatus
 * Description: Configures the sequence that must be sent between frames for
 *              the specified Port. The allowed sequence is ones and flags
 *              (01111110).
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPTxInterframe(void *handle, TEword Port, TEword Value)
{
    TEword AddressTx;
    TEbyte Data, Mask = TE_BIT5;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Selection parameter*/
    switch(Value)
    {
        case TE_FLAGS:
            Data =  TE_BIT5;
            break;
        case TE_ONES:
            Data = TE_CLEAR_BITS;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,AddressTx,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perfrom set operation*/
    return (IXF3204WriteBit(handle,AddressTx,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPTxInterframe
 * Returns: TEstatus
 * Description: Returns the configured sequence transmitted between frames for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPTxInterframe(void *handle,TEword Port, void *Results)
{
    TEword Address;
    TEbyte Data, Mask = TE_BIT5; 
    TEstatus Status;
    TEdword  *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operation*/
    Address = IXF3204FindPortTxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Data*/
    switch(Data)
    {
        case TE_BIT5:
            ReturnValue[0] = TE_FLAGS;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_ONES;
            break;
        default :   
            return TE_STATUS_BADREAD;  
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPTxFIFOSend
 * Returns: TEstatus
 * Description: Signals the MOP module for the specified Port that the host 
 *              has completed filling half of the FIFO (TE_FIFO) or the message
 *              is complete (TE_EOM) so the module can start sending it. It
 *              can also abort the sending of the current message (TE_ABORT).
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPTxFIFOSend(void *handle, TEword Port, TEword Value)
{
    TEword AddressTx;
    TEbyte dev_release;
    TEbyte Data, Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);

    /*Get offsets to perform the set operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Validate the Selection parameter*/
    switch(Value)
    {
        case TE_EOM:
            Data = TE_BIT0;
            break;
        case TE_FIFO:
            Data = TE_BIT1;
            break;
        case TE_ABORT:
            Data = TE_BIT2;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    AddressTx += IXF3204_TXFDL_HCR;

    /* this code is only for pre-C0 silicon the algorithm is described in */
    /* the errata Restricted use of tx FDL (or HDLC) engine               */
    CHK_STATUS(IXF3204ReadByte(handle,IXF3204_ADDR_HW_ID,&dev_release));

    if(dev_release == 0xA1)
    {
        TEbyte TS_Position; 
        TEbyte DisableArray[IXF3204_MAX_PORT];
        TEword Loop, Tries;          
        TEdword TS_Array, Current_TS;

        /* Read and store the slipbuffer Disable status */
        for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
        {
            CHK_STATUS(IXF3204ReadBit(handle,(TEword)(IXF3204_SBUSET +
                                                      (IXF3204_PORT_REG_OFFSET *
                                                       Loop)),
                                       TE_BIT1, &DisableArray[Loop]));
        }

        Tries = TE_CLEAR_BITS;
        do{
            TS_Array = TE_CLEAR_BITS;
            Tries++;
            if(Tries == IXF3204_MAX_TRIES)
            {
                return TE_STATUS_BUSY;
            }
            for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
            {
                /* if Slip buffer is disable does not check it */ 
                if(DisableArray[Loop])
                {
                    continue;
                }
                CHK_STATUS(IXF3204ReadByte(handle,(TEword)(IXF3204_SBRPTRR +
                                                  (IXF3204_PORT_REG_OFFSET * 
                                                   Loop)),&TS_Position));
                Current_TS = (TEdword)(0x00000001L << (TS_Position 
                                                       & (TE_BIT4 | TE_BIT0_3)));
                TS_Array |= Current_TS;
            }
        }while((TS_Array & 0xFFF00001L));
    }
    /* end of code */

    /*Perform set operation*/
    return(IXF3204WriteBit(handle,AddressTx,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPTxLength
 * Returns: TEstatus
 * Description: Sets the length of the MOP to be transmitted for the specified
 *              Port.
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPTxLength(void *handle, TEword Port, TEbyte Length)
{
    TEword AddressTx;

    /*Get offsets to perform the set operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;  
    }

    /*validate Length parameter*/
    if(Length > (TE_BIT6 | TE_BIT5 |TE_BIT4 | TE_BIT0_3))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Perform the Set operation*/
    AddressTx += IXF3204_TXFDL_HLR;
    return(IXF3204WriteByte(handle,AddressTx,Length));
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPTxLength
 * Returns: TEstatus
 * Description: Returns the configured length of the MOP to be transmitted for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPTxLength(void *handle, TEword Port, void *Results)
{
    TEword AddressTx;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the read operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;  
    }

    /*Perform the Get operation*/
    AddressTx += IXF3204_TXFDL_HLR;
    if((Status = IXF3204ReadByte(handle,AddressTx,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPTxData
 * Returns: TEstatus
 * Description: Writes a byte in the transmit FIFO for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPTxData(void *handle, TEword Port, TEbyte Data)
{
    TEword AddressTx;

    /*Get offsets to perform the read operations*/
    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    /*Perform the set operation*/
    AddressTx += IXF3204_TXFDL_FIFOWR;
    return (IXF3204WriteByte(handle,AddressTx, Data));
}

/*******************************************************************************
 * Function Name: IXF3204SetFDLMOPRxEnable
 * Returns: TEstatus
 * Description: This message enables or disables the reception of MOP messages.
 ******************************************************************************/
TEstatus IXF3204SetFDLMOPRxEnable(void *handle, TEword Port, TEword Mode)
{
    TEword Address;
    TEbyte data, mask = TE_BIT7;

    /*Get offsets to perform the write operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            data = TE_CLEAR_BITS;
            break;
        case TE_DISABLE:
            data = TE_BIT7;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        CHK_STATUS(IXF3204WriteBitAllPortsRegs(handle,Address,data,mask));

        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/

    CHK_STATUS(IXF3204WriteBit(handle,Address,data,mask));

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetFDLMOPRxEnable
 * Returns: TEstatus
 * Description: This message returns if the module is enabled or not to receive
 *              MOP messages.
 ******************************************************************************/
TEstatus IXF3204GetFDLMOPRxEnable(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEbyte data, mask = TE_BIT7;
    TEdword  *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the read operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    
    /*Perform the Get operation*/
    CHK_STATUS(IXF3204ReadBit(handle,Address,mask,&data));

    /*Validate data read parameter*/
    if(data)
    {
        ReturnValue[0] = TE_DISABLE;
    }
    else
    {
        ReturnValue[0] = TE_ENABLE;
    }

    return TE_STATUS_SUCCESS;
}

