/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/*****************************************************************************
 * File Name:   perf3204.c
 * Description: this file contains performance elements functions
 *
 *****************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetPEClearCounter
 * Returns: TEstatus
 * Description: Clears all the counters associated with the specified Port for 
 *              the specified Database.
 ******************************************************************************/
TEstatus IXF3204SetPEClearCounter (void *handle, TEword port, TEword database)
{
    TEword AddressPE, Address;
    TEstatus Status;

    /*Get offsets to perform the read operations*/
    AddressPE = IXF3204FindPortPE(port);
    if (AddressPE == TE_STATUS_BAD_PARAMS)  
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the database parameter*/
    switch(database)
    {
        case TE_NEAREND:
            /* Clear all the T1 counters */
            for(Address = (TEword)((IXF3204_T1NEAREND_INIOFFSET + AddressPE));
                Address <= (IXF3204_T1NEAREND_ENDOFFSET + AddressPE); Address++)
            {
                /*Clear counters*/
                if((Status = IXF3204WriteByte(handle,Address,TE_CLEAR_BITS)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            /* Clear all E1 counters */
            for(Address = (TEword)((IXF3204_E1NEAREND_INIOFFSET + AddressPE));
                Address <= (IXF3204_E1NEAREND_ENDOFFSET + AddressPE) ; Address++)
            {
                /*Clear counters*/
                if((Status = IXF3204WriteByte(handle,Address,TE_CLEAR_BITS)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        case TE_FAREND:
            /* Clear all T1 counters */
            for(Address = (TEword)((IXF3204_T1FAREND_INIOFFSET + AddressPE));
                Address <= (IXF3204_T1FAREND_ENDOFFSET + AddressPE);Address++)
            {
                 /*Clear counters*/
                if((Status = IXF3204WriteByte(handle,Address,TE_CLEAR_BITS)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            /* Clear all E1 counters */
            for(Address = (TEword)(IXF3204_E1FAREND_INIOFFSET + AddressPE);
                Address <= (IXF3204_E1FAREND_ENDOFFSET + AddressPE);Address++)
            {
                /*Clear counters*/
                if((Status = IXF3204WriteByte(handle,Address,TE_CLEAR_BITS)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
      case TE_GENERIC:
            /* Clear generic counters */
            for(Address = (TEword)((IXF3204_GENERIC_INIOFFSET + AddressPE));
                Address <= (IXF3204_GENERIC_ENDOFFSET + AddressPE) ; Address++)
            {
                /*Clear counters*/
                if((Status = IXF3204WriteByte(handle,Address,TE_CLEAR_BITS)) != 
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetPECounterValues
 * Returns: TEstatus
 * Description: Returns the value of the specified counter for the specified
 *              Database and Port. The number of bytes returned is related to
 *              the counter's size. See the counter size list.
 *              The Generic elements do not have any bin associated with them. 
 *              For these, Bin is ignored. Only the current values are
 *              displayed. This applies for both E1 and T1. 
 ******************************************************************************/
TEstatus IXF3204GetPECounterValues(void *handle, TEword Port, TEword Database,
                                   TEword Element, TEword Bin, void *Results)
{
    TEword Address, AddressOffset, Data;
    TEdword *ReturnValue = (TEdword *)Results;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortPE(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Bin param*/
    if(!((Bin >= TE_CURRENTDAY) && (Bin <= TE_RECENT15MIN30)) && (Bin != TE_NA))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Database param*/
    switch(Database)
    {
        case TE_NEAREND:
        case TE_FAREND:
        case TE_NA:
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    /*validate Element param*/
    switch(Element)
    {
        case TE_ESR:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204_NE_ESRL;
                    break;
                case TE_FAREND:
                    AddressOffset = IXF3204_FE_ESRL;
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }
 
            /*read performance counter */  
            Address = (TEword)(AddressOffset + Address);
            return (IXF3204ReaddWord(handle,Address,ReturnValue));
        case TE_SESR:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204_NE_SESRL;
                    break;
                case TE_FAREND:
                    AddressOffset = IXF3204_FE_SESRL;
                    break;
                default :  
                    return TE_STATUS_BAD_PARAMS;
            }
            /*read performance counter */  
            Address = (TEword)(AddressOffset + Address);
            return (IXF3204ReaddWord(handle,Address,ReturnValue));
        case TE_BBER:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204_NE_BBERL;
                    break;
                case TE_FAREND:
                    AddressOffset = IXF3204_FE_BBERL;
                    break;
                default :  
                    return TE_STATUS_BAD_PARAMS;
            }

            /*read performance counter */  
            Address = (TEword)(AddressOffset + Address);
            return (IXF3204ReaddWord(handle,Address,ReturnValue));
        case TE_ETIME:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204_NE_TIMEL;
                    break;
                case TE_FAREND:
                    AddressOffset = IXF3204_FE_TIMEL;
                    break;
                default :  
                    return TE_STATUS_BAD_PARAMS;
            }

            /*read performance counter */  
            Address = (TEword)(AddressOffset + Address);
            return (IXF3204ReaddWord(handle,Address,ReturnValue));
        case TE_UAS:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204_NE_UASTIMEL;
                    break;
                case TE_FAREND:
                    AddressOffset = IXF3204_FE_UASTIMEL;
                    break;
                default :
                    return TE_STATUS_BAD_PARAMS; 
            }

            /*read performance counter */  
            Address = (TEword)(AddressOffset + Address);
            return (IXF3204ReaddWord(handle,Address,ReturnValue));
        case TE_AVAILR:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204_NE_ARTIMEL;
                    break;
                case TE_FAREND:
                    AddressOffset = IXF3204_FE_ARTIMEL;
                    break;
                default:  
                    return TE_STATUS_BAD_PARAMS;
            }

            /*read performance counter */  
            Address = (TEword)(AddressOffset + Address);
            return (IXF3204ReaddWord(handle,Address,ReturnValue));
        case TE_AVAILT:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204_NE_ATIMEL;
                    break;
                case TE_FAREND:
                    AddressOffset = IXF3204_FE_ATIMEL;
                    break;
                default:  
                    return TE_STATUS_BAD_PARAMS;
            }

            /*read performance counter */  
            Address = (TEword)(AddressOffset + Address);
            return (IXF3204ReaddWord(handle,Address,ReturnValue));
        case TE_CV_L:
            AddressOffset = IXF3204FindCVLBin(Bin);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(AddressOffset + Address);
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                /*read double word */  
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;

            return TE_STATUS_SUCCESS;
        case TE_ES_L:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204Find_NE_ESL15(Bin);
                    if(AddressOffset == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                case TE_FAREND:
                    AddressOffset = IXF3204Find_FE_ESL15(Bin);
                    if(AddressOffset == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_SES_L:
            /*Get offset to perform the get operation*/            
            AddressOffset = IXF3204Find_NE_SESL15(Bin);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_CV_P:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204Find_NE_CVP(Bin);
                    if(AddressOffset == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                case TE_FAREND:
                    /*Get offset to perform the get operation*/
                    AddressOffset = IXF3204Find_FE_CVP(Bin);
                    if(AddressOffset == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin ==TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_ES_P:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204Find_NE_ESP(Bin);
                    if(Address == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }				
                    break;
                case TE_FAREND:
                    /*Get offset to perform the get operation*/
                    AddressOffset = IXF3204Find_FE_ESP(Bin);
                    if(Address == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                default:
                    return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_SES_P:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204Find_NE_SESP(Bin);
                    if(Address == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                case TE_FAREND:
                    AddressOffset=IXF3204Find_FE_SESP(Bin);
                    if(Address == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                default:  
                    return TE_STATUS_BAD_PARAMS; 
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_SAS_P:
            AddressOffset = IXF3204Find_NE_SASP(Bin);
            if (Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_UAS_P:
            /*validate Database param*/            
            switch(Database)
            {
                case TE_NEAREND:
                    AddressOffset = IXF3204Find_NE_UASP(Bin);
                    if(Address == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                case TE_FAREND:
                    /*Get offset to perform the get operation*/
                    AddressOffset = IXF3204Find_FE_UASP(Bin);
                    if(Address == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS; 
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_CSS_P:
            /*validate Database param*/
            switch(Database)
            {
                case TE_NEAREND:
                    /*Get offset to perform the get operation*/
                    AddressOffset = IXF3204Find_NE_CSSP(Bin);
                    if(Address == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                case TE_FAREND:
                    /*Get offset to perform the get operation*/
                    AddressOffset = IXF3204Find_FE_CSSP(Bin);
                    if(Address == TE_STATUS_BAD_PARAMS)
                    {
                        return TE_STATUS_BAD_PARAMS;
                    }
                    break;
                default: 
                    return TE_STATUS_BAD_PARAMS; 
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_ESA_P:
            /*Get offset to perform the get operation*/
            AddressOffset = IXF3204Find_FE_ESAP(Bin);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_ESB_P:
            /*Get offset to perform the get operation*/
            AddressOffset = IXF3204Find_FE_ESBP(Bin);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }
            
            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_SEFS_P:
            /*Get offset to perform the get operation*/
            AddressOffset = (TEword)IXF3204Find_FE_SEFSP(Bin);
            if(Address == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(AddressOffset + Address);
            /*read performance counter */  
            if((Bin == TE_CURRENTDAY) || (Bin == TE_PREVIOUSDAY))
            {
                return (IXF3204ReaddWord(handle,Address,ReturnValue));
            }
            else
            {
                if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[0] = Data;
                return TE_STATUS_SUCCESS;
            }
        case TE_CRC:
            /*read performance counter */  
            Address += IXF3204_CRCL;
            if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            return TE_STATUS_SUCCESS;
        case TE_FE:
            Address = (TEword)(Address + IXF3204_FEL);
            /*read performance counter */  
            if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            return TE_STATUS_SUCCESS;
        case TE_LCV:
            Address = (TEword)(Address + IXF3204_LCVL);

            /*read performance counter */  
            if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            return TE_STATUS_SUCCESS;
        case TE_OOF:
            Address = (TEword)(Address + IXF3204_OOFL);

            /*read performance counter */  
            if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            return TE_STATUS_SUCCESS;
        case TE_CSR:
            Address = (TEword)(Address + IXF3204_CSRL);

            /*read performance counter */  
            if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            return TE_STATUS_SUCCESS;
        case TE_CST:
            Address = (TEword)(Address + IXF3204_CSLL);

            /*read performance counter */  
            if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            return TE_STATUS_SUCCESS;
        case TE_COFA:
            Address = (TEword)(Address + IXF3204_COFAL);

            /*read performance counter */  
            if((Status = IXF3204ReadWord(handle,Address,&Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            return TE_STATUS_SUCCESS;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }
}


