/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability. Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   pli3204.c
 * Description: this file contains LIU functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetLIUPolarity
 * Returns: TEstatus
 * Description: Configures the following for the specified Port when the port
 *              is configured for unipolar mode (See the IXF3204SetLIULineMode 
 *              function to set unipolar mode):
 *              TPOS, TNEG, RPOS, and RNEG polarity.
 *              The transmit clock edge used to deliver data to the line.
 *              The receive clock edge used to sample data coming from the line.
 *              When RxPos/RxNeg is set to TE_INV, all input on the specified
 *              port is internally inverted. When TxPos/TxNeg is set to TE_INV,
 *             output is inverted.
 ******************************************************************************/
TEstatus IXF3204SetLIUPolarity(void *handle, TEword Port,TEword TxPos,
                               TEword TxNeg, TEword TxClk, TEword RxPos,
                               TEword RxNeg, TEword RxClk)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate TxPos parameter*/
    switch(TxPos)
    {
        case TE_INV:   
            Data = TE_BIT5;
            break;
        case TE_NONINV: 
            Data = TE_CLEAR_BITS;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate TXNeg parameter*/
    switch(TxNeg)
    {
        case TE_INV: 
            Data |= TE_BIT6;
            break;
        case TE_NONINV: 
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate TxClk parameter*/
    switch(TxClk)
    {
        case TE_FALLING: 
            Data |= TE_BIT4;
            break;
        case TE_RISING: 
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate RxPos parameter*/
    switch(RxPos)
    {
        case TE_INV: 
            Data |= TE_BIT1;
            break;
        case TE_NONINV: 
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate RxNeg parameter*/
    switch(RxNeg)
    {
        case TE_INV: 
            Data |= TE_BIT2;
            break;
        case TE_NONINV: 
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate RxClk parameter*/
    switch(RxClk)
    {
        case TE_FALLING: 
            Data |= TE_BIT0;
            break;
        case TE_RISING: 
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT2 | TE_BIT1 | TE_BIT0);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetLIUPolarity
 * Returns: TEstatus
 * Description: Returns the polarity value for the specified Port when the 
 *              port is configured for unipolar mode.
 ******************************************************************************/
TEstatus IXF3204GetLIUPolarity(void *handle, TEword Port, void *Results)
{
    TEbyte Data, Mask;
    TEstatus Status;
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT6 | TE_BIT5 | TE_BIT4 | TE_BIT2 | TE_BIT1 | TE_BIT0);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /* TxPos */
    if(Data & TE_BIT5)
    {
        ReturnValue[0] = TE_INV;
    }
    else
    {
        ReturnValue[0] = TE_NONINV;
    }
    /* TxNeg */
    if(Data & TE_BIT6)
    {
        ReturnValue[1] = TE_INV;
    }
    else
    {
        ReturnValue[1] = TE_NONINV;
    }
    /* TxClk */
    if(Data & TE_BIT4)
    {
        ReturnValue[2] = TE_FALLING;
    }
    else
    {
        ReturnValue[2] = TE_RISING;
    }
    /* RxPos */
    if(Data & TE_BIT1)
    {
        ReturnValue[3] = TE_INV;
    }
    else
    {
        ReturnValue[3] = TE_NONINV;
    }
    /* RxNeg */
    if(Data & TE_BIT2)
    {
        ReturnValue[4] = TE_INV;
    }
    else
    {
        ReturnValue[4] = TE_NONINV;
    }
    /* RxClk */ 
    if(Data & TE_BIT0)
    {
        ReturnValue[5] = TE_FALLING;
    }
    else
    {
        ReturnValue[5] = TE_RISING;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLIUNegUnipolarMode
 * Returns: TEstatus
 * Description: Configures the TNEG and RNEG functionality for the specified 
 *              Port when the port is configured for unipolar mode. See the
 *              IXF3204SetLIULineMode function to set unipolar mode. In
 *              unipolar mode, T1/E1/J1 data flows through TPOS and RPOS
 *              leaving TNEG and RNEG available for other specific functions
 *              described below:
 *              TNEG can be used to toggle every frame (Fsync), transmit a
 *              general purpose byte or indicate a BPV.
 *              RNEG can be used to detect BPVs or is not used. Every 1
 *              received implies a BPV and the internal counters increments.
 *              Pattern is used only when TNegValue is TE_GRAL.
 *              When the port is not in unipolar mode, this function has no
 *              effect.
 ******************************************************************************/
TEstatus IXF3204SetLIUNegUnipolarMode(void *handle, TEword Port,
                                      TEword TNegValue, TEword RNegValue,
                                      TEbyte Pattern)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;    
    TEword PartialAddress; 

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate TNegValue parameter*/
    switch(TNegValue)
    {
        case TE_GRAL:
            Data = TE_CLEAR_BITS;
            break;
        case TE_BPV:
            Data = TE_BIT4;
            break;
        case TE_FSYNC:
            Data = TE_BIT5;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate RNegValue parameter*/
    switch(RNegValue)
    {
        case TE_IGNORE:
            break;
        case TE_BPV:
            Data |= TE_BIT0;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress + IXF3204_LINEGU);
    Mask = (TE_BIT5 | TE_BIT4 | TE_BIT0);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*perform set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    Data = Pattern;
    Address = (TEword)(PartialAddress + IXF3204_LIGPB);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Data))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return IXF3204WriteByte(handle,Address,Data);
}

/*******************************************************************************
 * Function Name: IXF3204GetLIUNegUnipolarMode
 * Returns: TEstatus
 * Description: Returns the configured values in the TNEG and RNEG and the 
 *              pattern (when configured) when in unipolar mode for the
 *              specified Port. When not in unipolar mode, the returned values
 *              are not valid.
 ******************************************************************************/
TEstatus IXF3204GetLIUNegUnipolarMode(void *handle, TEword Port,void *Results)
{
    TEbyte Data, Data1;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress + IXF3204_LINEGU);
    Mask = (TE_BIT5 | TE_BIT4 | TE_BIT0);

    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Data1 = (TEbyte)(Data & TE_BIT0_3);
    /*validate Data1 parameter and get RNegValue*/
    switch (Data1)
    {
        case TE_CLEAR_BITS:
            ReturnValue[1] = TE_IGNORE;
            break;
        case TE_BIT0:
            ReturnValue[1] = TE_BPV;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    Data1 = (TEbyte)(Data & TE_BIT4_7);
    /*validate Data1 with different mask parameter*/
    switch (Data1){
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_GRAL;
            break;
        case TE_BIT4:
            ReturnValue[0] = TE_BPV;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_FSYNC;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    /* read the pattern */
    Address = (TEword)(PartialAddress + IXF3204_LIGPB);

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[2] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLIULineMode
 * Returns: TEstatus
 * Description: Defines several parameters in the LIU module for the specified 
 *              Port. The parameters are:
 *              Zero code suppression on the transmit side where the options
 *              are TE_ENABLE and TE_DISABLE.
 *              Transmit and receive AMI mode where the options are:
 *              AMI disable or non-return to zero (NRZ). Unipolar mode.
 *              AMI without postprocessing.
 *              AMI with postprocessing (B8ZS in T1 framer byte format or HDB3
 *              in E1 framer byte format).
 *              ISDN mode where the valid options are no ISDN compliance 
 *              (disable), or LOS and AIS compliance with ISDN (enable).
 ******************************************************************************/
TEstatus IXF3204SetLIULineMode(void *handle, TEword Port,TEword ZCSMode,
                               TEword TxAMIMode, TEword isdn, TEword RxAMIMode)
{
    TEbyte Data;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate ZCSMode parameter*/
    switch(ZCSMode) 
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_ENABLE:
            Data = TE_BIT7;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate TxAMIMode parameter*/
    switch(TxAMIMode)
    {
        case TE_NRZ:
            break;
        case TE_AMI:
            Data |= TE_BIT5;
            break;
        case TE_AMIPOST:
            Data |= (TE_BIT5 | TE_BIT4);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    /*validate ISDN parameter*/
    switch(isdn) 
    {
        case TE_DISABLE:
            break;
        case TE_ENABLE:
            Data |= TE_BIT3;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate RxAMIMode parameter*/
    switch(RxAMIMode)
    {
        case TE_NRZ:
            break;
        case TE_AMI:
            Data |= TE_BIT1;
            break;
        case TE_AMIPOST:
            Data |= (TE_BIT1 | TE_BIT0);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }

    Address = (TEword)(PartialAddress + IXF3204_LIMODE);
    Mask = (TE_BIT7 | TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT1 | TE_BIT0);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetLIULineMode
 * Returns: TEstatus
 * Description: Returns the LIU mode for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLIULineMode(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS) 
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address = (TEword)(Address + IXF3204_LIMODE);
    Mask = (TE_BIT7 | TE_BIT5 | TE_BIT4 | TE_BIT3 | TE_BIT1 | TE_BIT0);
    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT7)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    switch(Data & (TE_BIT4 | TE_BIT5)) 
    {
        case (TE_BIT4 | TE_BIT5):   
            ReturnValue[1] = TE_AMIPOST;
            break;
        case TE_BIT5:
            ReturnValue[1] = TE_AMI;
            break;
        case TE_BIT4:
        case TE_CLEAR_BITS:
            ReturnValue[1] = TE_NRZ;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    if(Data & TE_BIT3)
    {
        ReturnValue[2] = TE_ENABLE;
    }
    else
    {
        ReturnValue[2] = TE_DISABLE;
    }

    switch(Data & (TE_BIT1 | TE_BIT0)) 
    {
        case (TE_BIT1 | TE_BIT0):   
            ReturnValue[3] = TE_AMIPOST;
            break;
        case TE_BIT1:
            ReturnValue[3] = TE_AMI;
            break;
        case TE_BIT0:
        case TE_CLEAR_BITS:
            ReturnValue[3] = TE_NRZ;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLIUClearCounter
 * Returns: TEstatus
 * Description: Clears all counters and shadow counters in the PLI module for 
 * the specified Port. 
 ******************************************************************************/
TEstatus IXF3204SetLIUClearCounter(void *handle, TEword Port)
{
    TEbyte Data = TE_BIT5;
    TEbyte Mask;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortPLI(Port);
    if(Address == TE_STATUS_BAD_PARAMS) 
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address = (TEword)(Address + IXF3204_LIERST);

    Mask = TE_BIT5;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return (IXF3204WriteBit(handle,Address,Data,Mask));
}

/*******************************************************************************
 * Function Name: IXF3204SetLIUForceError
 * Returns: TEstatus
 * Description: Forces a single error in the output data for the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetLIUForceError(void *handle, TEword Port)
{
    TEword Address;
    TEbyte Data, Mask;
    TEstatus Status;

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortPLI(Port);

    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Data = TE_BIT4;
    Address = (TEword)(Address + IXF3204_LIERST);
    Mask = TE_BIT4;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204SetLIULOSDetection
 * Returns: TEstatus
 * Description: Configures several parameters to declare and clear LOS for the
 *              specified Port:
 *              Analog and Digital enables or disables the analog and digital
 *              LOS detection respectively. 
 *              The SetWin parameter defines the number of consecutive zeros
 *              needed to declare LOS. ClearWin defines the window length used
 *              to sample for LOS detection.
 *              DensityWin defines the number of ones needed to clear LOS
 *              NoEXZ, when enabled, clears LOS when it receives enough ones
 *              even when conditions are met to declare EXZ. Ignores EXZ when
 *              the previously defined criteria are met.
 ******************************************************************************/
TEstatus IXF3204SetLIULOSDetection(void *handle, TEword Port, TEword Analog,
                                   TEword Digital, TEword SetWin, 
                                   TEword ClearWin, TEword DensityWin,
                                   TEword NoEXZ)
{
    TEbyte Data, Mask;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {  
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Analog parameter*/
    switch(Analog)
    {
        case TE_ENABLE: 
            Data = TE_BIT7;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate Digital parameter*/
    switch (Digital)
    {
        case TE_ENABLE: 
            Data |= TE_BIT6;
            break;
        case TE_DISABLE: 
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    /*validate ClearWin parameter*/
    switch(ClearWin)
    {
        case TE_W16:
            Data |=  TE_CLEAR_BITS;
            break;
        case TE_W32:
            Data |= TE_BIT0;
            break;
        case TE_W64:
            Data |= TE_BIT1;
            break;
        case TE_W128:
            Data |= (TE_BIT1 | TE_BIT0);
            break;
        case TE_W256:
            Data |= TE_BIT2;
            break;
        case TE_W1024:
            Data |= (TE_BIT2 | TE_BIT0);
            break;
        case TE_W2048:
            Data |= (TE_BIT2 | TE_BIT1);
            break;
        case TE_W4096:
            Data |= (TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate SetWin parameter*/
    switch (SetWin) 
    {
        case TE_W16:
            Data |=  TE_CLEAR_BITS;
            break;
        case TE_W32:
            Data |= TE_BIT3;
            break;
        case TE_W64:
            Data |= TE_BIT4;
            break;
        case TE_W128:
            Data |= (TE_BIT4 | TE_BIT3);
            break;
        case TE_W256:
            Data |= TE_BIT5;
            break;
        case TE_W1024:
            Data |= (TE_BIT5 | TE_BIT3);
            break;
        case TE_W2048:
            Data |= (TE_BIT5 | TE_BIT4);
            break;
        case TE_W4096:
            Data |= (TE_BIT5 | TE_BIT4 | TE_BIT3);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress + IXF3204_LISET0);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteByteAllPortsRegs(handle,Address,Data))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*perform set operation*/
        if((Status = IXF3204WriteByte(handle,Address,Data)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /*validate DensityWin parameter*/
    switch (DensityWin)
    {
        case TE_DE1:
            Data =  TE_CLEAR_BITS;
            break;
        case TE_DE2:
            Data = TE_BIT0;
            break;
        case TE_DE4:
            Data = TE_BIT1;
            break;
        case TE_DE8:
            Data = (TE_BIT1 | TE_BIT0);
            break;
        case TE_DE16:
            Data = TE_BIT2;
            break;
        case TE_DE32:
            Data = (TE_BIT2 | TE_BIT0);
            break;
        case TE_DE256:
            Data = (TE_BIT2 | TE_BIT1);
            break;
        case TE_DE512:
            Data = (TE_BIT2 | TE_BIT1 | TE_BIT0);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS; 
    }

    /*validate NoEXZ parameter*/
    switch(NoEXZ)
    {
        case TE_ENABLE: 
            Data |= TE_BIT3;
            break;
        case TE_DISABLE: 
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    Address = (TEword)(PartialAddress + IXF3204_LISET1);
    Mask = TE_BIT0_3;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetLIULOSDetection
 * Returns: TEstatus
 * Description: Returns the settings for LOS detection for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLIULOSDetection(void *handle, TEword Port, 
                                   void *Results)
{
    TEword Address;
    TEstatus Status;
    TEword PartialAddress;
    TEbyte Data, Mask;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress + IXF3204_LISET0);
    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* Analog  LOS*/
    if(Data & TE_BIT7)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    /* Digital LOS*/
    if(Data & TE_BIT6)
    {
        ReturnValue[1] = TE_ENABLE;
    }
    else
    {
        ReturnValue[1] = TE_DISABLE;
    }

    /* Set time window*/
    switch(Data & (TE_BIT5 | TE_BIT4 | TE_BIT3))
    {
        case TE_CLEAR_BITS:
            ReturnValue[2] = TE_W16;
            break;
        case TE_BIT3:
            ReturnValue[2] = TE_W32;
            break;
        case TE_BIT4:
            ReturnValue[2] = TE_W64;
            break;
        case (TE_BIT4 | TE_BIT3):
            ReturnValue[2] = TE_W128;
            break;
        case TE_BIT5:
            ReturnValue[2] = TE_W256;
            break;
        case (TE_BIT5 | TE_BIT3):
            ReturnValue[2] = TE_W1024;
            break;
        case (TE_BIT5 | TE_BIT4):
            ReturnValue[2] = TE_W2048;
            break;
        case (TE_BIT5 | TE_BIT4 | TE_BIT3):
            ReturnValue[2] = TE_W4096;
            break;
        default:   
            return TE_STATUS_BADREAD; 
    }

     /* Clearance time window*/
    switch(Data & (TE_BIT2 | TE_BIT1 | TE_BIT0))
    {
        case TE_CLEAR_BITS:
            ReturnValue[3] = TE_W16;
            break;
        case TE_BIT0:
            ReturnValue[3] = TE_W32;
            break;
        case TE_BIT1:
            ReturnValue[3] = TE_W64;
            break;
        case (TE_BIT1 | TE_BIT0):
            ReturnValue[3] = TE_W128;
            break;
        case TE_BIT2:
            ReturnValue[3] = TE_W256;
            break;
        case (TE_BIT2 | TE_BIT0):
            ReturnValue[3] = TE_W1024;
            break; 
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[3] = TE_W2048;
            break;
        case (TE_BIT2 | TE_BIT1 | TE_BIT0):
            ReturnValue[3] = TE_W4096;
            break;
        default:   
            return TE_STATUS_BADREAD;  
    }

    Address = (TEword)(PartialAddress + IXF3204_LISET1);
    Mask = TE_BIT0_3;
    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

     /*  LOS density*/
    switch(Data & (TE_BIT2 | TE_BIT1 | TE_BIT0))
    {
        case TE_CLEAR_BITS:
            ReturnValue[4] = TE_DE1;
            break;
        case TE_BIT0:
            ReturnValue[4] = TE_DE2;
            break;
        case TE_BIT1:
            ReturnValue[4] = TE_DE4;
            break;
        case (TE_BIT1 | TE_BIT0):
            ReturnValue[4] = TE_DE8;
            break;
        case TE_BIT2:
            ReturnValue[4] = TE_DE16;
            break;
        case (TE_BIT2 | TE_BIT0):
            ReturnValue[4] = TE_DE32;
            break;
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[4] = TE_DE256;
            break;
        case (TE_BIT2 | TE_BIT1 | TE_BIT0):
            ReturnValue[4] = TE_DE512;
            break;
        default:   
            return TE_STATUS_BADREAD;  
    }

    /* EXZ */
    if(Data & TE_BIT3)
    {
        ReturnValue[5] = TE_ENABLE;
    }
    else
    {
        ReturnValue[5] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetLIUAISDetection
 * Returns: TEstatus
 * Description: Configures the time window used to declare AIS condition and 
 *              the use of OOF in declaring AIS for the specified Port.
 *              AISWindow specifies the time window in milliseconds where 99.9%
 *              of ones is required to declare AIS. 
 *              UseOOF, when enabled, indicates that the AIS condition is
 *              asserted only when the OOF condition exists and the required 
 *              99.9% of ones is present.
 ******************************************************************************/
TEstatus IXF3204SetLIUAISDetection(void *handle, TEword Port,TEword AISWindow,
                                   TEword UseOOF)
{
    TEbyte Data, Mask;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);

    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate AISWindow parameter*/
    switch(AISWindow)
    {
        case TE_MS3:
            Data = TE_CLEAR_BITS;
            break;
        case TE_MS9:
            Data = TE_BIT4;
            break;
        case TE_MS15:
            Data = TE_BIT5;
            break;
        case TE_MS21:
            Data = (TE_BIT5 | TE_BIT4);
            break;
        case TE_MS27:
            Data = TE_BIT6;
            break;
        case TE_MS33:
            Data = (TE_BIT6 | TE_BIT4);
            break;
        case TE_MS39:
            Data = (TE_BIT6 | TE_BIT5);
            break;
        case TE_MS42:
            Data = (TE_BIT6 | TE_BIT5 | TE_BIT4);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /*validate UseOOF parameter*/
    switch (UseOOF)
    {
        case TE_ENABLE: 
            Data |= TE_BIT7;
            break;
        case TE_DISABLE: 
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    Address = (TEword)(PartialAddress + IXF3204_LISET1);
    Mask= TE_BIT4_7;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetLIUAISDetection
 * Returns: TEstatus
 * Description: Returns the values configured to declare AIS for the specified 
 *              Port.
 ******************************************************************************/
TEstatus IXF3204GetLIUAISDetection(void *handle, TEword Port, void *Results)
{
    TEbyte Data, Mask;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);

    if(PartialAddress == TE_STATUS_BAD_PARAMS) 
    {
        return TE_STATUS_BAD_PARAMS; 
    }

    Address = (TEword)(PartialAddress + IXF3204_LISET1);
    Mask = TE_BIT4_7;
    /*perform get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    switch(Data & (TE_BIT6 | TE_BIT5 | TE_BIT4))
    {
        case TE_CLEAR_BITS:
            ReturnValue[0]= TE_MS3;
            break; 
        case TE_BIT4:
            ReturnValue[0] = TE_MS9;
            break;
        case TE_BIT5:
            ReturnValue[0] = TE_MS15;
            break;
        case (TE_BIT4 | TE_BIT5):
            ReturnValue[0] = TE_MS21;
            break;
        case TE_BIT6:
            ReturnValue[0] = TE_MS27;
            break;
        case (TE_BIT6 | TE_BIT4):
            ReturnValue[0] = TE_MS33;
            break;
        case (TE_BIT6 | TE_BIT5):
            ReturnValue[0] = TE_MS39;
            break;
        case (TE_BIT6 | TE_BIT5 | TE_BIT4):
            ReturnValue[0] = TE_MS42;
            break;
        default:
            return TE_STATUS_BADREAD;
    }

    if(Data & TE_BIT7)
    {
        ReturnValue[1] = TE_ENABLE;
    }
    else
    {
        ReturnValue[1] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetLIUStatus
 * Returns: TEstatus
 * Description: Returns the current status of the LIU module for the specified
 *              Port. The value returned is a byte. Each bit has a different 
 *              meaning as detailed in the table below.
 ******************************************************************************/
TEstatus IXF3204GetLIUStatus(void *handle, TEword Port, void *Results)
{

    TEbyte Data;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);

    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress + IXF3204_LILSTS);
    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetLIUCounter
 * Returns: TEstatus
 * Description: Returns the current count of the specified Counter for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetLIUCounter(void *handle, TEword Port, TEword Counter,
                              void *Results)
{
    TEstatus Status;
    TEbyte ValCount;
    TEword Address, Data, PartialAddress;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*validate Counter parameter*/
    switch(Counter)
    {
        case TE_EXZCOUNT:
            ValCount = (TE_BIT3 | TE_BIT2);
            break;
        case TE_BPVCOUNT:
            ValCount = (TE_BIT3 | TE_BIT2 | TE_BIT1);
            break;
        case TE_HDB3COUNT:
            ValCount = TE_BIT4;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortPLI(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address = (TEword)(PartialAddress + ValCount);
    /*perform get operation*/
    if((Status = IXF3204ReadWord(handle,Address,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

