/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   prm3204.c
 * Description: This file contains PRM functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetPRMMode
 * Returns: TEstatus
 * Description: Enables or disables the reception and transmission of PRMs for 
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetPRMMode(void *handle, TEword Port, TEword Mode)
{
    TEword Address, AddressPE ;
    TEbyte Data, Mask = TE_BIT7;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressPE = IXF3204FindPortPE( Port);
    if(AddressPE == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Mode parameter*/
    switch(Mode)
    {
        case TE_ENABLE:
            Data = TE_BIT7;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*perform set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask))!=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /* if Mode is disable, disable also the PRM database generation */
    if(Mode == TE_DISABLE)
    {
        Data = TE_CLEAR_BITS;
    }
    else
    {
        Data = TE_BIT0;
    }
    Mask = TE_BIT0;
    AddressPE += IXF3204_PRM_ENABLE_REG;

    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPE(handle,AddressPE,Data,Mask,
                                          IXF3204_PERF_OFFSET))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    if((Status = IXF3204WriteBit(handle,AddressPE,Data,Mask)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetPRMMode
 * Returns: TEstatus
 * Description: Returns the PRM mode for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetPRMMode(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data, Mask = TE_BIT7;
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*perform set operation*/
    if((Status = IXF3204ReadBit(handle,Address, Mask,&Data )) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /*validate Mode parameter*/
    switch(Data)
    {
        case TE_BIT7:
            ReturnValue[0] = TE_ENABLE;
            break;
        case TE_CLEAR_BITS:
            ReturnValue[0] = TE_DISABLE;
            break;
        default:   
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetPRMRxStatus
 * Returns: TEstatus
 * Description: Returns the current status of the received PRM for the 
 *              specified Port.
 ******************************************************************************/
TEstatus IXF3204GetPRMRxStatus(void *handle,TEword Port, void *Results)
{
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data, Mask = (TE_BIT2 | TE_BIT1 | TE_BIT0);
    TEstatus Status;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    Address = IXF3204FindPortRxFDL(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    Address += IXF3204_RXFDL_PRMSR;
    if((Status = IXF3204ReadBit(handle,Address, Mask, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetPRMAddress
 * Returns: TEstatus
 * Description: Sets the address and control fields that the transmit or 
 *              receive side (Direction) is to use to send or receive PRM for
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204SetPRMAddress(void *handle, TEword Port, TEword Direction,
                              TEbyte Byte1, TEbyte Byte2,TEbyte Byte3)
{
    TEword AddressRx, AddressTx;
    TEstatus Status;

    /*Get offsets to perform the set operations*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX  :
            AddressRx += IXF3204_RXFDL_PRMAD1R;
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressRx,
                                                          Byte1))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteByte(handle,AddressRx, Byte1)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            /*perform set operation*/
            AddressRx++;
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressRx,
                                                          Byte2))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteByte(handle,AddressRx, Byte2)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            AddressRx++;
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressRx,
                                                          Byte3))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteByte(handle,AddressRx, Byte3)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        case TE_TX:
            AddressTx += IXF3204_TXFDL_PRMAD1R;
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressTx,
                                                          Byte1))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteByte(handle,AddressTx, Byte1)) 
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            /*perform set operation*/
            AddressTx++;
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressTx,
                                                          Byte2))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteByte(handle,AddressTx, Byte2)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            AddressTx++;
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteByteAllPortsRegs(handle,AddressTx,
                                                          Byte3))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteByte(handle,AddressTx, Byte3)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return TE_STATUS_SUCCESS;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetPRMAddress
 * Returns: TEstatus
 * Description: Returns the address and control fields that the transmit or 
 *              receive side (Direction) is using to send or receive PRM for
 *              the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetPRMAddress(void *handle, TEword Port,TEword Direction,
                              void *Results)
{
    TEword AddressRx,AddressTx;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data, Loop;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Direction)
    {
        case TE_RX  :
            /*perform get operation*/
            AddressRx += IXF3204_RXFDL_PRMAD1R;
            for(Loop = 0; Loop < IXF3204_PRM_ADDRESS_NUM; Loop++, AddressRx++)
            {
                if((Status = IXF3204ReadByte(handle,AddressRx,&Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                ReturnValue[Loop] = Data;
            }
            return TE_STATUS_SUCCESS;
        case TE_TX:
            /*perform get operation*/
            AddressTx += IXF3204_TXFDL_PRMAD1R;
            for(Loop = 0; Loop < IXF3204_PRM_ADDRESS_NUM; Loop++, AddressTx++)
            {
                if((Status = IXF3204ReadByte(handle,AddressTx,&Data)) !=
                   TE_STATUS_SUCCESS){
                    return Status;
                }
                
                ReturnValue[Loop] = Data;
            }
            return TE_STATUS_SUCCESS;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetPRMLastMessage
 * Returns: TEstatus
 * Description: Returns the last PRM received or transmitted for the specified
 *              Port and Direction. The transmitted PRM is generated internally
 *              when it is enabled.
 ******************************************************************************/
TEstatus IXF3204GetPRMLastMessage(void *handle, TEword Port,TEword Direction,
                                  void *Results)
{
    TEword AddressRx, AddressTx;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data, Loop;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    AddressRx = IXF3204FindPortRxFDL(Port);
    if(AddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    AddressTx = IXF3204FindPortTxFDL(Port);
    if(AddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    switch(Direction)
    {
        case TE_RX  :
            /*perform get operations for rx side*/
            AddressRx += IXF3204_RXFDL_PRMAD1R;
            for(Loop = 0; Loop < IXF3204_MAX_PRM_MESSAGE_SIZE; 
                Loop++, AddressRx++)
            {
                if((Status = IXF3204ReadByte(handle,AddressRx, &Data)) !=
                   TE_STATUS_SUCCESS){
                    return Status;
                }
                
                ReturnValue[Loop] = Data;
            }
            return TE_STATUS_SUCCESS;
        case TE_TX:
            /*perform get operation for tx side*/
            AddressTx += IXF3204_TXFDL_PRMAD1R;
            for(Loop = 0; Loop < IXF3204_MAX_PRM_MESSAGE_SIZE; 
                Loop++, AddressTx++)
            {
                if((Status = IXF3204ReadByte(handle,AddressTx, &Data)) !=
                   TE_STATUS_SUCCESS){
                    return Status;
                }

                ReturnValue[Loop] = Data;
            }
            return TE_STATUS_SUCCESS;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }
}



