/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   sig3204.c
 * Description: this file contains signaling functions
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetSignalingMask
 * Returns: TEstatus
 * Description: Enables or disables signaling on the specified time slot  
 *              (or all time slots when TE_ALL is selected) for the specified 
 *              Port. This applies to both the transmit and receive paths.
 ******************************************************************************/
TEstatus IXF3204SetSignalingMask(void *handle, TEword Port, TEword TimeSlot, 
                                 TEword Value)
{
    TEstatus Status;
    TEbyte Mask, NMask;
    TEbyte OffsetAddress;
    TEword Address;
    TEword PartialAddressRx, PartialAddressTx;
    TEdword Data;

    Data = TE_ALL_BITS;
    /*get offsets to perform the set operations*/
    PartialAddressRx = IXF3204FindPortRxSignaling_CDR(Port);
    if(PartialAddressRx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    PartialAddressTx = IXF3204FindPortTxSignaling_CDR(Port);
    if(PartialAddressTx == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Value parameter*/
    if((Value != TE_DISABLE) && (Value != TE_ENABLE)) 
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* validating parameter TS */
    if( TimeSlot != TE_ALL)
    {
        if(!((TimeSlot >= TE_TS0) && (TimeSlot <= TE_TS31)))
        {
            return TE_STATUS_BAD_PARAMS;
        }
    }

    if(TimeSlot == TE_ALL)
    {
        if(Value == TE_ENABLE)
        {
            Address = (TEword)(PartialAddressRx);
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,
                                                         TE_CLEAR_BITS,TE_BIT0))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                if((Status = IXF3204WriteBit(handle,Address,
                                             TE_CLEAR_BITS,TE_BIT0))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(PartialAddressTx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,
                                                         TE_CLEAR_BITS,TE_BIT1))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteBit(handle,Address,
                                             TE_CLEAR_BITS,TE_BIT1))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(IXF3204_RXSIG_CHGMASK0 + PartialAddressRx);
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllPortsRegs(handle,Address,
                                                          (TEword)Data))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteWord(handle,Address,(TEword)Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(IXF3204_RXSIG_CHGMASK2 + PartialAddressRx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllPortsRegs(handle,Address,
                                                          (TEword)Data))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteWord(handle,Address,(TEword)Data)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(IXF3204_TXSIG_DIS0 + PartialAddressTx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllPortsRegs(handle,Address,
                                                          TE_CLEAR_BITS))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteWord(handle,Address,TE_CLEAR_BITS)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(IXF3204_TXSIG_DIS2 + PartialAddressTx);
            /*Perform the Set operation*/
            return IXF3204WriteByte(handle,Address,TE_CLEAR_BITS);
        }

        if(Value == TE_DISABLE)
        {
            Address = (TEword)(PartialAddressRx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,TE_BIT0,
                                                         TE_BIT0))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteBit(handle,Address,TE_BIT0,TE_BIT0))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            Address = (TEword)(PartialAddressTx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,TE_BIT1,
                                                         TE_BIT1))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteBit(handle,Address,TE_BIT1,TE_BIT1))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(IXF3204_RXSIG_CHGMASK0 + PartialAddressRx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllPortsRegs(handle,Address,
                                                          TE_CLEAR_BITS))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteWord(handle,Address,TE_CLEAR_BITS))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(IXF3204_RXSIG_CHGMASK2 + PartialAddressRx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllPortsRegs(handle,Address,
                                                          TE_CLEAR_BITS))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteWord(handle,Address,TE_CLEAR_BITS))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(IXF3204_TXSIG_DIS0 + PartialAddressTx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteWordAllPortsRegs(handle,Address,
                                                          (TEword)Data))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteWord(handle,Address,(TEword)Data))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(IXF3204_TXSIG_DIS2 + PartialAddressTx);

            /*Perform the Set operation*/
            return IXF3204WriteByte(handle,Address,(TEbyte)Data);
        } 
        else 
        { 
            return TE_STATUS_BAD_PARAMS;
        }
    }
    
    Address = (TEword)(PartialAddressRx);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,TE_CLEAR_BITS,
                                                 TE_BIT0))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteBit(handle,Address,TE_CLEAR_BITS,TE_BIT0))!=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    Address = (TEword)(PartialAddressTx);

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,TE_CLEAR_BITS,
                                                 TE_BIT1))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }
    else
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteBit(handle,Address,TE_CLEAR_BITS,TE_BIT1))!=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    if(TimeSlot <= TE_TS7) 
    {
        OffsetAddress = IXF3204_RXSIG_CHGMASK0;
    }
    else if((TimeSlot > TE_TS7) && (TimeSlot <=  TE_TS15))
    {
        OffsetAddress = IXF3204_RXSIG_CHGMASK1;
    }
    else if((TimeSlot > TE_TS15) && (TimeSlot <= TE_TS23))
    {
        OffsetAddress = IXF3204_RXSIG_CHGMASK2;
    }
    else
    {
        OffsetAddress = IXF3204_RXSIG_CHGMASK3;
    }

    /*Validate the Value parameter*/
    switch(Value)
    {
        case TE_ENABLE :
            Mask = (TEbyte)(TE_BIT0 << (IXF3204GetTimeSlot(TimeSlot) & 
                                        (TE_BIT2 | TE_BIT1 | TE_BIT0)));
            Address = (TEword)(OffsetAddress + PartialAddressRx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Mask,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Mask,Mask))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            NMask = (TEbyte)~Mask;
            if(OffsetAddress != IXF3204_RXSIG_CHGMASK3)
            {
                Address = (TEword)(OffsetAddress + 4 + PartialAddressTx);

                /* Set all the ports with the same configuration */
                if(Port == TE_ALL)
                {
                    if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,
                                                             NMask,Mask))
                       != TE_STATUS_SUCCESS)
                    {
                        return Status;
                    }
                    return TE_STATUS_SUCCESS;
                }

                /*Perform the Set operation*/
                return IXF3204WriteBit(handle,Address,NMask,Mask);
            }
            break;
        case TE_DISABLE:
            Mask = (TEbyte)(TE_BIT0 << (IXF3204GetTimeSlot(TimeSlot) & 
                                        (TE_BIT2 | TE_BIT1 | TE_BIT0)));
            NMask = (TEbyte)~Mask;
            Address = (TEword)(OffsetAddress + PartialAddressRx);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,NMask,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*Perform the Set operation*/
                if((Status = IXF3204WriteBit(handle,Address,NMask,Mask))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            if(OffsetAddress != IXF3204_RXSIG_CHGMASK3)
            {
                Address = (TEword)(OffsetAddress + 4 + PartialAddressTx);

                /* Set all the ports with the same configuration */
                if(Port == TE_ALL)
                {
                    if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,
                                                             Mask,Mask))
                       != TE_STATUS_SUCCESS)
                    {
                        return Status;
                    }
                    return TE_STATUS_SUCCESS;
                }

                /*Perform the Set operation*/
                return IXF3204WriteBit(handle,Address,Mask,Mask);
            }
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingMask
 * Returns: TEstatus
 * Description: Returns the configured signaling mode of the specified time 
 *              slot for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetSignalingMask(void *handle, TEword Port, TEword TimeSlot,
                                 void *Results)
{
    TEbyte Data;
    TEbyte OffsetAddress;
    TEbyte Mask;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortRxSignaling_CDR(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the TS parameter*/
    if((TimeSlot < TE_TS0) || (TimeSlot > TE_TS31))
    {
        return TE_STATUS_BAD_PARAMS;
    }
    else if(TimeSlot <= TE_TS7) 
    {
        OffsetAddress = IXF3204_RXSIG_CHGMASK0;
    }
    else if((TimeSlot > TE_TS7)  && (TimeSlot <= TE_TS15))
    {
        OffsetAddress = IXF3204_RXSIG_CHGMASK1;
    }
    else if((TimeSlot > TE_TS15) && (TimeSlot <= TE_TS23))
    {
        OffsetAddress = IXF3204_RXSIG_CHGMASK2;
    }
    else
    {
        OffsetAddress = IXF3204_RXSIG_CHGMASK3;
    }
    
    Mask = (TEbyte)(TE_BIT0 << (IXF3204GetTimeSlot(TimeSlot) & 
                                (TE_BIT2 | TE_BIT1 | TE_BIT0)));
    Address = (TEword)(OffsetAddress + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data) 
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetSignalingForce
 * Returns: TEstatus
 * Description: Forces the value of the Outgoing_Port and time slot 
 *              specified (or all time slots when TE_ALL is selected). 
 *              The function IXF3204SetSignalingRelease must be used to remove
 *              the forced action.
 ******************************************************************************/
TEstatus IXF3204SetSignalingForce(void *handle, TEword Outgoing_Port,
                                  TEword TimeSlot, TEbyte Value)
{
    TEstatus Status;
    TEbyte Mask;
    TEbyte OffsetAddress;
    TEbyte ValueInv;
    TEword Address;
    TEword PartialAddressRx_CDM;
    TEword PartialAddressTx_CDR;
    TEword PartialAddressTx_SMD;
    TEdword Data;
    TEbyte Inv[] = {0x00,0x08,0x04,0x0C,0x02,0x0A,0x06,0x0E,0x01,0x09,0x05
                        ,0x0D,0x03,0x0B,0x07,0x0F};

    ValueInv = Inv[Value];
    Data = TE_ALL_BITS;

    /* validate TS */
    if (TimeSlot!= TE_ALL)
    {
        if(!((TimeSlot >= TE_TS0) && (TimeSlot <= TE_TS31)))
        {
            return TE_STATUS_BAD_PARAMS;
        }
    }

    /*Validate the Direction parameter*/
    if((!((Outgoing_Port >= TE_PORT0) && (Outgoing_Port <= TE_PORT7)) && 
        !((Outgoing_Port >= TE_BP0) && (Outgoing_Port <= TE_BP7)))||
       (Value > TE_BIT0_3))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    if((Outgoing_Port >= TE_PORT0)  && (Outgoing_Port <= TE_PORT7))
    {
        /*get offsets to perform the set operations*/
        PartialAddressTx_CDR = IXF3204FindPortTxSignaling_CDR(Outgoing_Port);
        if(PartialAddressTx_CDR == TE_STATUS_BAD_PARAMS)
        {
            return TE_STATUS_BAD_PARAMS;
        }

        PartialAddressTx_SMD = IXF3204FindPortTxSignaling_SMD(Outgoing_Port);
        if(PartialAddressTx_SMD == TE_STATUS_BAD_PARAMS)
        {
            return TE_STATUS_BAD_PARAMS;
        }

        /*if TS is equal to TE_ALL and TE_PORT is used*/
        if(TimeSlot == TE_ALL)
        {
            /*Perform the Set operation*/
            Address = (TEword)(IXF3204_TXSIG_FORCE0 + PartialAddressTx_CDR);
            if((Status = IXF3204WriteWord(handle,Address,(TEword)Data)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            Address = (TEword)(IXF3204_TXSIG_FORCE2 + PartialAddressTx_CDR);
            /*Perform the Set operation*/
            if((Status = IXF3204WriteWord(handle,Address,(TEword)Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            /* Write the forced value, the upper nibbles are ts0 to ts15
             * and the lower nibbles are ts16 to ts31
             */
            ValueInv |= (ValueInv << 4);
            for(OffsetAddress = TE_CLEAR_BITS; 
                OffsetAddress < IXF3204_MAX_TS_TXMEM_ADDR;OffsetAddress++)
            {
                Address = (TEword)(OffsetAddress + PartialAddressTx_SMD);
                if((Status = IXF3204WriteByte(handle,Address,ValueInv))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            return Status;
        }

        /*if TS is not equal to TE_ALL do the rest*/
        if(TimeSlot <= TE_TS7)
        {
            OffsetAddress = IXF3204_TXSIG_FORCE0;
        }
        else if((TimeSlot > TE_TS7) && (TimeSlot <= TE_TS15))
        {
            OffsetAddress = IXF3204_TXSIG_FORCE1;
        }
        else if((TimeSlot > TE_TS15) && (TimeSlot <= TE_TS23))
        {
            OffsetAddress = IXF3204_TXSIG_FORCE2;
        }
        else
        {
            OffsetAddress = IXF3204_TXSIG_FORCE3;
        }

        Mask = (TEbyte)(TE_BIT0 << (IXF3204GetTimeSlot(TimeSlot) & 
                                    (TE_BIT2 | TE_BIT1 | TE_BIT0)));
        Address = (TEword)(OffsetAddress + PartialAddressTx_CDR);
        if((Status = IXF3204WriteBit(handle,Address,Mask,Mask))!=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }

        /*get offsets to perform the set operations*/
        Address = (TEword)((IXF3204GetTimeSlot(TimeSlot) & TE_BIT0_3)
                           + PartialAddressTx_SMD);
        if(TimeSlot > TE_TS15) 
        {
            Mask = TE_BIT0_3;
            return (IXF3204WriteBit(handle,Address,ValueInv,Mask));
        } 
        else 
        {
            Mask = TE_BIT4_7;
            ValueInv <<= 4;
            return (IXF3204WriteBit(handle,Address,ValueInv,Mask));
        }
    }
    /*case of BP*/
    else
    {
        PartialAddressRx_CDM = IXF3204FindPortRxSysSigIF_CDM(Outgoing_Port);

        if(PartialAddressRx_CDM == TE_STATUS_BAD_PARAMS)
        {
            return (TEstatus)(PartialAddressRx_CDM);
        }
        
        Mask = (TE_BIT4_7 | TE_BIT3);
        /*case of BP and TS equal to TE_ALL*/
        if(TimeSlot == TE_ALL)
        {
            for(OffsetAddress = TE_CLEAR_BITS;
                OffsetAddress < IXF3204_MAX_TS_RXMEM_ADDR;OffsetAddress++)
            {
                Address = (TEword)(OffsetAddress + PartialAddressRx_CDM);
                ValueInv = (TEbyte)((ValueInv << 4) | TE_BIT3);
                /*Perform the Set operation*/
                if((Status = IXF3204WriteBit(handle,Address,ValueInv,Mask)) !=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }

            }
            return TE_STATUS_SUCCESS;
        }
        else
        {
            Address = (TEword)(IXF3204GetTimeSlot(TimeSlot) 
                               + PartialAddressRx_CDM);
            ValueInv = (TEbyte)((ValueInv << 4) | TE_BIT3);
            /*Perform the Set operation*/
            return IXF3204WriteBit(handle,Address,ValueInv,Mask);
        }
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingForce
 * Returns: TEstatus
 * Description: Returns the value that is forced in the specified time slot
 *              for the specified Port. When the returned value is 0xFF,
 *              the time slot is not forced. 
 ******************************************************************************/
TEstatus IXF3204GetSignalingForce(void *handle, TEword Outgoing_Port,
                                  TEword TimeSlot, void *Results)
{
    TEbyte Data;
    TEbyte OffsetAddress;
    TEbyte Mask;
    TEword Address;
    TEword PartialAddressTx_CDR;
    TEword PartialAddressTx_SMD;
    TEword PartialAddressRx_CDM;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Inv[] = {0x00,0x08,0x04,0x0C,0x02,0x0A,0x06,0x0E,0x01,0x09,0x05
                        ,0x0D,0x03,0x0B,0x07,0x0F};

    CHK_NULL(ReturnValue);

    /*Validate the TS parameter*/
    if(!((TimeSlot >= TE_TS0) && (TimeSlot <= TE_TS31)))
    {
        return TE_STATUS_BAD_PARAMS;
    }


    /*Validate the Outgoing_Port parameter*/
    if((!((Outgoing_Port >= TE_PORT0) && (Outgoing_Port <= TE_PORT7)) && 
        !((Outgoing_Port >= TE_BP0) && (Outgoing_Port <= TE_BP7))))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    if((Outgoing_Port >= TE_PORT0) && (Outgoing_Port <= TE_PORT7))
    {
        /*get offsets to perform the get operations*/
        PartialAddressTx_CDR = IXF3204FindPortTxSignaling_CDR(Outgoing_Port);

        if(PartialAddressTx_CDR == TE_STATUS_BAD_PARAMS)
        {
            return TE_STATUS_BAD_PARAMS;
        }

        if(TimeSlot <= TE_TS7) 
        {
            OffsetAddress = IXF3204_TXSIG_FORCE0;
        }
        else if((TimeSlot > TE_TS7) && (TimeSlot <= TE_TS15))
        {
            OffsetAddress = IXF3204_TXSIG_FORCE1;
        }
        else if((TimeSlot > TE_TS15) && (TimeSlot <= TE_TS23))
        {
            OffsetAddress = IXF3204_TXSIG_FORCE2;
        } 
        else
        {
            OffsetAddress = IXF3204_TXSIG_FORCE3;
        }
        
        Address = (TEword)(OffsetAddress + PartialAddressTx_CDR);
        Mask = (TEbyte)(TE_BIT0 << (IXF3204GetTimeSlot(TimeSlot) & 
                                    (TE_BIT2 | TE_BIT1 | TE_BIT0)));
        if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        /* The time slot is not forced, return 0xFF */
        if(Data == 0)
        {
            ReturnValue[0] = TE_BIT0_7;
        }
        else
        {
            PartialAddressTx_SMD = IXF3204FindPortTxSignaling_SMD(Outgoing_Port);
            if(PartialAddressTx_SMD == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(PartialAddressTx_SMD + 
                               (IXF3204GetTimeSlot(TimeSlot) & TE_BIT0_3));

            if(TimeSlot > TE_TS15)
            {
                Mask = TE_BIT0_3;
                Status = IXF3204ReadBit(handle,Address,Mask,&Data);
                if(Status != TE_STATUS_SUCCESS) 
                {
                    return Status;
                }
                ReturnValue[0] = Inv[Data];
            }
            else
            {
                Mask = TE_BIT4_7;
                Status = IXF3204ReadBit(handle,Address,Mask,&Data);
                if(Status != TE_STATUS_SUCCESS) 
                {
                    return Status;
                }
                ReturnValue[0] = Inv[Data >> 4];
            }
        }
    }
    else
    {

        /*get offsets to perform the get operations*/
        PartialAddressRx_CDM = IXF3204FindPortRxSysSigIF_CDM(Outgoing_Port);
        if(PartialAddressRx_CDM == TE_STATUS_BAD_PARAMS)
        {
            return TE_STATUS_BAD_PARAMS;
        }
        Address = (TEword)(IXF3204GetTimeSlot(TimeSlot) + PartialAddressRx_CDM);
        Mask = (TE_BIT4_7 | TE_BIT3);
        /*Perform the get operation*/
        if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }

        if(TE_BIT3 & Data)
        {
            Data &= TE_BIT4_7;
            ReturnValue[0] = Inv[Data >> 4];
        }
        else
        {  
            ReturnValue[0] = TE_BIT0_7;
        }
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetSignalingForceBP
 * Returns: TEstatus
 * Description: Forces all the time slots to a value in Pattern. This force 
 *              only works to the backplane. It is commonly used to send alarms.
 *              There are two registers available to use to force a pattern so
 *              one of them must be selected using the Register parameter.
 ******************************************************************************/
TEstatus IXF3204SetSignalingForceBP(void *handle, TEword Backplane, 
                                    TEword Select, TEword Pattern)
{
    TEbyte Status;
    TEbyte Data, Mask;
    TEword Address;
    TEword PartialAddress;

    /*get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortRxSysSigIF_CDR(Backplane);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXSYS_CR1 + PartialAddress);

    /*Validate the Select parameter*/
    switch(Select)
    {
        case TE_R1:
            Data = TE_BIT7;
            Mask = TE_BIT7;
            break; 
        case TE_R2: 
            Data = TE_BIT6;
            Mask = TE_BIT6;
            break; 
        default: 
            return TE_STATUS_BAD_PARAMS;
    }

    if(Pattern == TE_NOFORCE)
    {
        /*Perform the Set operation*/
        return IXF3204WriteByte(handle,Address,TE_CLEAR_BITS);
    }
    else
    {
        /*Perform the Set operation*/
        if((Status = IXF3204WriteBit(handle,Address,Data,Mask)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }

        Address = (TEword)(IXF3204_RXSYS_CR2 + PartialAddress);
        if(Mask == TE_BIT7)
        {
            Pattern <<= 4;
            Mask = TE_BIT4_7;
            /*Perform the Set operation*/
            return IXF3204WriteBit(handle,Address,(TEbyte)Pattern,Mask);
        }
        else
        {
            Mask = TE_BIT0_3;
            /*Perform the Set operation*/
            return IXF3204WriteBit(handle,Address,(TEbyte)Pattern,Mask);
        }
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingForceBP
 * Returns: TEstatus
 * Description: Returns the force value used in the specified register. When 
 *              forced is not enabled, it returns 0xFF.
 ******************************************************************************/
TEstatus IXF3204GetSignalingForceBP(void *handle, TEword Backplane, 
                                    TEword Select, void *Results)
{
    TEbyte Data;
    TEbyte Mask, Mask1;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Data = TE_BIT0;

    /*get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortRxSysSigIF_CDR(Backplane);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Select parameter*/
    switch(Select)
    {
        case TE_R1:
            Mask = TE_BIT7;
            Mask1 = TE_BIT4_7;
            break;
        case TE_R2:
            Mask = TE_BIT6;
            Mask1 = TE_BIT0_3;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXSYS_CR1 + PartialAddress);

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(!Data)
    {
        ReturnValue[0]= TE_BIT0_7;
        return TE_STATUS_SUCCESS;
    } 
    else
    {
        Address = (TEword)(IXF3204_RXSYS_CR2 + PartialAddress);

        /*Perform the get operation*/
        if((Status = IXF3204ReadBit(handle,Address,Mask1,&Data)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }

        ReturnValue[0] = Data;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetSignalingRelease
 * Returns: TEstatus
 * Description: Removes the signaling forced on the specified Outgoing_Port 
 *              and time slot (ts).
 ******************************************************************************/
TEstatus IXF3204SetSignalingRelease(void *handle, TEword Outgoing_Port,
                                    TEword TimeSlot)
{
    TEbyte Mask;
    TEbyte NMask;
    TEbyte OffsetAddress;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte Loop;

    /*Validate the TimeSlot parameter*/
    if( TimeSlot != TE_ALL)
    {
        if(!((TimeSlot >= TE_TS0) && (TimeSlot <= TE_TS31)))
        {
            return TE_STATUS_BAD_PARAMS;
        }
    }

    /*Validate the Outgoing_Port parameter*/
    if(!((Outgoing_Port >= TE_PORT0) && (Outgoing_Port <= TE_PORT7)) && 
       !((Outgoing_Port >= TE_BP0) && (Outgoing_Port <= TE_BP7)))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    if(TimeSlot == TE_ALL)
    {
        if((Outgoing_Port >= TE_PORT0) && (Outgoing_Port <= TE_PORT7))
        {
            /*get offsets to perform the set operations*/
            PartialAddress = IXF3204FindPortTxSignaling_CDR(Outgoing_Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(PartialAddress + IXF3204_TXSIG_FORCE0);

            /*Perform the Set operation*/
            if((Status = IXF3204WriteWord(handle,Address,TE_CLEAR_BITS)) != 
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            Address = (TEword)(PartialAddress + IXF3204_TXSIG_FORCE2);

            /*Perform the Set operation*/
            if((Status=IXF3204WriteWord(handle,Address,TE_CLEAR_BITS)) !=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        else
        {
            /*get offsets to perform the set operations*/
            PartialAddress = IXF3204FindPortRxSysSigIF_CDM(Outgoing_Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            for (Loop = 0; Loop < IXF3204_MAX_TS; Loop++)
            {
                Address = (TEword)(PartialAddress + IXF3204_RXBUSIF_MEM_TS1);

                /*Perform the Set operation*/
                if((Status = IXF3204WriteBit(handle,Address,TE_CLEAR_BITS,
                                             TE_BIT3))!= TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
        }
        return TE_STATUS_SUCCESS;
    }
    else
    {
        if((Outgoing_Port >= TE_PORT0) && (Outgoing_Port <= TE_PORT7))
        {
            /*get offsets to perform the set operations*/
            PartialAddress = IXF3204FindPortTxSignaling_CDR(Outgoing_Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            if(TimeSlot <= TE_TS7)
            { 
                OffsetAddress = IXF3204_TXSIG_FORCE0;
            }
            else if((TimeSlot > TE_TS7) && (TimeSlot <= TE_TS15))
            { 
                OffsetAddress = IXF3204_TXSIG_FORCE1;
            }
            else if((TimeSlot > TE_TS15) && (TimeSlot <= TE_TS23))
            { 
                OffsetAddress = IXF3204_TXSIG_FORCE2;
            }
            else
            {
                OffsetAddress = IXF3204_TXSIG_FORCE3;
            }

            Mask = (TEbyte)(TE_BIT0 << (IXF3204GetTimeSlot(TimeSlot) & 
                                        (TE_BIT2 | TE_BIT1 | TE_BIT0)));
            NMask = (TEbyte)(~Mask);
            Address = (TEword)(OffsetAddress + PartialAddress);
            return IXF3204WriteBit(handle,Address,NMask,Mask);
        } 
        else
        {
            /*get offsets to perform the set operations*/
            PartialAddress = IXF3204FindPortRxSysSigIF_CDM(Outgoing_Port);
            if(PartialAddress == TE_STATUS_BAD_PARAMS)
            {
                return TE_STATUS_BAD_PARAMS;
            }

            Address = (TEword)(IXF3204GetTimeSlot(TimeSlot) + PartialAddress);

            /*Perform the Set operation*/
            return IXF3204WriteBit(handle,Address,TE_CLEAR_BITS,TE_BIT3);
        }
    }
}

/*******************************************************************************
 * Function Name: IXF3204SetSignalingByteOnBackplane
 * Returns: TEstatus
 * Description: Enables or disables the transmission of the stuff nibble and 
 *              defines the position in the byte for the stuff nibble to be
 *              transmitted.
 ******************************************************************************/
TEstatus IXF3204SetSignalingByteOnBackplane(void *handle, TEword Backplane,
                                            TEword Layout)
{
    TEstatus Status;
    TEbyte Data, Mask;
    TEword Address;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxSysSigIF_CDR(Backplane);
    if(Address == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }
    
    if(Layout == TE_DISABLE)
    {
        /* Set all the ports with the same configuration */
        if(Backplane == TE_ALL)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,
                                                     TE_CLEAR_BITS,TE_BIT1))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
            return TE_STATUS_SUCCESS;
        }

        /*Perform the Set operation*/
        return IXF3204WriteBit(handle,Address,TE_CLEAR_BITS,TE_BIT1);
    }
    else
    {
        Data = TE_BIT1;
        
        /*Validate the Layout parameter*/        
        switch(Layout)
        {
            case TE_STUFFSTUFF:
                break;
            case TE_STUFFSIG:
                Data |= TE_BIT2;
                break;
            case TE_SIGSTUFF:
                Data |= TE_BIT3;
                break;
            case TE_SIGSIG:
                Data |= (TE_BIT3 | TE_BIT2);
                break;
            default:
                return TE_STATUS_BAD_PARAMS;
        }

        Mask = (TE_BIT3 | TE_BIT2 | TE_BIT1);

        /* Set all the ports with the same configuration */
        if(Backplane == TE_ALL)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }
        else
        {
            /*Perform the Set operation*/
            if((Status = IXF3204WriteBit(handle,Address,Data,Mask))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }
        }

        /*get offsets to perform the set operations*/
        Address = IXF3204FindPortTxSysSigIF_CDR(Backplane);
        if(Address == TE_STATUS_BAD_PARAMS)
        {
            return TE_STATUS_BAD_PARAMS;
        }

        /*Validate the Layout parameter*/        
        switch(Layout)
        {
            case TE_STUFFSTUFF:
                Data = TE_CLEAR_BITS;
                break;
            case TE_STUFFSIG:
                Data = TE_BIT6;
                break;
            case TE_SIGSTUFF:
                Data = TE_BIT7;
                break;
            case TE_SIGSIG:
                Data = (TE_BIT7 | TE_BIT6);
                break;
            default:
                return TE_STATUS_BAD_PARAMS;
        }

        Mask = (TE_BIT7 | TE_BIT6);

        /* Set all the ports with the same configuration */
        if(Backplane == TE_ALL)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
            return TE_STATUS_SUCCESS;
        }
        /*Perform the Set operation*/
        return IXF3204WriteBit(handle,Address,Data,Mask);
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingByteOnBackplane
 * Returns: TEstatus
 * Description: Returns the position of the stuff nibble configured to send to 
 *              the backplane.
 ******************************************************************************/
TEstatus IXF3204GetSignalingByteOnBackplane(void *handle, TEword Backplane,
                                            void *Results)
{
    TEbyte Data, Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    
    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operations*/
    Address = IXF3204FindPortRxSysSigIF_CDR(Backplane);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = (TE_BIT3 | TE_BIT2 | TE_BIT1);

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* Verify if disable */
    if(!(Data & TE_BIT1)) 
    {
        ReturnValue[0] = TE_DISABLE;
        return TE_STATUS_SUCCESS;
    }

    /*Validate the Data parameter*/        
    switch (Data)
    {
        case TE_BIT1:
            ReturnValue[0] = TE_STUFFSTUFF;
            break;
        case (TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_STUFFSIG;
            break;
        case (TE_BIT3 | TE_BIT1):
            ReturnValue[0] = TE_SIGSTUFF;
            break;
        case (TE_BIT3 | TE_BIT2 | TE_BIT1):
            ReturnValue[0] = TE_SIGSIG;
            break;
        default: 
            return TE_STATUS_BADREAD;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetSignalingStuffValue
 * Returns: TEstatus
 * Description: Configures the stuff nibble to be sent to the Backplane for 
 *              the specified time slot (ts) (or all time slots when TE_ALL is
 *              selected).
 ******************************************************************************/
TEstatus IXF3204SetSignalingStuffValue(void *handle, TEword Backplane, 
                                       TEword TimeSlot, TEbyte Value)
{
    TEbyte Mask, IntValue;
    TEstatus Status;
    TEword Address;

    if(TimeSlot != TE_ALL)
    {
        if(!((TimeSlot >= TE_TS0) && (TimeSlot <= TE_TS31)))
        {
            return TE_STATUS_BAD_PARAMS;
        }
    }

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxSignaling_SDM(Backplane);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0_3;

    /*Validate the TimeSlot parameter*/        
    if(TimeSlot == TE_ALL)
    {
        /*get offsets to perform the set operations*/
        Address = IXF3204FindPortRxSysSigIF_CDR(Backplane);
        if(Address == TE_STATUS_BAD_PARAMS)
        {
            return TE_STATUS_BAD_PARAMS;
        }
        
        IntValue = (TEbyte)(Value << 4);
        Mask = TE_BIT4_7;

        /* Set all the ports with the same configuration */
        if(Backplane == TE_ALL)
        {
            if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,IntValue,
                                                     Mask))
               != TE_STATUS_SUCCESS)
            {
                return Status;
            }
            return TE_STATUS_SUCCESS;
        }
        /*Perform the Set operation*/
        return IXF3204WriteBit(handle,Address,IntValue,Mask);
    }

    Address |= IXF3204GetTimeSlot(TimeSlot);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Backplane == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsMem(handle,Address,Value,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Value,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingStuffValue
 * Returns: TEstatus
 * Description: Returns the configured stuff nibble for the specified time slot.
 ******************************************************************************/
TEstatus IXF3204GetSignalingStuffValue(void *handle, TEword Backplane,
                                       TEword TimeSlot, void *Results)
{
    TEbyte Data, Mask;
    TEstatus Status;
    TEword Address;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Validate the TimeSlot parameter*/        
    if(TimeSlot != TE_ALL)
    {
        if(!((TimeSlot >= TE_TS0) && (TimeSlot <= TE_TS31)))
        {
            return TE_STATUS_BAD_PARAMS;
        }
    }

    if(TimeSlot == TE_ALL )
    {
        /*get offsets to perform the get operations*/
        Address = IXF3204FindPortRxSysSigIF_CDR(Backplane);
        if(Address == TE_STATUS_BAD_PARAMS)
        { 
            return TE_STATUS_BAD_PARAMS;
        }
        Mask = TE_BIT4_7;

        /*Perform the get operation*/
        if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
           TE_STATUS_SUCCESS)
        {
            return Status;
        }

        ReturnValue[0] = (Data >> 4);

        return TE_STATUS_SUCCESS;
    }

    /*get offsets to perform the get operations*/
    Address = IXF3204FindPortRxSignaling_SDM(Backplane);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT0_3;

    /*select TimeSlot*/
    Address |= IXF3204GetTimeSlot(TimeSlot);

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetSignalingFreeze
 * Returns: TEstatus
 * Description: Sets the criteria used to declare freeze for the specified 
 *              Port. When the signaling module is in a freeze state, the last
 *              stable value is kept. More than one criteria can be set by
 *              using this function again for each criteria to be set. For
 *              example, the LOS criteria can be set by using this function.
 *              Then use it again to set the AIS criteria. When TE_ALL is
 *              selected, all the listed criteria cause freeze to be declared.
 ******************************************************************************/
TEstatus IXF3204SetSignalingFreeze(void *handle, TEword Port,
                                   TEword Criteria, TEword Value)
{
    TEbyte Mask;
    TEbyte Data;
    TEstatus Status;
    TEword Address;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Value parameter*/        
    switch(Value)
    {
        case TE_ENABLE:
            Data = TE_BIT0;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;        
    }

    /*Validate the Criteria parameter*/        
    switch(Criteria)
    {
        case TE_LOF: 
            Data = (TEbyte)(Data << 7);
            Mask = TE_BIT7;
            break; 
        case TE_LOS:
            Data = (TEbyte)(Data << 6);
            Mask = TE_BIT6;
            break;
        case TE_LOCASMF:
            Data = (TEbyte)(Data << 5);
            Mask = TE_BIT5;
            break;
        case TE_AIS:
            Data = (TEbyte)(Data << 4);
            Mask = TE_BIT4;
            break;
        case TE_ALL:
            if(Data != TE_CLEAR_BITS)
            {
                Data = TE_BIT4_7;
            }
            Mask = TE_BIT4_7;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingFreeze
 * Returns: TEstatus
 * Description: Returns the current status for the Port and freeze Criteria
 *              specified. The function must be used once for each criteria.
 *              To check all four freeze criteria, the function must be used 
 *              four times specifying the desired criteria.
 ******************************************************************************/
TEstatus IXF3204GetSignalingFreeze(void *handle, TEword Port, TEword Criteria,
                                   void *Results)
{
    TEbyte Data;
    TEbyte Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operations*/
    Address = IXF3204FindPortRxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Criteria parameter*/        
    switch(Criteria)
    {
        case TE_LOF:
            Mask = TE_BIT7;
            break;
        case TE_LOS:
            Mask = TE_BIT6;
            break;
        case TE_LOCASMF:
            Mask = TE_BIT5;
            break; 
        case TE_AIS:
            Mask = TE_BIT4;
            break; 
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetSignalingDebounce
 * Returns: TEstatus
 * Description: Enables or disables the debouncing of received signalling for
 *              the specified Port. When enabled, two consecutive signals with
 *              the same value are considered valid.
 ******************************************************************************/
TEstatus IXF3204SetSignalingDebounce(void *handle, TEword Port, TEword Value)
{
    TEbyte Data, Mask;
    TEstatus Status;
    TEword Address;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the Value parameter*/        
    switch(Value)
    {
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        case TE_ENABLE:
            Data = TE_BIT2;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT2;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingDebounce
 * Returns: TEstatus
 * Description: Returns the debounce mode for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetSignalingDebounce(void *handle, TEword Port, void *Results)
{
    TEbyte Data, Mask;
    TEword Address;
    TEstatus Status;
    TEbyte *ReturnValue = (TEbyte *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operations*/
    Address = IXF3204FindPortRxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS) 
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT2;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data)
    {
        ReturnValue[0] = TE_ENABLE;
    }
    else
    {
        ReturnValue[0] = TE_DISABLE;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetSignalingSLC96Mode
 * Returns: TEstatus
 * Description: Configures the number of states that the signaling module
 *              expects when configured in SLC96 mode for the specified Port.
 *              When 9 states is selected, the toggling of the bits is taken as
 *              a state. In this case the debouncing uses a different method.
 *              When 4 states is selected, the two signaling bits give four
 *              possible states and toggling is not used.
 ******************************************************************************/
TEstatus IXF3204SetSignalingSLC96Mode(void *handle, TEword Port, TEword State)
{
    TEbyte Mask, Data;
    TEstatus Status;
    TEword Address;

    /*get offsets to perform the set operations*/
    Address = IXF3204FindPortRxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*Validate the State parameter*/        
    switch(State)
    {
        case TE_STATE_4:
            Data = TE_CLEAR_BITS;
            break;
        case TE_STATE_9:
            Data = TE_BIT1;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;   
    }

    Mask = TE_BIT1;

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }
    /*Perform the Set operation*/
    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingSLC96Mode
 * Returns: TEstatus
 * Description: Returns the number of states that the module recognizes in
 *              SLC96 mode for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetSignalingSLC96Mode(void *handle, TEword Port, 
                                      void *Results)
{
    TEbyte Mask, Data;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operations*/
    Address = IXF3204FindPortRxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS) 
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Mask = TE_BIT1;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data)
    {
        ReturnValue[0] = TE_STATE_9;
    }
    else
    {
        ReturnValue[0] = TE_STATE_4;
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingChange
 * Returns: TEstatus
 * Description: The returned value shows the time slot that has changed for
 *              the specified Port. To enable this option the mask must be set
 *              with IXF3204SetSignalingMask. 
 ******************************************************************************/
TEstatus IXF3204GetSignalingChange(void *handle, TEword Port, void *Results)
{
    TEword Address;
    TEstatus Status;
    TEdword Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*get offsets to perform the get operations*/
    Address = IXF3204FindPortRxSignaling_CDR(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204_RXSIG_CHGSTATUS0 + Address);

    /*Perform the get operation*/
    if((Status = IXF3204ReaddWord( handle, Address, &Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetSignalingStable
 * Returns: TEstatus
 * Description: Returns the current received signaling of the specified Port 
 *              and time slot (TS). When debouncing is enabled, the Results 
 *              have been debounced and are considered stable. When debouncing
 *              is disabled, all received data is considered stable.
 ******************************************************************************/
TEstatus IXF3204GetSignalingStable(void *handle, TEword Port, TEword TimeSlot,
                                   void *Results)
{
    TEbyte Data, Mask;
    TEbyte ValueInv;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Inv[] = {0x00,0x08,0x04,0x0C,0x02,0x0A,0x06,0x0E,0x01,0x09,0x05
                        ,0x0D,0x03,0x0B,0x07,0x0F};

    CHK_NULL(ReturnValue);

    if(!((TimeSlot >= TE_TS0) && (TimeSlot <= TE_TS31)))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*get offsets to perform the get operations*/
    Address = IXF3204FindPortRxSignaling_SDM(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(IXF3204GetTimeSlot(TimeSlot) + Address);
    Mask = TE_BIT4_7;

    /*Perform the get operation*/
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }
    /* shift it to the low nibble */
    Data >>= 4;
    /* The value is storaged as DCBA                 */
    /* and returned as ABCD so inversion is required */
    ValueInv = Inv[Data];
    ReturnValue[0] = ValueInv;

    return TE_STATUS_SUCCESS;
}

