/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   slpb3204.c
 * Description: this file contains slip buffer functions
 *
 ******************************************************************************/

#include "ixf3204.h"


/*******************************************************************************
 * Function Name: IXF3204SetSlipBufferClearCounters
 * Returns: TEstatus
 * Description: Sets all the internal and shadow counters to zero for the 
 *              specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204SetSlipBufferClearCounters(void *handle, TEword Port,
                                           TEword Direction)
{
    TEbyte Data = TE_BIT2, Mask = TE_BIT2;
    TEstatus Status;
    TEword Address;

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortSlipBuffer(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                return TE_STATUS_SUCCESS;
            }
            /*perform set operation*/
            return IXF3204WriteBit(handle,Address,Data,Mask);
        case TE_TX:
            Address = (TEword)(Address + IXF3204_SB_SBUSET);
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                return TE_STATUS_SUCCESS;
            }
            /*perform set operation*/
            return IXF3204WriteBit(handle,Address,Data,Mask);
        case TE_BOTH:
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }
            else
            {
                /*perform set operation*/
                if((Status = IXF3204WriteBit(handle,Address,Data,Mask))!=
                   TE_STATUS_SUCCESS)
                {
                    return Status;
                }
            }

            Address = (TEword)(Address + IXF3204_SB_SBUSET);
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                return TE_STATUS_SUCCESS;
            }
            /*perform set operation*/
            return IXF3204WriteBit(handle,Address,Data,Mask);
        default:  
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204SetSlipBufferConfig
 * Returns: TEstatus
 * Description: Configures the slip buffer mode (distance) and centering on 
 *              COFA for the specified Port. The slip buffer mode refers to
 *              the distance allowed between LINE and SYSTEM (Normal flow) or
 *              between RECEIVE and TRANSMIT when in payload loopback mode.
 *              When in DFM (Double Frame Mode), if the distance grows too much
 *              (to almost 2 frames) or if it shrinks too much (to almost 0
 *              frames) a slip or skip will occur so that the distance is set
 *              back to something near one frame. In MDM (Minimum Delay Mode), 
 *              if the distance grows above 7 or shrinks to 0 bytes, then only
 *              a warning will be issued.
 ******************************************************************************/
TEstatus IXF3204SetSlipBufferConfig(void *handle, TEword Port, TEword Mode,
                                    TEword AutoCenter)
{
    TEbyte Data, Mask = (TE_BIT3 | TE_BIT2 | TE_BIT0);
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortSlipBuffer(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    Address = (TEword)(PartialAddress + IXF3204_SB_SBMODER);

    /*validate AutoCenter parameter*/
    switch(AutoCenter)
    {
        case TE_ENABLE:
            Data = TE_BIT0;
            break;
        case TE_DISABLE:
            Data = TE_CLEAR_BITS;
            break;
        default: 
            return TE_STATUS_BAD_PARAMS;  
    }

    /*validate Mode parameter*/
    switch(Mode)
    {
        case TE_TWOFRAMES:
            Data |= TE_BIT3;
            break;
        case TE_MINDELAY:
            Data |= (TE_BIT3 | TE_BIT2);
            break;
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

    /* Set all the ports with the same configuration */
    if(Port == TE_ALL)
    {
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }

        Address = (TEword)(PartialAddress + IXF3204_SB_SBMODET);
        if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,Mask))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
        return TE_STATUS_SUCCESS;
    }

    /*perform set operation*/
    if((Status = IXF3204WriteBit(handle,Address,Data,Mask))!=
        TE_STATUS_SUCCESS)
    {
        return Status;
    }

    Address = (TEword)(PartialAddress + IXF3204_SB_SBMODET);

    return IXF3204WriteBit(handle,Address,Data,Mask);
}

/*******************************************************************************
 * Function Name: IXF3204GetSlipBufferConfig
 * Returns: TEstatus
 * Description: Returns the configured slip buffer mode (distance) and 
 *              centering on COFA for the specified Port.
 ******************************************************************************/
TEstatus IXF3204GetSlipBufferConfig(void *handle, TEword Port, void *Results)
{
    TEbyte Data, Mask;
    TEword Address;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    Address = IXF3204FindPortSlipBuffer(Port);
    if(Address == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }
    Address = (TEword)(Address + IXF3204_SB_SBMODER);

    /*perform set operation*/
    Mask = (TE_BIT3 | TE_BIT2 | TE_BIT0);
    if((Status = IXF3204ReadBit(handle,Address,Mask,&Data))
       != TE_STATUS_SUCCESS)
    {
        return Status;
    }

    if(Data & TE_BIT0)
    {
        ReturnValue[1] = TE_ENABLE;
    }
    else
    {
        ReturnValue[1] = TE_DISABLE;
    }

    if(Data & TE_BIT3)
    {
        if(Data & TE_BIT2)
        {
            ReturnValue[0] = TE_MINDELAY;
        }
        else
        {
            ReturnValue[0] = TE_TWOFRAMES;
        }
    }

    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204SetSlipBufferCenterPointer
 * Returns: TEstatus
 * Description: Programs the slip buffer to move the pointer to the farthest 
 *              time slot when the read pointer reaches the end of a frame for
 *              the specified Port and Direction. This only works when working
 *              in two frames mode.
 ******************************************************************************/
TEstatus IXF3204SetSlipBufferCenterPointer(void *handle, TEword Port, 
                                           TEword Direction)
{
    TEbyte Data = TE_BIT1;
    TEbyte Mask = TE_BIT1;
    TEstatus Status;
    TEword Address;
    TEword PartialAddress;

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortSlipBuffer(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBMODER);
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                return TE_STATUS_SUCCESS;
            }
            /*perform set operation*/
            return IXF3204WriteBit(handle,Address,Data,Mask);
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBMODET);
            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                return TE_STATUS_SUCCESS;
            }
            /*perform set operation*/
            return IXF3204WriteBit(handle,Address,Data,Mask);
        case TE_BOTH:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBMODER);

            /* Set all the ports with the same configuration */
            if(Port == TE_ALL)
            {
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }
                Address = (TEword)(PartialAddress + IXF3204_SB_SBMODET);
                if((Status = IXF3204WriteBitAllPortsRegs(handle,Address,Data,
                                                         Mask))
                   != TE_STATUS_SUCCESS)
                {
                    return Status;
                }

                return TE_STATUS_SUCCESS;
            }

            /*perform set operation*/
            if((Status = IXF3204WriteBit(handle,Address,Data,Mask))!=
                TE_STATUS_SUCCESS)
            {
                return Status;
            }

            Address = (TEword)(PartialAddress + IXF3204_SB_SBMODET);
            return IXF3204WriteBit(handle,Address,Data,Mask);
        default: 
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetSlipBufferSlip
 * Returns: TEstatus
 * Description: Returns the status of the slip buffer for the specified Port 
 *              and Direction.
 ******************************************************************************/
TEstatus IXF3204GetSlipBufferSlip(void *handle, TEword Port,
                                  TEword Direction, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the set operations*/
    PartialAddress = IXF3204FindPortSlipBuffer(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {  
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBSTSR);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBSTST);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data)) !=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetSlipBufferDistance
 * Returns: TEstatus
 * Description: Returns the distance between the read and write pointers with
 *              the appropriate circular adjustment for the specified Port and 
 *              Direction. 
 ******************************************************************************/
TEstatus IXF3204GetSlipBufferDistance(void *handle,  TEword Port,
                                      TEword Direction, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortSlipBuffer(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    { 
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBDSTR);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBDSTT);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetSlipBufferReadStatus
 * Returns: TEstatus
 * Description: Returns a byte that contains the status of the read pointer in 
 *              the slip buffer for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetSlipBufferReadStatus(void *handle, TEword Port,
                                        TEword Direction, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEbyte Data;
    TEdword *ReturnValue = (TEdword*)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortSlipBuffer(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBRPTRR);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBRPTRT);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetSlipBufferWriteStatus
 * Returns: TEstatus
 * Description: Returns a byte that contains the status of the write pointer 
 *              in the slip buffer for the specified Port and Direction.
 ******************************************************************************/
TEstatus IXF3204GetSlipBufferWriteStatus(void *handle, TEword Port, 
                                         TEword Direction, void *Results)
{
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;
    TEbyte Data;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortSlipBuffer(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBWPTRR);
            break;
        case TE_TX: 
            Address = (TEword)(PartialAddress + IXF3204_SB_SBWPTRT);
            break;
        default:   
            return TE_STATUS_BAD_PARAMS;
    }

    /*perform get operation*/
    if((Status = IXF3204ReadByte(handle,Address,&Data))!=
       TE_STATUS_SUCCESS)
    {
        return Status;
    }

    ReturnValue[0] = Data;
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204GetSlipBufferCount
 * Returns: TEstatus
 * Description: Returns the number of controlled slips or the under/overflows 
 *              when working in minimum delay for the specified Port and
 *              Direction. The counter is updated every second by the hardware 
 *              with the count of the previous second. A 16-bit value is 
 *              returned. This is a non-rollover counter.
 ******************************************************************************/
TEstatus IXF3204GetSlipBufferCount(void *handle, TEword Port,
                                   TEword Direction, void *Results)
{
    TEbyte Data;
    TEword Address;
    TEword PartialAddress;
    TEstatus Status;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    /*Get offsets to perform the get operations*/
    PartialAddress = IXF3204FindPortSlipBuffer(Port);
    if(PartialAddress == TE_STATUS_BAD_PARAMS)
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /*validate Direction parameter*/
    switch(Direction)
    {
        case TE_RX:
            /*perform get operation*/
            Address = (TEword)(PartialAddress + IXF3204_SB_SBCSCLR);
            if((Status = IXF3204ReadByte(handle,Address,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            Address = (TEword)(PartialAddress + IXF3204_SB_SBCSCHR);
            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] |= (TEdword)(Data << 8);
            break;
        case TE_TX:
            Address = (TEword)(PartialAddress + IXF3204_SB_SBCSCLT);
            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] = Data;
            Address = (TEword) (PartialAddress + IXF3204_SB_SBCSCHT);
            /*perform get operation*/
            if((Status = IXF3204ReadByte(handle,Address,&Data))!=
               TE_STATUS_SUCCESS)
            {
                return Status;
            }

            ReturnValue[0] |= (TEdword)(Data << 8);
            break;
        default:  
            return TE_STATUS_BAD_PARAMS;
    }

    return TE_STATUS_SUCCESS;
}

