/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN "AS IS" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */
/******************************************************************************
 * File Name:   util3204.c
 * Description: this file  contains utils functions for port offsets
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204FindPortPLI
 * Returns: TEword
 * Description: This function returne the base address of the LIU
 ******************************************************************************/
TEword IXF3204FindPortPLI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_PLIOffset1;  
        case TE_PORT1:   
            return IXF3204_PLIOffset2;  
        case TE_PORT2:   
            return IXF3204_PLIOffset3;  
        case TE_PORT3:   
            return IXF3204_PLIOffset4;  
        default:
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204TxFindPort
 * Returns: TEword
 * Description: Returns the base address of the transmit framer
 ******************************************************************************/
TEword IXF3204TxFindPort(TEword Port)
{
    switch (Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:   
            return IXF3204_TxFramerOffset1; 
        case TE_PORT1:   
            return IXF3204_TxFramerOffset2; 
        case TE_PORT2:   
            return IXF3204_TxFramerOffset3; 
        case TE_PORT3:   
            return IXF3204_TxFramerOffset4; 
        default:   
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204RxFindPort
 * Returns: TEword
 * Description: Returns the base address of the receive framer
 ******************************************************************************/
TEword IXF3204RxFindPort(TEword Port)
{
    switch(Port) 
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:   
            return IXF3204_RxFramerOffset1;  
        case TE_PORT1:   
            return IXF3204_RxFramerOffset2;  
        case TE_PORT2:   
            return IXF3204_RxFramerOffset3;  
        case TE_PORT3:   
            return IXF3204_RxFramerOffset4;  
        default:   
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortBackPlane
 * Returns: TEword
 * Description: Returns the base addres of the selected backplane
 ******************************************************************************/
TEword IXF3204FindPortBackPlane(TEword Port)
{
    switch (Port)
    {
        /* if TE_ALL return the offset of backplane 0 */
        case TE_ALL:   
        case TE_BP0:   
        case TE_PORT0:   
            return IXF3204_BackplaneOffset1;  
        case TE_BP1:   
        case TE_PORT1:   
            return IXF3204_BackplaneOffset2;  
        case TE_BP2:   
        case TE_PORT2:   
            return IXF3204_BackplaneOffset3;  
        case TE_BP3:   
        case TE_PORT3:   
            return IXF3204_BackplaneOffset4;  
        default:  
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRAMRx
 * Returns: TEword
 * Description:This function returns offset of  Backplane_RAM_Rx
 ******************************************************************************/
TEword IXF3204FindPortRAMRx(TEword Port)
{
    switch (Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:
        case TE_BP0:   
            return IXF3204_Backplane_RAM_Rxoffset1; 
        case TE_PORT1:
        case TE_BP1:   
            return IXF3204_Backplane_RAM_Rxoffset2; 
        case TE_PORT2:
        case TE_BP2:   
            return IXF3204_Backplane_RAM_Rxoffset3; 
        case TE_PORT3:
        case TE_BP3:   
            return IXF3204_Backplane_RAM_Rxoffset4; 
        default:   
            return TE_STATUS_BAD_PARAMS; 
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRAMTx
 * Returns: TEword
 * Description: This function returns offset of Backplane_RAM_Tx
 ******************************************************************************/
TEword IXF3204FindPortRAMTx(TEword Port)
{
    switch(Port) 
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:
        case TE_BP0:   
            return IXF3204_Backplane_RAM_Txoffset1; 
        case TE_PORT1:
        case TE_BP1:   
            return IXF3204_Backplane_RAM_Txoffset2; 
        case TE_PORT2:
        case TE_BP2:   
            return IXF3204_Backplane_RAM_Txoffset3;  
        case TE_PORT3:
        case TE_BP3:   
            return IXF3204_Backplane_RAM_Txoffset4;  
        default:   
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortSlipBuffer
 * Returns: TEword
 * Description: Returns the base address of the slip buffer
 ******************************************************************************/
TEword IXF3204FindPortSlipBuffer(TEword Port)
{
    switch (Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:   
            return IXF3204_SlipBufferOffset1;  
        case TE_PORT1:   
            return IXF3204_SlipBufferOffset2;  
        case TE_PORT2:   
            return IXF3204_SlipBufferOffset3;  
        case TE_PORT3:   
            return IXF3204_SlipBufferOffset4;  
        default:   
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortClockSelector
 * Returns: TEword
 * Description: Returns the base address of the clock selector
 ******************************************************************************/
TEword IXF3204FindPortClockSelector(TEword Port)
{
    switch (Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:    
            return IXF3204_PerPortRegister1;
        case TE_PORT1:    
            return IXF3204_PerPortRegister2;  
        case TE_PORT2:    
            return IXF3204_PerPortRegister3;  
        case TE_PORT3:    
            return IXF3204_PerPortRegister4;  
        default: 
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindBERT
 * Returns: TEword
 * Description: Returns the base address of the selecte BERT
 ******************************************************************************/
TEword IXF3204FindBERT(TEword BERT)
{
    switch (BERT)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_BERT0:  
            return IXF3204_BERTOffset1;  
        case TE_BERT1:  
            return IXF3204_BERTOffset2;  
        case TE_BERT2:  
            return IXF3204_BERTOffset3;  
        case TE_BERT3:  
            return IXF3204_BERTOffset4;  
        default: 
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTxSignaling_CDR
 * Returns: TEword
 * Description: Returns the base address of the Tx signaling registers
 ******************************************************************************/
TEword IXF3204FindPortTxSignaling_CDR(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_TxSignaling_CDR_Offset1; 
        case TE_PORT1:  
            return IXF3204_TxSignaling_CDR_Offset2; 
        case TE_PORT2:  
            return IXF3204_TxSignaling_CDR_Offset3; 
        case TE_PORT3:  
            return IXF3204_TxSignaling_CDR_Offset4; 
        default: 
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTxSignaling_SMD
 * Returns: TEword
 * Description: Returns the base address of Tx signaling data memory
 ******************************************************************************/
TEword IXF3204FindPortTxSignaling_SMD(TEword Port){

    switch (Port){
        case TE_PORT0:  
            return IXF3204_TxSignaling_SDM_Offset1;  
        case TE_PORT1:  
            return IXF3204_TxSignaling_SDM_Offset2;  
        case TE_PORT2:  
            return IXF3204_TxSignaling_SDM_Offset3;  
        case TE_PORT3:  
            return IXF3204_TxSignaling_SDM_Offset4;  
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }

}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxSignaling_SDM
 * Returns: TEword
 * Description: Returns the base address of Rx signaling data memory
 ******************************************************************************/
TEword IXF3204FindPortRxSignaling_SDM(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:
        case TE_BP0: 
            return IXF3204_RxSignaling_SDM_Offset1;  
        case TE_PORT1:
        case TE_BP1:  
            return IXF3204_RxSignaling_SDM_Offset2;  
        case TE_PORT2:
        case TE_BP2:  
            return IXF3204_RxSignaling_SDM_Offset3;  
        case TE_PORT3:
        case TE_BP3:  
            return IXF3204_RxSignaling_SDM_Offset4;  
        default:    
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTxSignaling_CDM
 * Returns: TEword
 * Description: Returns the base address of Tx signaling configuratio memory
 ******************************************************************************/
TEword IXF3204FindPortTxSignaling_CDM(TEword Port)
{
    switch(Port)
    {
        case TE_PORT0:  
            return IXF3204_TxSignaling_CDM_Offset1;  
        case TE_PORT1:  
            return IXF3204_TxSignaling_CDM_Offset2;  
        case TE_PORT2:  
            return IXF3204_TxSignaling_CDM_Offset3;  
        case TE_PORT3:  
            return IXF3204_TxSignaling_CDM_Offset4;  
        default: 
            return TE_STATUS_BAD_PARAMS; 
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTxSignalingLine_CDM
 * Returns: TEword
 * Description: Returns the base address of Tx line signaling configuration
 *              memory
 ******************************************************************************/
TEword IXF3204FindPortTxSignalingLine_CDM(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_TxSignaling_CDM_Line_Offset1;
        case TE_PORT1:
            return IXF3204_TxSignaling_CDM_Line_Offset2;  
        case TE_PORT2:  
            return IXF3204_TxSignaling_CDM_Line_Offset3; 
        case TE_PORT3:
            return IXF3204_TxSignaling_CDM_Line_Offset4;
        default:
            return TE_STATUS_BAD_PARAMS; 
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxSignaling_CDR
 * Returns: TEword
 * Description: Returns the base address of Rx signaling configuration regs
 ******************************************************************************/
TEword IXF3204FindPortRxSignaling_CDR(TEword Port)
{
    switch (Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
        case TE_BP0:    
            return IXF3204_RxSignaling_CDR_Offset1;  
        case TE_PORT1:  
        case TE_BP1:    
            return IXF3204_RxSignaling_CDR_Offset2;  
        case TE_PORT2:  
        case TE_BP2:    
            return IXF3204_RxSignaling_CDR_Offset3;  
        case TE_PORT3:  
        case TE_BP3:    
            return IXF3204_RxSignaling_CDR_Offset4;  
        default:      
            return TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxSysSigIF_CDR
 * Returns: TEword
 * Description: Returns the base address of Rx signaling sys configuration regs
 ******************************************************************************/
TEword IXF3204FindPortRxSysSigIF_CDR(TEword BP)
{
    switch(BP)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_BP0:
        case TE_PORT0: 
            return IXF3204_RxSysSigIF_CDR_Offset1;  
        case TE_BP1:  
        case TE_PORT1:  
            return IXF3204_RxSysSigIF_CDR_Offset2;  
        case TE_BP2:  
        case TE_PORT2: 
            return IXF3204_RxSysSigIF_CDR_Offset3;   
        case TE_BP3:  
        case TE_PORT3: 
            return IXF3204_RxSysSigIF_CDR_Offset4;   
        default:  
            return TE_STATUS_BAD_PARAMS;  
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxSysSigIF_CDM
 * Returns: TEword
 * Description: Returns the base address of Rx signaling sys conf. memory
 ******************************************************************************/
TEword IXF3204FindPortRxSysSigIF_CDM(TEword BP)
{
    switch(BP)
    {
        case TE_BP0:  
            return IXF3204_RxSysSigIF_CDM_Offset1;  
        case TE_BP1:  
            return IXF3204_RxSysSigIF_CDM_Offset2;  
        case TE_BP2:  
            return IXF3204_RxSysSigIF_CDM_Offset3;  
        case TE_BP3:  
            return IXF3204_RxSysSigIF_CDM_Offset4;  
        default: 
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTxSysSigIF_CDR
 * Returns: TEword
 * Description: Returns the base address of Tx signaling sys configuration regs
 ******************************************************************************/
TEword IXF3204FindPortTxSysSigIF_CDR(TEword BP)
{
    switch(BP)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:
        case TE_BP0:
            return IXF3204_TxSysSigIF_CDR_Offset1;
        case TE_PORT1:
        case TE_BP1: 
            return IXF3204_TxSysSigIF_CDR_Offset2;  
        case TE_PORT2:
        case TE_BP2:
            return IXF3204_TxSysSigIF_CDR_Offset3;
        case TE_PORT3:
        case TE_BP3: 
            return IXF3204_TxSysSigIF_CDR_Offset4;  
        default: 
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxBusIF
 * Returns: TEword
 * Description: Returns the base address of the Rx bus interface
 ******************************************************************************/
TEword IXF3204FindPortRxBusIF(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_RxBusIFOffset1;   
        case TE_PORT1:  
            return IXF3204_RxBusIFOffset2;   
        case TE_PORT2:  
            return IXF3204_RxBusIFOffset3;   
        case TE_PORT3:  
            return IXF3204_RxBusIFOffset4;   
        default:   
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxBusIF_CDM
 * Returns: TEword
 * Description: Returns the base address of the rx bus interface memory
 ******************************************************************************/
TEword IXF3204FindPortRxBusIF_CDM(TEword Port)
{
    switch(Port)
    {
        case TE_PORT0:  
            return IXF3204_RxBusIF_CDM_Offset1;  
        case TE_PORT1:  
            return IXF3204_RxBusIF_CDM_Offset2;  
        case TE_PORT2:  
            return IXF3204_RxBusIF_CDM_Offset3;  
        case TE_PORT3:  
            return IXF3204_RxBusIF_CDM_Offset4;  
        default:   
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxHDLC
 * Returns: TEword
 * Description: Returns the base address of the receive HDLC's
 ******************************************************************************/
TEword IXF3204FindPortRxHDLC(TEword HDLC)
{
    switch(HDLC)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_HDLC0:   
            return IXF3204_Rx_HDLC1_Offset;   
        case TE_HDLC1:   
            return IXF3204_Rx_HDLC2_Offset;   
        case TE_HDLC2:   
            return IXF3204_Rx_HDLC3_Offset;   
        case TE_HDLC3:   
            return IXF3204_Rx_HDLC4_Offset;   
        case TE_HDLC4:   
            return IXF3204_Rx_HDLC5_Offset;   
        case TE_HDLC5:   
            return IXF3204_Rx_HDLC6_Offset;   
        case TE_HDLC6:   
            return IXF3204_Rx_HDLC7_Offset;   
        case TE_HDLC7:   
            return IXF3204_Rx_HDLC8_Offset;   
        case TE_HDLC8:   
            return IXF3204_Rx_HDLC9_Offset;   
        case TE_HDLC9:   
            return IXF3204_Rx_HDLC10_Offset;  
        case TE_HDLC10:  
            return IXF3204_Rx_HDLC11_Offset;  
        case TE_HDLC11:  
            return IXF3204_Rx_HDLC12_Offset;  
        case TE_HDLC12:  
            return IXF3204_Rx_HDLC13_Offset;  
        case TE_HDLC13:  
            return IXF3204_Rx_HDLC14_Offset;  
        case TE_HDLC14:  
            return IXF3204_Rx_HDLC15_Offset;  
        case TE_HDLC15:  
            return IXF3204_Rx_HDLC16_Offset;  
        default:     
            return  TE_STATUS_BAD_PARAMS;  
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTxHDLC
 * Returns: TEword
 * Description: Returns the base address of the transmit HDLC's
 ******************************************************************************/
TEword IXF3204FindPortTxHDLC(TEword HDLC)
{
    switch(HDLC)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_HDLC0:  
            return IXF3204_Tx_HDLC1_Offset; 
        case TE_HDLC1:  
            return IXF3204_Tx_HDLC2_Offset; 
        case TE_HDLC2:   
            return IXF3204_Tx_HDLC3_Offset; 
        case TE_HDLC3:  
            return IXF3204_Tx_HDLC4_Offset; 
        case TE_HDLC4:  
            return IXF3204_Tx_HDLC5_Offset; 
        case TE_HDLC5:  
            return IXF3204_Tx_HDLC6_Offset; 
        case TE_HDLC6:   
            return IXF3204_Tx_HDLC7_Offset; 
        case TE_HDLC7:   
            return IXF3204_Tx_HDLC8_Offset;   
        case TE_HDLC8:   
            return IXF3204_Tx_HDLC9_Offset; 
        case TE_HDLC9:  
            return IXF3204_Tx_HDLC10_Offset; 
        case TE_HDLC10:  
            return IXF3204_Tx_HDLC11_Offset; 
        case TE_HDLC11:  
            return IXF3204_Tx_HDLC12_Offset; 
        case TE_HDLC12:  
            return IXF3204_Tx_HDLC13_Offset; 
        case TE_HDLC13:  
            return IXF3204_Tx_HDLC14_Offset; 
        case TE_HDLC14:  
            return IXF3204_Tx_HDLC15_Offset; 
        case TE_HDLC15:  
            return IXF3204_Tx_HDLC16_Offset; 
        default:    
            return  TE_STATUS_BAD_PARAMS;  
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTxFDL
 * Returns: TEword
 * Description: Returns the base address of transmit FDL
 ******************************************************************************/
TEword IXF3204FindPortTxFDL(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_Tx_FDL1_Offset;  
        case TE_PORT1:  
            return IXF3204_Tx_FDL2_Offset;  
        case TE_PORT2:  
            return IXF3204_Tx_FDL3_Offset;  
        case TE_PORT3:  
            return IXF3204_Tx_FDL4_Offset;  
        default:  
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxFDL
 * Returns: TEword
 * Description: Returns the base address of the receive FDL
 ******************************************************************************/
TEword IXF3204FindPortRxFDL(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_Rx_FDL1_Offset;  
        case TE_PORT1:  
            return IXF3204_Rx_FDL2_Offset;  
        case TE_PORT2:  
            return IXF3204_Rx_FDL3_Offset;  
        case TE_PORT3:  
            return IXF3204_Rx_FDL4_Offset;  
        default:  
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortPE
 * Returns: TEword
 * Description: Returns the base address of the performance elements database
 ******************************************************************************/
TEword IXF3204FindPortPE(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_PE_Port1_Offset; 
        case TE_PORT1:  
            return IXF3204_PE_Port2_Offset; 
        case TE_PORT2:  
            return IXF3204_PE_Port3_Offset; 
        case TE_PORT3:  
            return IXF3204_PE_Port4_Offset; 
        default:  
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortRxAlarms
 * Returns: TEstatus
 * Description: Returns the base address of rx alarms
 ******************************************************************************/
TEword IXF3204FindPortRxAlarms(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_Rx_Alarms1_Offset;  
        case TE_PORT1:  
            return IXF3204_Rx_Alarms2_Offset;   
        case TE_PORT2:  
            return IXF3204_Rx_Alarms3_Offset;  
        case TE_PORT3:  
            return IXF3204_Rx_Alarms4_Offset;  
        default:  
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTxAlarms
 * Returns: TEword
 * Description: Returns the base address of tx alarms
 ******************************************************************************/
TEword IXF3204FindPortTxAlarms(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
        case TE_BP0:
            return IXF3204_Tx_Alarms1_Offset;  
        case TE_PORT1:  
        case TE_BP1:
            return IXF3204_Tx_Alarms2_Offset;  
        case TE_PORT2:  
        case TE_BP2:
            return IXF3204_Tx_Alarms3_Offset;  
        case TE_PORT3:  
        case TE_BP3:
            return IXF3204_Tx_Alarms4_Offset;  
        default:  
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortNational
 * Returns: TEword
 * Description: Returns the base address of the national bits module
 ******************************************************************************/
TEword IXF3204FindPortNational(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_National_Offset0;  
        case TE_PORT1:   
            return IXF3204_National_Offset1;  
        case TE_PORT2:  
            return IXF3204_National_Offset2;  
        case TE_PORT3:  
            return IXF3204_National_Offset3;  
        default: 
            return  TE_STATUS_BAD_PARAMS;     
    }
}

/***************************************************************************
 * Function Name: IXF3204FindPortMISRC
 * Returns:  TEword  
 * Description: Returns the base address of the interrupts mask
 ***************************************************************************/
TEword IXF3204FindPortMISRC(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_MISRC0;  
        case TE_PORT1:  
            return IXF3204_MISRC1;  
        case TE_PORT2:  
            return IXF3204_MISRC2;    
        case TE_PORT3:   
            return IXF3204_MISRC3;  
        default:   
            return  TE_STATUS_BAD_PARAMS;  
    }
}


/*******************************************************************************
 * Function Name: IXF3204FindPortISRC
 * Returns: TEword
 * Description: Returns the base address of the interrupts indicators
 ******************************************************************************/
TEword IXF3204FindPortISRC(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_ISRC0; 
        case TE_PORT1:  
            return IXF3204_ISRC1; 
        case TE_PORT2:  
            return IXF3204_ISRC2; 
        case TE_PORT3:  
            return IXF3204_ISRC3; 
        default: 
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortMR1ALRMI
 * Returns: TEword
 * Description: Returns the address of the alarms int mask
 ******************************************************************************/
TEword IXF3204FindPortMR1ALRMI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_MR1ALRMI0;  
        case TE_PORT1:  
            return IXF3204_MR1ALRMI1;  
        case TE_PORT2:  
            return IXF3204_MR1ALRMI2;  
        case TE_PORT3:   
            return IXF3204_MR1ALRMI3;  
        default:  
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortMR2ALRMI
 * Returns: TEword
 * Description: Returns the address of the alarms (2) int mask 
 ******************************************************************************/
TEword IXF3204FindPortMR2ALRMI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_MR2ALRMI0;  
        case TE_PORT1:  
            return IXF3204_MR2ALRMI1;  
        case TE_PORT2:  
            return IXF3204_MR2ALRMI2;  
        case TE_PORT3:  
            return IXF3204_MR2ALRMI3;  
        default: 
            return  TE_STATUS_BAD_PARAMS;      
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortMTXALRM
 * Returns: TEword
 * Description: Returns the address of the tx alarms int mask
 ******************************************************************************/
TEword IXF3204FindPortMTXALRM(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:   
            return IXF3204_MTXALRM0;  
        case TE_PORT1:   
            return IXF3204_MTXALRM1;  
        case TE_PORT2:   
            return IXF3204_MTXALRM2;  
        case TE_PORT3:   
            return IXF3204_MTXALRM3;  
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortR1ALRMI
 * Returns: TEword
 * Description: Returns the address of int alarms (1) indicators
 ******************************************************************************/
TEword IXF3204FindPortR1ALRMI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_R1ALRMI0; 
        case TE_PORT1:  
            return IXF3204_R1ALRMI1; 
        case TE_PORT2:  
            return IXF3204_R1ALRMI2; 
        case TE_PORT3:  
            return IXF3204_R1ALRMI3; 
        default: 
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortR2ALRMI
 * Returns: TEword
 * Description: Returns the address of int alarms (2) indicators
 ******************************************************************************/
TEword IXF3204FindPortR2ALRMI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_R2ALRMI0; 
        case TE_PORT1:  
            return IXF3204_R2ALRMI1; 
        case TE_PORT2:  
            return IXF3204_R2ALRMI2; 
        case TE_PORT3:  
            return IXF3204_R2ALRMI3; 
        default: 
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortTXALRM
 * Returns: TEword
 * Description: Returns the address of int tx alarms indicators
 ******************************************************************************/
TEword IXF3204FindPortTXALRM(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_TXALRM0;  
        case TE_PORT1:  
            return IXF3204_TXALRM1;  
        case TE_PORT2:  
            return IXF3204_TXALRM2;  
        case TE_PORT3:  
            return IXF3204_TXALRM3;  
        default: 
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortMFRAMEI
 * Returns: TEword
 * Description: returns the address of the int framer mask
 ******************************************************************************/
TEword IXF3204FindPortMFRAMEI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_MFRAMEI0; 
        case TE_PORT1:  
            return IXF3204_MFRAMEI1; 
        case TE_PORT2:  
            return IXF3204_MFRAMEI2; 
        case TE_PORT3:  
            return IXF3204_MFRAMEI3; 
        default:  
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortFRAMEI
 * Returns: TEword
 * Description: returns the address of the framer indicators
 ******************************************************************************/
TEword IXF3204FindPortFRAMEI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0: 
            return IXF3204_FRAMEI0;  
        case TE_PORT1: 
            return IXF3204_FRAMEI1;  
        case TE_PORT2: 
            return IXF3204_FRAMEI2;  
        case TE_PORT3: 
            return IXF3204_FRAMEI3;  
        default: 
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortMFDLI
 * Returns: TEword
 * Description: returns the address of the FDL indicator mask
 ******************************************************************************/
TEword IXF3204FindPortMFDLI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_MFDLI0;  
        case TE_PORT1:  
            return IXF3204_MFDLI1;  
        case TE_PORT2:  
            return IXF3204_MFDLI2;  
        case TE_PORT3:  
            return IXF3204_MFDLI3;  
        default: 
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortFDLI
 * Returns: TEword
 * Description: returns the address of the FDL indicators
 ******************************************************************************/
TEword IXF3204FindPortFDLI(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:   
            return IXF3204_FDLI0;  
        case TE_PORT1:   
            return IXF3204_FDLI1;  
        case TE_PORT2:   
            return IXF3204_FDLI2;  
        case TE_PORT3:   
            return IXF3204_FDLI3;  
        default:  
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortMEVENTS1I
 * Returns: TEword
 * Description: returns the address of the events 1 indicators mask
 ******************************************************************************/
TEword IXF3204FindPortMEVENTS1I(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_MEVENTS1I0;  
        case TE_PORT1:  
            return IXF3204_MEVENTS1I1;  
        case TE_PORT2:  
            return IXF3204_MEVENTS1I2;  
        case TE_PORT3:  
            return IXF3204_MEVENTS1I3;  
        default:  
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortEVENTS1I
 * Returns: TEword
 * Description: returns the address of the events 1 indicators
 ******************************************************************************/
TEword IXF3204FindPortEVENTS1I(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_EVENTS1I0;  
        case TE_PORT1:  
            return IXF3204_EVENTS1I1;  
        case TE_PORT2:   
            return IXF3204_EVENTS1I2;  
        case TE_PORT3:  
            return IXF3204_EVENTS1I3;  
        default:  
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortMEVENTS2I
 * Returns: TEword
 * Description: returns the address of the events2 indicator mask
 ******************************************************************************/
TEword IXF3204FindPortMEVENTS2I(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:  
            return IXF3204_MEVENTS2I0; 
        case TE_PORT1:  
            return IXF3204_MEVENTS2I1; 
        case TE_PORT2:  
            return IXF3204_MEVENTS2I2; 
        case TE_PORT3:  
            return IXF3204_MEVENTS2I3; 
        default:  
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortEVENTS2I
 * Returns: TEword
 * Description: returns the address of the events 2 indicators
 ******************************************************************************/
TEword IXF3204FindPortEVENTS2I(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:   
            return IXF3204_EVENTS2I0;  
        case TE_PORT1:   
            return IXF3204_EVENTS2I1;  
        case TE_PORT2:   
            return IXF3204_EVENTS2I2;  
        case TE_PORT3:   
            return IXF3204_EVENTS2I3;  
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204FindPortStatus
 * Returns: TEword
 * Description: returns the address of the status indicators
 ******************************************************************************/
TEword IXF3204FindPortStatus(TEword Port)
{
    switch(Port)
    {
        /* if TE_ALL return the offset of port 0 */
        case TE_ALL:   
        case TE_PORT0:   
            return IXF3204_R1ALARMS0;  
        case TE_PORT1:   
            return IXF3204_R1ALARMS1;
        case TE_PORT2:   
            return IXF3204_R1ALARMS2;
        case TE_PORT3:   
            return IXF3204_R1ALARMS3;
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}


/*******************************************************************************
 * Function Name: IXF3204FindCVLBin
 * Returns: TEword
 * Description: returns the address of the near end code violation data 
 ******************************************************************************/
TEword IXF3204FindCVLBin(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_NE_CVLCURRDAYL; 
        case TE_PREVIOUSDAY:  
            return IXF3204_NE_CVLPREVDAYL; 
        case TE_CURRENT15MIN:  
            return IXF3204_NE_CVL15C_L;  
        case TE_PREVIOUS15MIN: 
            return IXF3204_NE_CVL15P_L;  
        case TE_RECENT15MIN0:  
            return IXF3204_NE_CVL15R_0L; 
        case TE_RECENT15MIN1:  
            return IXF3204_NE_CVL15R_1L; 
        case TE_RECENT15MIN2:  
            return IXF3204_NE_CVL15R_2L; 
        case TE_RECENT15MIN3:  
            return IXF3204_NE_CVL15R_3L; 
        case TE_RECENT15MIN4:  
            return IXF3204_NE_CVL15R_4L; 
        case TE_RECENT15MIN5:  
            return IXF3204_NE_CVL15R_5L; 
        case TE_RECENT15MIN6:  
            return IXF3204_NE_CVL15R_6L; 
        case TE_RECENT15MIN7:  
            return IXF3204_NE_CVL15R_7L; 
        case TE_RECENT15MIN8:  
            return IXF3204_NE_CVL15R_8L; 
        case TE_RECENT15MIN9:  
            return IXF3204_NE_CVL15R_9L; 
        case TE_RECENT15MIN10: 
            return IXF3204_NE_CVL15R_10L; 
        case TE_RECENT15MIN11: 
            return IXF3204_NE_CVL15R_11L; 
        case TE_RECENT15MIN12: 
            return IXF3204_NE_CVL15R_12L; 
        case TE_RECENT15MIN13: 
            return IXF3204_NE_CVL15R_13L; 
        case TE_RECENT15MIN14: 
            return IXF3204_NE_CVL15R_14L; 
        case TE_RECENT15MIN15: 
            return IXF3204_NE_CVL15R_15L; 
        case TE_RECENT15MIN16: 
            return IXF3204_NE_CVL15R_16L; 
        case TE_RECENT15MIN17: 
            return IXF3204_NE_CVL15R_17L; 
        case TE_RECENT15MIN18: 
            return IXF3204_NE_CVL15R_18L; 
        case TE_RECENT15MIN19: 
            return IXF3204_NE_CVL15R_19L; 
        case TE_RECENT15MIN20: 
            return IXF3204_NE_CVL15R_20L; 
        case TE_RECENT15MIN21: 
            return IXF3204_NE_CVL15R_21L; 
        case TE_RECENT15MIN22: 
            return IXF3204_NE_CVL15R_22L; 
        case TE_RECENT15MIN23: 
            return IXF3204_NE_CVL15R_23L; 
        case TE_RECENT15MIN24: 
            return IXF3204_NE_CVL15R_24L; 
        case TE_RECENT15MIN25: 
            return IXF3204_NE_CVL15R_25L; 
        case TE_RECENT15MIN26: 
            return IXF3204_NE_CVL15R_26L; 
        case TE_RECENT15MIN27: 
            return IXF3204_NE_CVL15R_27L; 
        case TE_RECENT15MIN28: 
            return IXF3204_NE_CVL15R_28L; 
        case TE_RECENT15MIN29: 
            return IXF3204_NE_CVL15R_29L;  
        case TE_RECENT15MIN30: 
            return IXF3204_NE_CVL15R_30L; 
        default: 
            return  TE_STATUS_BAD_PARAMS;           
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_NE_ESL15
 * Returns: TEword
 * Description: returns the address of the near end ES_L data
 ******************************************************************************/
TEword IXF3204Find_NE_ESL15(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_NE_ESLCURRDAYL; 
        case TE_PREVIOUSDAY:  
            return IXF3204_NE_ESLPREVDAYL; 
        case TE_CURRENT15MIN:  
            return IXF3204_NE_ESL15C_L;  
        case TE_PREVIOUS15MIN: 
            return IXF3204_NE_ESL15P_L;  
        case TE_RECENT15MIN0:  
            return IXF3204_NE_ESL15R_0L; 
        case TE_RECENT15MIN1:  
            return IXF3204_NE_ESL15R_1L; 
        case TE_RECENT15MIN2:  
            return IXF3204_NE_ESL15R_2L; 
        case TE_RECENT15MIN3:  
            return IXF3204_NE_ESL15R_3L; 
        case TE_RECENT15MIN4:  
            return IXF3204_NE_ESL15R_4L; 
        case TE_RECENT15MIN5:  
            return IXF3204_NE_ESL15R_5L;   
        case TE_RECENT15MIN6:  
            return IXF3204_NE_ESL15R_6L; 
        case TE_RECENT15MIN7:  
            return IXF3204_NE_ESL15R_7L; 
        case TE_RECENT15MIN8:  
            return IXF3204_NE_ESL15R_8L; 
        case TE_RECENT15MIN9:  
            return IXF3204_NE_ESL15R_9L; 
        case TE_RECENT15MIN10: 
            return IXF3204_NE_ESL15R_10L; 
        case TE_RECENT15MIN11: 
            return IXF3204_NE_ESL15R_11L; 
        case TE_RECENT15MIN12: 
            return IXF3204_NE_ESL15R_12L; 
        case TE_RECENT15MIN13: 
            return IXF3204_NE_ESL15R_13L; 
        case TE_RECENT15MIN14: 
            return IXF3204_NE_ESL15R_14L; 
        case TE_RECENT15MIN15: 
            return IXF3204_NE_ESL15R_15L; 
        case TE_RECENT15MIN16: 
            return IXF3204_NE_ESL15R_16L; 
        case TE_RECENT15MIN17: 
            return IXF3204_NE_ESL15R_17L; 
        case TE_RECENT15MIN18: 
            return IXF3204_NE_ESL15R_18L; 
        case TE_RECENT15MIN19: 
            return IXF3204_NE_ESL15R_19L; 
        case TE_RECENT15MIN20: 
            return IXF3204_NE_ESL15R_20L; 
        case TE_RECENT15MIN21: 
            return IXF3204_NE_ESL15R_21L; 
        case TE_RECENT15MIN22: 
            return IXF3204_NE_ESL15R_22L; 
        case TE_RECENT15MIN23: 
            return IXF3204_NE_ESL15R_23L; 
        case TE_RECENT15MIN24: 
            return IXF3204_NE_ESL15R_24L; 
        case TE_RECENT15MIN25: 
            return IXF3204_NE_ESL15R_25L; 
        case TE_RECENT15MIN26: 
            return IXF3204_NE_ESL15R_26L; 
        case TE_RECENT15MIN27: 
            return IXF3204_NE_ESL15R_27L; 
        case TE_RECENT15MIN28: 
            return IXF3204_NE_ESL15R_28L; 
        case TE_RECENT15MIN29: 
            return IXF3204_NE_ESL15R_29L; 
        case TE_RECENT15MIN30: 
            return IXF3204_NE_ESL15R_30L; 
        default:   
            return  TE_STATUS_BAD_PARAMS;   
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_FE_ESL15
 * Returns: TEword
 * Description: returns the address of the far end ES_L data
 ******************************************************************************/
TEword IXF3204Find_FE_ESL15(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_FE_ESLCURRDAYL;  
        case TE_PREVIOUSDAY:  
            return IXF3204_FE_ESLPREVDAYL;  
        case TE_CURRENT15MIN:  
            return IXF3204_FE_ESL15C_L;   
        case TE_PREVIOUS15MIN: 
            return IXF3204_FE_ESL15P_L;   
        case TE_RECENT15MIN0:  
            return IXF3204_FE_ESL15R_0L;  
        case TE_RECENT15MIN1:  
            return IXF3204_FE_ESL15R_1L;  
        case TE_RECENT15MIN2:  
            return IXF3204_FE_ESL15R_2L;  
        case TE_RECENT15MIN3:  
            return IXF3204_FE_ESL15R_3L;  
        case TE_RECENT15MIN4:  
            return IXF3204_FE_ESL15R_4L;  
        case TE_RECENT15MIN5:  
            return IXF3204_FE_ESL15R_5L;  
        case TE_RECENT15MIN6:  
            return IXF3204_FE_ESL15R_6L;    
        case TE_RECENT15MIN7:  
            return IXF3204_FE_ESL15R_7L;  
        case TE_RECENT15MIN8:  
            return IXF3204_FE_ESL15R_8L;  
        case TE_RECENT15MIN9:  
            return IXF3204_FE_ESL15R_9L;  
        case TE_RECENT15MIN10: 
            return IXF3204_FE_ESL15R_10L;  
        case TE_RECENT15MIN11: 
            return IXF3204_FE_ESL15R_11L;  
        case TE_RECENT15MIN12: 
            return IXF3204_FE_ESL15R_12L;  
        case TE_RECENT15MIN13: 
            return IXF3204_FE_ESL15R_13L;  
        case TE_RECENT15MIN14: 
            return IXF3204_FE_ESL15R_14L;  
        case TE_RECENT15MIN15: 
            return IXF3204_FE_ESL15R_15L;  
        case TE_RECENT15MIN16: 
            return IXF3204_FE_ESL15R_16L;  
        case TE_RECENT15MIN17: 
            return IXF3204_FE_ESL15R_17L;  
        case TE_RECENT15MIN18: 
            return IXF3204_FE_ESL15R_18L;  
        case TE_RECENT15MIN19: 
            return IXF3204_FE_ESL15R_19L;  
        case TE_RECENT15MIN20: 
            return IXF3204_FE_ESL15R_20L;  
        case TE_RECENT15MIN21: 
            return IXF3204_FE_ESL15R_21L;  
        case TE_RECENT15MIN22: 
            return IXF3204_FE_ESL15R_22L;  
        case TE_RECENT15MIN23: 
            return IXF3204_FE_ESL15R_23L;  
        case TE_RECENT15MIN24: 
            return IXF3204_FE_ESL15R_24L;  
        case TE_RECENT15MIN25: 
            return IXF3204_FE_ESL15R_25L;  
        case TE_RECENT15MIN26: 
            return IXF3204_FE_ESL15R_26L;  
        case TE_RECENT15MIN27: 
            return IXF3204_FE_ESL15R_27L;  
        case TE_RECENT15MIN28: 
            return IXF3204_FE_ESL15R_28L;  
        case TE_RECENT15MIN29: 
            return IXF3204_FE_ESL15R_29L;  
        case TE_RECENT15MIN30: 
            return IXF3204_FE_ESL15R_30L;  
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_NE_SESL15
 * Returns: TEword
 * Description: returns the address of the near end SES_L data
 ******************************************************************************/
TEword IXF3204Find_NE_SESL15(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:    
            return IXF3204_NE_SESLCURRDAYL; 
        case TE_PREVIOUSDAY:  
            return IXF3204_NE_SESLPREVDAYL; 
        case TE_CURRENT15MIN:  
            return IXF3204_NE_SESL15C_L;  
        case TE_PREVIOUS15MIN: 
            return IXF3204_NE_SESL15P_L;  
        case TE_RECENT15MIN0:  
            return IXF3204_NE_SESL15R_0L;    
        case TE_RECENT15MIN1:  
            return IXF3204_NE_SESL15R_1L;  
        case TE_RECENT15MIN2:  
            return IXF3204_NE_SESL15R_2L;  
        case TE_RECENT15MIN3:  
            return IXF3204_NE_SESL15R_3L;  
        case TE_RECENT15MIN4:  
            return IXF3204_NE_SESL15R_4L;  
        case TE_RECENT15MIN5:  
            return IXF3204_NE_SESL15R_5L;  
        case TE_RECENT15MIN6:  
            return IXF3204_NE_SESL15R_6L;  
        case TE_RECENT15MIN7:  
            return IXF3204_NE_SESL15R_7L;   
        case TE_RECENT15MIN8:  
            return IXF3204_NE_SESL15R_8L;  
        case TE_RECENT15MIN9:  
            return IXF3204_NE_SESL15R_9L;  
        case TE_RECENT15MIN10: 
            return IXF3204_NE_SESL15R_10L;  
        case TE_RECENT15MIN11: 
            return IXF3204_NE_SESL15R_11L;  
        case TE_RECENT15MIN12: 
            return IXF3204_NE_SESL15R_12L;  
        case TE_RECENT15MIN13: 
            return IXF3204_NE_SESL15R_13L;  
        case TE_RECENT15MIN14: 
            return IXF3204_NE_SESL15R_14L;  
        case TE_RECENT15MIN15: 
            return IXF3204_NE_SESL15R_15L;  
        case TE_RECENT15MIN16: 
            return IXF3204_NE_SESL15R_16L;  
        case TE_RECENT15MIN17: 
            return IXF3204_NE_SESL15R_17L;  
        case TE_RECENT15MIN18: 
            return IXF3204_NE_SESL15R_18L;  
        case TE_RECENT15MIN19: 
            return IXF3204_NE_SESL15R_19L;  
        case TE_RECENT15MIN20: 
            return IXF3204_NE_SESL15R_20L;  
        case TE_RECENT15MIN21: 
            return IXF3204_NE_SESL15R_21L;  
        case TE_RECENT15MIN22: 
            return IXF3204_NE_SESL15R_22L;  
        case TE_RECENT15MIN23: 
            return IXF3204_NE_SESL15R_23L;  
        case TE_RECENT15MIN24: 
            return IXF3204_NE_SESL15R_24L;  
        case TE_RECENT15MIN25: 
            return IXF3204_NE_SESL15R_25L;  
        case TE_RECENT15MIN26: 
            return IXF3204_NE_SESL15R_26L;  
        case TE_RECENT15MIN27: 
            return IXF3204_NE_SESL15R_27L;  
        case TE_RECENT15MIN28: 
            return IXF3204_NE_SESL15R_28L;  
        case TE_RECENT15MIN29: 
            return IXF3204_NE_SESL15R_29L;  
        case TE_RECENT15MIN30: 
            return IXF3204_NE_SESL15R_30L;  
        default:  
            return  TE_STATUS_BAD_PARAMS;      
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_NE_CVP
 * Returns: TEword
 * Description: returns the address of the near end CV_P data
 ******************************************************************************/
TEword IXF3204Find_NE_CVP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_NE_CVPCURRDAYL; 
        case TE_PREVIOUSDAY:  
            return IXF3204_NE_CVPPREVDAYL; 
        case TE_CURRENT15MIN:  
            return IXF3204_NE_CVP15C_L;  
        case TE_PREVIOUS15MIN: 
            return IXF3204_NE_CVP15P_L;  
        case TE_RECENT15MIN0:  
            return IXF3204_NE_CVP15R_0L; 
        case TE_RECENT15MIN1:  
            return IXF3204_NE_CVP15R_1L; 
        case TE_RECENT15MIN2:  
            return IXF3204_NE_CVP15R_2L; 
        case TE_RECENT15MIN3:  
            return IXF3204_NE_CVP15R_3L; 
        case TE_RECENT15MIN4:  
            return IXF3204_NE_CVP15R_4L;  
        case TE_RECENT15MIN5:  
            return IXF3204_NE_CVP15R_5L;  
        case TE_RECENT15MIN6:  
            return IXF3204_NE_CVP15R_6L; 
        case TE_RECENT15MIN7:  
            return IXF3204_NE_CVP15R_7L; 
        case TE_RECENT15MIN8:  
            return IXF3204_NE_CVP15R_8L; 
        case TE_RECENT15MIN9:  
            return IXF3204_NE_CVP15R_9L; 
        case TE_RECENT15MIN10: 
            return IXF3204_NE_CVP15R_10L; 
        case TE_RECENT15MIN11: 
            return IXF3204_NE_CVP15R_11L; 
        case TE_RECENT15MIN12: 
            return IXF3204_NE_CVP15R_12L; 
        case TE_RECENT15MIN13: 
            return IXF3204_NE_CVP15R_13L; 
        case TE_RECENT15MIN14: 
            return IXF3204_NE_CVP15R_14L; 
        case TE_RECENT15MIN15: 
            return IXF3204_NE_CVP15R_15L; 
        case TE_RECENT15MIN16: 
            return IXF3204_NE_CVP15R_16L; 
        case TE_RECENT15MIN17: 
            return IXF3204_NE_CVP15R_17L; 
        case TE_RECENT15MIN18: 
            return IXF3204_NE_CVP15R_18L; 
        case TE_RECENT15MIN19: 
            return IXF3204_NE_CVP15R_19L; 
        case TE_RECENT15MIN20: 
            return IXF3204_NE_CVP15R_20L; 
        case TE_RECENT15MIN21: 
            return IXF3204_NE_CVP15R_21L; 
        case TE_RECENT15MIN22: 
            return IXF3204_NE_CVP15R_22L; 
        case TE_RECENT15MIN23: 
            return IXF3204_NE_CVP15R_23L; 
        case TE_RECENT15MIN24: 
            return IXF3204_NE_CVP15R_24L; 
        case TE_RECENT15MIN25: 
            return IXF3204_NE_CVP15R_25L; 
        case TE_RECENT15MIN26: 
            return IXF3204_NE_CVP15R_26L; 
        case TE_RECENT15MIN27: 
            return IXF3204_NE_CVP15R_27L; 
        case TE_RECENT15MIN28: 
            return IXF3204_NE_CVP15R_28L; 
        case TE_RECENT15MIN29: 
            return IXF3204_NE_CVP15R_29L; 
        case TE_RECENT15MIN30: 
            return IXF3204_NE_CVP15R_30L; 
        default:  
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_FE_CVP
 * Returns: TEword
 * Description: returns the address of the far end CV_P data
 ******************************************************************************/
TEword IXF3204Find_FE_CVP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_FE_CVPCURRDAYL;  
        case TE_PREVIOUSDAY:  
            return IXF3204_FE_CVPPREVDAYL;  
        case TE_CURRENT15MIN:  
            return IXF3204_FE_CVP15C_L;   
        case TE_PREVIOUS15MIN: 
            return IXF3204_FE_CVP15P_L;   
        case TE_RECENT15MIN0:  
            return IXF3204_FE_CVP15R_0L;  
        case TE_RECENT15MIN1:  
            return IXF3204_FE_CVP15R_1L;  
        case TE_RECENT15MIN2:  
            return IXF3204_FE_CVP15R_2L;  
        case TE_RECENT15MIN3:  
            return IXF3204_FE_CVP15R_3L;  
        case TE_RECENT15MIN4:  
            return IXF3204_FE_CVP15R_4L;  
        case TE_RECENT15MIN5:  
            return IXF3204_FE_CVP15R_5L;  
        case TE_RECENT15MIN6:  
            return IXF3204_FE_CVP15R_6L;  
        case TE_RECENT15MIN7:  
            return IXF3204_FE_CVP15R_7L;  
        case TE_RECENT15MIN8:  
            return IXF3204_FE_CVP15R_8L;  
        case TE_RECENT15MIN9:  
            return IXF3204_FE_CVP15R_9L;  
        case TE_RECENT15MIN10: 
            return IXF3204_FE_CVP15R_10L;  
        case TE_RECENT15MIN11: 
            return IXF3204_FE_CVP15R_11L;  
        case TE_RECENT15MIN12: 
            return IXF3204_FE_CVP15R_12L;  
        case TE_RECENT15MIN13: 
            return IXF3204_FE_CVP15R_13L;  
        case TE_RECENT15MIN14: 
            return IXF3204_FE_CVP15R_14L;  
        case TE_RECENT15MIN15: 
            return IXF3204_FE_CVP15R_15L;  
        case TE_RECENT15MIN16: 
            return IXF3204_FE_CVP15R_16L;  
        case TE_RECENT15MIN17: 
            return IXF3204_FE_CVP15R_17L;  
        case TE_RECENT15MIN18: 
            return IXF3204_FE_CVP15R_18L;  
        case TE_RECENT15MIN19: 
            return IXF3204_FE_CVP15R_19L;  
        case TE_RECENT15MIN20: 
            return IXF3204_FE_CVP15R_20L;  
        case TE_RECENT15MIN21: 
            return IXF3204_FE_CVP15R_21L;   
        case TE_RECENT15MIN22: 
            return IXF3204_FE_CVP15R_22L;  
        case TE_RECENT15MIN23: 
            return IXF3204_FE_CVP15R_23L;  
        case TE_RECENT15MIN24: 
            return IXF3204_FE_CVP15R_24L;  
        case TE_RECENT15MIN25: 
            return IXF3204_FE_CVP15R_25L;    
        case TE_RECENT15MIN26: 
            return IXF3204_FE_CVP15R_26L;   
        case TE_RECENT15MIN27: 
            return IXF3204_FE_CVP15R_27L;  
        case TE_RECENT15MIN28: 
            return IXF3204_FE_CVP15R_28L;  
        case TE_RECENT15MIN29: 
            return IXF3204_FE_CVP15R_29L;  
        case TE_RECENT15MIN30: 
            return IXF3204_FE_CVP15R_30L;   
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_FE_ESP
 * Returns: TEword
 * Description: returns the address of the far end ES_P data
 ******************************************************************************/
TEword IXF3204Find_FE_ESP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_FE_ESPCURRDAYL; 
        case TE_PREVIOUSDAY: 
            return IXF3204_FE_ESPPREVDAYL; 
        case TE_CURRENT15MIN:
            return IXF3204_FE_ESP15C_L;  
        case TE_PREVIOUS15MIN:
            return IXF3204_FE_ESP15P_L;  
        case TE_RECENT15MIN0: 
            return IXF3204_FE_ESP15R_0L;    
        case TE_RECENT15MIN1: 
            return IXF3204_FE_ESP15R_1L;  
        case TE_RECENT15MIN2: 
            return IXF3204_FE_ESP15R_2L;  
        case TE_RECENT15MIN3: 
            return IXF3204_FE_ESP15R_3L;  
        case TE_RECENT15MIN4: 
            return IXF3204_FE_ESP15R_4L;  
        case TE_RECENT15MIN5: 
            return IXF3204_FE_ESP15R_5L;  
        case TE_RECENT15MIN6: 
            return IXF3204_FE_ESP15R_6L;  
        case TE_RECENT15MIN7: 
            return IXF3204_FE_ESP15R_7L;  
        case TE_RECENT15MIN8: 
            return IXF3204_FE_ESP15R_8L;  
        case TE_RECENT15MIN9: 
            return IXF3204_FE_ESP15R_9L;  
        case TE_RECENT15MIN10:
            return IXF3204_FE_ESP15R_10L; 
        case TE_RECENT15MIN11:
            return IXF3204_FE_ESP15R_11L; 
        case TE_RECENT15MIN12:
            return IXF3204_FE_ESP15R_12L; 
        case TE_RECENT15MIN13:
            return IXF3204_FE_ESP15R_13L; 
        case TE_RECENT15MIN14:
            return IXF3204_FE_ESP15R_14L; 
        case TE_RECENT15MIN15:
            return IXF3204_FE_ESP15R_15L; 
        case TE_RECENT15MIN16:
            return IXF3204_FE_ESP15R_16L; 
        case TE_RECENT15MIN17:
            return IXF3204_FE_ESP15R_17L; 
        case TE_RECENT15MIN18:
            return IXF3204_FE_ESP15R_18L; 
        case TE_RECENT15MIN19:
            return IXF3204_FE_ESP15R_19L; 
        case TE_RECENT15MIN20:
            return IXF3204_FE_ESP15R_20L; 
        case TE_RECENT15MIN21:
            return IXF3204_FE_ESP15R_21L; 
        case TE_RECENT15MIN22:
            return IXF3204_FE_ESP15R_22L; 
        case TE_RECENT15MIN23:
            return IXF3204_FE_ESP15R_23L; 
        case TE_RECENT15MIN24:
            return IXF3204_FE_ESP15R_24L; 
        case TE_RECENT15MIN25:
            return IXF3204_FE_ESP15R_25L; 
        case TE_RECENT15MIN26:
            return IXF3204_FE_ESP15R_26L; 
        case TE_RECENT15MIN27:
            return IXF3204_FE_ESP15R_27L; 
        case TE_RECENT15MIN28:
            return IXF3204_FE_ESP15R_28L; 
        case TE_RECENT15MIN29:
            return IXF3204_FE_ESP15R_29L; 
        case TE_RECENT15MIN30:
            return IXF3204_FE_ESP15R_30L; 
        default:  
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_NE_ESP
 * Returns: TEword
 * Description: returns the address of the near end ES_P data
 ******************************************************************************/
TEword IXF3204Find_NE_ESP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:   
            return IXF3204_NE_ESPCURRDAYL; 
        case TE_PREVIOUSDAY:  
            return IXF3204_NE_ESPPREVDAYL;  
        case TE_CURRENT15MIN: 
            return IXF3204_NE_ESP15C_L;  
        case TE_PREVIOUS15MIN:
            return IXF3204_NE_ESP15P_L;  
        case TE_RECENT15MIN0: 
            return IXF3204_NE_ESP15R_0L;  
        case TE_RECENT15MIN1: 
            return IXF3204_NE_ESP15R_1L;  
        case TE_RECENT15MIN2: 
            return IXF3204_NE_ESP15R_2L;  
        case TE_RECENT15MIN3: 
            return IXF3204_NE_ESP15R_3L;  
        case TE_RECENT15MIN4: 
            return IXF3204_NE_ESP15R_4L;  
        case TE_RECENT15MIN5: 
            return IXF3204_NE_ESP15R_5L;  
        case TE_RECENT15MIN6: 
            return IXF3204_NE_ESP15R_6L;  
        case TE_RECENT15MIN7: 
            return IXF3204_NE_ESP15R_7L;  
        case TE_RECENT15MIN8: 
            return IXF3204_NE_ESP15R_8L;  
        case TE_RECENT15MIN9: 
            return IXF3204_NE_ESP15R_9L;  
        case TE_RECENT15MIN10:
            return IXF3204_NE_ESP15R_10L; 
        case TE_RECENT15MIN11:
            return IXF3204_NE_ESP15R_11L; 
        case TE_RECENT15MIN12:
            return IXF3204_NE_ESP15R_12L; 
        case TE_RECENT15MIN13:
            return IXF3204_NE_ESP15R_13L; 
        case TE_RECENT15MIN14:
            return IXF3204_NE_ESP15R_14L; 
        case TE_RECENT15MIN15:
            return IXF3204_NE_ESP15R_15L; 
        case TE_RECENT15MIN16:
            return IXF3204_NE_ESP15R_16L; 
        case TE_RECENT15MIN17:
            return IXF3204_NE_ESP15R_17L; 
        case TE_RECENT15MIN18:
            return IXF3204_NE_ESP15R_18L; 
        case TE_RECENT15MIN19:
            return IXF3204_NE_ESP15R_19L; 
        case TE_RECENT15MIN20:
            return IXF3204_NE_ESP15R_20L; 
        case TE_RECENT15MIN21:
            return IXF3204_NE_ESP15R_21L; 
        case TE_RECENT15MIN22:
            return IXF3204_NE_ESP15R_22L; 
        case TE_RECENT15MIN23:
            return IXF3204_NE_ESP15R_23L; 
        case TE_RECENT15MIN24:
            return IXF3204_NE_ESP15R_24L; 
        case TE_RECENT15MIN25:
            return IXF3204_NE_ESP15R_25L; 
        case TE_RECENT15MIN26:
            return IXF3204_NE_ESP15R_26L; 
        case TE_RECENT15MIN27:
            return IXF3204_NE_ESP15R_27L; 
        case TE_RECENT15MIN28:
            return IXF3204_NE_ESP15R_28L; 
        case TE_RECENT15MIN29:
            return IXF3204_NE_ESP15R_29L; 
        case TE_RECENT15MIN30:
            return IXF3204_NE_ESP15R_30L; 
        default:  
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_FE_SESP
 * Returns: TEword
 * Description: returns the address of the far end SES_P data
 ******************************************************************************/
TEword IXF3204Find_FE_SESP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:   
            return IXF3204_FE_SESPCURRDAYL;  
        case TE_PREVIOUSDAY:  
            return IXF3204_FE_SESPPREVDAYL;  
        case TE_CURRENT15MIN: 
            return IXF3204_FE_SESP15C_L;   
        case TE_PREVIOUS15MIN:
            return IXF3204_FE_SESP15P_L;   
        case TE_RECENT15MIN0: 
            return IXF3204_FE_SESP15R_0L;  
        case TE_RECENT15MIN1: 
            return IXF3204_FE_SESP15R_1L;  
        case TE_RECENT15MIN2: 
            return IXF3204_FE_SESP15R_2L;  
        case TE_RECENT15MIN3: 
            return IXF3204_FE_SESP15R_3L;  
        case TE_RECENT15MIN4: 
            return IXF3204_FE_SESP15R_4L;  
        case TE_RECENT15MIN5: 
            return IXF3204_FE_SESP15R_5L;  
        case TE_RECENT15MIN6: 
            return IXF3204_FE_SESP15R_6L;  
        case TE_RECENT15MIN7: 
            return IXF3204_FE_SESP15R_7L;  
        case TE_RECENT15MIN8: 
            return IXF3204_FE_SESP15R_8L;  
        case TE_RECENT15MIN9: 
            return IXF3204_FE_SESP15R_9L;  
        case TE_RECENT15MIN10:
            return IXF3204_FE_SESP15R_10L;  
        case TE_RECENT15MIN11:
            return IXF3204_FE_SESP15R_11L;  
        case TE_RECENT15MIN12:
            return IXF3204_FE_SESP15R_12L;  
        case TE_RECENT15MIN13:
            return IXF3204_FE_SESP15R_13L;  
        case TE_RECENT15MIN14:
            return IXF3204_FE_SESP15R_14L;  
        case TE_RECENT15MIN15:
            return IXF3204_FE_SESP15R_15L;  
        case TE_RECENT15MIN16:
            return IXF3204_FE_SESP15R_16L;  
        case TE_RECENT15MIN17:
            return IXF3204_FE_SESP15R_17L;  
        case TE_RECENT15MIN18:
            return IXF3204_FE_SESP15R_18L;  
        case TE_RECENT15MIN19:
            return IXF3204_FE_SESP15R_19L;  
        case TE_RECENT15MIN20:
            return IXF3204_FE_SESP15R_20L;  
        case TE_RECENT15MIN21:
            return IXF3204_FE_SESP15R_21L;  
        case TE_RECENT15MIN22:
            return IXF3204_FE_SESP15R_22L;  
        case TE_RECENT15MIN23:
            return IXF3204_FE_SESP15R_23L;  
        case TE_RECENT15MIN24:
            return IXF3204_FE_SESP15R_24L;  
        case TE_RECENT15MIN25:
            return IXF3204_FE_SESP15R_25L;  
        case TE_RECENT15MIN26:
            return IXF3204_FE_SESP15R_26L;  
        case TE_RECENT15MIN27:
            return IXF3204_FE_SESP15R_27L;  
        case TE_RECENT15MIN28:
            return IXF3204_FE_SESP15R_28L;  
        case TE_RECENT15MIN29:
            return IXF3204_FE_SESP15R_29L;  
        case TE_RECENT15MIN30:
            return IXF3204_FE_SESP15R_30L;  
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_NE_SESP
 * Returns: TEword
 * Description: returns the address of the near end SES_P data
 ******************************************************************************/
TEword IXF3204Find_NE_SESP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_NE_SESPCURRDAYL;  
        case TE_PREVIOUSDAY:  
            return IXF3204_NE_SESPPREVDAYL;  
        case TE_CURRENT15MIN:  
            return IXF3204_NE_SESP15C_L;   
        case TE_PREVIOUS15MIN: 
            return IXF3204_NE_SESP15P_L;   
        case TE_RECENT15MIN0:  
            return IXF3204_NE_SESP15R_0L;   
        case TE_RECENT15MIN1:  
            return IXF3204_NE_SESP15R_1L;   
        case TE_RECENT15MIN2:  
            return IXF3204_NE_SESP15R_2L;   
        case TE_RECENT15MIN3:  
            return IXF3204_NE_SESP15R_3L;   
        case TE_RECENT15MIN4:  
            return IXF3204_NE_SESP15R_4L;   
        case TE_RECENT15MIN5:  
            return IXF3204_NE_SESP15R_5L;   
        case TE_RECENT15MIN6:  
            return IXF3204_NE_SESP15R_6L;   
        case TE_RECENT15MIN7:  
            return IXF3204_NE_SESP15R_7L;   
        case TE_RECENT15MIN8:  
            return IXF3204_NE_SESP15R_8L;    
        case TE_RECENT15MIN9:  
            return IXF3204_NE_SESP15R_9L;   
        case TE_RECENT15MIN10: 
            return IXF3204_NE_SESP15R_10L;   
        case TE_RECENT15MIN11: 
            return IXF3204_NE_SESP15R_11L;   
        case TE_RECENT15MIN12: 
            return IXF3204_NE_SESP15R_12L;   
        case TE_RECENT15MIN13: 
            return IXF3204_NE_SESP15R_13L;   
        case TE_RECENT15MIN14: 
            return IXF3204_NE_SESP15R_14L;   
        case TE_RECENT15MIN15: 
            return IXF3204_NE_SESP15R_15L;   
        case TE_RECENT15MIN16: 
            return IXF3204_NE_SESP15R_16L;    
        case TE_RECENT15MIN17: 
            return IXF3204_NE_SESP15R_17L;   
        case TE_RECENT15MIN18: 
            return IXF3204_NE_SESP15R_18L;   
        case TE_RECENT15MIN19: 
            return IXF3204_NE_SESP15R_19L;   
        case TE_RECENT15MIN20: 
            return IXF3204_NE_SESP15R_20L;   
        case TE_RECENT15MIN21: 
            return IXF3204_NE_SESP15R_21L;   
        case TE_RECENT15MIN22: 
            return IXF3204_NE_SESP15R_22L;   
        case TE_RECENT15MIN23: 
            return IXF3204_NE_SESP15R_23L;   
        case TE_RECENT15MIN24: 
            return IXF3204_NE_SESP15R_24L;   
        case TE_RECENT15MIN25: 
            return IXF3204_NE_SESP15R_25L;   
        case TE_RECENT15MIN26: 
            return IXF3204_NE_SESP15R_26L;    
        case TE_RECENT15MIN27: 
            return IXF3204_NE_SESP15R_27L;   
        case TE_RECENT15MIN28: 
            return IXF3204_NE_SESP15R_28L;   
        case TE_RECENT15MIN29: 
            return IXF3204_NE_SESP15R_29L;   
        case TE_RECENT15MIN30: 
            return IXF3204_NE_SESP15R_30L;   
        default:    
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_NE_SASP
 * Returns: TEword
 * Description: returns the address of the near end SAS_P data
 ******************************************************************************/
TEword IXF3204Find_NE_SASP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:   
            return IXF3204_NE_SASPCURRDAYL;  
        case TE_PREVIOUSDAY:  
            return IXF3204_NE_SASPPREVDAYL;  
        case TE_CURRENT15MIN: 
            return IXF3204_NE_SASP15C_L;    
        case TE_PREVIOUS15MIN:
            return IXF3204_NE_SASP15P_L;   
        case TE_RECENT15MIN0: 
            return IXF3204_NE_SASP15R_0L;  
        case TE_RECENT15MIN1: 
            return IXF3204_NE_SASP15R_1L;  
        case TE_RECENT15MIN2: 
            return IXF3204_NE_SASP15R_2L;  
        case TE_RECENT15MIN3: 
            return IXF3204_NE_SASP15R_3L;  
        case TE_RECENT15MIN4: 
            return IXF3204_NE_SASP15R_4L;  
        case TE_RECENT15MIN5: 
            return IXF3204_NE_SASP15R_5L;  
        case TE_RECENT15MIN6: 
            return IXF3204_NE_SASP15R_6L;  
        case TE_RECENT15MIN7: 
            return IXF3204_NE_SASP15R_7L;  
        case TE_RECENT15MIN8: 
            return IXF3204_NE_SASP15R_8L;  
        case TE_RECENT15MIN9: 
            return IXF3204_NE_SASP15R_9L;  
        case TE_RECENT15MIN10:
            return IXF3204_NE_SASP15R_10L;  
        case TE_RECENT15MIN11:
            return IXF3204_NE_SASP15R_11L;  
        case TE_RECENT15MIN12:
            return IXF3204_NE_SASP15R_12L;  
        case TE_RECENT15MIN13:
            return IXF3204_NE_SASP15R_13L;  
        case TE_RECENT15MIN14:
            return IXF3204_NE_SASP15R_14L;  
        case TE_RECENT15MIN15:
            return IXF3204_NE_SASP15R_15L;  
        case TE_RECENT15MIN16:
            return IXF3204_NE_SASP15R_16L;  
        case TE_RECENT15MIN17:
            return IXF3204_NE_SASP15R_17L;  
        case TE_RECENT15MIN18:
            return IXF3204_NE_SASP15R_18L;  
        case TE_RECENT15MIN19:
            return IXF3204_NE_SASP15R_19L;  
        case TE_RECENT15MIN20:
            return IXF3204_NE_SASP15R_20L;  
        case TE_RECENT15MIN21:
            return IXF3204_NE_SASP15R_21L;  
        case TE_RECENT15MIN22:
            return IXF3204_NE_SASP15R_22L;  
        case TE_RECENT15MIN23:
            return IXF3204_NE_SASP15R_23L;  
        case TE_RECENT15MIN24:
            return IXF3204_NE_SASP15R_24L;  
        case TE_RECENT15MIN25:
            return IXF3204_NE_SASP15R_25L;  
        case TE_RECENT15MIN26:
            return IXF3204_NE_SASP15R_26L;  
        case TE_RECENT15MIN27:
            return IXF3204_NE_SASP15R_27L;  
        case TE_RECENT15MIN28:
            return IXF3204_NE_SASP15R_28L;  
        case TE_RECENT15MIN29:
            return IXF3204_NE_SASP15R_29L;  
        case TE_RECENT15MIN30:
            return IXF3204_NE_SASP15R_30L;  
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_NE_UASP
 * Returns: TEword
 * Description: returns the address of the near end UAS_P data
 ******************************************************************************/
TEword IXF3204Find_NE_UASP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_NE_UASPCURRDAYL; 
        case TE_PREVIOUSDAY:   
            return IXF3204_NE_UASPPREVDAYL; 
        case TE_CURRENT15MIN:  
            return IXF3204_NE_UASP15C_L;  
        case TE_PREVIOUS15MIN: 
            return IXF3204_NE_UASP15P_L;  
        case TE_RECENT15MIN0:  
            return IXF3204_NE_UASP15R_0L;  
        case TE_RECENT15MIN1:  
            return IXF3204_NE_UASP15R_1L;  
        case TE_RECENT15MIN2:  
            return IXF3204_NE_UASP15R_2L;  
        case TE_RECENT15MIN3:  
            return IXF3204_NE_UASP15R_3L;  
        case TE_RECENT15MIN4:  
            return IXF3204_NE_UASP15R_4L;  
        case TE_RECENT15MIN5:  
            return IXF3204_NE_UASP15R_5L;  
        case TE_RECENT15MIN6:  
            return IXF3204_NE_UASP15R_6L;  
        case TE_RECENT15MIN7:  
            return IXF3204_NE_UASP15R_7L;  
        case TE_RECENT15MIN8:  
            return IXF3204_NE_UASP15R_8L;  
        case TE_RECENT15MIN9:  
            return IXF3204_NE_UASP15R_9L;  
        case TE_RECENT15MIN10: 
            return IXF3204_NE_UASP15R_10L;  
        case TE_RECENT15MIN11: 
            return IXF3204_NE_UASP15R_11L;  
        case TE_RECENT15MIN12: 
            return IXF3204_NE_UASP15R_12L;  
        case TE_RECENT15MIN13: 
            return IXF3204_NE_UASP15R_13L;  
        case TE_RECENT15MIN14: 
            return IXF3204_NE_UASP15R_14L;  
        case TE_RECENT15MIN15: 
            return IXF3204_NE_UASP15R_15L;  
        case TE_RECENT15MIN16: 
            return IXF3204_NE_UASP15R_16L;  
        case TE_RECENT15MIN17: 
            return IXF3204_NE_UASP15R_17L;  
        case TE_RECENT15MIN18: 
            return IXF3204_NE_UASP15R_18L;  
        case TE_RECENT15MIN19: 
            return IXF3204_NE_UASP15R_19L;  
        case TE_RECENT15MIN20: 
            return IXF3204_NE_UASP15R_20L;  
        case TE_RECENT15MIN21: 
            return IXF3204_NE_UASP15R_21L;  
        case TE_RECENT15MIN22: 
            return IXF3204_NE_UASP15R_22L;  
        case TE_RECENT15MIN23: 
            return IXF3204_NE_UASP15R_23L;  
        case TE_RECENT15MIN24: 
            return IXF3204_NE_UASP15R_24L;  
        case TE_RECENT15MIN25: 
            return IXF3204_NE_UASP15R_25L;  
        case TE_RECENT15MIN26: 
            return IXF3204_NE_UASP15R_26L;  
        case TE_RECENT15MIN27: 
            return IXF3204_NE_UASP15R_27L;  
        case TE_RECENT15MIN28: 
            return IXF3204_NE_UASP15R_28L;  
        case TE_RECENT15MIN29: 
            return IXF3204_NE_UASP15R_29L;  
        case TE_RECENT15MIN30: 
            return IXF3204_NE_UASP15R_30L;  
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name:
 * Returns: TEword
 * Description: returns the address of the far end UAS_P data
 ******************************************************************************/
TEword IXF3204Find_FE_UASP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:   
            return IXF3204_FE_UASPCURRDAYL;  
        case TE_PREVIOUSDAY:   
            return IXF3204_FE_UASPPREVDAYL;  
        case TE_CURRENT15MIN:     
            return IXF3204_FE_UASP15C_L;     
        case TE_PREVIOUS15MIN:  
            return IXF3204_FE_UASP15P_L;   
        case TE_RECENT15MIN0:   
            return IXF3204_FE_UASP15R_0L;   
        case TE_RECENT15MIN1:   
            return IXF3204_FE_UASP15R_1L;   
        case TE_RECENT15MIN2:   
            return IXF3204_FE_UASP15R_2L;    
        case TE_RECENT15MIN3:    
            return IXF3204_FE_UASP15R_3L;    
        case TE_RECENT15MIN4:   
            return IXF3204_FE_UASP15R_4L;   
        case TE_RECENT15MIN5:   
            return IXF3204_FE_UASP15R_5L;   
        case TE_RECENT15MIN6:   
            return IXF3204_FE_UASP15R_6L;   
        case TE_RECENT15MIN7:   
            return IXF3204_FE_UASP15R_7L;   
        case TE_RECENT15MIN8:   
            return IXF3204_FE_UASP15R_8L;   
        case TE_RECENT15MIN9:   
            return IXF3204_FE_UASP15R_9L;   
        case TE_RECENT15MIN10:  
            return IXF3204_FE_UASP15R_10L;   
        case TE_RECENT15MIN11:    
            return IXF3204_FE_UASP15R_11L;   
        case TE_RECENT15MIN12:  
            return IXF3204_FE_UASP15R_12L;    
        case TE_RECENT15MIN13:  
            return IXF3204_FE_UASP15R_13L;   
        case TE_RECENT15MIN14:  
            return IXF3204_FE_UASP15R_14L;   
        case TE_RECENT15MIN15:  
            return IXF3204_FE_UASP15R_15L;   
        case TE_RECENT15MIN16:  
            return IXF3204_FE_UASP15R_16L;   
        case TE_RECENT15MIN17:  
            return IXF3204_FE_UASP15R_17L;   
        case TE_RECENT15MIN18:  
            return IXF3204_FE_UASP15R_18L;   
        case TE_RECENT15MIN19:  
            return IXF3204_FE_UASP15R_19L;   
        case TE_RECENT15MIN20:  
            return IXF3204_FE_UASP15R_20L;   
        case TE_RECENT15MIN21:  
            return IXF3204_FE_UASP15R_21L;   
        case TE_RECENT15MIN22:  
            return IXF3204_FE_UASP15R_22L;   
        case TE_RECENT15MIN23:  
            return IXF3204_FE_UASP15R_23L;   
        case TE_RECENT15MIN24:  
            return IXF3204_FE_UASP15R_24L;   
        case TE_RECENT15MIN25:  
            return IXF3204_FE_UASP15R_25L;   
        case TE_RECENT15MIN26:  
            return IXF3204_FE_UASP15R_26L;   
        case TE_RECENT15MIN27:  
            return IXF3204_FE_UASP15R_27L;   
        case TE_RECENT15MIN28:  
            return IXF3204_FE_UASP15R_28L;   
        case TE_RECENT15MIN29:  
            return IXF3204_FE_UASP15R_29L;   
        case TE_RECENT15MIN30:  
            return IXF3204_FE_UASP15R_30L;   
        default:  
            return TE_STATUS_BAD_PARAMS;           
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_FE_CSSP
 * Returns: TEword
 * Description: returns the address of the far end CSS_P data
 ******************************************************************************/
TEword IXF3204Find_FE_CSSP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:   
            return IXF3204_FE_CSSPCURRDAYL; 
        case TE_PREVIOUSDAY:  
            return IXF3204_FE_CSSPPREVDAYL; 
        case TE_CURRENT15MIN: 
            return IXF3204_FE_CSSP15C_L;   
        case TE_PREVIOUS15MIN:
            return IXF3204_FE_CSSP15P_L;  
        case TE_RECENT15MIN0: 
            return IXF3204_FE_CSSP15R_0L; 
        case TE_RECENT15MIN1: 
            return IXF3204_FE_CSSP15R_1L; 
        case TE_RECENT15MIN2: 
            return IXF3204_FE_CSSP15R_2L; 
        case TE_RECENT15MIN3: 
            return IXF3204_FE_CSSP15R_3L; 
        case TE_RECENT15MIN4: 
            return IXF3204_FE_CSSP15R_4L;   
        case TE_RECENT15MIN5: 
            return IXF3204_FE_CSSP15R_5L; 
        case TE_RECENT15MIN6: 
            return IXF3204_FE_CSSP15R_6L; 
        case TE_RECENT15MIN7: 
            return IXF3204_FE_CSSP15R_7L; 
        case TE_RECENT15MIN8: 
            return IXF3204_FE_CSSP15R_8L; 
        case TE_RECENT15MIN9: 
            return IXF3204_FE_CSSP15R_9L; 
        case TE_RECENT15MIN10:
            return IXF3204_FE_CSSP15R_10L; 
        case TE_RECENT15MIN11:
            return IXF3204_FE_CSSP15R_11L; 
        case TE_RECENT15MIN12:
            return IXF3204_FE_CSSP15R_12L; 
        case TE_RECENT15MIN13:
            return IXF3204_FE_CSSP15R_13L; 
        case TE_RECENT15MIN14:
            return IXF3204_FE_CSSP15R_14L;  
        case TE_RECENT15MIN15:
            return IXF3204_FE_CSSP15R_15L; 
        case TE_RECENT15MIN16:
            return IXF3204_FE_CSSP15R_16L; 
        case TE_RECENT15MIN17:
            return IXF3204_FE_CSSP15R_17L; 
        case TE_RECENT15MIN18:
            return IXF3204_FE_CSSP15R_18L; 
        case TE_RECENT15MIN19:
            return IXF3204_FE_CSSP15R_19L; 
        case TE_RECENT15MIN20:
            return IXF3204_FE_CSSP15R_20L; 
        case TE_RECENT15MIN21:
            return IXF3204_FE_CSSP15R_21L; 
        case TE_RECENT15MIN22:
            return IXF3204_FE_CSSP15R_22L; 
        case TE_RECENT15MIN23:
            return IXF3204_FE_CSSP15R_23L; 
        case TE_RECENT15MIN24:
            return IXF3204_FE_CSSP15R_24L; 
        case TE_RECENT15MIN25:
            return IXF3204_FE_CSSP15R_25L; 
        case TE_RECENT15MIN26:
            return IXF3204_FE_CSSP15R_26L; 
        case TE_RECENT15MIN27:
            return IXF3204_FE_CSSP15R_27L; 
        case TE_RECENT15MIN28:
            return IXF3204_FE_CSSP15R_28L; 
        case TE_RECENT15MIN29:
            return IXF3204_FE_CSSP15R_29L; 
        case TE_RECENT15MIN30:
            return IXF3204_FE_CSSP15R_30L; 
        default:  
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_NE_CSSP
 * Returns: TEword
 * Description: returns the address of the near end CSS_P data
 ******************************************************************************/
TEword IXF3204Find_NE_CSSP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_NE_CSSPCURRDAYL; 
        case TE_PREVIOUSDAY:  
            return IXF3204_NE_CSSPPREVDAYL; 
        case TE_CURRENT15MIN:  
            return IXF3204_NE_CSSP15C_L;  
        case TE_PREVIOUS15MIN: 
            return IXF3204_NE_CSSP15P_L;  
        case TE_RECENT15MIN0:  
            return IXF3204_NE_CSSP15R_0L;  
        case TE_RECENT15MIN1:  
            return IXF3204_NE_CSSP15R_1L;  
        case TE_RECENT15MIN2:  
            return IXF3204_NE_CSSP15R_2L;  
        case TE_RECENT15MIN3:  
            return IXF3204_NE_CSSP15R_3L;  
        case TE_RECENT15MIN4:  
            return IXF3204_NE_CSSP15R_4L;  
        case TE_RECENT15MIN5:  
            return IXF3204_NE_CSSP15R_5L;  
        case TE_RECENT15MIN6:  
            return IXF3204_NE_CSSP15R_6L;  
        case TE_RECENT15MIN7:  
            return IXF3204_NE_CSSP15R_7L;  
        case TE_RECENT15MIN8:  
            return IXF3204_NE_CSSP15R_8L;  
        case TE_RECENT15MIN9:  
            return IXF3204_NE_CSSP15R_9L;  
        case TE_RECENT15MIN10: 
            return IXF3204_NE_CSSP15R_10L;  
        case TE_RECENT15MIN11: 
            return IXF3204_NE_CSSP15R_11L;   
        case TE_RECENT15MIN12: 
            return IXF3204_NE_CSSP15R_12L;  
        case TE_RECENT15MIN13: 
            return IXF3204_NE_CSSP15R_13L;  
        case TE_RECENT15MIN14: 
            return IXF3204_NE_CSSP15R_14L;  
        case TE_RECENT15MIN15: 
            return IXF3204_NE_CSSP15R_15L;  
        case TE_RECENT15MIN16: 
            return IXF3204_NE_CSSP15R_16L;   
        case TE_RECENT15MIN17: 
            return IXF3204_NE_CSSP15R_17L;  
        case TE_RECENT15MIN18: 
            return IXF3204_NE_CSSP15R_18L;  
        case TE_RECENT15MIN19: 
            return IXF3204_NE_CSSP15R_19L;  
        case TE_RECENT15MIN20: 
            return IXF3204_NE_CSSP15R_20L;  
        case TE_RECENT15MIN21: 
            return IXF3204_NE_CSSP15R_21L;  
        case TE_RECENT15MIN22: 
            return IXF3204_NE_CSSP15R_22L;  
        case TE_RECENT15MIN23: 
            return IXF3204_NE_CSSP15R_23L;  
        case TE_RECENT15MIN24: 
            return IXF3204_NE_CSSP15R_24L;  
        case TE_RECENT15MIN25: 
            return IXF3204_NE_CSSP15R_25L;  
        case TE_RECENT15MIN26: 
            return IXF3204_NE_CSSP15R_26L;  
        case TE_RECENT15MIN27: 
            return IXF3204_NE_CSSP15R_27L;  
        case TE_RECENT15MIN28: 
            return IXF3204_NE_CSSP15R_28L;  
        case TE_RECENT15MIN29: 
            return IXF3204_NE_CSSP15R_29L;  
        case TE_RECENT15MIN30: 
            return IXF3204_NE_CSSP15R_30L;  
        default:   
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_FE_ESAP
 * Returns: TEword
 * Description: returns the address of the far end ESA_P data
 ******************************************************************************/
TEword IXF3204Find_FE_ESAP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:   
            return IXF3204_FE_ESAPCURRDAYL;  
        case TE_PREVIOUSDAY:   
            return IXF3204_FE_ESAPPREVDAYL;    
        case TE_CURRENT15MIN:     
            return IXF3204_FE_ESAP15C_L;     
        case TE_PREVIOUS15MIN:  
            return IXF3204_FE_ESAP15P_L;   
        case TE_RECENT15MIN0:   
            return IXF3204_FE_ESAP15R_0L;   
        case TE_RECENT15MIN1:   
            return IXF3204_FE_ESAP15R_1L;   
        case TE_RECENT15MIN2:   
            return IXF3204_FE_ESAP15R_2L;   
        case TE_RECENT15MIN3:   
            return IXF3204_FE_ESAP15R_3L;   
        case TE_RECENT15MIN4:    
            return IXF3204_FE_ESAP15R_4L;   
        case TE_RECENT15MIN5:     
            return IXF3204_FE_ESAP15R_5L;   
        case TE_RECENT15MIN6:   
            return IXF3204_FE_ESAP15R_6L;   
        case TE_RECENT15MIN7:   
            return IXF3204_FE_ESAP15R_7L;   
        case TE_RECENT15MIN8:   
            return IXF3204_FE_ESAP15R_8L;   
        case TE_RECENT15MIN9:   
            return IXF3204_FE_ESAP15R_9L;   
        case TE_RECENT15MIN10:  
            return IXF3204_FE_ESAP15R_10L;   
        case TE_RECENT15MIN11:  
            return IXF3204_FE_ESAP15R_11L;   
        case TE_RECENT15MIN12:  
            return IXF3204_FE_ESAP15R_12L;   
        case TE_RECENT15MIN13:  
            return IXF3204_FE_ESAP15R_13L;   
        case TE_RECENT15MIN14:  
            return IXF3204_FE_ESAP15R_14L;   
        case TE_RECENT15MIN15:  
            return IXF3204_FE_ESAP15R_15L;   
        case TE_RECENT15MIN16:  
            return IXF3204_FE_ESAP15R_16L;   
        case TE_RECENT15MIN17:  
            return IXF3204_FE_ESAP15R_17L;   
        case TE_RECENT15MIN18:    
            return IXF3204_FE_ESAP15R_18L;    
        case TE_RECENT15MIN19:    
            return IXF3204_FE_ESAP15R_19L;   
        case TE_RECENT15MIN20:  
            return IXF3204_FE_ESAP15R_20L;   
        case TE_RECENT15MIN21:  
            return IXF3204_FE_ESAP15R_21L;   
        case TE_RECENT15MIN22:  
            return IXF3204_FE_ESAP15R_22L;   
        case TE_RECENT15MIN23:  
            return IXF3204_FE_ESAP15R_23L;   
        case TE_RECENT15MIN24:  
            return IXF3204_FE_ESAP15R_24L;   
        case TE_RECENT15MIN25:  
            return IXF3204_FE_ESAP15R_25L;   
        case TE_RECENT15MIN26:  
            return IXF3204_FE_ESAP15R_26L;   
        case TE_RECENT15MIN27:  
            return IXF3204_FE_ESAP15R_27L;   
        case TE_RECENT15MIN28:  
            return IXF3204_FE_ESAP15R_28L;   
        case TE_RECENT15MIN29:  
            return IXF3204_FE_ESAP15R_29L;   
        case TE_RECENT15MIN30:  
            return IXF3204_FE_ESAP15R_30L;   
        default:   
            return  TE_STATUS_BAD_PARAMS;       
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_FE_ESBP
 * Returns: TEword
 * Description: returns the address of the far end ESB_P data
 ******************************************************************************/
TEword IXF3204Find_FE_ESBP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_FE_ESBPCURRDAYL;  
        case TE_PREVIOUSDAY:  
            return IXF3204_FE_ESBPPREVDAYL;  
        case TE_CURRENT15MIN:  
            return IXF3204_FE_ESBP15C_L;   
        case TE_PREVIOUS15MIN: 
            return IXF3204_FE_ESBP15P_L;   
        case TE_RECENT15MIN0:  
            return IXF3204_FE_ESBP15R_0L;   
        case TE_RECENT15MIN1:  
            return IXF3204_FE_ESBP15R_1L;   
        case TE_RECENT15MIN2:  
            return IXF3204_FE_ESBP15R_2L;   
        case TE_RECENT15MIN3:  
            return IXF3204_FE_ESBP15R_3L;   
        case TE_RECENT15MIN4:  
            return IXF3204_FE_ESBP15R_4L;   
        case TE_RECENT15MIN5:  
            return IXF3204_FE_ESBP15R_5L;   
        case TE_RECENT15MIN6:  
            return IXF3204_FE_ESBP15R_6L;   
        case TE_RECENT15MIN7:  
            return IXF3204_FE_ESBP15R_7L;   
        case TE_RECENT15MIN8:  
            return IXF3204_FE_ESBP15R_8L;   
        case TE_RECENT15MIN9:  
            return IXF3204_FE_ESBP15R_9L;   
        case TE_RECENT15MIN10: 
            return IXF3204_FE_ESBP15R_10L;   
        case TE_RECENT15MIN11: 
            return IXF3204_FE_ESBP15R_11L;    
        case TE_RECENT15MIN12: 
            return IXF3204_FE_ESBP15R_12L;   
        case TE_RECENT15MIN13: 
            return IXF3204_FE_ESBP15R_13L;   
        case TE_RECENT15MIN14: 
            return IXF3204_FE_ESBP15R_14L;   
        case TE_RECENT15MIN15: 
            return IXF3204_FE_ESBP15R_15L;   
        case TE_RECENT15MIN16: 
            return IXF3204_FE_ESBP15R_16L;   
        case TE_RECENT15MIN17: 
            return IXF3204_FE_ESBP15R_17L;   
        case TE_RECENT15MIN18: 
            return IXF3204_FE_ESBP15R_18L;   
        case TE_RECENT15MIN19: 
            return IXF3204_FE_ESBP15R_19L;   
        case TE_RECENT15MIN20: 
            return IXF3204_FE_ESBP15R_20L;   
        case TE_RECENT15MIN21: 
            return IXF3204_FE_ESBP15R_21L;   
        case TE_RECENT15MIN22: 
            return IXF3204_FE_ESBP15R_22L;   
        case TE_RECENT15MIN23: 
            return IXF3204_FE_ESBP15R_23L;   
        case TE_RECENT15MIN24: 
            return IXF3204_FE_ESBP15R_24L;   
        case TE_RECENT15MIN25: 
            return IXF3204_FE_ESBP15R_25L;   
        case TE_RECENT15MIN26: 
            return IXF3204_FE_ESBP15R_26L;   
        case TE_RECENT15MIN27: 
            return IXF3204_FE_ESBP15R_27L;   
        case TE_RECENT15MIN28: 
            return IXF3204_FE_ESBP15R_28L;   
        case TE_RECENT15MIN29: 
            return IXF3204_FE_ESBP15R_29L;    
        case TE_RECENT15MIN30: 
            return IXF3204_FE_ESBP15R_30L;   
        default:    
            return  TE_STATUS_BAD_PARAMS;    
    }
}

/*******************************************************************************
 * Function Name: IXF3204Find_FE_SEFSP
 * Returns: TEword
 * Description: returns the address of the far end SES_P data
 ******************************************************************************/
TEword IXF3204Find_FE_SEFSP(TEword Bin)
{
    switch(Bin)
    {
        case TE_CURRENTDAY:  
            return IXF3204_FE_SEFSPCURRDAYL;  
        case TE_PREVIOUSDAY:  
            return IXF3204_FE_SEFSPPREVDAYL;  
        case TE_CURRENT15MIN:  
            return IXF3204_FE_SEFSP15C_L;   
        case TE_PREVIOUS15MIN: 
            return IXF3204_FE_SEFSP15P_L;   
        case TE_RECENT15MIN0:  
            return IXF3204_FE_SEFSP15R_0L;   
        case TE_RECENT15MIN1:  
            return IXF3204_FE_SEFSP15R_1L;   
        case TE_RECENT15MIN2:  
            return IXF3204_FE_SEFSP15R_2L;   
        case TE_RECENT15MIN3:  
            return IXF3204_FE_SEFSP15R_3L;   
        case TE_RECENT15MIN4:  
            return IXF3204_FE_SEFSP15R_4L;   
        case TE_RECENT15MIN5:  
            return IXF3204_FE_SEFSP15R_5L;   
        case TE_RECENT15MIN6:  
            return IXF3204_FE_SEFSP15R_6L;   
        case TE_RECENT15MIN7:  
            return IXF3204_FE_SEFSP15R_7L;   
        case TE_RECENT15MIN8:  
            return IXF3204_FE_SEFSP15R_8L;   
        case TE_RECENT15MIN9:  
            return IXF3204_FE_SEFSP15R_9L;   
        case TE_RECENT15MIN10: 
            return IXF3204_FE_SEFSP15R_10L;  
        case TE_RECENT15MIN11: 
            return IXF3204_FE_SEFSP15R_11L;  
        case TE_RECENT15MIN12: 
            return IXF3204_FE_SEFSP15R_12L;  
        case TE_RECENT15MIN13: 
            return IXF3204_FE_SEFSP15R_13L;  
        case TE_RECENT15MIN14: 
            return IXF3204_FE_SEFSP15R_14L;  
        case TE_RECENT15MIN15: 
            return IXF3204_FE_SEFSP15R_15L;  
        case TE_RECENT15MIN16: 
            return IXF3204_FE_SEFSP15R_16L;  
        case TE_RECENT15MIN17: 
            return IXF3204_FE_SEFSP15R_17L;  
        case TE_RECENT15MIN18: 
            return IXF3204_FE_SEFSP15R_18L;  
        case TE_RECENT15MIN19: 
            return IXF3204_FE_SEFSP15R_19L;  
        case TE_RECENT15MIN20: 
            return IXF3204_FE_SEFSP15R_20L;  
        case TE_RECENT15MIN21: 
            return IXF3204_FE_SEFSP15R_21L;  
        case TE_RECENT15MIN22: 
            return IXF3204_FE_SEFSP15R_22L;  
        case TE_RECENT15MIN23: 
            return IXF3204_FE_SEFSP15R_23L;  
        case TE_RECENT15MIN24: 
            return IXF3204_FE_SEFSP15R_24L;  
        case TE_RECENT15MIN25: 
            return IXF3204_FE_SEFSP15R_25L;  
        case TE_RECENT15MIN26: 
            return IXF3204_FE_SEFSP15R_26L;  
        case TE_RECENT15MIN27: 
            return IXF3204_FE_SEFSP15R_27L;  
        case TE_RECENT15MIN28: 
            return IXF3204_FE_SEFSP15R_28L;  
        case TE_RECENT15MIN29: 
            return IXF3204_FE_SEFSP15R_29L;  
        case TE_RECENT15MIN30: 
            return IXF3204_FE_SEFSP15R_30L;  
        default:  
            return  TE_STATUS_BAD_PARAMS;      
    }
}

/*******************************************************************************
 * Function Name: IXF3204GetTimeSlot
 * Returns: TEbyte
 * Description: returns an offset depending of the selected time slot
 ******************************************************************************/
TEbyte IXF3204GetTimeSlot(TEword TS)
{
    switch(TS)
    {
        case TE_TS0:   
            return 0;  
        case TE_TS1:   
            return 1;  
        case TE_TS2:   
            return 2;  
        case TE_TS3:   
            return 3;  
        case TE_TS4:   
            return 4;  
        case TE_TS5:   
            return 5;  
        case TE_TS6:   
            return 6;  
        case TE_TS7:   
            return 7;  
        case TE_TS8:   
            return 8;  
        case TE_TS9:   
            return 9;  
        case TE_TS10:   
            return 10;  
        case TE_TS11:   
            return 11;  
        case TE_TS12:   
            return 12;  
        case TE_TS13:   
            return 13;  
        case TE_TS14:   
            return 14;  
        case TE_TS15:   
            return 15;  
        case TE_TS16:   
            return 16;  
        case TE_TS17:   
            return 17;  
        case TE_TS18:   
            return 18;  
        case TE_TS19:   
            return 19;  
        case TE_TS20:   
            return 20;  
        case TE_TS21:   
            return 21;  
        case TE_TS22:   
            return 22;  
        case TE_TS23:   
            return 23;  
        case TE_TS24:   
            return 24;  
        case TE_TS25:   
            return 25;  
        case TE_TS26:   
            return 26;  
        case TE_TS27:   
            return 27;  
        case TE_TS28:   
            return 28;  
        case TE_TS29:   
            return 29;  
        case TE_TS30:   
            return 30;  
        case TE_TS31:   
            return 31;  
        default:
            return TE_STATUS_BAD_PARAMS;
    }
}

/*******************************************************************************
 * Function Name: IXF3204WriteBitAllPortsRegs
 * Returns: TEstatus
 * Description: Write to all ports in register area
 ******************************************************************************/
TEstatus IXF3204WriteBitAllPortsRegs(void *handle, TEword Address, TEbyte Data,
                                     TEbyte Mask)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for Port 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WriteBit(handle,WriteAddress,Data,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_REG_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteByteAllPortsRegs
 * Returns: TEstatus
 * Description: Write to all ports in register area
 ******************************************************************************/
TEstatus IXF3204WriteByteAllPortsRegs(void *handle, TEword Address, TEbyte Data)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for Port 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WriteByte(handle,WriteAddress,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_REG_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteWordAllPortsRegs
 * Returns: TEstatus
 * Description: Write to all ports in register area
 ******************************************************************************/
TEstatus IXF3204WriteWordAllPortsRegs(void *handle, TEword Address, TEword Data)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for Port 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WriteWord(handle,WriteAddress,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_REG_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WritedWordAllPortsRegs
 * Returns: TEstatus
 * Description: Write to all ports in register area
 ******************************************************************************/
TEstatus IXF3204WritedWordAllPortsRegs(void *handle, TEword Address, 
                                       TEdword Data)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for Port 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WritedWord(handle,WriteAddress,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_REG_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteBitAllPortsMem
 * Returns: TEstatus
 * Description: Write to all ports in memory area
 ******************************************************************************/
TEstatus IXF3204WriteBitAllPortsMem(void *handle, TEword Address, TEbyte Data,
                                    TEbyte Mask)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for Port 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WriteBit(handle,WriteAddress,Data,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_MEM_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteByteAllPortsMem
 * Returns: TEstatus
 * Description: Write to all ports in memory area
 ******************************************************************************/
TEstatus IXF3204WriteByteAllPortsMem(void *handle, TEword Address, TEbyte Data)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for Port 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WriteByte(handle,WriteAddress,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_MEM_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteWordAllPortsMem
 * Returns: TEstatus
 * Description: Write to all ports in memory area
 ******************************************************************************/
TEstatus IXF3204WriteWordAllPortsMem(void *handle, TEword Address, TEword Data)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for Port 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WriteWord(handle,WriteAddress,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_MEM_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteBitAllHDLCMem
 * Returns: TEstatus
 * Description: Write to all HDLC in memory area
 ******************************************************************************/
TEstatus IXF3204WriteBitAllHDLCMem(void *handle, TEword Address, TEbyte Data,
                                   TEbyte Mask)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for HDLC 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_HDLC; Loop++)
    {
        if((Status = IXF3204WriteBit(handle,WriteAddress,Data,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_MEM_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204WriteByteAllHDLCMem
 * Returns: TEstatus
 * Description: Write to all HDLC in memory area
 ******************************************************************************/
TEstatus IXF3204WriteByteAllHDLCMem(void *handle, TEword Address, TEbyte Data)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for HDLC 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_HDLC; Loop++)
    {
        if((Status = IXF3204WriteByte(handle,WriteAddress,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_MEM_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}


/*******************************************************************************
 * Function Name: IXF3204WriteWordAllHDLCMem
 * Returns: TEstatus
 * Description: Write to all HDLC in memory area
 ******************************************************************************/
TEstatus IXF3204WriteWordAllHDLCMem(void *handle, TEword Address, TEword Data)
{
    TEbyte Loop;
    TEword WriteAddress;
    TEstatus Status;

    /* this must be the address for HDLC 0 */
    WriteAddress = Address;
    for(Loop = 0; Loop < IXF3204_MAX_HDLC; Loop++)
    {
        if((Status = IXF3204WriteWord(handle,WriteAddress,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress += IXF3204_PORT_MEM_OFFSET;
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteByteAllInt
 * Returns: TEstatus
 * Description: Write to all interrupt registers
 ******************************************************************************/
TEstatus IXF3204WriteByteAllInt(void *handle, TEword Address, TEbyte Data,
                                TEbyte Offset)
{
    TEbyte Loop;
    TEword WriteAddress, Increment;
    TEstatus Status;

    WriteAddress = Address;
    Increment = Offset;

    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WriteByte(handle,WriteAddress,Data)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress = (TEword)(WriteAddress + Increment);
    }
    return TE_STATUS_SUCCESS;
}

/*******************************************************************************
 * Function Name: IXF3204WriteBitAllPE
 * Returns: TEstatus
 * Description: Write to all interrupt registers
 ******************************************************************************/
TEstatus IXF3204WriteBitAllPE(void *handle, TEword Address, TEbyte Data, 
                              TEbyte Mask, TEword Offset)
{
    TEbyte Loop;
    TEword WriteAddress, Increment;
    TEstatus Status;

    WriteAddress = Address;
    Increment = Offset;

    for(Loop = 0; Loop < IXF3204_MAX_PORT; Loop++)
    {
        if((Status = IXF3204WriteBit(handle,WriteAddress,Data,Mask)) != 
           TE_STATUS_SUCCESS)
        {
            return Status;
        }
        WriteAddress = (TEword)(WriteAddress + Increment);
    }
    return TE_STATUS_SUCCESS;
}

