/*
 * Copyright  Intel Corporation (2002).
 *
 * INTEL MAKES NO WARRANTY OF ANY KIND REGARDING THE CODE.  THIS CODE
 * IS LICENSED ON AN \"AS IS\" BASIS AND INTEL WILL NOT PROVIDE ANY
 * SUPPORT, ASSISTANCE, INSTALLATION, TRAINING OR OTHER SERVICES.
 * INTEL DOES NOT PROVIDE ANY UPDATES, ENHANCEMENTS OR EXTENSIONS.
 * INTEL SPECIFICALLY DISCLAIMS ANY WARRANTY OF MERCHANTABILITY,
 * NONINFRINGEMENT, FITNESS FOR ANY PARTICULAR PURPOSE, OR ANY OTHER
 * WARRANTY.  Further, Intel disclaims all liability of any kind,
 * including but not limited to liability for infringement of any
 * proprietary rights, relating to the use of the code, even if Intel
 * is notified of the possibility of such liability.  Except as expressly
 * stated in an Intel license agreement provided with this code and
 * agreed upon with Intel, no license, express or implied, by estoppel
 * or otherwise, to any intellectual property rights is granted herein.
 *
 */

/******************************************************************************
 * File Name:   wacc3204.c
 * Description: This file contains functions to use the windows access
 *              mechanism
 *
 ******************************************************************************/

#include "ixf3204.h"

/*******************************************************************************
 * Function Name: IXF3204SetWinAccess
 * Returns: TEstatus
 * Description: Implements the protocol needed to read or write internal 
 *              memory trough the window access mechanism.
 ******************************************************************************/
TEstatus IXF3204SetWinAccess(void *handle, TEword Address, TEbyte Value,
                             TEword Command)
{
    TEbyte Data, WriteFlag;
    TEbyte Mask;
    TEstatus Status;

    Mask = TE_BIT0;

    /* 
     * validate address, if it is out of the range 
     * returns bad parameter error
     */
    if((Address < IXF3204_WIN_LOW_LIMIT) || (Address > IXF3204_WIN_HIGH_LIMIT))
    {
        return TE_STATUS_BAD_PARAMS;
    }

    /* validate Command parameter */
    switch(Command)
    {
        case TE_READ:
            Data = TE_BIT0;
            WriteFlag = TE_FALSE;
            break;
        case TE_WRITE:
            Data = TE_CLEAR_BITS;
            WriteFlag = TE_TRUE;
            break;
        default:
            return TE_STATUS_BAD_PARAMS;
    }

    /* write the desired address */
    if((Status = IXF3204WriteWord(handle,IXF3204_WIN_ADDRESS_LOW,Address))
       != TE_STATUS_SUCCESS)
    {
        return Status;
    }

    /* write the data if TE_WRITE */
    if(WriteFlag)
    {
        if((Status = IXF3204WriteByte(handle,IXF3204_WIN_DATA,Value))
           != TE_STATUS_SUCCESS)
        {
            return Status;
        }
    }

    /* write the command */
    return IXF3204WriteBit(handle,IXF3204_WIN_COMMAND,Mask,Data);
}

/*******************************************************************************
 * Function Name: IXF3204GetWinAccessStatus
 * Returns: TEstatus
 * Description: Reads the status indicating if the operation has been executed
 *              successfully, if the operation in IXF3204SetWinAccess was 
 *              TE_READ, the value found in the corresponding register is 
 *              returned in Results.
 *              Notice that if Status is TE_INVALID the value found in Results
 *              does not correspond to the one in the desired address.
 ******************************************************************************/
TEstatus IXF3204GetWinAccessStatus(void *handle, TEword *Status, void *Results)
{
    TEbyte Data, Mask;
    TEstatus rStatus;
    TEdword *ReturnValue = (TEdword *)Results;

    CHK_NULL(ReturnValue);

    Mask = TE_BIT0;

    /* read the operation status */
    if((rStatus = IXF3204ReadBit(handle,IXF3204_WIN_STATUS,Mask,&Data)) 
       != TE_STATUS_SUCCESS)
    {
        return rStatus;
    }

    /* check the value read */
    if(Data)
    {
        *Status = TE_VALID;
        if((rStatus = IXF3204ReadByte(handle,IXF3204_WIN_DATA,&Data))
           != TE_STATUS_SUCCESS)
        {
            return rStatus;
        }
        
        ReturnValue[0] = Data;
    }
    else
    {
        *Status = TE_INVALID;
        ReturnValue[0] = TE_CLEAR_BITS;
    }

    return TE_STATUS_SUCCESS;
}

