#!/bin/sh
#
# srcdconf.sh
#
# Interactive configuration script for 
# RAID Configuration service 'srcd'.
#
# Copyright (c) 2000-2003 Intel Corporation
# All rights reserved.
#

true=1
false=0

VERSION=1.12.3
SRCD_CONF=/etc/srcd.conf
TMP_PERM_FILE=/tmp/srcperm.tmp
TMP_PERM_FILE2=/tmp/srcperm2.tmp
TMP_IP_FILE=/tmp/srcipconn.tmp
TMP_TRGT_FILE=/tmp/srctrgt.tmp
TMP_EXEC_FILE=/tmp/srcexec.tmp
AWK=/usr/bin/awk
GREP=/usr/bin/grep
SED=/usr/bin/sed
SRCDBIN=/usr/sbin/srcd
SRCDSCRIPT=srcd.sh
RCSCRIPTDIR=/usr/local/etc
RCSCRIPT=$RCSCRIPTDIR/$SRCDSCRIPT
RCSCRIPT5X=/etc/rc.d/srcd
RCSTARTLNK=$RCSCRIPTDIR/rc.d/99$SRCDSCRIPT
#RCSTOPLNK=/etc/rc.d/rc3.d/K01$SRCDSCRIPT
#RCSTARTLNK2=/etc/rc.d/rc5.d/S99$SRCDSCRIPT
#RCSTOPLNK2=/etc/rc.d/rc5.d/K01$SRCDSCRIPT
SRCDINSTALL=$0
FBSD_VERSION=`uname -r|sed 's/\..*//'`

perm_count=0


rc_gen()
{
if [ ! -d $RCSCRIPTDIR ]; then
    mkdir -p $RCSCRIPTDIR
fi
cat << EOF > $RCSCRIPT
#!/bin/sh
#
#
# Start RAID Configuration service srcd.
#

# load config
. /etc/rc.conf

SRCD=$SRCDBIN
RETVAL=0
rc_done=" done"
rc_failed=" failed"

return="\$rc_done"

case "\$1" in
    'start')
	if [ -x \$SRCD ]; then
		# start it.
		echo -n "Starting srcd..."
		\$SRCD || return="\$rc_failed"
		RETVAL=\$?
		echo -e "\$return"
	else
		echo "Could not find srcd."
		RETVAL=1
	fi
	;;

    'stop')
	echo -n "Shutting down srcd..."
	killall srcd || return="$rc_failed"
	RETVAL=\$?
	echo -e "\$return"
	;;

    'restart'|'reload')
	\$0 stop
	\$0 start
	RETVAL=\$?
	;;

    *)
	echo "Usage: \$0 { start | stop | restart | reload }"
	exit 1
	;;
esac
exit \$RETVAL
EOF
chmod 755 $RCSCRIPT

if [ -L $RCSTARTLNK ]; then
	rm -f $RCSTARTLNK
fi
#if [ -L $RCSTOPLNK ]; then
#	rm -f $RCSTOPLNK
#fi
#if [ -L $RCSTARTLNK2 ]; then
#	rm -f $RCSTARTLNK2
#fi
#if [ -L $RCSTOPLNK2 ]; then
#	rm -f $RCSTOPLNK2
#fi

ln -s ../$SRCDSCRIPT $RCSTARTLNK
#ln -s ../$SRCDSCRIPT $RCSTOPLNK
#ln -s ../$SRCDSCRIPT $RCSTARTLNK2
#ln -s ../$SRCDSCRIPT $RCSTOPLNK2
}

#
# Start script for FreeBSD 5.x
#
rc5_gen()
{
cat << EOF > $RCSCRIPT5X
#!/bin/sh
#

# PROVIDE: srcd
# KEYWORD: FreeBSD

. /etc/rc.subr

name="srcd"
command="/usr/sbin/srcd"

load_rc_config \$name
run_rc_command "\$1"

EOF
chmod 555 $RCSCRIPT5X
}

#
# global ask function
#

ask()
{
	if [ -z "$2" ]; then
		echo -n "$1 "
	else
		echo -n "$1 [$2] "
	fi
	read answer
	if [ "$answer" = "Q" ]; then
		echo "Installation aborted."
		exit 1
	fi
	if [ -z "$answer" -a ! -z "$2" ]; then
		answer="$2"
	fi
}

#
# get current permissions from $SRCD_CONF
#
perm_get()
{
	if [ -r $TMP_PERM_FILE ]; then
		rm $TMP_PERM_FILE
	fi
	$AWK 'BEGIN{FS=" "; RS="\n"} $1 ~ /^user/ { print $0} $1 ~ /^group/ { print $0 }' $SRCD_CONF > $TMP_PERM_FILE
}

#
# get current IP connection config from $SRCD_CONF
#
ipconn_get()
{
	if [ -r $TMP_IP_FILE ]; then
		rm $TMP_IP_FILE
	fi
	$AWK 'BEGIN{FS=" "; RS="\n"} $1 ~ /^allow/ { print $0} $1 ~ /^deny/ { print $0 }' $SRCD_CONF > $TMP_IP_FILE
}

#
# get current SNMP targets and community from $SRCD_CONF
#
target_get()
{
	if [ -r $TMP_TRGT_FILE ]; then
		rm $TMP_TRGT_FILE
	fi
	$AWK 'BEGIN{FS=" "; RS="\n"} $1 ~ /^target/ { print $0} $1 ~ /^community/ { print $0 }' $SRCD_CONF > $TMP_TRGT_FILE
}

#
# get current execution parameters from $SRCD_CONF
#
execute_get()
{
	if [ -r $TMP_EXEC_FILE ]; then
		rm $TMP_EXEC_FILE
	fi
	$AWK 'BEGIN{FS=" "; RS="\n"} $1 ~ /^execute/ { print $0 }' $SRCD_CONF > $TMP_EXEC_FILE
}

#
# print temporary permissions
#
print_permissions()
{
	if [ -s $TMP_PERM_FILE ]; then
		$AWK 'BEGIN{i=1;FS="\n"} {print i":", $1; i++}' $TMP_PERM_FILE
	else
		echo "($SRCD_CONF does not exist or has no permissions defined)"
	fi
}

#
# ask for user name
#
perm_user()
{
user_ok=0
while [ $user_ok -eq 0 ]
do
	usergroup="user"
	echo ""
	ask "Enter user name or 'L' for Unix user list: " 
	if [ "$answer" = "" ]; then
		usergroup=""
		user_ok=1
	elif [ "$answer" = "L" ]; then
		echo ""
		$AWK 'BEGIN{FS=":"} {print "   ", $1}' /etc/passwd
	else
		dummy=`$GREP "^$answer:" /etc/passwd`
		if [ -z "$dummy" ]; then
			echo "user $answer does not exist in /etc/passwd! Please enter an existing user."
		elif [ ! -z "`$GREP \"^user $answer\" $TMP_PERM_FILE`" ]; then
			echo "Permission definition for user $answer already exists:"
			$GREP "^user $answer" $TMP_PERM_FILE
			echo "Please enter a user name, for which a permission is not yet defined."
		else
			name=$answer
			user_ok=1
		fi
	fi
done
}


#
# ask for group name
#
perm_group()
{
group_ok=0
while [ $group_ok -eq 0 ]
do
        usergroup="group"
	echo ""
        ask "Enter group name or 'L' for Unix group list: "
        if [ "$answer" = "" ]; then
                usergroup=""
                group_ok=1
        elif [ "$answer" = "L" ]; then
        	echo ""
                $AWK 'BEGIN{FS=":"} {print "    ",$1}' /etc/group
        else
                dummy=`$GREP "^$answer:" /etc/group`
                if [ -z "$dummy" ]; then
                        echo "Group $answer does not exist in /etc/group! Please enter an existing group."
                elif [ ! -z "`$GREP \"^group $answer\" $TMP_PERM_FILE`" ]; then
                        echo "Permission definition for group $answer already exists:"
                        $GREP "^group $answer" $TMP_PERM_FILE
                        echo "Please enter a group name, for which permission is not yet defined."
                else
                        name=$answer
                        group_ok=1
		fi
	fi
done
}


#
# define access type
#
perm_access()
{
acc_ok=0
while [ $acc_ok -eq 0 ]
do
	echo ""
	echo "Do you want to grant full (f) or readonly (r)"
	ask "remote access for $1? (f/r/Q)" r
	if [ "$answer" = "r" ]; then
		access="readonly"
		acc_ok=1
	elif [ "$answer" = "f" ]; then
		access="full"
		acc_ok=1
	fi
done
}


#
# add permission entry
#
perm_add()
{
add_ok=0
while [ $add_ok -eq 0 ]
do
	echo ""
	ask "User(u) or group(g) access permission? (u/g/Q)" u
	case "$answer" in
	    'u')
		perm_user
		add_ok=1
		;;
	    'g')
		perm_group
		add_ok=1
		;;
	    '')
		return 0
		;;
	esac
done

if [ -n "$usergroup" ]; then

	perm_access $name

	echo "$usergroup $name access $access" >> $TMP_PERM_FILE
	perm_count=$perm_count+1
fi
}

#
# delete permission entry
#
perm_delete()
{
        ask "Entry number to delete, or (q):" q
	if [ $answer = "q" ]; then
		echo "abort delete."
	else
		sed "${answer}d" $TMP_PERM_FILE > $TMP_PERM_FILE2 && cp $TMP_PERM_FILE2 $TMP_PERM_FILE
	fi
}

#
# interactive access configuration
#

access_configuration()
{
perm_get
# save IP connection config
ipconn_get
# save SNMP targets/community
target_get
# save execution parameters
execute_get
ok=0
while [ $ok -eq 0 ]
do
	echo ""
        echo "Permissions configured so far:"
        echo ""
        print_permissions
	echo ""
	echo "Add (a), delete (d) a permission entry or"
	ask "continue (c) installation (a/d/c/Q)?" a

	case "$answer" in
	    'a')
		perm_add
		;;
	    'd')
		perm_delete
		;;
	    'c')
		ok=1
		;;
	esac
done

echo ""
ask "Save the configuration in $SRCD_CONF? (y/n/Q)" y
if [ "$answer" = "y" ]; then

echo ""
echo "Writing configuration to $SRCD_CONF..."
cat << EOF > $SRCD_CONF
#################################################################
#
# This is the configuration file for the
# srcd srcd.
#
#################################################################
#
# login configuration
#
# Syntax: user | group <user-or-group> access <access-type>
#
# The user and/or group with name <user-or-group> must exist in /etc/passwd
# or /etc/group, respectively. PAM is used for authentification.
# Possible values for <access-type> are 'full' and 'readonly'.
#
# Example:
# user jack access readonly
# group srcadm access full
EOF
/bin/cat $TMP_PERM_FILE >> $SRCD_CONF
cat << EOF >> $SRCD_CONF
#
#################################################################
#
# IP connection configuration
#
# Syntax: allow | deny <IP-address>
#
# You can allow/deny connects from certain IP addresses.
# IP-address can be a complete address (e.g. 192.168.0.10) or a
# part of an address with a ending '*' (e.g. 192.168.*).
# No allow/deny entry means that connections from all addresses are allowed.
# Only one 'allow' entry entry means that all remaining addresses are denied.
# Only one 'deny' entry means that all remaining addresses are allowed.
# If mixed allow/deny entries are given, the last matching entry wins.
#
# Example:
# deny *
# allow 192.168.*
EOF
if [ -e $TMP_IP_FILE ]; then
	/bin/cat $TMP_IP_FILE >> $SRCD_CONF
	rm $TMP_IP_FILE
fi
cat << EOF >> $SRCD_CONF
#
#################################################################
#
# SNMP trap configuration
#
# Syntax: target <target-address>
#         community <snmp-community>
#
# srcd can send SNMP (v1) traps, if target hosts are defined
# like shown below. Default community is 'public'.
#
# Example:
# target 127.0.0.1
# target snmp.intranet.xyz.com
# community xyz
EOF
if [ -e $TMP_TRGT_FILE ]; then
	/bin/cat $TMP_TRGT_FILE >> $SRCD_CONF
	rm $TMP_TRGT_FILE
fi
cat << EOF >> $SRCD_CONF
#
#################################################################
#
# External program start configuration
#
# Syntax: execute <execution-command-line>
#         executeuser <user-name>
#
# You can invoke an external command by 'execute' every time, when a
# controller event occurs. You can use following variables:
#   %H   Host name.
#   %C   Controller index (counting from 0)
#   %T   Time string.
#   %S   Severity (informational, warning, critical)
#   %E   Event string.
#   %I   Event ID.
# Use 'executeuser' to define the user to run the external command as.
#
# Example:
# executeuser nobody
# execute echo -e "Host      : %H\nController: %C\nTime      : %T\nSeverity  : %S\nEvent ID  : %I\n\nEvent text: %E" | mail -s "RAID: %S event" joerg@kermit
EOF
if [ -e $TMP_EXEC_FILE ]; then
	/bin/cat $TMP_EXEC_FILE >> $SRCD_CONF
	rm $TMP_EXEC_FILE
fi

echo "Configuration successfully saved."

else
	echo "Configuration not saved."
fi
}

#
# Ask for configuration
#

echo ""
echo "----------------------------------------------------------"
echo " srcd configuration"
echo "----------------------------------------------------------"
echo ""
echo "This installation procedure guides you through the configuration"
echo "of the RAID Configuration service (srcd). This"
echo "service is needed if you want to manage Storage RAID Controllers"
echo "installed in this machine from a remote workstation."
echo ""
echo "You must have superuser rights to run this script successfully!"
echo ""
echo "Do you want to manage the Storage RAID Controller installed"
ask  "in this machine from a remote workstation? (y/n)" y

if [ "$answer" = "n" -o "$answer" = "N" ]; then
	echo ""
	echo "Installation terminated by user."
	echo "This installation can be restarted from $SRCDINSTALL."
	echo ""
	exit 0
fi

echo ""
echo "You can abort the installation during input by typing 'Q' or"
echo "at any time by <ctrl> C."
echo ""
echo "Remote managability of Storage RAID Controllers can be granted"
echo "only to already exisiting Unix users or groups. This installation"
echo "procedure does not create Unix users or groups."
echo ""
echo "Do you want to terminate the installation in order to create"
ask  "Unix users or groups now? (y/n)" n

if [ "$answer" = "y" -o "$answer" = "Y" ]; then
	echo ""
	echo "Installation terminated by user."
	echo "This installation can be restarted from $SRCDINSTALL."
	echo ""
	exit 0
fi

echo ""

ask "Do you want 'srcd' to be automatically started at startup? (y/n/Q)" y
automatic_start=$answer

if [ "$automatic_start" != "y" ]; then
    if [ "$FBSD_VERSION" = "4" ]
    then
        echo "No automatic start."
        if [ -L $RCSTARTLNK ]; then
           rm $RCSTARTLNK
        fi
        if [ -L $RCSTOPLNK ]; then
           rm $RCSTOPLNK
        fi
        if [ -L $RCSTARTLNK2 ]; then
           rm $RCSTARTLNK2
        fi
        if [ -L $RCSTOPLNK2 ]; then
           rm $RCSTOPLNK2
        fi
    else
        if [ -f /etc/rc.d/srcd ]; then
            rm /etc/rc.d/srcd
        fi
    fi
else
    echo "Automatic start."
    if [ "$FBSD_VERSION" = "4" ]
    then
        rc_gen
    else
        rc5_gen
    fi
fi

#
# ask for manual or automatic permission definition
#
 
echo ""
echo "The RAID Configuration service 'srcd' uses"
echo "the file $SRCD_CONF for its configuration at startup time."
echo "This file can now be created if it does not exist, or an existing file"
echo "can now be modified. All modifications are only written to $SRCD_CONF" 
echo "at the end of the installation procedure."
echo "$SRCD_CONF defines access permissions of users and groups which have"
echo "remote access to the Storage RAID Controllers installed in this machine."
echo "These access permissions affect only 'srcd', they do not modify user or"
echo "group permissions for the operating system in any way."
echo "The shadow passwd system is used for authentification."
echo ""
echo "Following access permissions can be granted:"
echo ""
echo "readonly      user or group can view controller configuration"
echo "full          user or group can view and change controller configuration"
echo ""

ok=0
interactive=0
while [ $ok -eq 0 ]
do
	ask "Do you want to create/modify $SRCD_CONF? (y/n/Q)" y
	if [ "$answer" = "y" ]; then
		ok=1
		interactive=1
	elif [ "$answer" = "n" -o "$answer" = "N" ]; then
		ok=1
		interactive=0
	fi
done

#
# interactive access configuration
#
if [ $interactive -eq 1 ]; then
	access_configuration
fi

#
# start service?
#
echo ""
ask "Do you want to start 'srcd' now? (y/n/Q)" y
if [ "$answer" = "y" ]; then
    if [ "$FBSD_VERSION" = "4" ]
    then
	$RCSCRIPT start
    else
        $RCSCRIPT5X start
    fi
else
	echo "'srcd' not started."
fi

#
# finishing
#
echo ""
echo "Installation of 'srcd' finished successfully."
echo ""
