/*******************************************************************************
 *     Copyright(c) 2002 - 2003  Intel Corporation. All Rights Reserved.
 *
 *
 *     Watchdog timer driver for Intel(R) ESB.
 *
 *     This program is free software; you can redistribute it and/or modify 
 *     it under the terms of the GNU General Public License as published by 
 *     the Free Software Foundation; either version 2 of the License, or 
 *     any later version.
 *		
 *     Intel does not admit liability nor provide warranty for this software. 
 *     This material is provided "AS-IS" and at no charge.
 *
 *
 *     Module Name:      esbwdt.c
 *
 *     Abstract:         This module support Intel ESB watchdog timer. 
 *                       All of the driver functions is provided in this file.
 * 
 *     Environment:      This file is intended to be specific to Linux 
 *                       operating system.
 *
 *
 *     Version 1.0      1st drop for beta linux driver packages for esb
 *     Jun 23 2003
 *******************************************************************************/


#include <linux/config.h>  
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/reboot.h>
#include <linux/ioport.h>
#include <linux/types.h>
#include <asm/uaccess.h>
#include <linux/fs.h>
#include <linux/pci.h>
#include <linux/mm.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <asm/io.h>
#include <linux/watchdog.h>
#include <linux/miscdevice.h>
#include "esbwdt.h"
	
static unsigned long  wdt_margin1 = TIMER_MARGIN;
static unsigned long  wdt_margin2 = TIMER_MARGIN;  
static unsigned char  wdt_mode = WDT_MODE;
static unsigned char  wdt_scale = HIGH_SCALE;
static unsigned char  wdt_count = 0;
static int nowayout = 0;
static struct pci_dev *wdt_pci;
static int wdt_irq = 34;
// static int wdt_irq = 9;
static spinlock_t wdt_lock;
static DECLARE_WAIT_QUEUE_HEAD(wdt_wait_queue); 
static int wdt_wq_active = 0;
static char wdt_expect_close = 0;

MODULE_PARM(wdt_irq, "i");
MODULE_PARM_DESC(wdt_irq, "ESB WDT Interrupt (default: wdt_irq = 34)");
MODULE_PARM(wdt_mode, "s");
MODULE_PARM_DESC(wdt_mode, "ESB Watchdog timer mode (default WDT mode).");
MODULE_PARM(wdt_scale, "s");
MODULE_PARM_DESC(wdt_scale, "ESB WDT scale ( default in steps of 1 ms).");
MODULE_PARM(wdt_margin1, "l");
MODULE_PARM_DESC(wdt_margin1, "First stage ESB WDT timeout in steps of 1 ms by default.");
MODULE_PARM(wdt_margin2, "l");
MODULE_PARM_DESC(wdt_margin2, "Second stage ESB WDT timeout in steps of 1 ms by default.");
MODULE_PARM(nowayout,"i");
MODULE_PARM_DESC(nowayout, "ESB WDT can't be stopped once started (default=0)");

static int wdt_get_mode(void);
static int wdt_set_mode(unsigned char mode);
static int wdt_get_scale(void);
static int wdt_set_scale(unsigned char scale);
static int wdt_enable(void);
static int wdt_disable(void);
static int wdt_getenable(void);
static int wdt_lk(void);
static int wdt_enable_m(void);
static int wdt_set_timeout1(unsigned long val);
static int wdt_set_timeout2(unsigned long val);
static void wdt_reload(void);
static int wdt_clear_toutsts(void);
static int wdt_sys_status(void);
static void wdt_start(void);
static void wdt_sethwdflt(void);
static int wdt_open (struct inode *inode, struct file *file);
static int wdt_release (struct inode *inode, struct file *file);
static ssize_t wdt_write (struct file *file, const char *data, size_t count, loff_t * pos);
static int wdt_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static void wdt_isr(int irq, void *dev_id, struct pt_regs *regs);
static void __exit wdt_cleanup(void);
static int __init wdt_init(void);
static int __init wdt_init_one (struct pci_dev *dev, const struct pci_device_id *ent);
static void __devexit wdt_remove_one (struct pci_dev *pdev);
static unsigned char __init wdt_device (void);
static int wdt_dis_opin(void);
static int wdt_en_opin(void);
static int wdt_get_irqstat(void);
static int wdt_clr_irqstat(void);
static unsigned long wdt_get_dc(void);

static struct pci_device_id wdt_pci_tbl[] __initdata = {
        { PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_CICH2_WDT, PCI_ANY_ID, PCI_ANY_ID, },
        { 0, },
};

static struct file_operations wdt_fops = {
	owner:		THIS_MODULE,
	write:		wdt_write,
	ioctl:		wdt_ioctl,
	open:	        wdt_open,
	release:	wdt_release,
};
static struct miscdevice wdt_miscdev = {
	minor:		WATCHDOG_MINOR,
	name:		"watchdog",
	fops:		&wdt_fops,
};
static struct pci_driver wdt_driver = {
	name:		"esbwdt",
	id_table:	wdt_pci_tbl,
	probe:		wdt_init_one,
	remove:		__devexit(wdt_remove_one),
};

static unsigned char __init wdt_device (void)
{      
	struct pci_dev *dev;
	pci_for_each_dev(dev) {
		if (pci_match_device(wdt_pci_tbl, dev)) {
			wdt_pci = dev;
			break;
		}
	}
	return 0;
}

/*
 * Function Name:   wdt_get_mode.
 * Parameter:       None.
 * Return:          Mode value - WDT_MODE or FREE_MODE (free running mode)
 * Description:     This function is used to get current mode.
 */
static int wdt_get_mode(void)
{   
        unsigned char val;  
        spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
        spin_unlock(&wdt_lock);
        if ( ( val &= MOSC_TEST ) == 0 )
                return WDT_MODE;
        return FREE_MODE;
}
/*
 * Function Name:   wdt_get_dc.
 * Parameter:       None.
 * Return:          ulong dcount
 * Description:     returns upper 20bits of 35bit downcounter.
 */
static unsigned long wdt_get_dc(void)
{   
        u32 val;  
        spin_lock(&wdt_lock);
        pci_read_config_dword(wdt_pci, WDT_DC_OFFSET, &val);
        spin_unlock(&wdt_lock);
        return val;
}
/*
 * Function Name:   wdt_get_opin.
 * Parameter:       None.
 * Return:          OPIN0_EN - Enabled OPIN1_DIS - Disabled
 * Description:     This function is used to get state if WDT_OUTPUT bit.
 */
static int wdt_get_opin(void)
{   
        unsigned char val;  
        spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_CONFIG_OFFSET, &val);
        spin_unlock(&wdt_lock);
        if ( ( val &= OPIN_MASK ) == 0 )
                return OPIN0_EN; 
        return OPIN1_DIS;
}

// wdt_dis_opin
// Set WDT_OUTPUT bit 5 in WDT_CONFIG register (PCI CFG SPC offset 61:60)
// WDT_OUTPUT =1 DISABLES WDT_OUTPUT Pin... prevents resets from occurring
static int wdt_dis_opin(void)
{
        unsigned char val;
        
	spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_CONFIG_OFFSET, &val);
        val |= 0x20;
        pci_write_config_byte(wdt_pci, WDT_CONFIG_OFFSET, val);
        spin_unlock(&wdt_lock);
	return 0;
}

// wdt_en_opin
// Clears WDT_OUTPUT bit 5 in WDT_CONFIG register (PCI CFG SPC offsett 61:60)
// WDT_OUTPUT = 0 ENABLES WDT_OUTPUT Pin... enables output pin to toggle
//
static int wdt_en_opin(void)
{
        unsigned char val;
        
	spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_CONFIG_OFFSET, &val);
        val &= 0xdf;
        pci_write_config_byte(wdt_pci, WDT_CONFIG_OFFSET, val);
        spin_unlock(&wdt_lock);
	return 0;
}

/*
 * Function Name:   wdt_set_mode.
 * Parameter:       The mode value to be set: 0 - WDT, 1 - free running.
 * Return:          0 - succesful, negative value - failed.
 * Description:     This function is used to set current mode.
 */      
static int wdt_set_mode(unsigned char mode)
{
        unsigned char val;
        
        switch(mode)
        {
                case WDT_MODE: 
                        spin_lock(&wdt_lock);
                        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
                        val &= MODE_MASK;
                        pci_write_config_byte(wdt_pci, WDT_LOCK_OFFSET, val);
                        spin_unlock(&wdt_lock);
	                if ( wdt_get_mode() == WDT_MODE ) { 
	                        wdt_mode = WDT_MODE; 
			        return 0;
                        }         
                        else { 
	                        return -1;
			}
                case FREE_MODE:
                        spin_lock(&wdt_lock);
                        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
                        val |= ~MODE_MASK;
                        pci_write_config_byte(wdt_pci, WDT_LOCK_OFFSET, val);
                        spin_unlock(&wdt_lock);
	                if ( wdt_get_mode() == FREE_MODE ) {
	                        wdt_mode = FREE_MODE; 
                                return 0;
                        }        
                        else {
	                        return -1;
                        }
                default:
                        return -EINVAL;  
        }
}
/*
 * Function Name:   wdt_set_opin.
 * Parameter:       The mode value to be set: 0 - enabled, 1 - disabled (backass backward in hw!)
 * Return:          0 - succesful, negative value - failed.
 * Description:     This function is used to en/dis WDT_OUTPUT pin
 */      
static int wdt_set_opin(unsigned char mode)
{
  //        unsigned char val;
        
        switch(mode)
        {
                case OPIN0_EN:
		  wdt_en_opin();
		  return 0;
                case OPIN1_DIS:
		  wdt_dis_opin();
		  return 0;
                default:
                        return -EINVAL;  
        }
}

/*
 * Function Name:   wdt_get_scale.
 * Parameter:       none.
 * Return:          Current scale.
 * Description:     This function is used to get scale.
 */
static int wdt_get_scale(void)
{   
        unsigned char val;  
        spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_CONFIG_OFFSET, &val);
        spin_unlock(&wdt_lock);
        if ( ( val &= MOSC_TEST ) == 0 )
                return HIGH_SCALE; 
        return LOW_SCALE;
}

/*
 * Function Name:   wdt_set_scale.
 * Parameter:       The scale value to be set.
 * Return:          0 - successful, negative value -  failed.
 * Description:     This function is used to set scale.
 */    
static int wdt_set_scale(unsigned char scale)
{
        unsigned char val;
        
        switch( scale )
        {
                case HIGH_SCALE: 
                        spin_lock(&wdt_lock);
                        pci_read_config_byte(wdt_pci, WDT_CONFIG_OFFSET, &val);
                        val &= MODE_MASK;
                        pci_write_config_byte(wdt_pci, WDT_CONFIG_OFFSET, val);
                        spin_unlock(&wdt_lock);
	                if ( wdt_get_scale() == HIGH_SCALE ) { 
	                        wdt_scale = HIGH_SCALE; 
			        return 0;
                        }         
                        else { 
	                        return -1;
			}
                case  LOW_SCALE:
                        spin_lock(&wdt_lock);
                        pci_read_config_byte(wdt_pci, WDT_CONFIG_OFFSET, &val);
                        val |= ~MODE_MASK;
                        pci_write_config_byte(wdt_pci, WDT_CONFIG_OFFSET, val);
                        spin_unlock(&wdt_lock);
	                if ( wdt_get_scale() ==  LOW_SCALE ) {
	                        wdt_scale =  LOW_SCALE; 
                                return 0;
                        }        
                        else {
	                        return -1;
                        }
                default:
                        return -EINVAL;  
        }
}

/*
 * Function Name:   wdt_enable.
 * Parameter:       none.
 * Return:          0 - enable successful, -1 - failed.
 * Description:     This function is used to enable WDT.
 */
static int wdt_enable(void)
{
        unsigned char val;
        spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
        val |= WDT_ENABLE;
        pci_write_config_byte(wdt_pci, WDT_LOCK_OFFSET, val);
        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
        spin_unlock(&wdt_lock);
        if( ( val & WDT_ENABLE ) == 0 )
                return -1;
        return 0;
}

/*
 * Function Name:   wdt_disable.
 * Parameter:       none.
 * Return:          0 - enable successful, -1 - failed.
 * Description:     This function is used to disable WDT.
 */
static int wdt_disable(void)
{
        unsigned char val;
        spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
        val &= WDT_MASK;
        pci_write_config_byte(wdt_pci, WDT_LOCK_OFFSET, val);
        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
        spin_unlock(&wdt_lock);
        if( ( val & WDT_ENABLE ) == 0 )
                return 0;
        return -1;
}

/*
 * Function Name:   wdt_disable.
 * Parameter:       none.
 * Return:          state of WDT_ENABLE bit1 @ pci config offset 0x68h
 * Description:     This function is used to read state of WDT_ENABLE bit
 */
static int wdt_getenable(void)
{
        unsigned char val;
        spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
        spin_unlock(&wdt_lock);
        if( ( val & WDT_ENABLE ) == 0 )    //WDT_ENABLE =0x02
                return 0;
        return 1;
}
/*
 * Function Name:   wdt_lk.
 * Parameter:       none.
 * Return:          0 - successful, -1 - failed.
 * Description:     This function is used to lock WDT.
 */ 
static int wdt_lk(void)
{
        unsigned char val;
        spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
        val |= WDT_LOCK;
        pci_write_config_byte(wdt_pci, WDT_LOCK_OFFSET, val);
        pci_read_config_byte(wdt_pci, WDT_LOCK_OFFSET, &val);
        spin_unlock(&wdt_lock);
        if( ( val & WDT_LOCK ) == 0 )
                return -1;
        return 0;
}

/*
 * Function Name:   wdt_enablem.
 * Parameter:       none.
 * Return:          0 - enable successful, -1 - failed.
 * Description:     This function is used to enable WDT memory access.
 */
static int wdt_enable_m(void)
{
        unsigned char val;
        spin_lock(&wdt_lock);
        pci_read_config_byte(wdt_pci, WDT_CMD_OFFSET, &val);  
        val |= WDT_ENABLE;                                  
        pci_write_config_byte(wdt_pci, WDT_CMD_OFFSET, val);
        pci_read_config_byte(wdt_pci, WDT_CMD_OFFSET, &val);
        spin_unlock(&wdt_lock);
        if( ( val & WDT_ENABLE ) == 0 )
                return -1;
        return 0;                             
}

/*
 * Function Name:   wdt_set_timeout1.
 * Parameter:       Time out value to be set.
 * Return:          0 - successful, -1 - failed.
 * Description:     This function is used to set WDT first stage timeout.
 */ 
static int wdt_set_timeout1(unsigned long val)
{    
        unsigned long  count, setval;
        if ( val < 1 || val > 1048575 )
                return -EINVAL;
	//        if ( wdt_scale == LOW_SCALE ) 
	//        count = val*33/32 - 1;
        //else 
        //        count = val*33000/32768 - 1;
	count = val;
        spin_lock(&wdt_lock);
        writeb(0x80,(unsigned long*) WDT_RLD);
        writeb(0x86,(unsigned long*)WDT_RLD);
        writel(count,(unsigned long*) WDT_PRELD1);
        setval = readl((unsigned long*)WDT_PRELD1);
        spin_unlock(&wdt_lock);
        if ( ( setval & 0x000fffff ) == count )
                return 0;
        return -1;
}

/*
 * Function Name:   wdt_set_timeout2.
 * Parameter:       Time out value to be set.
 * Return:          0 - successful, -1 - failed.
 * Description:     This function is used to set WDT second stage timeout.
 */ 
static int wdt_set_timeout2(unsigned long val)
{
        unsigned long  count, setval;
        if ( val < 1 || val > 1048575 )
                return -EINVAL;
	//        if ( wdt_scale == LOW_SCALE ) 
	//        count = val*33/32 - 1;
        //else 
        //        count = val*33000/32768 - 1;
	count = val;
        spin_lock(&wdt_lock);
        writeb(0x80, (unsigned long*)WDT_RLD);
        writeb(0x86, (unsigned long*)WDT_RLD);
        writel(count, (unsigned long*)WDT_PRELD2);
        setval = readl((unsigned long*)WDT_PRELD2);
        spin_unlock(&wdt_lock);
        if ( ( setval & 0x000fffff ) == count )
                return 0;
        return -1;
}

/*
 * Function Name:   wdt_reload.
 * Parameter:       None.
 * Return Value:    None.
 * Description:     This function is used to reset WDT.
 */ 
static void wdt_reload(void)
{
        spin_lock(&wdt_lock);
        writeb(0x80, (unsigned long*)WDT_RLD);
        writeb(0x86, (unsigned long*)WDT_RLD);
        writew(0x0100, (unsigned long*)WDT_RLD);
        spin_unlock(&wdt_lock);
        return;
}

// Read IRQ Status bit in Gen_IRQ_Stat Reg (mmio base+08)
// per cspec 1.0 no register unlocking bs needed to r/w this one

static int wdt_get_irqstat(void)
{
        int status;
        spin_lock(&wdt_lock);
        status = readw((unsigned long*)WDT_INT_STS);
        spin_unlock(&wdt_lock);
	//	printk("ESB WDT driver version 0.1: WDT Gen IRQ Stat Reg =  %d\n", status);
        if ( ( status & 0x0001 ) == 0 )
                return 0;
        return -1;
}

// Write 1 to clear IRQ Status bit in Gen_IRQ_Stat Reg (mmio base+08)
// per cspec 1.0 no register unlocking bs needed to r/w this one

static int wdt_clr_irqstat(void)
{      
        int val;
        spin_lock(&wdt_lock);
        val = readb(WDT_INT_STS);
        val |= 0x01; 
        writeb(val, (unsigned long*)WDT_INT_STS);
        spin_unlock(&wdt_lock);
        return wdt_get_irqstat();
}

/*
 * Function Name:   wdt_clear_toutsts.
 * Parameter:       None.
 * Return Value:    state of WDT_TIMEOUT bit ( 0 noTO 1 TO )
 * Description:     This function is used to clear timeout bit.
 */ 
static int wdt_clear_toutsts(void)
{      
        int val;
        spin_lock(&wdt_lock);
        //val = readw(WDT_RLD);
        //val |= WDT_CLEAR_STS; // WDT_CLEAR_STS = 0x200h (set bit 9 =1)
	val = WDT_CLEAR_STS;    // set ONLY bit 9 =1 
        writeb(0x80, (unsigned long*)WDT_RLD);
        writeb(0x86, (unsigned long*)WDT_RLD);
	//        writeb(val, (unsigned long*)WDT_RLD);
	writew(val, (unsigned long*)WDT_RLD);
        spin_unlock(&wdt_lock);
        return wdt_sys_status();
}

/*
 * Function Name:   wdt_sys_status.
 * Parameter:       None.
 * Return:          0 if WDT_TIMEOUT=0 (sys is stable - no timeouts)
 *                  1 if WDT_TIMEOUT=1 (sys is unstable - timeouts occured)
 * Description:     This function is used to get system status.
 */ 
static int wdt_sys_status(void)
{
        int status;
        spin_lock(&wdt_lock);
        status = readw((unsigned long*)WDT_RLD);
	//	printk("ESBWDT wdt_sys_status(): status=%x\n",status);
        spin_unlock(&wdt_lock);
        if ( ( status & WDT_CLEAR_STS ) == 0 )
                return 0;
        return 1;
}

/*
 * Function Name:   wdt_start.
 * Parameter:       None.
 * Return:          0 - successful, -1 - failed.
 * Description:     This function is used to start the timer.
 */ 
static void wdt_start(void)
{
  unsigned char tmp;
        wdt_enable_m();
	// clear timeout status bit
	// WDT_TIMEOUT bit: RELOAD.9 = [base+C].9
	// WDT_TIMEOUT bit lives in RTC well (cmos pwrd) 
	// - it's value is not lost when the system resets
        wdt_clear_toutsts();
	// set default configurations in hardware
	tmp = wdt_mode;
	wdt_set_mode(tmp);
	tmp = wdt_scale;
	wdt_set_scale(tmp);
        if (wdt_mode ==  FREE_MODE) {
                wdt_set_timeout2(wdt_margin2);
        }
        else {
                wdt_set_timeout1(wdt_margin1);
                wdt_set_timeout2(wdt_margin2);
	}
	// iad - debug -- gen irqs without reseting
	//	wdt_dis_opin();
	//---------------------------
	// enable the output pin
	wdt_en_opin();
	// start the wdt
	wdt_enable();
}

/*
 * Function Name:   wdt_sethwdflt.
 * Parameter:       None.
 * Return:          0 - successful, -1 - failed.
 * Description:     This function is used to set hw config at insmod time
 */ 

static void wdt_sethwdflt(void)
{
  unsigned char tmp;
        wdt_enable_m();
	// enable the BAR in CMD reg... is this really necessary??

	wdt_disable();
	// STOP the WDT

	// clear all status bits

        wdt_clear_toutsts();
	// clear timeout status bit
	// WDT_TIMEOUT bit: RELOAD.9 = [base+C].9
	// WDT_TIMEOUT bit lives in RTC well (cmos pwrd) 
	// - it's value is not lost when the system resets

	// note: Hardware defaults to Interrupts Enabled (STUPID!!!)
	// ideally we should disable interrupts until after the ISR is installed
	// for now, assume hw is in power-on state and wdt is stopped

	tmp = 0x01;
	writeb(tmp, (unsigned long*)WDT_INT_STS);
	// clear irq status

	// set default configurations in hardware
	tmp = wdt_mode;
	wdt_set_mode(tmp);

	tmp = wdt_scale;
	wdt_set_scale(tmp);

        if (wdt_mode ==  FREE_MODE) {
                wdt_set_timeout2(wdt_margin2);
        }
        else {
                wdt_set_timeout1(wdt_margin1);
                wdt_set_timeout2(wdt_margin2);
	}

	wdt_en_opin();
	// enable wdt output pin
}

/*
 * Function Name:   wdt_open.
 * Parameter:       struct inode *inode, struct file *file.
 * Return Value:    0 - successful, negative value - failed.
 * Description:     This function is used to open the device.
 */
static int wdt_open (struct inode *inode, struct file *file)
{
        if (test_and_set_bit(0, &wdt_count))
  	        return -EBUSY;
	//        wdt_addr = (unsigned long)ioremap(wdt_base,16);

	if (wdt_getenable() == 0) { // if STOPPED
	  wdt_enable();             // start with insmod dflts 
	}                           // or with last progd hw configs

        // if RUNNING, leave it alone...assume user/wdt app will
	// stop/config/restart as needed

        return 0;        
}

/*
 * Function Name:   wdt_release.
 * Parameter:       struct inode *inode, struct file *file.
 * Return Value:    0 - successful.
 * Description:     This function is used to release the device @ close
 */
static int wdt_release (struct inode *inode, struct file *file)
{
  if (wdt_expect_close == 42 && !nowayout) {
    // 'V' was written prior to close & nowayout=0
    // stop the wdt 
    wdt_disable();
  } else {
    // 'V' wasn't written prior to close OR nowayout=1
    // leave wdt running 
    wdt_reload();
    printk("esbwdt: unexpected close, not stopping watchdog!!\n");
  }
  //stuff that gets done on all closes
  clear_bit(0, &wdt_count);
  wdt_expect_close = 0;
  //  iounmap((void*)wdt_addr);
  return 0;
}

/*
 * Function Name:   wdt_write.
 * Parameter:       struct file *file, const char *data, size_t count, loff_t * pos.
 * Return Value:    count.
 * Description:     This function is used to open the device.
 */
static ssize_t wdt_write (struct file *file, const char *data, size_t count, loff_t * pos)
{
  // Can't seek (pwrite) on this device
  if ( pos != &file->f_pos )
    return -ESPIPE;
  //reload on all writes
  //app writing magic char 'V' prior to closing /dev/watchdog stops wdt when
  // started with nowayout=1, otherwise wdt stays running after close
  if ( count ) {
    size_t i;
    wdt_expect_close=0;
    for (i = 0; i != count; i++) {
      u8 c;
      if (get_user(c, data+i))
	return -EFAULT;
      if (c == 'V')
	wdt_expect_close=42;
    }
    wdt_reload ();
    return 1;
  }
  return 0;
}

/*
 * Function Name:   wdt_ioctl.
 * Parameter:       struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg.
 * Return Value:    0 - successful, negative value - failed.
 * Description:     This function is used to provide IO interface.
 */
static int wdt_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
        unsigned char mode, scale;
        unsigned long u_margin, dcount;

        static  struct watchdog_info ident = {
                options:	        WDIOF_SETMODE | WDIOF_KEEPALIVEPING | WDIOF_SETTIMEOUT | WDIOF_SETTIMEOUT2,
	        firmware_version:	0.1,
	        identity:		"ESB Watchdog Timer",
        };
        switch (cmd) 
        {
	        case WDIOC_GETSUPPORT:
		        if ( copy_to_user((struct watchdog_info *) arg, &ident, sizeof (ident)) )
		                return -EFAULT;
		        return 0;    
		case WDIOC_GETSTATUS:
			return put_user(wdt_sys_status(),(int *)arg);

	        case WDIOC_CLRSTATUS:
		  return put_user(wdt_clear_toutsts(),(int *)arg);

		case WDIOC_GETBOOTSTATUS:
			return put_user(0, (int *)arg);    
	        case WDIOC_DISABLE:
                        wdt_reload ();
		        return wdt_disable();
		case WDIOC_ENABLE:
	                return wdt_enable();
	        case WDIOC_GETWDTENABLE:
                        return put_user(wdt_getenable(),(int *)arg);
                case WDIOC_SETMODE:
                        if ( get_user(mode, (unsigned char *) arg) )
		                return -EFAULT;
                        return wdt_set_mode(mode);
	        case WDIOC_GETMODE:
                        return put_user(wdt_get_mode(),(int *)arg);
                case WDIOC_SETOPIN:
                        if ( get_user(mode, (unsigned char *) arg) )
		                return -EFAULT;
                        return wdt_set_opin(mode);
	        case WDIOC_GETOPIN:
                        return put_user(wdt_get_opin(),(int *)arg);
	        case WDIOC_GETIRQSTAT:
                        return put_user(wdt_get_irqstat(),(int *)arg);
	        case WDIOC_CLRIRQSTAT:
		        wdt_clr_irqstat();
		        return 0;
                case WDIOC_SETSCALE:
                        if ( get_user(scale, (unsigned char *) arg) )
		                return -EFAULT;
                        return wdt_set_scale(scale);
                case WDIOC_GETSCALE:
                        return put_user(wdt_get_scale(),(int *)arg);         
                case WDIOC_LOCK:
                        if ( get_user(scale, (unsigned char *) arg) )
		                return -EFAULT;
                        return wdt_lk();
	        case WDIOC_KEEPALIVE:
		        wdt_reload ();
		        return 0;
                case WDIOC_SETTIMEOUT:
		        if ( get_user(u_margin, (unsigned long *) arg) )
		                return -EFAULT;
		        if ( wdt_set_timeout1(u_margin) )
		                return -EINVAL;
		        wdt_margin1 = u_margin;
		        wdt_reload ();
                        return 0;
	        case WDIOC_SETTIMEOUT2:
		        if ( get_user(u_margin, (unsigned long *) arg) )
		                return -EFAULT;
		        if ( wdt_set_timeout2(u_margin) )
		                return -EINVAL;
		        wdt_margin2 = u_margin;
		        wdt_reload ();
                        return 0;
                case WDIOC_GETTIMEOUT:
		        return put_user(wdt_margin1, (unsigned long *) arg);
	        case WDIOC_GETTIMEOUT2:
	                return put_user(wdt_margin2, (unsigned long *) arg);
                case WDIOC_NOTIFY:
		        wdt_wq_active = 1;
                        interruptible_sleep_on(&wdt_wait_queue);
			wdt_wq_active = 0;
                        return 0;
                case WDIOC_GETDC:
		  dcount = wdt_get_dc();
                  put_user(dcount, (unsigned long *) arg);                       
                  return 0;
                case WDIOC_GETIRQVEC:
		  put_user(wdt_irq, (unsigned long *) arg);                       
                  return 0;

	case WDIOC_CLRNOTIFY:
	  if (wdt_wq_active == 1) {
	    wake_up_interruptible(&wdt_wait_queue);
	    wdt_wq_active = 0;
	  }
	  return 0;

                default:
		        return -ENOTTY;
        }
}

/*
 * Function Name:   wdt_isr.
 * Parameter:       int irq - irq number, void *dev_id, struct pt_regs *regs
 * Return Value::   None.
 * Description:     This is the interrupt service routine of the WDT.
 */
static void wdt_isr(int irq, void *dev_id, struct pt_regs *regs)
{
    unsigned char val;
    val = readb((unsigned long*)WDT_INT_STS);
    if( ( val & 0x01 ) == 0 )
       return; 
       writeb(val, (unsigned long*)WDT_INT_STS);
    wake_up_interruptible(&wdt_wait_queue); 
    // printk("esbwdt stage1 timeout irq34 detected...\n");
    return;
}

MODULE_DEVICE_TABLE (pci, wdt_pci_tbl);

/*
 * Function Name:   wdt_init_one.
 * Parameter:       struct pci_dev *dev, const struct pci_device_id *id_ptr
 * Return Value:    0 - successful, negative value - failed.
 * Description:     This function is used to init the device and request resources.
 */ 
static int __init wdt_init_one (struct pci_dev *dev, const struct pci_device_id *id_ptr)
{
        spin_lock_init(&wdt_lock); 

        wdt_device();
	// pci_match_device(wdt_pci_tbl, dev); wdt_pci = dev; ...
	// ... initializes [ struct pci_dev *wdt_pci ] for use with pci_config_r/w function
 
	//wdt_irq = dev->irq;  can't use irq info from pci *dev struct cuz hw is broken...have to hardcode irq

	wdt_base = pci_resource_start(dev, 0);
	// wdt_base = physical address in BAR0
	printk("esbwdt: WDT at base address %lX, Interrupt %d\n", wdt_base, wdt_irq);

	if ( pci_enable_device(dev) )
	        return -EIO;
	// pci_enable_device assigns its interrupt line and I/O regions. (assume it enables BAR decoders via CMD Reg)

	request_mem_region(wdt_base, 16, "esbwdt"); 
	// allocate i/o memory region (starting @ wdt_base, len=16, for "esbwdt") in kernel's registery
	// ideally one would check_mem_region 1st, however, since we are using bios assigned base address assume its ok
	// note that this is for mainly 'housekeeping', we still can't access the physical address yet

	wdt_addr = (unsigned long)ioremap(wdt_base,16);
	// ioremap does physical to virtual mappings... wdt_addr=virtual addr for accessing wdt_base=physical addr

	// configure hardware with driver defaults, but don't start the wdt timer
	wdt_sethwdflt();

        if ( request_irq(wdt_irq, wdt_isr, SA_INTERRUPT | SA_SHIRQ, "esbwdt", &wdt_miscdev) ) 
        {
                printk ("IRQ %d is not free.\n", wdt_irq);
                release_mem_region(wdt_base, 16);
                return -EIO;
        }     
        if ( misc_register(&wdt_miscdev) != 0 ) 
        {
                release_mem_region(wdt_base, 16);
                free_irq(wdt_irq, &wdt_miscdev);
	        printk("esbwdt: cannot register miscdev\n");
	        return -EIO;
        }        
        return 0;
}

/*
 * Function Name:   wdt_remove_one.
 * Parameter:       struct pci_dev *pdev
 * Return Value:    None.
 * Description:     This function is used to release resources at rmmod time.
 */
static void __devexit wdt_remove_one (struct pci_dev *pdev)
{

  // stop the WDT
  wdt_reload();
  wdt_disable();
  // deallocate resources that were assigned @ insmod time (in wdt_add_one)
         misc_deregister(&wdt_miscdev);
         free_irq(wdt_irq, &wdt_miscdev);
	 iounmap((void*)wdt_addr);
         release_mem_region(wdt_base, 16);
}

/*
 * Function Name:   wdt_init.
 * Parameter:       None.
 * Return Value:    0 - successful, negative value - failed.
 * Description:     This function is used to register the driver.
 */
static int __init wdt_init(void)
{
	if ( pci_register_driver(&wdt_driver) < 1 )
	     return -ENODEV;
	return 0;
}

/*
 * Function Name:   wdt_cleanup.
 * Parameter:       None.
 * Return Value:    None.
 * Description:     This function is used to deregister the driver.
 */
static void __exit wdt_cleanup(void)
{
	pci_unregister_driver(&wdt_driver);
}

 
module_init(wdt_init);
module_exit(wdt_cleanup);
MODULE_AUTHOR("ike.a.dean@intel.com");
MODULE_DESCRIPTION("WDT timer driver for Intel(R) ESB");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;



