#!/bin/bash
##############################################################
# version that configures drive X to boot as drive X
# KEL - Tue Nov 27 23:54:31 GMT 2001
#     - Added initrd
# KEL - Wed Oct 31 17:10:15 MST 2001
#     - implemented partiton-*-size parameters for makepartitions style
# KEL - Tue Oct 30 16:39:30 MST 2001
#     - previous version checked in without cvs comment....do it now
# KEL - Tue Oct 30 14:22:44 MST 2001
#     - Added partition size/type/minor node selection for root var data and swap
#     - Added debug_only flag to not execute anything but rather print what
#     -  would be done
#     - Added Seagate firmware identification
# KEL - Thu Oct  4 19:52:05 MDT 2001
#     - changed the creation of timezone
#     - fully specified path to rm
##############################################################
# TODO:
##############################################################
set -e

##############################################################
# Functions
##############################################################
what_drive ()
{
    case $1 in
    /dev/hda*|hda*|a)
        what_drive_ret=hda
    ;;
    /dev/hdb*|hdb*|b)
        what_drive_ret=hdb
    ;;
    /dev/hdc*|hdc*|c)
        what_drive_ret=hdc
    ;;
    /dev/hdd*|hdd*|d)
        what_drive_ret=hdd
    ;;
    *)
        echo "ERROR:  drive $1 $2 is invalid"
		what_drive_ret=UNKNOWN
        usage
    ;;
	esac
}

what_md ()
{
    case $1 in
    hda|hdb|hdc|hdd)
        what_os_md_ret=/dev/md200
        what_var_md_ret=/dev/md199
    ;;
    *)
        echo "ERROR:  drive $1 is invalid...unable to determine corresponding mds"
        what_os_md_ret=UNKNOWN
        what_var_md_ret=UNKNOWN
        usage
    ;;
	esac
}
what_md_x ()
{
    case $1 in
    hda)
        what_os_md_ret=/dev/md100
        what_var_md_ret=/dev/md101
    ;;
    hdb)
        what_os_md_ret=/dev/md110
        what_var_md_ret=/dev/md111
    ;;
    hdc)
        what_os_md_ret=/dev/md120
        what_var_md_ret=/dev/md121
    ;;
    hdd)
        what_os_md_ret=/dev/md130
        what_var_md_ret=/dev/md131
    ;;
    *)
        echo "ERROR:  drive $1 is invalid...unable to determine corresponding mds"
        what_os_md_ret=UNKNOWN
        what_var_md_ret=UNKNOWN
        usage
    ;;
	esac
}

clean_size ()
{
    case $1 in
	[0-9]*[Kk])
		clean_size_ret=${1/[Kk]}
		clean_size_ret=$((1024 / 512 * clean_size_ret))
	;;
	[0-9]*[Kk][Bb])
		clean_size_ret=${1/[Kk][Bb]}
		clean_size_ret=$((1024 / 512 * clean_size_ret))
	;;
	[0-9]*[Mm])
		clean_size_ret=${1/[Mm]}
		clean_size_ret=$((1024 / 512 * 1024 * clean_size_ret))
	;;
	[0-9]*[Mm][Bb])
		clean_size_ret=${1/[Mm][Bb]}
		clean_size_ret=$((1024 / 512 * 1024 * clean_size_ret))
	;;
	[0-9]*[Gg])
		clean_size_ret=${1/[Gg]}
		clean_size_ret=$((1024 / 512 * 1024 * 1024 * clean_size_ret))
	;;
	[0-9]*[Gg][Bb])
		clean_size_ret=${1/[Gg][Bb]}
		clean_size_ret=$((1024 / 512 * 1024 * 1024 * clean_size_ret))
	;;
    [0-9]*)
        clean_size_ret=$1
    ;; 
    NA)
        clean_size_ret=NA
    ;;
    *)
        echo "ERROR:  size $1 is invalid"
		clean_size_ret=NA
        usage
    ;;
	esac
}

function usage()
{
	echo "usage: $0 <drive to image> <firmware>"
	echo "    [--boot-as <drive letter for lilo boot>]"
	echo "    [--drive <drive letter to image>]"
	echo "    [--firmware <11A | 51A | QUACK| 15A>]"
	echo "    [--software <software version to use>]"
	echo "    [--sw-version <software version to use>]"
	echo "    [--partition-signature-slice <0-9>]"
	echo "    [--partition-os-slice <0-9>]"
	echo "    [--partition-upgrade-slice <0-9>]"
	echo "    [--partition-var-slice <0-9>]"
	echo "    [--partition-data-slice <0-9>]"
	echo "    [--partition-swap-slice <0-9>]"
	echo "    [--partition-os-size [####GB|####MB|####KB|#### (512 byte blocks)]"
	echo "    [--partition-var-size [####GB|####MB|####KB|#### (512 byte blocks)]"
	echo "    [--partition-data-size [####GB|####MB|####KB|#### (512 byte blocks)]"
	echo "    [--partition-swap-size [####GB|####MB|####KB|#### (512 byte blocks)]"
	echo "    [--partition-gig-os] [--partition-meg-os]"
	echo "    [--filesystem-os <xfs | reiserfs>]"
	echo "    [--filesystem-boot <xfs | ext2>]"
	echo "    [--filesystem-var <xfs | reiserfs>]"
	echo "    [--filesystem-data <xfs | reiserfs>]"
	echo "    [--old-partition] [--new-partition]"
	echo "    [--debug-only] [--no-debug-only]"
	echo "    [--do-custom-os] [--skip-custom-os]"
	echo "    [--do-os] [--skip-os]"
	echo "    [--do-lhn] [--skip-lhn]"
	echo "    [--do-umount] [--skip-umount]"
	echo "    [--do-partition] [--skip-partition]"
	echo "    [--do-mkfs] [--skip-mkfs]"
	echo "    [--do-swap] [--skip-swap]"
	echo "    [--do-mount] [--skip-mount]"
	echo "    [--do-initrd] [--skip-initrd]"
	echo "    [--force-single-user] [--allow-any-level]"

	echo " drive letters: a b c d (NOTE:  a is probably your boot disk!!!)"
	echo " Firmware (Unique model) supported: 11A 11 51A 51 A08.1500 QUACK 15A 15"
	exit
}

##############################################################
# set defaults
##############################################################
debug_only=0
drive_to_image=x
kernel_to_use=vmlinuz-2.4.9-xfs
boot_as_drive=hda
raid_partition=md4
drive_firmware=x
swversion=0.112.19
filesystem_os=xfs
filesystem_boot=ext2
filesystem_var=xfs
filesystem_data=xfs
partition_signature_slice=1
partition_os_slice=5
partition_upgrade_slice=6
partition_var_slice=7
partition_data_slice=2
partition_swap_slice=6
partition_boot_slice=3
partition_os_size=2048000
partition_var_size=NA
partition_data_size=NA
partition_swap_size=NA
partition_style=makepartitions
partition_template=NA
do_custom_os=0
do_single=0
do_newos=1
do_umount=1
do_mkfs=1
do_swap=0
do_mount=1
do_initrd=0
disks_to_image="hdb hdc hdd"
do_lhn=1

##############################################################
# Parse command line options
##############################################################
while [ $# -gt 0 ] ; do
	case $1 in
	--boot-as)
		if [ $# -gt 1 ] ; then
			boot_as_drive=$2
			shift
		else
			echo "ERROR:  --boot-as requires a parameter"
		fi
	;;
	--drive)
		if [ $# -gt 1 ] ; then
			drive_to_image=$2
			shift
		else
			echo "ERROR:  --drive requires a parameter"
		fi
	;;
	--firmware)
		if [ $# -gt 1 ] ; then
			drive_firmware=$2
			shift
		else
			echo "ERROR:  --firmware requires a parameter"
		fi
	;;
	--software|--sw-version)
		if [ $# -gt 1 ] ; then
			swversion=$2
			shift
		else
			echo "ERROR:  --software requires a parameter"
		fi
	;;
	--partition-signature-slice)
		if [ $# -gt 1 ] ; then
			partition_signature_slice=$2
			shift
		else
			echo "ERROR:  --partition-signature-slice requires a parameter"
		fi
	;;
	--partition-os-slice)
		if [ $# -gt 1 ] ; then
			partition_os_slice=$2
			shift
		else
			echo "ERROR:  --partition-os-slice requires a parameter"
		fi
	;;
	--partition-upgrade-slice)
		if [ $# -gt 1 ] ; then
			partition_upgrade_slice=$2
			shift
		else
			echo "ERROR:  --partition-upgrade-slice requires a parameter"
		fi
	;;
	--partition-var-slice)
		if [ $# -gt 1 ] ; then
			partition_var_slice=$2
			shift
		else
			echo "ERROR:  --partition-var-slice requires a parameter"
		fi
	;;
	--partition-data-slice)
		if [ $# -gt 1 ] ; then
			partition_data_slice=$2
			shift
		else
			echo "ERROR:  --partition-data-slice requires a parameter"
		fi
	;;
	--partition-swap-slice)
		if [ $# -gt 1 ] ; then
			partition_swap_slice=$2
			shift
		else
			echo "ERROR:  --partition-swap-slice requires a parameter"
		fi
	;;
	--partition-os-size)
		if [ $# -gt 1 ] ; then
			partition_style=makepartitions
			partition_os_size=$2
			shift
		else
			echo "ERROR:  --partition-os-size requires a parameter"
		fi
	;;
	--partition-var-size)
		if [ $# -gt 1 ] ; then
			partition_style=makepartitions
			partition_var_size=$2
			shift
		else
			echo "ERROR:  --partition-var-size requires a parameter"
		fi
	;;
	--partition-data-size)
		if [ $# -gt 1 ] ; then
			partition_style=makepartitions
			partition_data_size=$2
			shift
		else
			echo "ERROR:  --partition-data-size requires a parameter"
		fi
	;;
	--partition-swap-size)
		if [ $# -gt 1 ] ; then
			partition_style=makepartitions
			partition_swap_size=$2
			shift
		else
			echo "ERROR:  --partition-swap-size requires a parameter"
		fi
	;;
	--partition-gig-os|--gig-os)
		partition_style=sfdisk
		partition_os_size=1024000
	;;
	--partition-meg-os|--meg-os)
		partition_style=sfdisk
		partition_os_size=1024
	;;
	--filesystem-os)
		if [ $# -gt 1 ] ; then
			filesystem_os=$2
			shift
		else
			echo "ERROR:  --filesystem-os requires a parameter"
		fi
	;;
	--filesystem-boot)
		if [ $# -gt 1 ] ; then
			filesystem_boot=$2
			shift
		else
			echo "ERROR:  --filesystem-boot requires a parameter"
		fi
	;;
	--filesystem-var)
		if [ $# -gt 1 ] ; then
			filesystem_var=$2
			shift
		else
			echo "ERROR:  --filesystem-var requires a parameter"
		fi
	;;
	--filesystem-data)
		if [ $# -gt 1 ] ; then
			filesystem_data=$2
			shift
		else
			echo "ERROR:  --filesystem-data requires a parameter"
		fi
	;;
	--old-partition)
		partition_style=sfdisk
	;;
	--new-partition)
		partition_style=makepartitions
	;;
	--debug-only)
		debug_only=1
	;;
	--no-debug-only)
		debug_only=0
	;;
	--do-os)
		do_newos=1
	;;
	--skip-os)
		do_newos=0
	;;
	--do-custom-os)
		do_custom_os=1
	;;
	--skip-custom-os)
		do_custom_os=1
	;;
	--force-single-user)
		do_single=1
	;;
	--allow-any-level)
		do_single=0
	;;
	--do-lhn)
		do_lhn=1
	;;
	--skip-lhn)
		do_lhn=0
	;;
	--do-umount)
		do_umount=1
	;;
	--skip-umount)
		do_umount=0
	;;
	--do-partition)
		partition_style=sfdisk
	;;
	--skip-partition)
		partition_style=NONE
	;;
	--do-mkfs)
		do_mkfs=1
	;;
	--skip-mkfs)
		do_mkfs=0
	;;
	--do-swap)
		do_swap=1
	;;
	--skip-swap)
		do_swap=0
	;;
	--do-mount)
		do_mount=1
	;;
	--skip-mount)
		do_mount=0
	;;
	--do-initrd)
		do_initrd=1
	;;
	--skip-initrd)
		do_initrd=0
	;;
	/dev/hd*|hd*|?)
		drive_to_image=$1
	;;
	11A|11|51A|51|A08.1500|QUACK|15A|15)
		drive_firmware=$1
	;;
	*)
		echo "unknown parameter $1"
		usage
		exit
	;;
	esac
	shift
done

##############################################################
# Validate drive choices
##############################################################
what_drive $drive_to_image "to image"
drive_to_image=$what_drive_ret

what_drive $boot_as_drive "to boot as"
boot_as_drive=$what_drive_ret

##############################################################
# Select the parameters needed for the partition style being used
##############################################################
makepartitions_parameters=""
sfdisk_parameters="-H255 -S63"
sfdisk_extension="x"
case ${partition_style} in
	makepartitions)
		clean_size ${partition_os_size}
		partition_os_size=${clean_size_ret}

		clean_size ${partition_var_size}
		partition_var_size=${clean_size_ret}

		clean_size ${partition_data_size}
		partition_data_size=${clean_size_ret}

		clean_size ${partition_swap_size}
		partition_swap_size=${clean_size_ret}

		if [ "${partition_os_size}" != "NA" ] ; then
			partition_parameters="${partition_parameters} -ossize ${partition_os_size}"
		fi
		if [ "${partition_var_size}" != "NA" ] ; then
			partition_parameters="${partition_parameters} -logsize ${partition_var_size}"
		fi
		if [ "${partition_data_size}" != "NA" ] ; then
			partition_parameters="${partition_parameters} -datasize ${partition_data_size}"
		fi
		echo "DEBUG:  Using sizes:  OS=${partition_os_size} VAR=${partition_var_size} DATA=${partition_data_size} SWAP=${partition_swap_size}"
	;;
	sfdisk)
		if [ "${drive_firmware}" == "x" ] ; then
			##############################################################
			# Automatically Determine firmware if not given
			##############################################################
			echo "Automatically determining firmware"
			drive_identify=`/sbin/hdparm -i /dev/${drive_to_image}`
			case ${drive_identify} in
				*Maxtor*LBAsects=156301487*)
					echo "Processing 80Gig Maxtor Drive with QuackStore firmware rev"
					drive_firmware=QUACK
				;;
				*MAXTOR*LBAsects=156301487*)
					echo "Processing 80Gig Maxtor Drive with QuackStore firmware rev"
					drive_firmware=QUACK
				;;
				*Maxtor*LBAsects=156301488*)
					echo "Processing 80Gig+1 Maxtor Drive with QuackStore firmware rev"
					drive_firmware=QUACK
				;;
				*MAXTOR*LBAsects=156301488*)
					echo "Processing 80Gig+1 Maxtor Drive with QuackStore firmware rev"
					drive_firmware=QUACK
				;;
				*Maxtor*LBAsects=160086528*)
					echo "Processing 80Gig Maxtor Drive with customer unique version 11A"
					drive_firmware=11A
				;;
				*Maxtor*LBAsects=156250000*)
					echo "Processing 80Gig Maxtor Drive with customer unique version 51A"
					drive_firmware=51A
				;;
				*Maxtor*LBAsects=80041248*)
					echo "Processing 40Gig Maxtor Drive with customer unique version 15A"
					drive_firmware=15A
				;;
				*ST380021A*FwRev=3.05*LBAsects=156301488*)
					echo "Processing 80Gig Seagate Drive"
					drive_firmware=Seagate
				;;
				*)
					echo "Unable to automatically determine firmware rev"
					echo "TRIED:${drive_identify}"
					usage
				;;
			esac
		#######end of firmware == x
		fi

		##############################################################
		# Validate firmware and pick partition settings
		##############################################################
		case ${drive_firmware} in
			A08.1500|QUACK)
				sfdisk_parameters="${sfdisk_parameters} -C9729"
				sfdisk_extension=156301487
			;;
			11A|11)
				sfdisk_parameters="${sfdisk_parameters} -C9964"
				sfdisk_extension=160086528
			;;
			51A|51)
				sfdisk_parameters="${sfdisk_parameters} -C9926"
				sfdisk_extension=156250000
			;;
			15A|15)
				sfdisk_parameters="${sfdisk_parameters} -C4982"
				sfdisk_extension=80041248
			;;
			*)
				echo "ERROR:  firmware ${drive_firmware} is invalid"
				usage
			;;
		esac

		clean_size ${partition_os_size}
		partition_os_size=${clean_size_ret}

		case ${partition_os_size} in
			1024000)
				partition_template=fdisk.gig
			;;
			1024)
				partition_template=fdisk.template
			;;
			*)
				echo "ERROR:  OS size ${partition_os_size} not valid for fdisk partitioning style"
				usage
			;;
		esac
	;;
	NONE)
		echo "WARNING:  partition style ${partition_style}...NOT PARTITIONING"
	;;
	*)
		echo "ERROR:  partition style ${partition_style} is invalid"
		usage
	;;
esac

##############################################################
# Tell user what we are going to execute
##############################################################
echo -n "Will: "
if [ ${do_umount} == 1 ] ; then
	echo -n " Unmount"
fi
if [ "${partition_style}" != "NONE" ] ; then
	echo -n " partition"
fi
if [ ${do_mkfs} == 1 ] ; then
	echo -n " mkfs"
fi
if [ ${do_swap} == 1 ] ; then
	echo -n " swap"
fi
if [ ${do_mount} == 1 ] ; then
	echo -n " mount"
fi
if [ ${do_initrd} == 1 ] ; then
	echo -n " initrd"
fi
if [ ${do_newos} == 1 ] ; then
	echo -n " os_load"
fi
if [ ${do_lhn} == 1 ] ; then
	echo -n " lefthand_additions"
fi
echo ""

##############################################################
# setup dependant variables
##############################################################
if [ ${do_initrd} == 1 ] ; then
	what_md ${drive_to_image}
	path_to_raw_os_device=${what_os_md_ret}
	path_to_raw_var_device=${what_var_md_ret}

	path_to_raw_device=/dev/${drive_to_image}
	path_to_raw_signature_device=${path_to_raw_device}${partition_signature_slice}
	path_to_raw_data_device=${path_to_raw_device}${partition_data_slice}
	path_to_raw_swap_device=${path_to_raw_device}${partition_swap_slice}
	path_to_raw_boot_device=${path_to_raw_device}${partition_boot_slice}
else
	path_to_raw_os_device=/dev/${drive_to_image}${partition_os_slice}
	path_to_raw_var_device=/dev/${drive_to_image}${partition_var_slice}

	path_to_raw_device=/dev/${drive_to_image}
	path_to_raw_signature_device=${path_to_raw_device}${partition_signature_slice}
	path_to_raw_data_device=${path_to_raw_device}${partition_data_slice}
	path_to_raw_swap_device=${path_to_raw_device}${partition_swap_slice}
fi
path_to_mount_point=/mnt/${drive_to_image}${partition_os_slice}

dir_to_images=/root/imager/${swversion}
raidtab_for_image_drive=/root/imager/${swversion}/raidtab.all
if [ ${do_custom_os} == 1 ] ; then
		OS_image=lhn_os.cpio.bz2
		required_os_images=""
		optional_os_images="ypbind"
else
		OS_image=lhn_os.cpio.gz
		required_os_images="glibc gcc kernel lm_sensors i2c pam samba ypbind"
		optional_os_images="kernel.src.objs"
fi
lefthand_additions="sysmods"
if [ ${do_initrd} == 1 ] ; then
	optional_files_to_copy="lhn_initrd.template.raw.gz lilo.conf.hda lilo.conf.hdb lilo.conf.hdc lilo.conf.hdd"
# "raidtab.hda raidtab.hdb raidtab.hdc raidtab.hdd"
#KELLY
#raidtab.hda.failed
#raidtab.hdb.failed
#raidtab.hdc.failed
#raidtab.hdd.failed
fi

##############################################################
# Verify that all required input files exist before doing anything
##############################################################
if [ ! -e "${dir_to_images}/${OS_image}" ] ; then
	echo "------ERROR:  LHN OS image ${dir_to_images}/${OS_image} does not exist"
	exit 
fi

for file in ${required_os_images} ; do
	if [ ! -e "${dir_to_images}/lhn_${file}.cpio.gz" ] ; then
		echo "------ERROR:  LHN ${file} image in ${dir_to_images}/lhn_${file}.cpio.gz does not exist"
		exit 
	fi
done

for file in ${optional_os_images} ; do
	if [ ! -e "${dir_to_images}/lhn_${file}.cpio.gz" ] ; then
		echo "------WARNING:  LHN ${file} image in ${dir_to_images}/lhn_${file}.cpio.gz does not exist"
	fi
done

for file in ${lefthand_additions} ; do
	if [ ! -e "${dir_to_images}/lhn_${file}_${swversion}.tar.gz" ] ; then
		echo "------WARNING:  LHN ${file} image ${dir_to_images}/lhn_${file}_${swversion}.tar.gz does not exist"
	fi
done

for file in ${optional_files_to_copy} ; do
	if [ ! -e "${dir_to_images}/${file}" ] ; then
		echo "------WARNING:  missing file ${dir_to_images}/${file}"
	fi
done

##############################################################
# Verify that mount point is usable
##############################################################
if [ -e ${path_to_mount_point} ] ; then
	if [ ! -d ${path_to_mount_point} ] ; then
		echo "------ERROR:  ${path_to_mount_point} is not a directory"
		exit
	fi
else
	mkdir ${path_to_mount_point}
fi

##############################################################
# Last minute print of what is going to happen
##############################################################
echo "PARTITIONING with ${partition_style}"
echo "	SIGN:  partition=${path_to_signature_device} type=${filesystem_os} size=${partition_os_size}"
echo "	BOOT:  partition=${path_to_boot_device} type=${filesystem_boot} size=${partition_boot_size}"
echo "	OS:  partition=${path_to_os_device} type=${filesystem_os} size=${partition_os_size}"
echo "	VAR:  partition=${path_to_var_device} type=${filesystem_var} size=${partition_var_size}"
echo "	DATA:  partition=${path_to_data_device} type=${filesystem_data} size=${partition_data_size}"
echo "	SWAP:  partition=${path_to_swap_device} type=${swap_filesystem} size=${partition_swap_size}"
if [ ${debug_only} == 1 ] ; then
	echo "NOT EXECUTING"
	exit
fi

##############################################################
# ensure that users are in single user mode.  This IS the defined
# process
##############################################################
if [ ${do_single} == 1 ] ; then
	run_level=`/sbin/runlevel`
	case ${run_level} in
		*[2345])
			echo "Run Level ${run_level} not valid...go to single user."
		;;
		*[1Ss])
			echo "Run Level is single user...good."
		;;
		default)
			echo "Run Level ${run_level} unknown...go to single user."
		;;
	esac
fi

##############################################################
# HERE GOES!!!!!!
# From here down things are actually done instead of being prep'ed
##############################################################

##############################################################
# make sure disk is not in use and unuse it if it is in use
##############################################################
if [ ${do_umount} == 1 ] ; then
	echo "stopping outgoing share services"
	/etc/init.d/httpd stop || echo "http stop failed...ignoring"
	/etc/init.d/smb stop || echo "smb stop failed...ignoring"
	/etc/init.d/nfs stop || echo "nfs stop failed...ignoring"
	/etc/init.d/proftpd stop || echo "proftpd stop failed...ignoring"

	# slices are listed in backwards order in the hopes that mount points
	# will be higher numbers on top of lower numbers...this works for / and /var
	for slice in 9 8 7 6 3 5 4 2 1; do
		if [ -e ${path_to_raw_device}${slice} ] ; then
			df_slice=`df |grep ${path_to_raw_device}${slice} || echo NA`
			if [ "${df_slice}" == "NA" ] ; then
				devfs=`ls -l ${path_to_raw_device}${slice} | awk '{print $NF}'`
				df_slice=`df |grep ${devfs} || echo NA`
				if [ "${df_slice}" == "NA" ] ; then
					echo "drive ${path_to_raw_device}${slice} not mounted"
				else
					echo "umounting ${path_to_raw_device}${slice}"
					umount ${path_to_raw_device}${slice}
				fi
				raiddev=`grep ${devfs} /proc/mdstat | awk '{print $1}'|| echo NA`
				if [ "${raiddev}" == "NA" -o "${raiddev}NA" == "NA" ] ; then
					echo "drive ${path_to_raw_device}${slice} not in use by raid"
				else
					echo "stopping raid /dev/${raiddev} in use by ${path_to_raw_device}${slice}"
					stoplist="4 2 1"
					if [ ${do_initrd} == 1 ] ; then
						stoplist="${stoplist} 101 102 100 111 112 110 121 122 120 131 132 130 199 10"
					fi
					for raiddev in ${stoplist}; do
						umount /dev/md${raiddev} || echo "Unmount /dev/md${raiddev} failed...ignoring"
						raidstop --configfile ${raidtab_for_image_drive} /dev/md${raiddev} || echo "raid stop /dev/md${raiddev} failed...ignoring"
					done
				fi
			else
				echo "umounting ${path_to_raw_device}${slice}"
				umount ${path_to_raw_device}${slice}
			fi
		else
			echo "partition ${path_to_raw_device}${slice} does not exist"
		fi
	done
else
	echo "Skipping unmount"
fi

##############################################################
# make sure hydra is not running or it may cause signature problems
##############################################################
hydra=`ps -eaf | grep hydra || echo "NA"`
if [ "${hydra}" == "NA" ] ; then
	echo "hydra not running"
else
	echo "stopping hydra"
	/etc/init.d/hydra stop || echo "hydra stop failed...ignoring"
fi


##############################################################
# partition the disk
##############################################################
case ${partition_style} in
	makepartitions)
		for disk in ${disks_to_image} ; do
			${dir_to_images}/makepartitions /dev/${disk} ${partition_parameters}
			sleep 3
			dd if=/dev/zero of=/dev/${disk}${partition_signature_slice} || echo "dd completed"
			${dir_to_images}/signdrives /dev/${disk}
		done;
	;;
	sfdisk)
		for disk in ${disks_to_image} ; do
			dd if=/dev/zero of=/dev/${disk} count=1024
			sfdisk ${sfdisk_parameters} /dev/${disk} < ${partition_template}.${sfdisk_extension}
		done;
	;;
	NONE)
		echo "WARNING:  partition style ${partition_style}...NOT PARTITIONING"
	;;
	*)
		echo "ERROR:  partition style ${partition_style} is invalid"
		usage
	;;
esac
if [ ${do_initrd} == 1 ] ; then
	#echo mkraid --really-force --configfile ${raidtab_for_image_drive} ${path_to_raw_os_device}
	mkraid --really-force --configfile ${raidtab_for_image_drive} ${path_to_raw_os_device}
#KELLY FIX HARD CODED RAID HOTADD DEVICES
	#raidhotadd --configfile ${raidtab_for_image_drive} /dev/md200 /dev/hdc5
	#raidhotadd --configfile ${raidtab_for_image_drive} /dev/md200 /dev/hdd5
	
	#echo mkraid --really-force --configfile ${raidtab_for_image_drive} ${path_to_raw_var_device}
	mkraid --really-force --configfile ${raidtab_for_image_drive} ${path_to_raw_var_device}
#KELLY FIX HARD CODED RAID HOTADD DEVICES
	#raidhotadd --configfile ${raidtab_for_image_drive} /dev/md199 /dev/hdc7
	#raidhotadd --configfile ${raidtab_for_image_drive} /dev/md199 /dev/hdd7
fi

##############################################################
# make the required filesystems
##############################################################
if [ ${do_mkfs} == 1 ] ; then
	##############################################################
	# make the root OS filesystem
	##############################################################
	echo "Making OS filesystem"
    case ${filesystem_os} in
	reiserfs)
		yes "y" | mkreiserfs ${path_to_raw_os_device}
	;;
	xfs)
		mkfs.xfs ${path_to_raw_os_device} -f
	;;
	*)
		echo "ERROR:  OS filesystem ${filesystem_os} is unknown"
		exit 1
	;;
	esac
	##############################################################
	# make the VAR filesystem
	##############################################################
	echo "Making var filesystem"
    case ${filesystem_var} in
	reiserfs)
		yes "y" | mkreiserfs ${path_to_raw_var_device}
	;;
	xfs)
		mkfs.xfs ${path_to_raw_var_device} -f
	;;
	*)
		echo "ERROR:  VAR filesystem ${filesystem_var} is unknown"
		exit 1
	;;
	esac
	if [ ${do_initrd} == 1 ] ; then
		##############################################################
		# make the BOOT filesystem
		##############################################################
		echo "Making boot filesystems"
		for disk in ${disks_to_image} ; do
			case ${filesystem_boot} in
			xfs)
				mkfs.xfs /dev/${disk}${partition_boot_slice} -f
			;;
			ext2)
				mke2fs -F /dev/${disk}${partition_boot_slice}
			;;
			*)
				echo "ERROR:  BOOT filesystem ${filesystem_boot} is unknown"
				exit 1
			;;
			esac
		done
	fi
else
	echo "Skipping filesystem creation"
fi
##############################################################
# make the swap filesystem
##############################################################
if [ ${do_swap} == 1 ] ; then
	mkswap ${path_to_raw_swap_device}
else
	echo "Skipping swap creation"
fi

##############################################################
# mount the VAR partition
##############################################################
if [ ${do_mount} == 1 ] ; then
	##############################################################
	# mount the OS partition
	##############################################################
	mount -t ${filesystem_os} ${path_to_raw_os_device} ${path_to_mount_point}

	##############################################################
	# Verify that the VAR mount point is usable
	##############################################################
	if [ -e ${path_to_mount_point}/var ] ; then
		if [ ! -d ${path_to_mount_point}/var ] ; then
			echo "------ERROR:  ${path_to_mount_point}/var is not a directory"
			exit
		fi
	else
		mkdir ${path_to_mount_point}/var
	fi
	##############################################################
	# mount the VAR partition
	##############################################################
	mount -t ${filesystem_var} ${path_to_raw_var_device} ${path_to_mount_point}/var

	if [ ${do_initrd} == 1 ] ; then
		##############################################################
		# Verify that the BOOT (used for initrd only) mount point is usable
		##############################################################
		for disk in ${disks_to_image} ; do
			if [ -e ${path_to_mount_point}/${disk} ] ; then
				if [ ! -d ${path_to_mount_point}/${disk} ] ; then
					echo "------ERROR:  ${path_to_mount_point}/${disk} is not a directory"
					exit
				fi
			else
				mkdir ${path_to_mount_point}/${disk}
			fi
			##############################################################
			# mount the BOOT partition
			##############################################################
			mount -t ${filesystem_boot} /dev/${disk}${partition_boot_slice} ${path_to_mount_point}/${disk}
		done
	fi
else
	echo "Skipping filesystem mount"
fi

##############################################################
# put the OS on the OS partition
##############################################################
if [ ${do_newos} == 1 ] ; then
	cd ${path_to_mount_point}
	if [ -e "${dir_to_images}/${OS_image}" ]; then
		echo "Installing ${OS_image} from ${dir_to_images} to ${path_to_mount_point}"
		if [ ${do_custom_os} == 1 ] ; then
				bzip2 -dc ${dir_to_images}/${OS_image} | \
					cpio --extract --no-absolute-filenames \
						 --make-directories --unconditional -m
		else
				gunzip -c ${dir_to_images}/${OS_image} | \
					cpio --extract --no-absolute-filenames \
						 --make-directories --unconditional -m
		fi
		echo "${OS_image} Installed."
	else
				echo "ERROR: Image ${dir_to_images}/${OS_image} not found."
				exit
	fi
else
	echo "Skipping os install"
fi

##############################################################
# put the LHN specifics on the OS partition
##############################################################
if [ ${do_lhn} == 1 ] ; then
	echo "Adding LHN specifics from ${dir_to_images} ..."
	osversion=${swversion}
	serial_number=`date +%Y%m%d%H%M%S`
	hostname=$serial_number
	domainname=localdomain
	ip_address=127.0.0.1
	gateway=
	netmask=

	cd ${path_to_mount_point}
	for image in ${required_os_images} ; do
		if [ -e "${dir_to_images}/lhn_${image}.cpio.gz" ]; then
			echo "    Installing lhn_${image}.cpio.gz to ${path_to_mount_point}"
			gunzip -c ${dir_to_images}/lhn_${image}.cpio.gz | cpio --make-directories --no-absolute-filenames --extract --unconditional
		else
			echo "ERROR: Image ${dir_to_images}/lhn_${image}.cpio.gz not found."
			exit
		fi
	done

	for image in ${optional_os_images} ; do
		if [ -e "${dir_to_images}/lhn_${image}.cpio.gz" ]; then
			echo "    Installing lhn_${image}.cpio.gz to ${path_to_mount_point}"
			gunzip -c ${dir_to_images}/lhn_${image}.cpio.gz | cpio --make-directories --no-absolute-filenames --extract --unconditional
		else
			echo "WARNING: Image ${dir_to_images}/lhn_${image}.cpio.gz not found."
		fi
	done
	for file in ${optional_files_to_copy} ; do
		if [ -e "${dir_to_images}/${file}" ]; then
			echo "    Copying ${file} to ${path_to_mount_point}"
			cp -p ${dir_to_images}/${file} ${path_to_mount_point}
		else
			echo "WARNING: Image ${dir_to_images}/${file} not found."
		fi
	done

	cd ${dir_to_images}
	for image in ${lefthand_additions} ; do
		if [ -e "${dir_to_images}/lhn_${image}_${swversion}.tar.gz" ] ; then
			cp ${dir_to_images}/lhn_${image}_${swversion}.tar.gz ${path_to_mount_point}/
		else
			echo "------WARNING:  LHN ${image} image ${dir_to_images}/lhn_${image}_${swversion}.tar.gz does not exist"
		fi
	done

##############################################################
# prepare a script to run from a chrooted environment that
# will finish preping the OS
##############################################################
	cat << EOF_LILOSCRIPT > ${path_to_mount_point}/etc/lhn_template2system_${swversion}.bash
#!/bin/bash
set -e
lhn_bootdisk=${boot_as_drive}
lhn_osversion=${osversion}
lhn_imagedisk=${drive_to_image}
lhn_osversion=${osversion}
lhn_bootdisk=${boot_as_drive}
lhn_remapdisk=${drive_to_image}
lhn_rootpartition=${path_to_raw_os_device}
lhn_bootpartition=${boot_as_drive}${partition_boot_slice}
lhn_varpartition=${path_to_raw_var_device}
lhn_datapartition=${boot_as_drive}2
lhn_raidpartition=${raid_partition}
lhn_swappartition=${boot_as_drive}${partition_swap_slice}
lhn_rootfstype=${filesystem_os}
lhn_bootfstype=${filesystem_boot}
lhn_varfstype=${filesystem_var}
lhn_datafstype=${filesystem_data}
lhn_hostname=${hostname}
lhn_domainname=${domainname}
lhn_serialnumber=${serial_number}
lhn_ipaddress=${ip_address}
lhn_gateway=${gateway}
lhn_netmask=${netmask}
lhn_version=${swversion}
lhn_additions="${lefthand_additions}"
lhn_do_initrd="${do_initrd}"
lhn_disks_to_image="${disks_to_image}"

# Change directories to the root of it all
cd /

# Bug 792 fix
FIXING=/etc/ntp.conf
if [ -e \${FIXING} ]; then
    mv -f \${FIXING} \${FIXING}.orig
    sed -e 's/^server.*$//g' \
		-e 's/^fudge.*$//g' \${FIXING}.orig > \${FIXING}
fi

# Remove the 'src' entry from the passwd file
for afile in passwd shadow ; do
    FIXING=/etc/\${afile}
    if [ -e \${FIXING} ]; then
		mv -f \${FIXING} \${FIXING}.orig
		sed -e 's/^src.*$//g' \${FIXING}.orig > \${FIXING}
    fi
done

# Backup the original /etc/lefthand area (if it exists)
if [ -d /etc/lefthand ]; then
    echo "    Backing up original /etc/lefthand tree ...."
    tar zcf /old_etc_lefthand.tar.gz /etc/lefthand
    echo "    Done"
fi

# Make sure our filesystem mount point is available
for adir in tmp proc storage /var/log/httpd /etc/xinetd.d /etc/rc.d ; do
    if [ ! -d /\${adir} ]; then
		echo "    Creating directory /\${adir}..."
		mkdir -m 0755 -p /\${adir}
    fi
done

# Now, setup the rc directories
for rc in 0 1 2 3 4 5 6 ; do
    if [ ! -d /etc/rc\$rc.d ]; then
		mkdir -p /etc/rc\$rc.d
    fi
    if [ ! -e /etc/rc.d/rc\$rc.d ]; then
		ln -s /etc/rc\$rc.d /etc/rc.d/rc\$rc.d
    fi
done

# Fix permissions on /tmp irregardless if the above occurs.
chmod -R 777 /tmp

# Change directories to the root of it all again
cd /

# Unzip/untar the LHN custom tar images
for image in \${lhn_additions} ; do
    if [ -e /lhn_\${image}_\${lhn_version}.tar.gz ]; then
		echo "    Installing /lhn_\${image}_\${lhn_version} ..."
		tar zxf /lhn_\${image}_\${lhn_version}.tar.gz
    else
		echo "  Error: Missing /lhn_\${image}_\${lhn_version}.tar.gz!!!"
		exit 1
    fi
done

cd /boot 
if [ -e vmlinux ] ; then
	rm vmlinux
fi
ln -s ${kernel_to_use} vmlinux
cd /

if [ ! -h /sbin/dhcpcd ]; then
    cd /
	ln -s /usr/sbin/dhcpcd /sbin/dhcpcd
fi

# Now transpose tokens for actual values in the /etc/lilo template files we just unpacked
echo "Transposing /etc/lilo.conf.template into /etc/lilo.conf.as_imaged ..."
sed \
    -e "s#LHN_OSVERSION#\${lhn_osversion}#g" \
    -e "s#LHN_BOOTDISK#\${lhn_imagedisk}#g" \
    -e "s#LHN_REMAPDISK#\${lhn_imagedisk}#g" \
    -e "s#LHN_ROOTPARTITION#\${lhn_rootpartition}#g" \
    /etc/lilo.conf.template > /etc/lilo.conf.as_imaged
#####
############## now setup boot sequence
#####
if [ \${lhn_do_initrd} == 1 ] ; then
	cdate=20011127
	localdir=/make_initrd
	initrd_template=/lhn_initrd.template.raw.gz
	#
	###### here we go
	#
	if [ ! -e \${initrd_template} ] ; then
		echo "ERROR:  missing requried file \${initrd_template}"
		exit
	fi
	rm -rf \${localdir}*
	mkdir \${localdir}

cat << EOF1 > /make_initrd.files.list.fromos
.
./bin
./bin/cat
./bin/echo
./bin/ls
./bin/mount
./bin/umount
./dev
./etc
./etc/devfsd.conf
./etc/ld.so.conf
#./lib
#./lib/ld-2.1.3.so
##./lib/ld-2.2.4.so
#./lib/ld-linux.so.1
#./lib/ld-linux.so.1.9.11
#./lib/ld-linux.so.2
#./lib/libc-2.1.3.so
#./lib/libc.so.6
#./lib/libcom_err.so.2
#./lib/libcom_err.so.2.0
#./lib/libcrypt-2.1.3.so
##./lib/libcrypt-2.2.4.so
#./lib/libcrypt.so.1
#./lib/libdl-2.1.3.so
##./lib/libdl-2.2.4.so
#./lib/libdl.so.2
#./lib/libhistory.so.4
#./lib/libhistory.so.4.1
#./lib/libpam.so.0
#./lib/libpam.so.0.72
#./lib/libreadline.so.4
#./lib/libreadline.so.4.1
#./lib/libtermcap.so
#./lib/libtermcap.so.2
#./lib/libtermcap.so.2.0.8
./lib/modules/2.4.14
./lib/modules/2.4.14/kernel
./lib/modules/2.4.14/kernel/arch
./lib/modules/2.4.14/kernel/arch/i386
./lib/modules/2.4.14/kernel/arch/i386/kernel
./lib/modules/2.4.14/kernel/arch/i386/kernel/cpuid.o
./lib/modules/2.4.14/kernel/arch/i386/kernel/microcode.o
./lib/modules/2.4.14/kernel/drivers
./lib/modules/2.4.14/kernel/drivers/md
./lib/modules/2.4.14/kernel/drivers/md/linear.o
./lib/modules/2.4.14/kernel/drivers/md/lvm-mod.o
./lib/modules/2.4.14/kernel/drivers/md/md.o
./lib/modules/2.4.14/kernel/drivers/md/multipath.o
./lib/modules/2.4.14/kernel/drivers/md/raid0.o
./lib/modules/2.4.14/kernel/drivers/md/raid1.o
./lib/modules/2.4.14/kernel/drivers/md/raid5.o
./lib/modules/2.4.14/kernel/drivers/md/xor.o
./lib/modules/2.4.14/kernel/drivers/net
./lib/modules/2.4.14/kernel/drivers/net/sis900.o
./lib/modules/2.4.14/misc
./lib/modules/2.4.14/modules.dep
./lib/modules/2.4.14/modules.pcimap
./proc
./sbin/devfsd
./sbin/insmod
./sbin/lsmod
./sbin/raidstart
EOF1


	dd if=/dev/zero of=\${localdir}.raw bs=1024 count=8192
	mke2fs -F \${localdir}.raw
	mount -o loop \${localdir}.raw \${localdir}
	cd /
	cat /make_initrd.files.list.fromos | cpio --make-directories --unconditional -m --pass-through \${localdir}
	if [ -e \${initrd_template} ] ; then
		echo "ERROR:  missing requried file /lhn
	fi
	gunzip -c \${initrd_template} | cpio --make-directories --pass-through --unconditional \${localdir}

	echo "Syncing..."
	sync
	echo "Unmounting..."
	umount \${localdir}.raw
	echo "Syncing..."
	sync
	echo "Compressing..."
	gzip -9 -c \${localdir}.raw > /boot/lhn_initrd.raw.gz
	echo "INITRD ready"

	cd /boot
	#mv /lilo.conf.hd* /boot
	for disk in \${lhn_disks_to_image} ; do
		if [ -e /\${disk} ] ; then
			if [ ! -d /\${disk} ] ; then
				echo "ERROR:  /\${disk} is not a directory...exiting"
				exit 1
			fi
		else
			mkdir /\${disk}
		fi
		find . -print | cpio --pass-through -m --make-directories --unconditional /\${disk}
		echo "running lilo for disk \${disk}"
		lilo -C /\${disk}/lilo.conf.\${disk}
		lilo -C /\${disk}/lilo.conf.\${disk} -v -q | grep -i time
	done
else
	#nothing to do here but run lilo
	lilo -C /etc/lilo.conf.as_imaged
	lilo -C /etc/lilo.conf.as_imaged -v -q | grep -i time
fi

# Now transpose tokens for actual values in the template files we just unpacked
for afile in /etc/fstab /etc/lilo.conf /etc/lefthand/lefthand.nas /etc/sysconfig/network /etc/hosts /etc/inittab /etc/conf.getty.ttyS0 ; do
	echo "Transposing \$afile.template file into \$afile...."
	sed \
		-e "s#LHN_OSVERSION#\${lhn_osversion}#g" \
		-e "s#LHN_BOOTDISK#\${lhn_bootdisk}#g" \
		-e "s#LHN_REMAPDISK#\${lhn_remapdisk}#g" \
		-e "s#LHN_ROOTPARTITION#\${lhn_rootpartition}#g" \
		-e "s#LHN_VARPARTITION#\${lhn_varpartition}#g" \
		-e "s#LHN_DATAPARTITION#\${lhn_datapartition}#g" \
		-e "s#LHN_SWAPPARTITION#\${lhn_swappartition}#g" \
		-e "s#LHN_RAIDPARTITION#\${lhn_raidpartition}#g" \
		-e "s#LHN_ROOTFSTYPE#\${lhn_rootfstype}#g" \
		-e "s#LHN_VARFSTYPE#\${lhn_varfstype}#g" \
		-e "s#LHN_HOSTNAME#\${lhn_hostname}#g" \
		-e "s#LHN_DOMAINNAME#\${lhn_domainname}#g" \
		-e "s#LHN_SERIALNUMBER#\${lhn_serialnumber}#g" \
		-e "s#LHN_IPADDRESS#\${lhn_ipaddress}#g" \
		-e "s#LHN_GATEWAY#\${lhn_gateway}#g" \
		-e "s#LHN_NETMASK#\${lhn_netmask}#g" \
		\$afile.template > \$afile
		rm -f \$afile.template
done

# First clean out any eth0 references included in the default modules.conf file...
for afile in /etc/modules.conf ; do
    if [ -e \$afile ] ; then
		\grep -v eth0 \$afile > \$afile.$$
		mv \$afile.$$ \$afile
    fi
done

# Overwrite the contents of our lhn modified files to any existing files.
for afile in /etc/modules.conf ; do
    if [ -e \$afile.add ] ; then
		echo "Overwriting \$afile.add information to \$afile ...."
		cp \$afile.add \$afile
		rm -f \$afile.add
    fi
done

# Set default timezone 
for afile in localtime ; do
    if [ -e /etc/\$afile ] ; then
		/bin/rm -f /etc/\$afile
    fi
    ln -s /usr/share/zoneinfo/GMT /etc/\$afile
done

# Append the contents of our lhn modified files to any existing files.
for afile in /etc/passwd /etc/shadow /etc/smbpasswd /etc/group ; do
    if [ -e \$afile.add ] ; then
		echo "Appending \$afile.add information to \$afile ...."
		cat \$afile.add >> \$afile
    fi
	if [ -e \${afile}_custom.add ]; then
		if [ ${do_custom_os} == 1 ] ; then
			echo "Appending \${afile}_custom.add information to \$afile ...."
			cat \${afile}_custom.add >> \$afile
    	fi
    fi
done

for afile in /etc/group /etc/lefthand/groups ; do
    echo "Adding group information...."
    awk -F: '{printf("%s:--------:No Description:\n",\$1);}' \$afile > \$afile.info
done

#####################################################
# All rc startup scripts are based off scripts
# found in a single /etc/rc.d/init.d area. 
#####################################################
cd /etc/rc.d/init.d

# Create a few symlinks for LHN scripts....
for alink in hydra storage mkraid ; do
    echo "Installing symlink lhn_\$alink to \$alink ..."
    /bin/rm -f \$alink && ln -s ./lhn_\$alink \$alink
done

# for the runlevels 0 1 2 6 create the necessary symlinks to STOP services
for rc in 0 1 2 6 ; do
    echo "Installing symlinks for runlevel \$rc ..."
    cd /etc/rc\$rc.d
    for alink in K40snmpd K45lhn_aebs_server K50xntpd K55ypbind K60portmap K65lhn_storage K75lhn_hydra; do
		afile=\`echo \$alink | sed 's/^...//'\`
		if [ -e /etc/rc.d/init.d/\$afile ]; then
	    	echo "  Installing \$afile ..."
	    	/bin/rm -f \$alink && ln -s /etc/rc.d/init.d/\$afile \$alink
		else
	    	echo "ERROR: Missing /etc/rc.d/init.d/\$afile"
	    	exit 1
		fi
    done
done

# for the runlevels 2 3 4 5 create the necessary symlinks to START services
for rc in 2 3 4 5 ; do
    echo "Installing symlinks for runlevel \$rc ..."
    cd /etc/rc\$rc.d
    for alink in S25lhn_hydra S45portmap S90snmpd S95xntpd S96ypbind S96lhn_aebs_server S97lhn_storage; do
		afile=\`echo \$alink | sed 's/^...//'\`
		if [ -e /etc/rc.d/init.d/\$afile ]; then
	    	echo "  Installing \$afile ..."
	    	/bin/rm -f \$alink && ln -s /etc/rc.d/init.d/\$afile \$alink
		else
	    	echo "ERROR: Missing /etc/rc.d/init.d/\$afile"
	    	exit 1
		fi
    done
done
echo "CHROOTED script ran successfully"

EOF_LILOSCRIPT

##############################################################
# run the just prepared script in a chrooted environment
##############################################################
    chmod oug+x ${path_to_mount_point}/etc/lhn_template2system_${swversion}.bash
    echo "Changing roots and running '/etc/lhn_template2system_${swversion}.bash'..."
    chroot ${path_to_mount_point} /etc/lhn_template2system_${swversion}.bash
    echo "NSM Installation complete!"
else
    echo "Skipping LHN specifics"
fi

exit
