#!/bin/sh

#
# Copyright (c) 2000 Sun Microsystems Inc.,
# All rights reserved.
#
# File         : cod-monitor.sh
# Version      : 1.13
# Last Checkin : 10/18/02
#

# This shell script monitors the indicated System Controller
# via the COD MIB and appends *new* log messages to a file.
#
# It expects one command line argument - name of the system controller
# An optional second argument - provides the path to a directory to
#     store the log files in.
#
# Two files are created:
#  cod.log --> a list of all messages
#  cod.lastOid --> keeps track of last OID that was logged
#
# It will make one pass (i.e. start from the next message till
# it hits 'no more messages').  It needs to be run periodically
# 

LASTOIDFILE=cod.lastOid
LOGFILE=cod.log
TIMEOUT=60
COMMUNITY=P-public
RETRIES=3
CODSTARTTIMEOID=1.3.6.1.4.1.42.2.70.19.4.0
CODLOGTABLEOID=1.3.6.1.4.1.42.2.70.19.3.1.2
CHASSIS_SYSTEM_SERIAL_NO_OID=1.3.6.1.4.1.42.2.70.1.1.1.25.0

validateOid() {
    TESTOID=$1
    # validate TESTOID, it should start with ${CODLOGTABLEOID}
    echo ${TESTOID} | grep "^${CODLOGTABLEOID}" > /dev/null 2>&1
    return $?
}

getSnmpValue() {
    # get the SNMP value of the indicated $1 OID and return it
    OBJOID=$1
    result=`${SNMPGETCMDLINE} ${OBJOID} 2>/dev/null`
    if [ $? -ne 0 ]
    then
	echo "Cannot communicate with ${scName}"
	return 3
    fi
    if [ "aa${result}" = "aa" ]
    then
	echo "reached unexpected end of mib with ${scName}"
	return 3
    fi
    # make sure there is no error
    errString=`echo "${result}" | grep "^Error: "`
    if [ "${errString}" != "Error: noError" ]
    then
	echo "error reading MIB '${errString}'"
	return 3
    fi
    # Get the value out
    snmpValue=`echo "${result}" | grep "^Value: " | sed -e "s/^Value: //"`
    echo ${snmpValue}
    return 0
}

if [ $# -lt 1 -o $# -gt 3 ]
then
    echo "Usage: $0 <ScName> [System Serial Number [community name]]"
    exit 1
fi

scName=$1
RELOCATIONDIR=`/usr/bin/pkgparam SUNWescom BASEDIR`
BASEDIR=$RELOCATIONDIR/SUNWsymon

check_osver() {
    if [ -z "$1" ]; then
        OSVERS=`/usr/bin/uname -r`
    else
        OSVERS="$1"
    fi

    case $OSVERS in
        5.5.1*) OSVERS=2.5  ;;
        5.6*)   OSVERS=2.6  ;;
        5.7*)   OSVERS=2.7  ;;
        5.8*)   OSVERS=2.8  ;;
        5.9*)   OSVERS=2.9  ;;
        *)      echolog 'Unsupported OS version: $2' "$OSVERS"
                exit 1      ;;
    esac
}

OSVERS=`/usr/bin/uname -r`
check_osver ${OSVERS}

codLogDir=/var/${BASEDIR}/cod
if [ $# -ge 2 ]
then
    pathName=${codLogDir}/$2
    if [ $# -eq 3 ]
    then
	COMMUNITY=$3
    fi
fi
SNMPNEXT=${BASEDIR}/util/bin/sparc-sun-solaris${OSVERS}/snmpnext
SNMPNEXTCMDLINE="${SNMPNEXT} -t ${TIMEOUT} -h ${scName} -c ${COMMUNITY}"

SNMPGET=${BASEDIR}/util/bin/sparc-sun-solaris${OSVERS}/snmpget
SNMPGETCMDLINE="${SNMPGET} -t ${TIMEOUT} -h ${scName} -c ${COMMUNITY}"
if [ $# -eq 1 ]
then
    # ask the SC for its system serial number - that is the pathName
    retVal=`getSnmpValue ${CHASSIS_SYSTEM_SERIAL_NO_OID}`
    if [ $? -ne 0 ]
    then
	echo ${retVal}
	exit $?
    fi
    if [ "aa${retVal}" = "aa" ]
    then
	# we couldn't get a proper value, use the scName itself ...
	pathName=${codLogDir}/${scName}
    else
	# eliminate the "'s in "P1.0-B70D05" and build
	# /var/opt/SUNWsymon/cod/P1.0-B70D05
	pathName=`eval echo ${codLogDir}/${retVal}`
    fi
fi

if [ ! -d ${pathName} ]
then
    mkdir -p ${pathName} > /dev/null 2>&1
    if [ $? -ne 0 ]
    then
	echo "unable to create directory ${pathName}"
	exit 1
    fi
fi

# Make sure we can write a file in the given directory
touch ${pathName}/testing > /dev/null 2>&1
if [ $? -ne 0 ]
then
    echo "unable to write to ${pathName}"
    exit 1
fi
/bin/rm -f ${pathName}/testing

# First, we need to find the current boottime of the SC.  If the
# boot time does not match the time recorded in LASTOIDFILE, then
# we do not want to use the LASTOIDFILE
# Second we need to find the last OID that we had successfully
# read.  This is contained in the file ${pathName}/cod.lastOid
# if the file does not exist, start from the beginning

# Find the last boottime from the SC
scStartTime=`getSnmpValue ${CODSTARTTIMEOID}`
lastOid=${CODLOGTABLEOID}

# find the starttime & lastOid from the file, if any
#
# LASTOIDFILE has 3 lines as follows:
# CodStartTime: xxxxxx
# LastOid: 1.3.4.......
# ScName: XXXXX
#
if [ -f ${pathName}/${LASTOIDFILE} ]
then
    lastStartTime=`grep "^CodStartTime: " ${pathName}/${LASTOIDFILE} | sed -e "s/^CodStartTime: //"`
    if [ "${scStartTime}" = "${lastStartTime}" ]
    then
	# we can trust the lastOid in the file, get it out
	thisOid=`grep "^LastOid: " ${pathName}/${LASTOIDFILE} | sed -e "s/^LastOid: //"`
	validateOid "${thisOid}"
	if [ $? -eq 0 ]
	then
	    # OKAY, file contains a good lastOid, use it
	    lastOid="${thisOid}"
	fi
    fi
fi
lastStartTime="${scStartTime}"

while :
do
    result=`${SNMPNEXTCMDLINE} ${lastOid} 2>/dev/null`
    if [ $? -ne 0 ]
    then
	# WAIT: there doesn't seem to be a valid exit status being returned by snmpnext
	echo "Cannot communicate with ${scName}"
	exit 3
    fi
    if [ "aa${result}" = "aa" ]
    then
	echo "reached end of mib with ${scName}"
	break;
    fi
    # result should look like:
    # Request Id: 0
    # Error: noError
    # Index: 0
    # Count: 1
    #
    # Name: 1.3.6.1.4.1.42.2.70.19.3.1.2.2
    # Kind: OctetString
    # Value: "Mon Aug 14 10:42:51 PDT 2000;966274971.516;P1.0-B70D05;P1.0-B70D05;01;0;1;0017: SC codd started;NaPip2ZMi2ow8Vz/5zEGjQ"

    # Note: when getting the very first message, the entire
    # table is returned by snmpnext, throw away everything but
    # the first result

    # make sure there is no error
    errString=`echo "${result}" | grep "^Error: " | head -1`
    if [ "${errString}" = "Error: noSuchName" ]
    then
	echo "reached end of mib with ${scName}"
	break;
    fi
    if [ "${errString}" != "Error: noError" ]
    then
	echo "error reading MIB '${errString}'"
	break;
    fi

    #
    # break it up into the two parts,
    #    add the message to the log file
    #    update lastOid file
    # 
    thisOid=`echo "${result}" | grep "^Name: " | head -1 | sed -e "s/^Name: //"`

    # Make sure that thisOid is still in the logtable range ...
    validateOid ${thisOid}
    if [ $? -ne 0 ]
    then
	# Ha, we walked out of the codLogTable
	echo `date "+%D %R`
	break;
    fi

    # Get the message out and append it to the log file
    # eliminate thisOid, then '= "' and finally last '"'
    # This ensures that we will not mess up if the message contains
    # either '=' or '"' etc
    #
    message=`echo "${result}" | grep "^Value: \"" | head -1 | sed -e "s/^Value: \"//" -e "s/\"$//"`
    echo "${message}" >> ${pathName}/${LOGFILE}

    # save this OID & lastStartTime in the lastOidFile
    lastOid=${thisOid}
    echo "CodStartTime: ${lastStartTime}" > ${pathName}/${LASTOIDFILE}
    echo "LastOid: ${lastOid}" >> ${pathName}/${LASTOIDFILE}
    echo "ScName: ${scName}" >> ${pathName}/${LASTOIDFILE}
done

