#!/usr/bin/perl

use strict;

package Logger::Term;
use Carp qw(confess);

sub new {
	my $self = {};
	my $class = shift;
	my $file = shift;
	$self->{fh} = "fh";

	no strict "refs";
	open $self->{fh}, ">>$file" or die "Can't open output log file $file: $!";
	use strict "refs";

	bless $self, $class;
}

sub log {
	my $self = shift;
	my $fh = $self->{fh};
	my $level;
	my $msg;
	if ( @_ == 1){
		$msg = shift;
		$level = 'err';
	}elsif ( @_ == 2){
		$level = shift;
		$msg = shift;
	}else{
		confess "Bad number of args";
	}
	chomp(my $date = `date +"%b %e %T"`);
	no strict "refs";
	select ((select($fh), $| = 1)[0]);
	print $fh "$date [$$]: $msg\n";
	use strict "refs";
}

package Logger::Sys;
use lib qw(/opt/SUNWs3khc/lib/perl ./lib/perl);
use Sys::Syslog qw(:DEFAULT setlogsock);
use Carp qw(confess);

sub new {
	my $self = {};
	my $class = shift;
	$self->{tag} = shift;
	my $output = shift;
	$self->{pid} = $$;
	$self->{output} = $output;
	setlogsock ('unix');
	bless $self, $class;
}

sub log {
	my $self = shift;
	my $output = $self->{output};
	my $level;
	my $msg;
	if ( @_ == 1){
		$msg = shift;
		$level = 'err';
	}elsif ( @_ == 2){
		$level = shift;
		$msg = shift;
	}else{
		confess "Bad number of args";
	}
	my @array = split (/\n/, $msg);
	openlog ($self->{tag}, 'cons,pid', 'user');
	for (@array){
		syslog ($level, "%s", $_);
	}
	$output->log("$msg");
	closelog();
}

package Device;
use lib qw(/opt/SUNWs3khc/lib/perl ./lib/perl);
use Date::Parse;

sub new {

	my $self = {};
	my $class = shift;
	$self->{device} = shift;
	bless $self, $class;

	$self->{disks} = [];
	$self->{diskch} = [];

	$self->{badCounts} = 0;

	return -1 if $self->getUID == -1;
	return -1 if (my $lesb = $self->getChannelCounts()) == -1;

	$self->{iommsg} = "Consider replacing the Sun StorEdge 3510 RAID controller";
	$self->{bkplmsg} = "Check the Sun StorEdge 3510 expansion FC Jumper Cables";

	# Set baselines
	for (@main::linkHeaders){
		$self->{base}{$_} = $lesb->{$_};
	}

	return -1 if (my $chan = $self->runCli("show channels")) == -1;
	for ( split /\n/, $chan){
		next unless /^ \d/;
		my ($ch, $type) = unpack "x A3 A8", $_;
		$ch  =~ s/^\s*//g;  # Strip leading white space
		$type =~ s/^\s*//g; # Strip leading white space
		push @{$self->{diskch}}, $ch if $type eq "Drive";
	}

	for (@main::rdch){
		my ($ch1, $ch2) = split (/-/, $_);

		$ch1 =~ s/\s*$//g;
		$ch1 =~ s/^\s*//g;
		$ch2 =~ s/\s*$//g;
		$ch2 =~ s/^\s*//g;

		if ( $ch1 == $ch2 and ((grep /$ch1/,  @{$self->{diskch}})  or (grep /$ch2/, @{$self->{diskch}})) ){
			$self->log("Channels $ch1 and $ch2 cannot be the same in Redundant Channel Map");
			return -1;
		}
		if ( (grep /$ch1/,  @{$self->{diskch}}) and (grep /$ch2/, @{$self->{diskch}}) ){
			$self->{rdch}{$ch1} = $ch2;
			$self->{rdch}{$ch2} = $ch1;
		}
		if (  (grep /$ch1/,  @{$self->{diskch}}) and !(grep /$ch2/, @{$self->{diskch}}) or
			 !(grep /$ch1/,  @{$self->{diskch}}) and (grep /$ch2/, @{$self->{diskch}}) 
		   ){
			$self->log("Channels $ch1 and $ch2 not both disk channels in Redundant Channel Map");
			return -1;
		}
	}

	
	$self;
}


sub parseEvent {

	my $self = shift;
	$self->debug("parseEvent: Enter");
	my $event = shift;
	my $date = $event->{date};
	my $mesg = $event->{mesg};

	# sprintf(ret,"#%d: %s\n",index,fmt); (build_fmt(), evt_interpreter.c)
	# interpreter_events(), evt_opt.c

	# printf("\n%s[%.2X%.2x] ", ctime((time_t *)&ptr->evt[i].event_timestamp),
	# p01->EventTypeCode,p01->ErrorCode);
	# evt_interpreter_type01(p01, evt_severity,evt_source, evt_idx, unit_id, strlen(unit_id));

	# evt_interpreter_type01(), evt_interpreter.c
	# build_fmt(index,msg,fmt);
	# evt_printf(fmt,unit_id);

	# [0181] #1: StorEdge Array SN#8900145 Controller NOTICE: controller initialization completed

	unless ( $mesg =~ /\[(....)\]\s+#(\d+):\s+(.*)/ ) {
		warn "parseEvent: Event parse fail: \n$mesg";
		return { code => "", idx => "", text => "" };
	}

	my ( $code, $idx, $text) = ($1, $2, $3);

	$self->debug("mesg = $mesg, code = $code, idx = $idx, text = $text");

	$self->debug("parseEvent: Leave");
	{ code => $code, idx => $idx, text => $text };
}

sub checkEvents {

	my $self = shift;
	$self->debug("checkEvents: Enter");

	return -1 if (my $events = $self->getEvents()) == -1;

	if ( scalar @$events <= 0 ){
		$self->log("Event: No Events found");
		return;
	}

	my $resetCounter = 1;
	for (@$events){
		if ( str2time($_->{date}) > str2time($self->{lastevent}) ){
			$self->log("Event: $_->{date}: $_->{mesg}");
			my $evt = $self->parseEvent($_);
			$self->debug("code = $evt->{code}, idx = $evt->{idx}, text = $evt->{text}");
			if ( $evt->{code} eq "0181" and $resetCounter){ # Controller NOTICE: controller initialization completed
				$self->log("Controller initialization detected, clearing LESB counters");
				my $lesb = $self->getChannelCounts();
				for (@main::linkHeaders){
					$self->{base}{$_} = $lesb->{$_};
				}
				$resetCounter = 0;
			}
		}
	}
	
	my $nev = scalar @$events - 1;
	$self->{lastevent} = $events->[$nev]{date};

	$self->debug("checkEvents: Leave");

}

sub initEvents {

	my $self = shift;
	$self->debug("initEvents: Enter");

	return -1 if (my $events = $self->getEvents()) == -1;

	if ( scalar @$events <= 0 ){
		$self->log("Event: No Events found");
		return;
	}

	my $nev = scalar @$events - 1;

	$self->{lastevent} = $events->[$nev]{date};

	for (@$events){
		$self->log("Event: $_->{date}: $_->{mesg}");
	}
	$self->debug("initEvents: Leave");
}


sub getEvents {

	my $self = shift;
	return -1 if (my $events = $self->runCli("show events")) == -1;

	my @events = split /\n\n/, $events;

	my $events = [];

	# The output of show events includes stderr.  
	# Shift this off

	shift @events;

	for (@events){
		my ($date, $mesg) = split (/\n/, $_); 
		push @$events, {date => $date, mesg => $mesg};
	}

	$events;
}

sub setBaseCounter {

	my $self = shift;
	$self->debug("setBaseCounter: Enter");
	my $param = shift;

	die "LESB counts not defined!" unless defined $self->{lesb};
	my $lesb = $self->{lesb};
#	return -1 if (my $lesb = $self->getChannelCounts()) == -1;

	$self->{base}{$param} = $lesb->{$param};
	$self->debug("setBaseCounter: Leave");
}

sub defLESB {


	my $self = shift;
	$self->debug("defLESB: Enter");

	return -1 if (my $lesb = $self->getChannelCounts()) == -1;
	$self->{lesb} = $lesb;

	$self->debug("defLESB: Leave");
}

sub undefLESB {

	my $self = shift;
	$self->debug("undefLESB: Enter");
	undef $self->{lesb};
	$self->debug("undefLESB: Leave");
}

sub checkDelta {

	my $self = shift;
	$self->debug("checkDelta: Enter");
	my $param = shift;
	my $ErrorCount = shift;
	my $lesb = $self->{lesb};

	my $current = $lesb->{$param};
	my $base = $self->{base}{$param};

	my $delta = {};

	my @ch = sort {$a <=> $b} keys %$current;

	foreach my $ch (@ch){
		my @id = sort {$a <=> $b} keys %{$current->{$ch}};
		foreach my $id (@id){
			my $value = $current->{$ch}{$id};
			if ( exists $base->{$ch}{$id} ){
				$self->debug("Diff $param: $ch, $id: current: $current->{$ch}{$id}, base: $base->{$ch}{$id}");
				$delta->{$ch}{$id} = $current->{$ch}{$id} - $base->{$ch}{$id};
				if ( $delta->{$ch}{$id} < 0 ){
					$self->log("Warning: link count decreased ch: $ch, id: $id, setting delta to zero");
					$delta->{$ch}{$id} = 0;
				}
				if ( $delta->{$ch}{$id} >= $ErrorCount ){
					$self->log("Alert: $param ErrorCount reached: ch: $ch, id: $id, ".
						"Threshold: $ErrorCount, current: $delta->{$ch}{$id}");
					$self->{badCounts} = 1;
				}
			}else{
				$self->debug("Diff $param: $ch, $id: current: $current->{$ch}{$id}");
				$self->log("Warning: New device detected, ch: $ch, id: $id, setting delta to zero");
				$delta->{$ch}{$id} = 0;
			}
		}
	}

	# This not needed
	#$self->{lesb}{delta}{$param} = $delta;
	$self->debug("checkDelta: Leave");
}


sub checkSES {

	my $self = shift;
	return -1 if $self->getSES == -1;

	for ( @{$self->{ses}} ){
		$self->log("SES: $_->{channel}, $_->{id}, $_->{chassis}");
	}
}

sub getSES {

	my $self = shift;
	return -1 if (my $ses = $self->runCli("show ses")) == -1;

	@{$self->{ses}} = ();

	for ( split /\n/, $ses ){
		next unless /^[ |\d]\d /;
		my ($ch, $id, $chassis) = unpack "a2xa3xa6", $_;
		push @{$self->{ses}}, 
			{channel => $ch, id => $id, chassis => $chassis};
	}
}

sub getUID {

	my $self = shift;
	return -1 if (my $uid = $self->runCli("show unique-identifier",1)) == -1;

	#unique-identifier: 004504

	for ( split /\n/, $uid ){
		next unless /unique-identifier/;
		$uid = (split (/:/, $_))[1];
		$uid =~ s/\s*$//g;
		$uid =~ s/^\s*//g;
	}
	$self->{uid} = $uid;
}

sub checkDiskData {

	my $self = shift;
	$self->debug("checkDisk: Enter");

	# s3khc_0034BC_042004.log --
	chomp (my $date = `/bin/date +%D_%T`);
	$date =~ s/\///g;
	$date =~ s/\://g;
	$self->{logprefix} = "$::config{logdir}/s3khc_$self->{uid}_$date";

	my $LOGSELECT = "4c 00 00 00 00 00 00 00 0e 00";
	my @LOGSENSE = ( "4d 00 3d 00 00 00 03 ff ff 00",
	                 "4d 00 3d 00 00 03 00 ff ff 00" );
	my @paramlist = (
		"3d 00 00 0a 00 01 00 06 00 00 01 00 00 00",
		"3d 00 00 0a 00 01 00 06 00 00 02 00 00 00",
		"3d 00 00 0a 00 01 00 06 00 00 03 00 00 00",
		"3d 00 00 0a 00 01 00 06 00 00 03 00 00 00",
		"3d 00 00 0a 00 01 00 06 00 00 04 00 00 00",
		"3d 00 00 0a 00 01 00 06 00 00 05 00 00 00",
		"3d 00 00 0a 00 01 00 06 00 00 06 00 00 00",
		"3d 00 00 0a 00 01 00 06 00 00 07 00 00 00",
		"3d 00 00 0a 00 01 00 06 00 00 08 00 00 00",
	);

	$self->getDisks();

	my $ndisks = scalar @{$self->{disks}};
	
	$self->debug("checkDisk: number of disks: $ndisks");

	die "Can't create temporary file" if ( (my $tmpfile = main::mktmp($self->{uid})) == -1);
	
	foreach my $disk ( @{$self->{disks}} ){
		my ($ch, $id, $vendor, $product, $rev, $serial) =  
			($disk->{ch}, $disk->{id}, $disk->{vendor},
			 $disk->{product}, $disk->{rev}, $disk->{serial});
		if ( $vendor ne "SEAGATE "){
			$self->log("checkDisk: Skipping non-SEAGATE drive: vendor: \"$vendor\",  ch: $ch, id: $id, serial: $serial");
			next;
		}
		# Do the "complete" log sense command
		my $outfile = "$self->{logprefix}_${ch}_${id}_00_$serial";
		my $logsense = $LOGSENSE[0];
		if ((my $sense = $self->runPassThru($ch, $id, $logsense, "in", 65535, $outfile )) == -1){
			$self->log("Log Sense command failed, ch: $ch, id: $id, serial: $serial");
		}

		foreach my $param (@paramlist){
			# Write the parameter to the temp file in binary
			open LOGS, ">$tmpfile" or die "Can't open file $tmpfile: $!";
			my @bytes = split " ", $param;
			my $byte = $bytes[10];
			for (@bytes){
				print LOGS pack "H2", $_;
			}
			close LOGS;
			my $cdb = "$LOGSELECT";
	
			chomp (my $od = `od -x $tmpfile`);
	
			$self->debug("checkDisk: Input buffer file: $od");

			if ((my $logselect = $self->runPassThru($ch, $id, $cdb, "out", 14, $tmpfile )) == -1){
				$self->log("Log Select command failed, ch: $ch, id: $id, serial: $serial");
				next;
			}

			my $outfile = "$self->{logprefix}_${ch}_${id}_${byte}_$serial";
			my $logsense = $LOGSENSE[1];
		
			if ((my $sense = $self->runPassThru($ch, $id, $logsense, "in", 65535, $outfile )) == -1){
				$self->log("Log Sense command failed, ch: $ch, id: $id, serial: $serial");
			}
		}
	}

	unlink $tmpfile;

	$self->debug("checkDisk: Leave");
}


sub checkFruData {

	my $self = shift;
	my $frudata =  shift; 

	$self->debug("checkFruData: Enter");

	my %fruTable = (
		CtrlDate =>
			{
				name => "FC_RAID_IOM",
				testval => $main::config{CtrlDate},
				param => "Manufacturing Date",
				compare => sub {str2time ($_[0]) < str2time ($_[1])},
				msg => $self->{iommsg},
			},
		CtrlRev =>
			{
				name => "FC_RAID_IOM",
				testval => $main::config{CtrlRev},
				param => "Revision",
				compare => sub { $_[0] < $_[1] },
				msg => $self->{iommsg},
			},
		ChassisDate =>
			{
				name => "FC_CHASSIS_BKPLN",
				testval => $main::config{ChassisDate},
				param => "Manufacturing Date",
				compare => sub {str2time ($_[0]) < str2time ($_[1])},
				msg => $self->{bkplmsg},
			},
		ChassisRev =>
			{
				name => "FC_CHASSIS_BKPLN",
				testval => $main::config{ChassisRev},
				param => "Revision",
				compare => sub { $_[0] < $_[1] },
				msg => $self->{bkplmsg},
			}
	);


	die "FRU data $frudata doesn't exist in FRU table" unless exists $fruTable{$frudata};

	my $name = $fruTable{$frudata}{name};
	my $testval = $fruTable{$frudata}{testval};
	my $param = $fruTable{$frudata}{param};
	my $compare = $fruTable{$frudata}{compare};
	my $msg = $fruTable{$frudata}{msg};

	$self->debug("checkFruData: frudata: $frudata, name: $name, ".
			"testval: $testval, param: $param, msg: $msg, compare: $compare");

	my $retval = 0;

	foreach my $fru ( @{$self->{fru}} ){
		next unless $fru =~ / Name: $name/;
		my @data = split (/\n/, $fru);
		my $serial = "";
		foreach my $datum  (@data){
			if ( $datum =~ /^ Serial/ ){
				$serial = (split /:/, $datum, 2)[1];
				$serial =~ s/\s*$//g; $serial =~ s/^\s*//g;
				$self->debug("checkFruData: serial number: $serial");
			}
			if ( $datum =~ /^ $param/ ){
				die "serial number not defined for FRU data" if $serial eq "";
				if ( $name eq "FC_CHASSIS_BKPLN" and $self->{uid} eq "$serial" ){
					$self->debug("Skipping $name, param: $param  for Raid controller, serial: $serial");
					next;
				}
				my $val = (split /:/, $datum, 2)[1];
				$val =~ s/\s*$//g; $val =~ s/^\s*//g;
				$self->debug("checkFruData: value: $val");
				if ( $compare->($val, $testval) ){
					$self->sysLog("Alert: Serial: $serial, $msg") if $frudata eq "ChassisDate";
					$retval = -1 if $frudata eq "CtrlRev";
				}else{
					$self->log("FRU: $name, serial: $serial, Param: $param, found: $val, OK");
				}
			}
		}
	}

	$self->debug("checkFruData: Leave");
	return $retval;
}


sub getFru {

	my $self = shift;
	$self->debug("getFru: Enter");
	return -1 if (my $fru = $self->runCli("show fru")) == -1;

	my @frudata = split /\n\n/, $fru;

	for (@frudata){
		push @{$self->{fru}}, $_ if /^ Name/;
	}
	$self->debug("getFru: Leave");
}

sub getDisks {

	# ift_display_show_disks_output(), 
	# "%2d %3d%10s          %-6s %-8.8s %28.28s %s\n",

	# Serial number field.
	# printf ("                                              S/N %s\n",

	my $self = shift;

	$self->debug("getDisks: Enter");

	return -1 if 
		(my $disks = $self->runCli("show disks")) == -1;

	$self->{disks} = [];

	my @disks = split /\n/, $disks;

	$self->debug("Number of disks: $#disks");
#	for ( split /\n/, $disks ){

	for ( my $i = 0; $i <= $#disks; $i++){
		next unless $disks[$i] =~ /^[ |\d]\d /;
		my $idx = $i + 1;  # Look ahead to get the next serial number.
		my ($ch, $id, $inq) = unpack "A2 x A3 \@42 a29", $disks[$i];
		$ch  =~ s/^\s*//g;  # Strip leading white space
		$id  =~ s/^\s*//g;  # Strip leading white space

		my $serial = (split (" ", $disks[$idx]))[1];
		
		my ($vendor, $product, $rev) = unpack "a8 a16 x a4", $inq;
		push @{$self->{disks}}, 
			{ch => $ch, id => $id, vendor => $vendor, product => $product, rev => $rev, serial => $serial};
		$self->debug("ch: $ch, id: $id, vendor: $vendor, product: $product, rev: $rev, serial: $serial");

	}

	$self->dumpDisks() if $::debug;

	$self->debug("getDisks: Leave");
}

sub dumpDisks {

	my $self = shift;
	$self->debug("dumpDisks: Enter");

	for (@{$self->{disks}}){
		$self->debug("dumpDisks: ch: $_->{ch}, id: $_->{id}, ".
			"vendor: $_->{vendor}, product: $_->{product}, ".
			"rev: $_->{rev}, serial: $_->{serial}");
	}

	$self->debug("dumpDisks: Leave");

}

sub getChannelCounts {

	my $self = shift;
	$self->debug("getChannelCounts: Enter");

	my @ch = ();
	my %lesb = {};

	# " %-3d%-8s%-8s%-8s%-8s"
	return -1 if 
		(my $chan = $self->runCli("show channels")) == -1;
	for ( split /\n/, $chan){
		next unless /^ \d/;
		my ($ch, $type) = unpack "x A3 A8", $_;
		$ch  =~ s/^\s*//g;  # Strip leading white space
		$type =~ s/^\s*//g; # Strip leading white space
		push @ch, $ch if $type eq "Drive";
	}

	# "%2d %3d  %-6s%-6d%-8d %-8d %-8d %-8d %-8d %-8d\n";

	foreach $chan (@ch){
		return -1 if 
			(my $lesb = $self->runCli("diag error channel $chan target all")) == -1;

		for ( split /\n/, $lesb ){
			next unless /^[ |\d]\d /;
			my ($ch, $id, $type, $lip, $link, $sync, $sig, $prim, $txw, $crc) = 
				unpack "A2 x A3 x2 A6 A6 A8 x A8 x A8 x A8 x A8 x A8", $_;
			$ch  =~ s/^\s*//g;  # Strip leading white space
			$id  =~ s/^\s*//g;  # Strip leading white space
			die "Didn't get channel asked for, wanted: $chan, got: $ch" if $chan != $ch;
			$lesb{LIP}{$ch}{$id}      = $lip;
			$lesb{LinkFail}{$ch}{$id} = $link;
			$lesb{LossOfSy}{$ch}{$id} = $sync;
			$lesb{LossOfSi}{$ch}{$id} = $sig;
			$lesb{PrimErr}{$ch}{$id}  = $prim;
			$lesb{InvalTxW}{$ch}{$id} = $txw;
			$lesb{InvalCRC}{$ch}{$id} = $crc;
		}
	}

	$self->dumpChannelCounts(\%lesb) if $::debug;
	$self->debug("getChannelCounts: Leave");
	\%lesb;
}

sub dumpChannelCounts {

	my $self = shift;
	my $lesb = shift;

	$self->debug("dumpChannelCounts: Enter");

	my @param = sort keys %$lesb;

	my %tmphash;

	foreach my $param  (@param){
		my @ch = sort {$a <=> $b} keys %{$lesb->{$param}};
		foreach my $ch (@ch){
			my @id = sort {$a <=> $b} keys %{$lesb->{$param}{$ch}};
			foreach my $id  (@id){
				$tmphash{$ch}{$id}{$param} = $lesb->{$param}{$ch}{$id};
			}
		}
	}

	my @ch = sort {$a <=> $b} keys %tmphash;

	foreach my $ch  (@ch){
		my $tmphash = $tmphash{$ch};
		my @id = sort {$a <=> $b} keys %$tmphash; 
		foreach my $id  (@id){
			my $counts = $tmphash->{$id};
			$self->log("CH: $ch, ID: $id, ".
				"LIP: $counts->{LIP}, ".
				"LinkFail: $counts->{LinkFail}, ".
				"LossOfSy: $counts->{LossOfSy}, ".
				"LossOfSi: $counts->{LossOfSi}, ".
				"PrimErr: $counts->{PrimErr}, ".
				"InvalTxW: $counts->{InvalTxW}, ".
				"InvalCRC: $counts->{InvalCRC}, "
			);
		}
	}

	$self->debug("dumpChannelCounts: Leave");
}

sub runCli {

	my $self = shift;
	my $cmd = shift;
	my $fail = shift;
	my $password = "";
	if ($main::config{password}){
		$password = "--password=$main::config{password}";
	}
	return main::runCmd("$main::config{cli} $password $self->{device} $cmd < /dev/null", $fail);

}

sub runPassThru {

	my $self = shift;
	my $ch = shift;
	my $id = shift;
	my $cmd = shift;
	my $dir = shift;
	my $len = shift;
	my $file = shift;
	
	my $doit = sub {
		my $ch = shift;
	
		my $password = "null";

		if ($::config{password}){
			$password = "$::config{password}";
		}
		my $result = main::runCmd("$main::config{passthru} $self->{device} $ch.$id \"$cmd\" $dir $len $file $password", 1);

		# ====== pco ====== len(0x040)
		# 00 00 02 01 FF FF 05 24 00 00 00 00 00 00 00 00 .......$........
		# 4D 00 3D 00 00 03 00 FF FF 00 00 00 00 00 00 00 M.=.............
		# 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 ................
		# 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 ................

		my @result = split (/\n/, $result);
		shift @result;  # Shift off ==== pco ====

		my @str = ();
		for (@result){
			my $str = substr $_, 0, 47;
			push @str, $str;
		}
		my $str = join (' ', @str);
		my $cdb = substr $str, 48, 47;

		$self->debug("runPassThru: cdb = \"$cdb\"");
		return ( (split " ", $str, 10)[0..8], "$cdb");
	};

	# NB This stuff is ASCII hex use string comparisons

	my $retry = 1;
	my $maxretries = 2;

	my  ( $status, $cmd_status, $tgt_status, $valid, $lenmsb, $lenlsb, $sense, $asc, $ascq, $cdb );

TRY:{


		( $status, $cmd_status, $tgt_status, $valid, $lenmsb, $lenlsb, $sense, $asc, $ascq, $cdb ) = $doit->($ch);

		$self->debug("runPassThru: stat: $status, cmd_stat: $cmd_status, tgt_stat: $tgt_status, ".
			"lenmsb: $lenmsb, lenlsb: $lenlsb, sense: $sense, asc: $asc, ascq: $ascq, cdb: \"$cdb\"");

		return 0 if ( $status eq "00" and $cmd_status eq "00" and $tgt_status eq "00");

		if ( $tgt_status eq "02" and  $valid and $sense eq "05" and $asc eq "24" ){
			$self->log("runPassThru: Expected Check Condition $sense/$asc/$ascq : ch: $ch, id: $id, cdb: $cdb");
			die "Can't open log file $file" unless open LOG, ">$file";
			print LOG "runPassThru: Expected Check Condition $sense/$asc/$ascq : ch: $ch, id: $id, cdb: $cdb\n";
			close LOG;
			return -1;
		}

		if ( $tgt_status eq "02" and $valid ){
			$self->log("runPassThru: Check Condition $sense/$asc/$ascq : ch: $ch, id: $id, cdb: $cdb");
			return -1;
		}

		if ( $retry < $maxretries ){
			$retry++;
			die "No redundant channel for channel \"$ch\"" unless exists $self->{rdch}{$ch};
			my $newch = $self->{rdch}{$ch};
			die "Redundant channel \"$newch\" not a disk channel" unless grep /^$newch$/, @{$self->{diskch}};

			$self->log("runPassThru: ch: $ch, id: $id pass through command failed, trying ch $newch");
			$self->log("runPassThru: stat: $status, cmd_stat: $cmd_status, tgt_stat: $tgt_status, ".
				"lenmsb: $lenmsb, lenlsb: $lenlsb, sense: $sense, asc: $asc, ascq: $ascq, cdb: \"$cdb\"");
		
			$ch = $newch;
			redo TRY;
		}
	}

	$self->log("runPassThru: ch: $ch, id: $id pass through command failed");
	$self->log("runPassThru: stat: $status, cmd_stat: $cmd_status, tgt_stat: $tgt_status, ".
		"lenmsb: $lenmsb, lenlsb: $lenlsb, sense: $sense, asc: $asc, ascq: $ascq, cdb: \"$cdb\"");

	return -1;
}

sub log {

	my $self = shift;
	my $msg = shift;
	my @array = split (/\n/, $msg);
	for (@array){
		$::logger->log("$self->{uid}: $_");
	}
}

sub sysLog {

	my $self = shift;
	my $msg = shift;
	my @array = split (/\n/, $msg);
	for (@array){
		$::sysLogger->log("$self->{uid}: $_");
	}
}

sub debug {
	my $self = shift;
	my $msg = shift;
	return unless $::debug;
	$self->log($msg);
}

package main;

use vars qw($PRODUCT $logger $sysLogger $debug %config @linkHeaders @rdch @tmpfiles $diskinfo);
use Carp qw(:DEFAULT cluck);
use Getopt::Long;
use File::Basename;
use Date::Parse;
use POSIX;

$PRODUCT = "Sun StorEdge 3510";

@tmpfiles = ();
@linkHeaders = qw(LIP LossOfSy LossOfSi PrimErr InvalTxW InvalCRC);

sub dieHandler {
	my $msg = shift;
	unlink @tmpfiles;
	my $longmess = Carp::longmess($msg);
	$sysLogger->log('err', "Script Error (exiting)");
	$logger->log('err', "Script Error (exiting): $longmess");
	exit(-1);
}

sub warnHandler {
	my $msg = shift;
	my $longmess = Carp::longmess($msg);
	$sysLogger->log('err', "Script Warning (continuing)");
	$logger->log('err', "Script Warning (continuing): $longmess");
}

sub termHandler {
	my $signal = shift;
	unlink @tmpfiles;
	$sysLogger->log('err', "Script Shutdown from $signal signal");
	exit(-1);
}


sub debug {
	my $msg = shift;
	return unless $::debug;
	$logger->log($msg);
}


sub mktmp {

	my $uid = shift;
	my $getfile = sub {

		chomp (my $date = `/bin/date +%D_%T`);
		$date =~ s/\///g;
		$date =~ s/\://g;
		my $tmp = "/tmp/s3khc_tmp_${uid}_${$}_$date";

		if ( -e $tmp ){
			return -1;
		}
		return $tmp;
	};

	my $retry = 1;
	my $maxretries = 15;

TRY:{

		my $tmpfile = $getfile->();

		unless ( $tmpfile == -1 ){
			push @tmpfiles, $tmpfile;
			return $tmpfile;  
		}

		if ( $tmpfile == -1 and ($retry < $maxretries)) {
			$retry++;
			sleep 1;
			redo TRY;
		}
		return -1;
	}
}



sub runCmd {

	my $cmd = shift;
	my $fail = shift;

	debug("runCmd: Enter");
	debug("runCmd: cmd: $cmd");

	local $SIG{ALRM} = sub { die "Cli command timeout: $cmd" };
	alarm 900;

	my $retry = 1;
	my $maxretries = 15;

	my $result;

TRY:{
		$result =  `$cmd 2>&1`;
		my $status = $? >> 8;
		debug ("run command: $cmd\nresult:\n$result");

		if ( $status == 0){
			alarm 0;
			return $result;
		}

		if ( ($status != 0) and ($retry < $maxretries)) {
			$retry++;
			chomp $result;
			debug("Command Failed: $! $result, retrying\n");
			redo TRY;
		}
	}# TRY

	#
	# This will end the script
	#
	alarm 0;
	return -1 if $fail;
	chomp $result;
	die "Failed to run command: $cmd: $!\nresult: $result";

	debug("runCmd: Leave");
}

sub runit {

	my $devObjs = shift;

	my $lesb = shift;
	my $event = shift;
	my $fru = shift;

	my %lesb = %$lesb;
	my %event = %$event;

	die "No systems to monitor" if @$devObjs == 0;

	$sysLogger->log("$::PRODUCT Health Check Initializing");

	my $runtime = $config{runtime};

	my $idx = 0;
	my @idxlist = ();
	my @devObjs = ();

	foreach my $dev (@$devObjs){
		$dev->initEvents();
		if ($diskinfo){
			$dev->sysLog("Checking Disk Drives");
			$dev->checkDiskData();
		}
		$dev->getFru();
		$dev->sysLog("Checking Sun StorEdge 3510 RAID Controller Rev");
		my $retval = $dev->checkFruData("CtrlRev");
		$dev->sysLog("Checking Sun StorEdge 3510 FC Expansion Cables");
		$dev->checkFruData("ChassisDate");
		$dev->dumpChannelCounts($dev->getChannelCounts());
		if ( $retval == -1 ){
			debug("Added on index $idx to check LESB counts");
			push @idxlist, $idx;  # these are out of rev controllers to check
		}
	}
	
	@devObjs = @$devObjs[@idxlist]; # Slice of controller to check

	for (;;){

		debug("Wakeup");
		
		foreach my $dev (@devObjs){

			my $lesbCheck = 0;

			debug("EventPeriod: $event{EventPeriod}, EventCurrent: $event{EventCurrent}");

			unless ( $event{EventPeriod} == 0) {
				if ( $event{EventPeriod} == $event{EventCurrent}){
					$dev->sysLog("Checking Sun StorEdge 3510 Events");
					$dev->checkEvents();
					$event{EventCurrent} = 0;
				}
				$event{EventCurrent}++;
			}

			foreach my $header (@linkHeaders){

				my $ErrorCount  = $lesb{$header}{ErrorCount};
				my $TimePeriod  = $lesb{$header}{TimePeriod};
				my $TimeCurrent  = $lesb{$header}{TimeCurrent};

				debug("$header: ErrorCount: $ErrorCount, TimePeriod: $TimePeriod, TimeCurrent: $TimeCurrent");

				unless ( $TimePeriod == 0 ){
					if ( $TimePeriod == $TimeCurrent){
						unless (defined $dev->{lesb}){
							$dev->checkEvents();
							$dev->sysLog("Checking Sun StorEdge 3510 LESB");
							$dev->defLESB();
							$lesbCheck = 1;
						}
						$dev->log("Checking Sun StorEdge 3510 LESB: $header");
						$dev->checkDelta($header, $ErrorCount);
						$lesb{$header}{TimeCurrent} = 0;
					}else{
						$lesb{$header}{TimeCurrent}++ if $lesb{$header}{QuietCurrent} == 0;
					}
				}


				my $QuietPeriod = $lesb{$header}{QuietPeriod};
				my $QuietCurrent = $lesb{$header}{QuietCurrent};
				debug("$header: QuietPeriod: $QuietPeriod, QuietCurrent: $QuietCurrent");

				unless ( $TimePeriod == 0 ){
					if ( $lesb{$header}{QuietCurrent} == $QuietPeriod ){
						unless (defined $dev->{lesb}){
							$dev->log("Zero LESB");
							$dev->defLESB() 
						}
						$dev->log("Zero LESB: $header");
						$dev->setBaseCounter($header);
						$lesb{$header}{QuietCurrent} = 0;
						$lesb{$header}{TimeCurrent}++;  # Fix off by one
					}else{
						$lesb{$header}{QuietCurrent}++ if $lesb{$header}{TimeCurrent} == 0;
					}
				}
			} # Loop over link headers
			if ( defined $dev->{lesb} ){
				if ( $dev->{badCounts} ){
					$dev->sysLog("Alert: LESB counts increasing, $dev->{iommsg}");
					$dev->dumpChannelCounts($dev->{lesb});
				}else{
					$dev->sysLog("Notice: LESB counts OK") if $lesbCheck;
				}
				$dev->undefLESB();
			}
			$dev->{badCounts} = 0;
		} # Loop over devices
		debug("Sleep");
		sleep 60;
		if ( ((( time() - $^T) >= $runtime) and $runtime > 0) or ( @devObjs == 0)  ){
			$sysLogger->log("$::PRODUCT Health Check Completed");
			exit(0);
		}
	}
}

sub usage {

	print "usage: $0 [--device=<dev,...>] [--password=<pwd>]\n";
	print "          [--runtime=<n>] [--cli=<fn>]\n";
	print "          [--output=<fn>] [--help]\n";
	print "\n";
	print "--device=<dev,...>   => Comma separated device list\n";
	print "--password=<pwd>     => Password for OOB interface\n";
	print "--runtime=n          => Run time for the utility\n";
	print "--cli=<fn>           => Location of the cli\n";
	print "--output=<fn>        => Redirect output to filename\n";
	print "--help               => Displays this message\n";
	print "\n";
	exit(0);
}

sub main {

	my $config;
	my $cli;
	my $device;
	my $logdir;
	my $output;
	my $help;
	my $rdch;
	my $password;
	my $passthru;
	my $runtime;
	
	$diskinfo = 0;

	usage if ($ARGV[0] eq "help");
	usage unless GetOptions (
		'config=s' => \$config,
		'device=s' => \$device,
		'logdir=s'    => \$logdir,
		'password=s'    => \$password,
		'passthru=s'    => \$passthru,
		'cli=s'    => \$cli,
		'debug'    => \$debug,
		'diskinfo'    => \$diskinfo,
		'output=s'    => \$output,
		'runtime=i'    => \$runtime,
		'rdch=s'    => \$rdch,
		'help'       => \$help,
	);
	usage if $help;

	my $program = basename($0, ".pl");

	my @configParams = qw(device cli runtime logdir output rdch passthru EventPeriod CtrlRev ChassisDate) ;

	my @lesbParams = ();

	foreach my $header (@linkHeaders){
		foreach my $param ( "ErrorCount", "TimePeriod", "QuietPeriod"){
			push @lesbParams, "$header$param";
		}
	}

	push @configParams, @lesbParams;

	my @numericFields = (@lesbParams, qw(runtime EventPeriod CtrlRev));
	my @dateFields = qw(ChassisDate);

	#
	# default configuration file parameters
	#
	
	%config = (

		# These have config file and command line overrides

		device   => "192.168.1.1",
		cli      => "/opt/SUNW${program}/libexec/sccli",
		runtime  => 150,
		logdir   => "/var/log",
		output   => "/var/log/s3khc.log",
		rdch => "2-3 4-5",
		passthru => "/opt/SUNW${program}/libexec/sm-pass",

		# These just have config file line overrides

		CtrlRev             => "05",

		ChassisDate         => "12/31/2003",

		EventPeriod         => 60,

		LIPErrorCount       => 2,
		LIPTimePeriod       => 50,
		LIPQuietPeriod      => 10,

		LossOfSyErrorCount  => 2,
		LossOfSyTimePeriod  => 50,
		LossOfSyQuietPeriod => 10,

		LossOfSiErrorCount  => 2,
		LossOfSiTimePeriod  => 50,
		LossOfSiQuietPeriod => 10,

		PrimErrErrorCount   => 2,
		PrimErrTimePeriod   => 50,
		PrimErrQuietPeriod  => 10,

		InvalTxWErrorCount  => 2,
		InvalTxWTimePeriod  => 50,
		InvalTxWQuietPeriod => 10,

		InvalCRCErrorCount  => 2,
		InvalCRCTimePeriod  => 50,
		InvalCRCQuietPeriod => 10,
	);

	$config = "/opt/SUNW${program}/etc/${program}.conf" unless defined $config;

	if ( open (CFG, $config) ){
		print "Configuration file found: $config\n";
		while (<CFG>){
			s/\s*$//g;
			s/^\s*//g;
			next if  /^#/;
			next if  /^$/;
			my ($key, $value) = split /=/, $_;
			$key =~ s/\s*$//g; $key =~ s/^\s*//g;
			$value =~ s/\s*$//g; $value =~ s/^\s*//g;
			unless ( grep /^$key$/, @configParams ){
				die "Warning: Bad config parameter: \"$key\"";
			}
			$config{$key} = $value;

		}
		close CFG;
	}

	# Command line overrides.

	$config{device}   = $device   if defined $device;
	$config{cli}      = $cli      if defined $cli;
	$config{runtime}  = $runtime  if defined $runtime;
	$config{logdir}   = $logdir   if defined $logdir;
	$config{output}   = $output   if defined $output;
	$config{password} = $password if defined $password;
	$config{rdch}     = $rdch     if defined $rdch;
	$config{passthru} = $passthru if defined $passthru;


	my $key;
	foreach $key (@configParams ){
		# Only need passthru for diskinfo
		next if $key eq "passthru" and ! defined $diskinfo;
		die "Parameter \"$key\" not defined" unless defined $config{$key};

		if ( grep /^$key$/, @numericFields){
			die "Bad value for parameter \"$key\": \"$config{$key}\" " unless ($config{$key}) =~ /^\d+$/;
		}

		if ( grep /^$key$/, @dateFields){
			die "Bad date spec for parameter \"$key\": \"$config{$key}\" " unless str2time($config{$key});
		}
	}

	$config{runtime}  = ( $config{runtime} *  60 );

	$logger = Logger::Term->new($config{output});
	$sysLogger = Logger::Sys->new($program, $logger);

	$SIG{__DIE__} = \&dieHandler;
	$SIG{__WARN__} = \&warnHandler;
	$SIG{TERM} = \&termHandler;
	$SIG{INT} = \&termHandler;

	if ($debug){
		debug ("Configuration Information:");
		my $kys;
		foreach $kys ( sort keys %config ){
			debug("$kys:  $config{$kys}");
		}
	}

	#
	# Redundant Channel map check
	#

	@rdch = split " ", $config{rdch};

	for (@rdch){
		my @ch = ();
		@ch = split/-/, $_;
		die "Must have two channels for Redundant Channel Map: \"$config{rdch}\"" unless scalar @ch == 2;
		$ch[0] =~ s/\s*$//g;
		$ch[0] =~ s/^\s*//g;
		$ch[1] =~ s/\s*$//g;
		$ch[1] =~ s/^\s*//g;
		die "Both channels must be numeric for Redundant Channel Map: \"$config{rdch}\"" 
			unless ( $ch[0] =~ /^\d+$/ and $ch[1] =~ /^\d+$/ );
	}

	# Only check this if we are log disk data, not used otherwise

	if ( $diskinfo ){
		# Log directory check
		die "Log directory \"$config{logdir}\" doesn't exist" unless -e $config{logdir};
		die "Log directory \"$config{logdir}\" not a directory" unless -d $config{logdir};
		die "Log directory \"$config{logdir}\" not writable" unless -w $config{logdir};
		# Passthru check
		die "Can't find passthru: $config{passthru}" unless -e $config{passthru};
		die "Cli not executable: $config{passthru}" unless -X $config{passthru};
	}

	# Cli check

	die "Can't find cli: $config{cli}" unless -e $config{cli};
	die "Cli not executable: $config{cli}" unless -X $config{cli};


	# Device Check, setup

	my @devObjs = ();
	my $devObj;

	my @devs = split /,/, $config{device};
	my @tmp = ();
	for (@devs){
		s/\s*$//g;
		s/^\s*//g;
		push @tmp, $_;
	}
	@devs = @tmp;

	foreach my $dev  (@devs){
		if ( ($devObj = Device->new($dev)) == -1){
			warn "Warning: Failed to initialize device $dev, skipping";
			print STDERR  "Warning: Failed to initialize device $dev, skipping\n";
			next;
		}else{
			push @devObjs, $devObj;
		}
	}

	# Setup LESB counters;

	my %lesb;
	for (@linkHeaders){
		$lesb{$_}{ErrorCount} = $config{"${_}ErrorCount"};
		$lesb{$_}{TimePeriod} = $config{"${_}TimePeriod"};
		$lesb{$_}{TimeCurrent} = 0;
		$lesb{$_}{QuietPeriod} = $config{"${_}QuietPeriod"};
		$lesb{$_}{QuietCurrent} = 0;
	}

	# Setup Events

	my %event;
	$event{EventPeriod} = $config{"EventPeriod"};
	$event{EventCurrent} = 0;

	if ($output eq "-" ){
		runit(\@devObjs, \%lesb, \%event); # This should never return.
		exit(-1);
	}else{
		if( my $pid = fork ){
			# Parent
		}elsif (defined $pid ){ # Child
			POSIX::setsid();
			runit(\@devObjs, \%lesb, \%event); # This should never return.
			exit(-1);
		}else{
			croak "Fork failed: $!";
		}
	}
	exit(0);
}
main;


=head1 NAME

s3khc.pl - Sun StorEdge 3000 Health Check

=head1 SYNOPSIS

s3khc.pl [options]

s3khc.pl help

=head1 DESCRIPTION 

***Only Sun Service Personnel should run the health check utility***

This health check utility, written in Perl language, will monitor the
health of the SE3510 array controller(s) and attached expansion chassis.
The utility provides the following functionalities:

=over

=item 1.

Poll the Link Error Status Block (LESB) counters in attempt to identify
possible excessive Fibre Channel signal jitter.

The Health Check periodically polls the LESB counters of all attached
disk drives and RAID controller drive channels in an attempt to
detect excessive jitter or other fibre channel problems. The LESB
counters include loss of sync, LIP, loss of signal, primitive error,
invalid transmission word, and invalid CRC.  The counters are polled
every 50 minutes and then compared to a predefined threshold value
within predefined time period.  If the threshold value is exceeded,
the utility will log an alert message to replace the controller(s)
in the /var/adm/messages and the /var/log/s3khc.log files.

=item 2.

Identify possible defective JBOD jumper cables.

If the utility detects the JBOD manufacturing date from the chassis
FRU-ID to be before 12/31/2003, it will log an alert message in the
/var/adm/messages and the /var/log/s3khc.log files of possible down
rev/ defective JBOD jumper cables (labeled TCC) that were shipped with
the unit.  Newer JBOD cables (labeled BIZ) should be used.


=item 3.

Record RAID Controller event log.

The health check utility periodically polls the RAID controller event
log for new events and records them to the /var/log/s3khc.log file,
if using the default file name.  The event log also is polled for a
reset of the RAID Controller. In the event of a RAID Controller reset,
LESB counters may reach excessively high levels. These increases in
the LESB counters are normal during the RAID controller reset process.
In the event of a RAID Controller reset, the health check discards all
current LESB error counts to avoid false triggers.

=item 4.

Log utility progress, warning and event messages, and LESB counters to
a log file.

By default, all progress, warning and event messages, and LESB counters
are logged to the /var/log/s3khc.log file.  This file is intended
for engineering use only.  Simplified messages are also logged in
the /var/adm/messages file.  The field is suggested to look at the
/var/adm/messages file to take any actions necessary.

=back

Note, the health check utility achieves all of the above functionalities
with very minimal impact to system performance and resource.

=head1  OPTIONS

=over

=item --device=I<IP address>

It is recommended that you run out-of-band by specifying an IP address.
One can run inband by specifying "/dev/rdsk/c#t#d#s2", but running inband
can take a long time if there is a high load of host I/O being generated
as utility commands are given lower priority.  Only 1 3510 IP address
can be specified per each utility execution.

=item --password=I<password>

Password for Out-of-Band interface only.  This option is needed if
running Out-of-Band and the RAID controller password is set.

=item --runtime=I<minutes>

This is length of time (minutes) to run the utility.  Default is 150 min
(2.5hr) if not specified. The system clock is used in the calculations.
Changes to the system clock will effect this run time.  A runtime of
"0" will cause the utility to run indefinitely.

=item --output=I</directory/filename>

Send output to a file.  By default, the utility will log a summary of the
utility execution and what actions to take in the /var/adm/messages file.
The field is expected to look into the /var/adm/messages file to determine
what to do after running the utility.

The utility will log more verbose and detail messages, intended for
engineering use, in the /var/log/s3khc.log file.  The "output" option
used here is to change the location and name of the verbose log file.
It is recommended that, for each different 3510 that the utility is run
against, a different log file name is assigned to each 3510.

If a filename of "-" is given, output will be written to standard output,
generally the terminal from which the utility was launched. The process
will also run in the foreground. Utility and sccli errors are directed
to Standard error.

If deciding to use default "output" /var/log/s3khc.log file name, it
will be appended to each time the utility is invoked, and thus the file
can become large.

=back

=head1 TERMINATION

Ordinarily, the Health Check will run in the background as daemon process
for 2 hours and 30 minutes.  The --runtime command line option can be
used to override this.

The proper way to prematurely terminate the process is to send a SIGTERM
to the process.  This signal is caught and a shutdown message is logged.

A sccli command failure or utility error will terminate the utility.

=head1 LIMITATIONS

=over

=item *

The Health Check  will not recognize new hardware configurations after
startup.

=item *

The Health Check  does not detect StorEdge devices automatically.
Devices must be specified on the command line.

=item *

Times may slowly drift.  At the end of each period certain operations
are performed.  These operations may take some time.  Polling intervals
will be extended naturally by this time.

=item *

Hardware reconfiguration can increase LESB counters.  No provision to
account for this is provided.

=item *

When monitoring more then one device, an error on one will terminate
the entire Health Check.  Separate Health Check processes may be started
for each device as a work around.

=back

=cut
