package Client::InstrReport;
use System;
use Util;
use Client;
use CIM;
use PDM::ConfigFile;
use Report;
use strict;
use NWS::Schema;
use Modules;

sub revision {
   my $rev = '$Revision: 1.13 $';
   $rev =~ s/\$//g;
   return $rev;
}

sub help {
   print Client->http_OK;
   print <<EOF;
<b>StorADE Client::InstrReport syntax</b><br>
<pre>
 This url can be used to retrieve a list of available reports or a specific
 report from a StorADE agent using different formats and filters.

<b>Examples:</b>
 http://hostname:7654/rashttp?GO=Client::InstrReport::list   # tab format
 http://hostname:7654/rashttp?GO=Client::InstrReport::list&format=xml   # xml format

 http://hostname:7654/rashttp?GO=Client::InstrReport::report&key=se:6910.80a3738d   
 http://hostname:7654/rashttp?GO=Client::InstrReport::report&key=se:6910.80a3738d&format=xml
 http://hostname:7654/rashttp?GO=Client::InstrReport::report&key=se:6910.80a3738d&include=slot.   

<b>Args:</b>
   format: valid formats are string,tab,xml,cim, default is 'tab'.
  include: include only entries that starts with the following.
  exclude: exclude entries that starts with the following.
      key: key is a key returned by InstrReport::list.
    level: s=system, d=device. Used in 'list' to return only system-level or device-level
           reports. System-level are currently 3900/6900 series.

<b>Errors:</b>
  Errors are returned in string/tab format with the following format:

#ERROR&lt;tab>ERR_NO&lt;tab>Text

  and in XML format:

&lt;?xml version="1.0"?>
&lt;ERROR no="ERR_NO">TEXT
&lt;/ERROR>

<b>Comments:</b>
  In tab/string format, entries that with a '#' are comments/error information.
  Basic Authentication is required (user=peer, password=peer).
     http://user:password\@hostname:7654/rashtp?GO=Client::...
  When querying a SE solution rack, the rack is returned by default when no key is specified.

  local Perl programs can be written using storade libraries  and the following equivalent syntax:
  #!/usr/bin/perl -I/iopt/SUNWstade/lib
  use System;
  use Client::InstrReport;

  Client::InstrReport::list({ format => "string"} );
  Client::InstrReport::report({key => "se:6910.80a3738d", format => "cim"}) ;
EOF
}

  
# level = s : system, d = devices

sub list {
  my($q) = @_;
  my (%WWN);
  my $format = $q->{format} || "tab";
  my $level  = $q->{level};
  my $out =  Client->http_OK();

  if ($format eq "xml" || $format eq "cim") {
     $out .= &xml_head();
     $out .= "<REPORTLIST>\n";
  } else {
     $out .= "#<pre>\n";
     $out .= "#FORMAT\thost\tkey\ttype\tdate\tname\twwn\n";
  }
     
  my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
  foreach my $d (@$devs) {
     $WWN{"$d->{type}:$d->{key}"} = $d->{wwn};
  }
  my $list = Report->reportList();

  foreach my $key (sort keys %$list) {
     my $v = $list->{$key};
     my $type = $v->[1];
     next if ($type eq "san" || $type eq "topo");
     next if ($level eq 'd' && $type eq "se");
     next if ($level eq 's' && $type ne "se");
     next if (!$WWN{$key});

     if ($format eq "xml" || $format eq "cim") {
        $out .= &xml_list($key, $v, $WWN{$key});
     } else {
        $out .= "$v->[0]\t$key\t$v->[1]\t$v->[2]\t$v->[4]\t$WWN{$key}\n";
     }
  }
  if ($format eq "xml") {
      $out .= "</REPORTLIST>\n";
  } else {
      $out .= "#DONE\n";
  }
  if ($q->{RETURN}) {
    return $out;
  } else {
    print $out;
  }
}

sub report {
  my($q) = @_;
  my $key = $q->{key};
  my $renv = System->get_renv();
  if (!$key && $renv->{solution_model}) {
     $key  =  "se:" . $renv->{solution_model} . "." . System->hostid();
  }

  my $format = $q->{format} || "tab";
  my ($err, $report) = Report->readReport($key, $q->{host});

  if ($err) {
     print Client->error($format, 101, "Error reading report $key: $err");
     return;
  }
  my $id = $report->id();
  my $rep = $report->content();

  if (!$report) {
     print Client->error($format, 102, "No report for $key on $q->{host}");
     return;
  }
  my $host = $q->{host} || $renv->{hostname};
  if ($format eq "xml") {
     my $out =  Client->http_OK();
     $out .= "<?xml version=\"1.0\"?>\n";
     $out .= "<REPORT host=\"$host\" key=\"$key\">\n";
     $out .= $report->toXML();
     $out .= "</REPORT>\n";
     print $out;

  } elsif ($format eq "cim") {
     print Client->http_OK();
     print "<?xml version=\"1.0\"?>\n";
     my($type, $k) = split(/\:/, $key);
     my $cim_class = "NWS::" . uc($type);
     Modules->loadOne($cim_class);

     NWS::Schema->init();
     CIM->version("1.1");
     my $cim_object = $cim_class->newSystem($rep);
     print "<CIMObject key=\"$key\">\n";
     print $cim_object->toXML();
     print "</CIMObject>\n";

  } elsif ($format eq "string") {
    print Client->http_OK();
    print "#<pre>\n";
    print "#revision = " . &revision() . "\n";
    print $report->toString({exclude => $q->{exclude}}, include => $q->{include} );
    print "#DONE\n";

  } else {
    print Client->http_OK();
    print "#<pre>\n";
    print "#revision\t" . &revision() . "\n";
    print $report->toTab({exclude => $q->{exclude},  include => $q->{include} } );
    print "#DONE\n";
  }

}


#########
#  XML
#########
# <!DOCTYPE RAIDBaseView SYSTEM "Raid.dtd">

sub xml_head {
  return <<EOF;
<?xml version ="1.0"?>
EOF
}

sub xml_list {
  my($key, $v, $wwn) = @_;
  return <<EOF;
  <REPORT host="$v->[0]" key="$key" type="$v->[1]" date="$v->[2]" name="$v->[4]" wwn="$wwn" />
EOF
}


1;
