#!/usr/bin/perl
# Copyright 2003 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
#ident  "@(#)migrate_subscribers.pl 1.3     03/10/08 SMI"

# This script reads log_template files and migrates
# the subscribers to the new format for log subscribers.
#
# The new file is written to standard output, and should
# typically be re-directed to be the file
# /opt/se6x20/resources/log_subscriptions.properties
#
# All of the existing log_template files should be passed
# in as arguments to the perl script. In particular, a resonable
# way of running the script is:
#
# cd /opt/se6x20/resources
# perl migrate_subscribers.pl log_template log_template_?? log_template_??_?? > log_subscriptions.properties
#
# Note, there is no need for the output file to have all events listed
# in the file. It only needs to have entries for subscribers. If event is
# not present, then it means there is no subscriber for the event.
#

my $usage = "Usage:\n\n$0 log_template log_template_fr log_template_jp log_template_zh_CN";
die $usage if (scalar(@ARGV) < 1);

# we store category => [event_id] => [ subscribers ]
%orig_categories = ();
            
my @data;
my $dataCnt = 0;
foreach $file (@ARGV) {

    $data[$dataCnt] = parseFile($file);
    $dataCnt++;
}

# Now determine the e-mail subscribers to each category
%category_subscribers = determine_cat_subscribers();

# Now we generate the new file of subscribers.

print "all=\n"; #no one is an all subscriber by default
# do the category subscribers first
foreach $cat (keys %category_subscribers) {
    print "$cat=", join(",", keys %{$category_subscribers{$cat}}), "\n";
}
#now do the event only subscribers
my %event_subs;
foreach $r_data (@data) {
    foreach $event (keys %$r_data) {
        my ($cat, $event_id) = split(/\./, $event);
        my $event_name = $$r_data{$event}->[1];
        my $event_subs = $$r_data{$event}->[4];
        my @F = split(/,/, $event_subs);
        grep(tr/A-Z/a-z/, @F); # make them all lower case
        foreach $sub (@F) {
            if (!exists($category_subscribers{$cat}{$sub})) {
                $event_subs{$event_name}{$sub}++;
            }
        }
    }
}
foreach $event (keys %event_subs) {
    print "$event=", join(",", keys (%{$event_subs{$event}})), "\n";
}


####################################################
# Subroutines
####################################################

# This subroute returns an Hash that looks like:
#    event_id => Array with the following indices
#                0 => event_id
#                 1 => friendly_name
#                 2 => number of arguments
#                 3 => message
#                 4 => string of subscribers for the event
#                 5 => Array of all subscribers
sub parseFile {
    my $file = $_[0];
    open(IN, $file) || die "Unable to open file: $file, $!\n\n$usage";
    my %data;
    while (<IN>) {
        next if /^\s*#/;
        next unless /:/;
        chomp;
        my @F = split(/:/);
        # Handle special case where the French file has an extra colon in it
        if (($file =~ /_fr/) && ($_ =~ /^arraynetwork.1:/) && (scalar(@F) >= 5)) {
            $F[3] = join(" ", $F[3], $F[4]);
            $F[4] = $F[5];
        }
        my @addrs = split(/[,;]/, $F[4]);
        $F[5] = \@addrs;

        $data{"$F[0]"} = \@F;

        grep(tr/A-Z/a-z/, @addrs); # make all lower case for comparisons
        my @category = split(/\./, $F[0]);
        $category_event_count{$dataCnt}{$category[0]}++;
        foreach $addr (@addrs) {
            $orig_categories{$category[0]}->[$category[1]]->{$addr}++;
        }
        
    }
    close (IN);
    return \%data;
}


# Determine which e-mail addresses are subscribers to an entire
# category--not just some events
sub determine_cat_subscribers() {
    my %members;
    foreach $key (keys %orig_categories) {
        my %event_counter;
        $a_ref = $orig_categories{$key};
        foreach $event (@$a_ref) {
            foreach $sub (keys %$event) {
                $event_counter{$sub}++;
            }
        }

        foreach $numFiles (keys %category_event_count) {
            $num_events = $category_event_count{$numFiles}{$key};
            foreach $addr (keys %event_counter) {
                $members{$key}{$addr}++ 
                    if ($event_counter{$addr} ==  $num_events);
            }
        }

    }
    return %members;
}
