package LockManager;

#<copyright>
# ----------------------------------------------------------
# Sun Proprietary/Confidential Code
# Copyright 2001, Sun Microsystems, Inc. All rights reserved.
# ----------------------------------------------------------
#</copyright>

#  $Id: LockManager.pm,v 1.6 2004/02/11 17:43:28 ccadieux Exp $
 
#  Taxonomy:
#     system                         # meaning the whole system
#     system~array00 system~dsp
#     system~rack
#     system~sp

#  Usage:
#  $lease = LockManager->new();
#  $lease->lock('luxadm', 60, 10);
#  $lease->lock(['dsp01', 'array00~fru1', 'array00~fru2'], 60, 10);
#  $lease->unlock();
#  before taking array01.fru1, 
#    - check that array01 is not taken
#    - check that aray01.* is not taken either.
#
#  This library allows to syncronize resource usage by locking file with the
#  corresponding resource name in /tmp/MgmtLocks directory.
#  Multiple locks can be taken at the same time. The '~' is used for 
#  hiearchycal locks. Before locking a~b, the library checks that 'a' is
#  not there and that no other lock starting with 'a~b~' exist.
#
#  A file called MASTERLOCK is used to sync operations with the use of flock.
#  The real resource locks are done by creating or removing (lock/unlock) the
#  files with the resource names. The lock files are not themselves locked and
#  can still exist after the process who created them has ended.
#  The lock files contain the expiration_time, process_id and info string,. 
#  The expiration_time is used to inform application of expired locks and to
#  removed old locks. By default, old locks are removed.
#  when requesting a lock , 2 times (in seconds) are passed, the expected duration
#  of the lock and the time the application is willing to wait for the lock.
#
#  Algorithm:
#    while (not_too_long) {
#      if (got_a_lock on MASTERLOCK) {
#         foreach lock requested {
#             if (lock already there, not expired and no parent||child present)
#                looking_good=1
#             } else {
#                looking_good=0
#             }
#         }
#         if (looking_good) {
#            foreach lock requested {
#                take lock (create file)
#            }
#            return success
#         }
#      }
#      release MASTERLOCK
#      wait a little
#      return failure if it's been too long
#    } # try again
#



use Fcntl qw(:flock);
use strict 'vars';

use vars qw($ERR @THERE $BASE);

$BASE = "/tmp/MgmtLocks";

sub new {
  my($class, $arg) = @_;

  mkdir $BASE, 0777 if (!-d $BASE);
  $ERR = "";

  my $lock = $arg || {} ;
  bless $lock, 'LockManager';
  
  return $lock;
}

sub read {
  my($lock, $file, $clean) = @_;

  my $found;
  if (open(O, "$BASE/$file.lock")) {
    my $file_expire = <O>; 
    my $pid         = <O>; 
    my $info        = <O>; 
    close(O);
    chop($file_expire); chop($pid); chop($info);
    if ($clean && ($file_expire +10 < time) ) {
       unlink "$BASE/$file.lock";
       $found = 0;
    } else {
       $found = { time_left => $file_expire - time, pid => $pid, info => $info};
    }
  }
  return $found;
}

sub lockList {
  my($class) = @_;
  my $out;
  if (wantarray) {
     return \@THERE;
  }
  foreach my $el (@THERE) {
     my($time, $pid, $info) = split(/,/, $el);
     $out .= "pid:$pid, time_left:$time ($info)\n";
  }
  return $out;
}

#
# remove stale locks
#
sub cleanLocks {
  my($class) = @_;
  opendir(O, $BASE);
  my @F = readdir(O); closedir(O);
  foreach my $f (sort @F) {
     next if (substr($f,-5) ne ".lock");
     $class->read(substr($f,0,-5), 1);
  }
     
}

sub error {
  return $ERR;
}
  

sub exists {
  my($lock, $file, $clean) = @_;
  my($file_expire, $pid, $info);

  $file = $lock->filter($file);
  my @parts = split(/\~/, $file);
  my($found, $x);

  for ($x=$#parts; $x >= 0; $x--) {
     my $sub = join("~", @parts[0..$x]);
     last if ($found = $lock->read($sub, $clean));
  }
  return $found if ($found);

  opendir(O, $BASE);
  my @F = readdir(O); closedir(O);
  foreach my $f (sort @F) {
     next if (substr($f,-5) ne ".lock");
     if (substr($f,0,length($file)+1) eq "$file~") {
        last if ($found = $lock->read(substr($f,0,-5), $clean));
     }
  }
  return $found;
}

sub write {
  my($lock, $file, $lease_time, $info) = @_;

  my $F = "$BASE/$file.lock";
  my $pid = $$;
  open(WW, ">$F");
  print WW (time + $lease_time) . "\n$pid\n$info\n";
  close(WW);
}

sub filter {
  my($lock, $file) = @_;
  $file =~ s/[^\w\.\-\_\,\#]/_/g;
  return $file;
}
  
  
#  file can be a value or a list: $name = [file1, file2, file3]

sub lock {
  my($lock ,$file, $lease_time, $wait, $info, $arg ) = @_;

  my ($err, $x);
  $file = $lock->filter($file);
  my $force = $arg->{force};  # lock anyway
  my $clean = 1;
  $clean    = 0 if ($arg->{never_expire});
  my @caller = caller;
  $info = "$caller[1]:$caller[0]:$caller[2]" if (!$info);
  
  if (ref($file) ne "ARRAY") {
     $file= [$file];
  }
  for ($x=0; $x <= $#$file; $x++) {
     $file->[$x] = $lock->filter($file->[$x]);
  }
  $lock->{file_list} = $file;
  my ($RC);
  my $time = 0;
  while (1) {
     if ($lock->flock1("MASTERLOCK", 5, 6)) {
        @THERE = ();
        foreach my $f (@$file) {
           my $rc = $lock->exists($f, $clean);
           if ($rc && !$force) {
              push(@THERE, $rc);
           }
        }
        if ($#THERE < 0) {
           foreach my $f (@$file) {
              $lock->write($f, $lease_time, $info);
           }
           $lock->funlock1("MASTERLOCK");
           return 1;
        }
        if ($time >= $wait) {
           $lock->funlock1("MASTERLOCK");
           return 0;
        }
        $lock->funlock1("MASTERLOCK");
        sleep 5;
        $time += 1;
     } else {
        $ERR = "Cannot get MASTERLOCK";
        return 0;
     }
  }
}




sub unlock {
  my($self, $file) = @_;

  $file = $self->filter($file);
  if (!$file) {
     $file = $self->{file_list};
  } elsif (ref($file) ne "ARRAY") {
     $file= [$file];
  }
  if ($self->flock1("MASTERLOCK", 5, 6)) {

     foreach my $f (sort @$file) {
          $file = $self->filter($file);
          unlink "$BASE/$f.lock";
     }
     return $self->funlock1("MASTERLOCK");

  } else {
    return 0;
  }
}



# 1 : success
# 0 : failure

sub flock1 {
  my($lock ,$file , $lease_time, $wait) = @_;

  $ERR = "";
  my($file_time, $expired , $file_expire);

  my $file_name = "$BASE/$file";

  if (open(O, $file_name)) {
     $file_expire = <O>; close(O);
     $expired = time > $file_expire;
  }
  $lock->{"FH_$file"} = "FH_$file";
  if (!open($lock->{"FH_$file"}, ">$file_name")) {
     $ERR = "Cannot open $file_name: $!";
     return 0;
  }
  my $time = 0;

  while (1) {
    my $rc = flock($lock->{"FH_$file"},  LOCK_EX|LOCK_NB);  # 1 : success, 0 : failure
    if ($rc) {
       my $pid = $$;
       my $F = $lock->{"FH_$file"};
       my $l1 = time + $lease_time . "\n$pid\n\n";
       print $F $l1;
       return 1; # GOT IT
    }
    if ($expired) {
       flock($lock->{"FH_$file"}, LOCK_UN); # unlock
       next;
    }
    return 0 if ($time >= $wait);
    sleep 1;
    $time++;
  }
  return 0;
}


sub funlock1 {
  my($self, $file) = @_;

  if (!flock($self->{"FH_$file"}, LOCK_UN)) {
     $ERR = $!;
     return 0;
  }
  if (!close($self->{"FH_$file"})) {
     $ERR = $!;
     return 0;
  }
  return 1;
}


1;
