/*
# The contents of this file are subject to the Netscape Public License
# Version 1.1 (the "License"); you may not use this file except in
# compliance with the License. You may obtain a copy of the License at
# http://www.mozilla.org/NPL/
#
# Software distributed under the License is distributed on an "AS IS"
# basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
# the License for the specific language governing rights and limitations
# under the License.
#
# The Original Code was released August, 1999.  The Initial Developer
# of the Original Code is Netscape Communications Corporation.  Portions
# created by Netscape are Copyright (C) 1999, 2000 Netscape Communications
# Corporation.  All Rights Reserved.
#
# Contributor(s): John M. Kristian <kristian@netscape.com>
#                 Richard Megginson <richm@netscape.com>
*/
#include <errno.h>
#include <stdio.h> /* fprintf fopen fread fwrite perror */
#include <stdlib.h> /* malloc */
#include <string.h>
#ifdef ARCH_AIX
#include <strings.h> /* for strcasecmp */
#endif
#include <sys/types.h> /* mode_t */

#ifdef ARCH_MSWin32
#define strcasecmp(L,R) stricmp(L,R)
#include <io.h> /* chmod */
#include <sys/stat.h> /* file modes */
#define READABLE (S_IREAD | S_IWRITE)
#define EXECUTABLE READABLE
#include <windows.h> /* CreateProcess */

#else /* Unix */
#include <sys/stat.h> /* chmod */
#include <unistd.h> /* execv */
#define READABLE (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)
#define EXECUTABLE (READABLE | S_IXUSR | S_IXGRP | S_IXOTH)
#endif

#include "interpret.h" /* interpret */

static char*
strnstrn (char* s, size_t sLen, char* ss, size_t ssLen)
{
    for ( ; sLen >= ssLen; ++s, --sLen) {
	if (!memcmp (s, ss, ssLen)) return s;
    }
    return NULL;
}

static char** exec_argv = NULL;
static int failed = 0;

static int
execute (const char* source, const char* op)
{
    if (!strcasecmp (op, "exec")) {
	auto const char* const p = strtok (NULL, "\n\r");
	if (!strcmp(p, exec_argv[0])) {
	    fprintf (stderr, "%s won't run itself recursively\n", p);
	    return -1;
	}
#ifdef ARCH_MSWin32
	/* NT supports execv(), but poorly.  In particular,
	   cmd.exe acts as though a program has terminated as soon as
	   the program calls execv(); cmd.exe resumes doing commands,
	   even if the program that was exec'd is still running.  Bad.
	   The following works better:
	*/
	auto const char* const fmt = "%s: CreateProcess(%s)";
	auto char* msg = (char*) malloc (strlen(fmt) + strlen(source) + strlen(p));
	sprintf (msg, fmt, source, p);

	auto STARTUPINFO child_info;
	GetStartupInfo (&child_info);
	child_info.lpReserved = NULL;
	auto PROCESS_INFORMATION child;

	if (CreateProcess(p, GetCommandLine(),
			NULL, // process security attributes
			NULL, // thread security attributes
			TRUE, // inherit handles
			0, // creation flags
			NULL, // environment block
			NULL, // current directory name
			&child_info,
			&child)) {
	    if (WAIT_FAILED != WaitForSingleObject (child.hProcess, INFINITE)) {
		auto DWORD status;
		if (GetExitCodeProcess (child.hProcess, &status)) {
		    ExitProcess (status);
		}
	    }
	    ExitProcess (GetLastError()); // BUG?
	}
	auto DWORD err = GetLastError();
	auto char* errMsg;
	FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
		      FORMAT_MESSAGE_FROM_SYSTEM |
		      FORMAT_MESSAGE_IGNORE_INSERTS,
		      NULL,
		      err,
		      MAKELANGID (LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
		      (LPTSTR)&errMsg,
		      0,
		      NULL );
	fprintf (stderr, "%s: %s\n", msg, errMsg);
	LocalFree (errMsg);
	
#else /* Unix */
	auto const char* const fmt = "%s: execv(%s)";
	auto char* msg = (char*) malloc (strlen(fmt) + strlen(source) + strlen(p));
	sprintf (msg, fmt, source, p);
	execv (p, exec_argv);
	/* This is tricky, although good for its intended purpose.
	   We're executing a new program, but its argv[0] refers to
	   this program.  So, this program is a very tight wrapper:
	   the wrapped program thinks it's this one.  Caveat user.
	 */
	failed = errno;
	perror (msg);
#endif
	free (msg);
    } else {
	fprintf (stderr, "%s: %s is not supported\n", source, op);
	return -1;
    }
    return 0;
}

static char*
fget_all (FILE* const f)
{
    auto size_t got = 0;
    auto size_t allSize = 1024;
    auto char* all = (char*) malloc (allSize);
    while (fgets (all + got, allSize - got, f)) {
	got += strlen (all + got);
	if (got + 1 >= allSize) {
	    allSize <<= 1;
	    all = (char*) realloc (all, allSize);
	}
    }
    if (!feof(f)) {
	errno = ferror (f);
	free (all);
	return NULL;
    }
    return all;
}

static char
script[] = "\0The buffer that contains the script."
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n"
"The rest of its initial value is merely unlikely to appear elsewhere in the program.\n";

int
main (int argc, char** argv)
{
    if (argc < 1) return EINVAL;
    if (*script) {
	exec_argv = argv;
	auto const int err = interpret (script, execute);
	return failed ? failed : err;
    }
    if (argc < 2) {
	fprintf (stderr, "usage: %s < script outfile\n", argv[0]);
	return EINVAL;
    }
    /* Output a copy of this (executable) program, except with
       the initial script replaced by whatever is read from stdin.
       This is tricky.
    */
    auto const size_t scriptSize = sizeof(script);
/*  fprintf (stderr, "%s: %li-byte script buffer\n", argv[0], long(scriptSize)); */
    auto char* const input = fget_all (stdin);
    if (!input) return errno;
    auto size_t inputLen = strlen (input);
/*  fprintf (stderr, "%s: %li bytes input\n", argv[0], long(inputLen)); */
    if (!inputLen) {
	fprintf (stderr, "%s: zero-length input\n", argv[0]);
	return EINVAL;
    }
    if (inputLen > scriptSize - 1) {
	fprintf (stderr, "%s: %li input bytes won't fit into my %li-byte buffer\n",
		 long(inputLen), long(scriptSize));
	return ENOMEM;
    }
    auto FILE* fin = fopen (argv[0], "rb"); /* Yes; we're reading this program file. */
    if (!fin) {
	perror (argv[0]);
	return errno;
    }
    auto FILE* fout = NULL;
    auto int failed = 0;
    auto char buf [(2 * sizeof(script)) - 1];
    auto size_t bufLen = 0;
    while(1) {
	auto const size_t got = fread (buf+bufLen, 1, sizeof(buf)-bufLen, fin);
/*	fprintf (stderr, "%s: %li bytes self\n", argv[0], long(got)); */
	if (!got || (bufLen += got) >= sizeof(buf)) {
	    auto char* newscript = strnstrn (buf, bufLen, script, scriptSize);
	    if (newscript) {
/*		fprintf (stderr, "%s: replace %li bytes at offset %li\n",
			 argv[0], long(inputLen+1),
			 long(ftell (fin) - (bufLen - (newscript - buf)))); */
		memcpy (newscript, input, inputLen+1);
	    }
	    if (!fout && !(fout = fopen (argv[1], "wb"))) {
		failed = errno;
		perror (argv[1]);
		break;
	    }
	    auto const size_t put = fwrite (buf, 1, scriptSize, fout);
	    if (put) memmove (buf, buf + put, bufLen -= put);
	    if (put != scriptSize) {
		failed = errno = ferror (fout);
		perror (argv[1]);
		break;
	    }
	}
	if (!got) break;
    }
    if (!failed && fout) {
	if (bufLen && (fwrite (buf, 1, bufLen, fout) != bufLen)) {
	    failed = errno = ferror (fout);
	    perror (argv[1]);
	} else if (fclose (fout)) {
	    failed = EOF;
	    perror (argv[1]);
	}
    }
    if (!failed && !feof(fin)) {
	failed = errno = ferror (fin);
	perror (argv[0]);
    }
    fclose (fin);
    free (input);
    chmod (argv[1], failed ? READABLE : EXECUTABLE);
    return failed;
}
