
; 
;                            CPUUNDOC
; 
;    Test all known undocumented instructions for this CPU.
;    Most tests verify instruction functions as described.
; 
;    (c) Copyright 1994  Frank van Gilluwe  All Rights Reserved. 

include undocpc.inc

cseg    segment para public
        assume  cs:cseg, ds:cseg, ss:stacka

; DATA AREA 

;  general data

        db      'CPUUNDOC v1.01 '
        db      '(c) 1994 Frank van Gilluwe',0

cpuhead db      CR, LF, CR, LF
        db      '  UNDOCUMENTED INSTRUCTION TESTS'
        db      '                         v1.01 (c) 1994 FVG'
        db      CR, LF
        db      '  '
        db      ''
        db      CR, LF, '$'

cpu_val  db     0                  ; CPU value from CPUVALUE
                                   ;   0 = 8088/8086 or V20/V30
                                   ;   1 = 80186/80188
                                   ;   2 = 80286
                                   ;   3 = 80386
                                   ;   4 = 80486
                                   ;   5 = Pentium

cpu_info db     0                  ; flags from CPUVALUE
                                   ;   bit 0 = 1 if CPUID ok 
                                   ;       1 = 1 if V20/V30 CPU

cpu_prot db     0                  ; protected state
                                   ;   0 = no protected mode 
                                   ;   1 = real mode
                                   ;   2 = protected mode
                                   ;   3 = V86 mode


; undocuemted instruction data

ud_flags     db 0                  ; bit 1 = 1 if INT 1 occured

old_int6_seg dw 0                  ; temp storage for old int 6
old_int6_off dw 0                  ;  vector (bad opcode)

badoff       dw 0                  ; temp return offset if bad offset
                                   ;  interrupt 6 called

;  text for undocumented instructions

ud_header db    CR, LF
          db    '  Undocumented Instruction Summary for this CPU'
          db    CR, LF, '$'
ud_popcs  db    '    Should support:    POP     CS      (0Fh)          $'
ud_ldall2 db    '    Should support:    LOADALL         (0Fh, 05h)     $'
ud_ldall3 db    '    Should support:    LOADALL         (0Fh, 07h)     $'
ud_umov   db    '    Might support :    UMOV            (0Fh, 10h-13h) $'
ud_rdtsc  db    '    Should support:    RDTSC           (0Fh, 31h)     $'
ud_xbts   db    '    Only 386 rev A:    XBTS            (0Fh, A6h)     $'
ud_ibts   db    '    Only 386 rev A:    IBTS            (0Fh, A7h)     $'
ud_shlal3 db    '    Should support:    SHL     AL,imm  (C0h, reg=110) $'
ud_shlax3 db    '    Should support:    SHL     AX,imm  (C1h, reg=110) $'
ud_shlal  db    '    Should support:    SHL     AL,1    (D0h, reg=110) $'
ud_shlax  db    '    Should support:    SHL     AX,1    (D1h, reg=110) $'
ud_setalc db    '    Should support:    SETALC          (D6h)          $'
ud_icebp  db    '    Should support:    ICEBP           (F1h)          $'
ud_testal db    '    Should support:    TEST    AL,1    (F6h, reg=001) $'
ud_testax db    '    Should support:    TEST    AX,1    (F7h, reg=001) $'

ud_tested db    ' Tested OK.', CR, LF, '$'
ud_untest db    ' Not tested.', CR, LF, '$'
ud_untstv db    ' Untested (V86 mode).', CR, LF, '$'
ud_failed db    ' Failed test.', CR, LF, '$'
ud_notA   db    ' Failed, not rev A.', CR, LF, '$'


; CODE START 

cpuundoc proc    far

start:  
        mov     ax, cs
        mov     ds, ax
        mov     es, ax

        OUTMSG  cpuhead            ; output header message

; get cpu type and cpumode, and save results

        call    cpuvalue           ; get CPU type 0 to 5 in al
        mov     [cpu_val], al      ; save
        mov     [cpu_info], ah     ; save flags
        call    cpumode            ; get current mode: V86, real,
                                   ;   protected into ax
        mov     [cpu_prot], al     ; save type of protection

; --- Undocumented instruction identification

undoc_check:
        call    chk_undoc          ; test and display undoc info

cpui_exit:
        mov     ah,4Ch
        int     21h                ; exit with al return code
cpuundoc endp


;
;    CHECK UNDOCUMENTED
;       Check and display undocumented instructions.  Most 
;       undocumented instructions are tested to ensure the
;       instruction works as described.  The result of the 
;       test is displayed.  
;
;       Specific undocumented instructions are only tested
;       when it is known to work for some or all CPUs in
;       a family.  For example, POP CS is only tested on
;       8088/8086 CPUs, as it conflicts with documented
;       instructions on all other CPUs.
;
;       Call with:      ds:[cpu_val] set with CPU
;                       ds:[cpu_info] set with CPU info
;                       ds:[cpu_prot] set with CPU protection 
;                           status
;
;       Returns:        display undocumented information
;
;       Regs used:      ax, bx, cx, dx

chk_undoc proc  near
        OUTMSG  ud_header          ; undocumented summary header

; first, if a 286 or later, trap the bad opcode interrupt 6

        cmp     [cpu_val], 2       ; 286 or later ?
        jb      ud_skp1
        call    hook_int6          ; hook the bad opcode interrupt

; begin tests for undocumented instructions

ud_skp1:
        cmp     [cpu_val], 0       ; 8088 or V20 ?
        jne     ud_skp2            ; jump if not
        test    [cpu_info], 2      ; V20/V30 ?
        jnz     ud_skp2            ; jump if so (no POP CS)
        OUTMSG  ud_popcs           ; undocumented POP CS 
        push    cs
        POPCS
        nop
        OUTMSG  ud_tested          ; tested ok

ud_skp2:
        cmp     [cpu_val], 2       ; 286 ?
        jne     ud_skp3            ; jump if not
        OUTMSG  ud_ldall2          ; LOADALL
        OUTMSG  ud_untest          ; not tested

ud_skp3:
        cmp     [cpu_val], 3       ; 386 ?
        jne     ud_skp4            ; jump if not
        OUTMSG  ud_ldall3          ; LOADALL
        OUTMSG  ud_untest          ; not tested
ud_skp4:
        cmp     [cpu_val], 3       ; 386 or later
        jb      ud_skp6
        OUTMSG  ud_umov            ; UMOV might be supported
        mov     [badoff], offset ud_skp5  ; for bad opcodes
        mov     bx, 5A5Ah
        xor     ax, ax
        db      0Fh, 10h, 0D8h     ; mov al, bl
        db      4 dup (90h)        ; nops
        cmp     al, 5Ah
        jne     ud_skp5            ; test failed
        xor     ax, ax
        db      0Fh, 11h, 0D8h     ; mov ax, bx
        db      4 dup (90h)        ; nops
        cmp     ax, 5A5Ah
        jne     ud_skp5            ; test failed
        xor     ax, ax
        db      0Fh, 12h, 0C3h     ; mov al, bl
        db      4 dup (90h)        ; nops
        cmp     al, 5Ah
        jne     ud_skp5            ; test failed
        xor     ax, ax
        db      0Fh, 13h, 0C3h     ; mov ax, bx
        db      4 dup (90h)        ; nops
        cmp     ax, 5A5Ah
        jne     ud_skp5            ; test failed
        OUTMSG  ud_tested          ; tested ok
        jmp     ud_skp6 
ud_skp5:
        OUTMSG  ud_failed          ; failed UMOV test

ud_skp6:
        cmp     [cpu_val], 5       ; Pentium ?
        jb      ud_skp9            ; jump if not
        OUTMSG  ud_rdtsc           ; RDTSC
        mov     [badoff], offset ud_skp8   ; for bad opcodes
.386
        mov     eax, 0
        mov     edx, 0
        RDTSC                      ; read time stamp register
        nop
        nop
        cmp     eax, 0             ; returned value in edx:eax
        jne     ud_skp7            ;  will be ticks after reset
        cmp     edx, 0             ;  so a value of zero means
        je      ud_skp8            ;  RDTSC is not working!
.8086
ud_skp7:
        OUTMSG  ud_tested          ; ok !
        jmp     ud_skp9

ud_skp8:
        OUTMSG  ud_failed          ; failed

ud_skp9:
        cmp     [cpu_val], 3       ; 386 only ?
        jne     ud_skp13
        OUTMSG  ud_xbts            ; undocumented xbts
        mov     [badoff], offset ud_skp10  ; for bad opcodes
        mov     ax, 1
        mov     bx, 1
        mov     cl, 1
        mov     dx, 1
        db      0Fh, 0A6h, 0DAh    ; xbts  bx, dx, ax, al
        nop
        nop
        OUTMSG  ud_tested
        jmp     ud_skp11
ud_skp10:
        OUTMSG  ud_notA            ; failed, not A step of 386

ud_skp11:
        OUTMSG  ud_ibts            ; undocumented ibts
        mov     [badoff], offset ud_skp12  ; for bad opcodes
        mov     ax, 1
        mov     bx, 1
        mov     cl, 1
        mov     dx, 1
        db      0Fh, 0A7h, 0DAh    ; ibts  bx, dx, ax, al
        nop
        nop
        OUTMSG  ud_tested
        jmp     ud_skp13
ud_skp12:
        OUTMSG  ud_notA            ; failed, not A step of 386

ud_skp13:
        cmp     [cpu_val], 2       ; 80286 or later 
        jae     ud_skp14           ; jump if so
        jmp     ud_skp22           ; jump if not

ud_skp14:
        OUTMSG  ud_shlal3          ; undocumented shl/sal
        mov     [badoff], offset ud_skp15 ; for bad opcodes
        mov     al, 1
        db      0C0h, 0F0h, 5      ; shl  al, 1
        db      4 dup (90h)        ; nops
        cmp     al, 20h            ; did shift occur ?
        jne     ud_skp15           ; jump if not
        OUTMSG  ud_tested
        jmp     ud_skp16
ud_skp15:
        OUTMSG  ud_failed

ud_skp16:
        OUTMSG  ud_shlax3          ; undocumented shl/sal
        mov     [badoff], offset ud_skp17 ; for bad opcodes
        mov     ax, 10h
        db      0C1h, 0F0h, 5      ; shl  ax, 5
        db      4 dup (90h)        ; nops
        cmp     ax, 200h           ; did shift occur ?
        jne     ud_skp17           ; jump if not
        OUTMSG  ud_tested
        jmp     ud_skp18
ud_skp17:
        OUTMSG  ud_failed
ud_skp18:
        OUTMSG  ud_shlal           ; undocumented shl/sal
        mov     [badoff], offset ud_skp19 ; for bad opcodes
        mov     al, 2
        db      0D0h, 0F0h         ; shl  al, 1
        db      4 dup (90h)        ; nops
        cmp     al, 4              ; did shift occur ?
        jne     ud_skp19           ; jump if not
        OUTMSG  ud_tested
        jmp     ud_skp20
ud_skp19:
        OUTMSG  ud_failed

ud_skp20:
        OUTMSG  ud_shlax           ; undocumented shl/sal
        mov     [badoff], offset ud_skp21 ; for bad opcodes
        mov     ax, 200h
        db      0D1h, 0F0h         ; shl  ax, 1
        db      4 dup (90h)        ; nops
        cmp     ax, 400h           ; did shift occur ?
        jne     ud_skp21           ; jump if not
        OUTMSG  ud_tested
        jmp     ud_skp22
ud_skp21:
        OUTMSG  ud_failed

ud_skp22:
        OUTMSG  ud_setalc          ; undocumented SETALC
        mov     [badoff], offset ud_skp23 ; for bad opcodes
        stc                        ; set carry
        SETALC                     ; set al from carry
        db      4 dup (90h)        ; nops
        cmp     al, 0FFh
        jne     ud_skp23           ; jump if not working
        clc
        SETALC                     ; set al from carry
        db      4 dup (90h)        ; nops
        cmp     al, 0
        jne     ud_skp23           ; jump if not working
        OUTMSG  ud_tested          ; tested ok
        jmp     ud_skp24 
ud_skp23:
        OUTMSG  ud_failed

ud_skp24:
        cmp     [cpu_val], 3       ; 386 or later
        jb      ud_skp27
        OUTMSG  ud_icebp           ; ICEBP might be supported
        cmp     [cpu_prot], 2      ; in protected/v86 mode ?
        jb      ud_skp25           ; jump if not (ok to test)
        OUTMSG  ud_untstv          ; untested message (v86)
        jmp     ud_skp27

ud_skp25:
        mov     [badoff], offset ud_skp26 ; for bad opcodes
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     dx, es:[4]         ; get offset of int 1
        mov     cx, es:[4+2]       ; get segment
        mov     ax, offset int1test
        mov     es:[4], ax         ; set new vector
        mov     ax, cs
        mov     es:[4+2], ax
        sti                        ; enable interrupts
.386P
        mov     eax, dr7           ; get debug register
        and     eax, 0FFFFEFFFh    ; clear bit 12 to allow ICEBP
        mov     dr7, eax           ; set it
        ICEBP                      ; issue int 1
        db      4 dup (90h)        ; nops
.8086
        cli                        ; disable interrupts
        mov     es:[4], dx         ; restore original int 1
        mov     es:[4+2], cx
        sti                        ; enable interrupts
        pop     es
        test    [ud_flags], 2      ; did int 1 occur ?
        jz      ud_skp26
        OUTMSG  ud_tested          ; tested ok
        jmp     ud_skp27

ud_skp26:
        mov     [ud_flags], 0      ; clear bad opcode flag
        OUTMSG  ud_failed          ; failed ICEBP test

ud_skp27:
        OUTMSG  ud_testal          ; undocumented test al
        mov     [badoff], offset ud_skp28 ; for bad opcodes
        mov     al, 1
        db      0F6h, 0C8h, 01h    ; test al, 1
        db      4 dup (90h)        ; nops
        jz      ud_skp28           ; jump if failed
        mov     al, 0
        db      0F6h, 0C8h, 01h    ; test al, 1
        db      4 dup (90h)        ; nops
        jnz     ud_skp28           ; jump if failed
        OUTMSG  ud_tested
        jmp     ud_skp29
ud_skp28:
        OUTMSG  ud_failed

ud_skp29:
        OUTMSG  ud_testax          ; undocumented test ax
        mov     [badoff], offset ud_skp30 ; for bad opcodes
        mov     ax, 100h
        db      0F7h, 0C8h, 0, 1   ; test ax, 100h
        db      4 dup (90h)        ; nops
        jz      ud_skp30           ; jump if failed
        mov     ax, 0
        db      0F7h, 0C8h, 0, 1   ; test ax, 100h
        db      4 dup (90h)        ; nops
        jnz     ud_skp30           ; jump if failed
        OUTMSG  ud_tested
        jmp     ud_skp31
ud_skp30:
        OUTMSG  ud_failed

ud_skp31:
        cmp     [cpu_val], 2       ; 286 or later ?
        jb      ud_skp32           ; jump if not

        call    restore_int6
ud_skp32:
        ret

;---------------------------------------------------------------
; Interrupt vector 1 comes here if the ICEBP instruction is 
;   supported.  

int1test:
        mov     cs:[ud_flags], 2   ; set flag that we got here
        iret

chk_undoc endp


;
;    HOOK INTERRUPT 6
;       Save the old interrupt 6 vector and replace it with  
;       a new vector to the bad_op_handler.  Vectors are handled 
;       directly without using DOS.
;
;       Called with:    nothing
;                      
;       Returns:        vector hooked
;                       old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Regs used:      none

hook_int6 proc    near
        push    ax
        push    cx
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     ax, es:[6*4]       ; get offset of int 6
        mov     cx, es:[6*4+2]     ; get segment
        mov     es:[6*4], offset bad_op_handler
        mov     word ptr es:[6*4+2], seg bad_op_handler
        sti                        ; enable interrupts
        mov     [old_int6_seg], cx ; save original vector
        mov     [old_int6_off], ax
        pop     es
        pop     cx
        pop     ax
        ret
hook_int6 endp


;
;    RESTORE INTERRUPT 6
;       Restore the previously saved old interrupt 6 vector.
;       Vectors handled directly without using DOS.
;
;       Called with:    old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;                      
;       Returns:        vector restored
;
;       Regs used:      none

restore_int6 proc    near
        push    ax
        push    cx
        push    dx
        mov     cx, [old_int6_seg] ; get original vector
        mov     dx, [old_int6_off]
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     es:[6*4], dx       ; restore original int 6
        mov     es:[6*4+2], cx
        sti                        ; enable interrupts
        pop     es
        pop     dx
        pop     cx
        pop     ax
        ret
restore_int6 endp


;
;    BAD OFFSET INTERRUPT HANDLER
;       If a bad opcode occurs (80286 or later) will come here.
;       The saved BADOFF offset is used to goto the routine
;       previously stored in BADOFF.
;
;       In a few cases, it is also used for double faults. A few
;       instructions (RDMSR & WRMSR) can issue a double fault if
;       not supported, so well come here as well.
;
;       Called with:    cs:[badoff] previously set
;
;       Returns:        returns to address stored in badoff


bad_op_handler proc far
        push    ax
        push    bp
        mov     ax, cs:[badoff]
        mov     bp, sp
        mov     ss:[bp+4], ax      ; insert new return offset
        pop     bp
        pop     ax
        iret
bad_op_handler endp


;
;    CPU IDENTIFICATION SUBROUTINE
;       Identify the CPU type, from 8088 to the Pentium.  Works 
;       even if the 386 or later CPU is in V86 mode.  Note that
;       interrupts are enabled at exit, even if they were 
;       disabled on entry.  If it is necessary to run this 
;       routine with interrupts disabled, just remove all CLI 
;       and STI instructions, so long as interrupts are 
;       always disabled before running.
;
;       Called with:    nothing
;
;       Returns:        al = CPU type
;                             0 if 8088/8086 or V20/V30
;                             1 if 80186/80188
;                             2 if 80286
;                             3 if 80386
;                             4 if 80486
;                             5 if Pentium
;                       ah =  bit 0 = 0 if CPUID unavailable
;                                     1 if CPUID ok
;                             bit 1 = 0 if not V20/V30
;                                     1 if NEC V20/V30
;
;       Regs used:      ax, bx (all)
;                       eax, ebx (386 or later)
;
;       Subs called:    hook_int6, restore_int6, bad_op_handler

.8086   ; all instructions 8088/8086 unless overridden later

cpuvalue proc    far
        push    cx
        push    dx
        push    ds
        push    es

; 8088/8086 test - Use rotate quirk - All later CPUs mask the CL
;   register with 0Fh, when shifting a byte by cl bits.  This 
;   test loads CL with a large value (20h) and shifts the AX
;   register right.  With the 8088, any bits in AX are shifted 
;   out, and becomes 0.  On all higher level processors, the
;   CL value of 20h is anded with 0Fh, before the shift.  This
;   means the effective number of shifts is 0, so AX is 
;   unaffected.

        mov     cl, 20h            ; load high CL value
        mov     ax, 1              ; load a non-zero value in AX
        shr     ax, cl             ; do the shift
        cmp     ax, 0              ; if zero, then 8088/86
        jne     up186              ; jump if not 8088/86

; V20/V30 test - It is now either a V20/V30 or a 8088.  I'll use
;   another undocumented trick to find out which.  On the 8088,
;   0Fh performs a POP CS.  On the V20/V30, it is the start of
;   a number of multi-byte instructions.  With the byte string
;   0Fh, 14h, C3h the CPU will perform the following:
;               8088/8086               V20/V30
;             pop     cs              set1   bl, cl  
;             adc     al, 0C3h

        xor     al, al             ; clear al and carry flag
        push    cs
        db      0Fh, 14h, 0C3h     ; instructions (see above)
        cmp     al, 0C3h           ; if al is C3h then 8088/8086
        jne     upV20
        mov     ax, 0              ; set 8088/8086 flag
        jmp     uP_Exit

upV20:
        pop     ax                 ; correct for lack of pop cs
        mov     ax, 200h           ; set V20/V30 flag
        jmp     uP_Exit

; 80186/80188 test - Check what is pushed onto the stack with a 
;   PUSH SP instruction.  The 80186 updates the stack pointer 
;   before the value of SP is pushed onto the stack.  With all
;   higher level processors, the current value of SP is pushed
;   onto the stack, and then the stack pointer is updated.
        
up186:  
        mov     bx, sp             ; save the current stack ptr
        push    sp                 ; do test
        pop     ax                 ; get the pushed value
        cmp     ax, bx             ; did SP change ?
        je      up286              ; if not, it's a 286+ 
        mov     ax, 1              ; set 80186 flag
        jmp     uP_Exit

; 80286 test A - We'll look at the top four bits of the EFLAGS 
;   register.  On a 286, these bits are always zero.  Later 
;   CPUs allow these bits to be changed.  During this test, 
;   We'll disable interrupts to ensure interrupts do not change 
;   the flags. 

up286: 
        cli                        ; disable interrupts
        pushf                      ; save the current flags

        pushf                      ; push flags onto stack 
        pop     ax                 ; now pop flags from stack
        or      ax, 0F000h         ; try and set bits 12-15 hi
        push    ax
        popf                       ; set new flags
        pushf
        pop     ax                 ; see if upper bits are 0

        popf                       ; restore flags to original
        sti                        ; enable interrupts
        test    ax, 0F000h         ; were any upper bits 1 ?
        jnz     up386              ; if so, not a 286

; 80286 test B - If the system was in V86 mode, (386 or higher) 
;   the POPF instruction causes a protection fault, and the 
;   protected mode software must emulate the action of POPF. If 
;   the protected mode software screws up, as occurs with a 
;   rarely encountered bug in Windows 3.1 enhanced mode, the 
;   prior test may look like a 286, but it's really a higher 
;   level processor. We'll check if the protected mode bit is 
;   on.  If not, it's guaranteed to be a 286.

.286P                              ; allow a 286 instruction
        smsw    ax                 ; get machine status word
        test    ax, 1              ; in protected mode ?
        jz      is286              ; jump if not (must be 286)

; 80286 test C - It's very likely a 386 or greater, but it is 
;   not guaranteed yet.  There is a small possibility the system 
;   could be in 286 protected mode so we'll do one last test. We 
;   will try out a 386 unique instruction, after vectoring the 
;   bad-opcode interrupt vector (int 6) to ourselves.  

        call    hook_int6          ; do it!
        mov     [badoff], offset upbad_op  ; where to go if bad
.386
        xchg    eax, eax           ; 32 bit nop (bad on 286)
        
        call    restore_int6       ; restore vector
        jmp     up386              ; only gets here if 386 
                                   ;  or greater!

; Interrupt vector 6 (bad opcode) comes here if system is a 
;   80286 (assuming the 286 protected mode interrupt 6 handler 
;   will execute the bad-opcode interrupt). 

upbad_op:
        call    restore_int6
is286:
        mov     ax, 2              ; set 80286 flag
        jmp     uP_Exit

; 80386 test - Bit 18 in EFLAGS is not settable on a 386, but is
;   changeable on the 486 and later CPUs.  Bit 18 is used to 
;   flag alignment faults. During this test, we'll disable 
;   interrupts to ensure no interrupt will change any flags.

.386                               ; allow 386 instructions

up386:
        cli                        ; disable interrupts
        pushfd                     ; push flags to look at
        pop     eax                ; get eflags
        mov     ebx, eax           ; save for later
        xor     eax, 40000h        ; toggle bit 18
        push    eax                                    
        popfd                      ; load modified eflags to CPU
        pushfd                     ; push eflags to look at
        pop     eax                ; get current eflags
        push    ebx                ; push original onto stack
        popfd                      ; restore original flags
        sti                        ; enable interrupts
        xor     eax, ebx           ; check if bit changed 
        jnz     up486              ; changed, so 486 or later
        mov     ax, 3              ; set 80386 flag
        jmp     uP_Exit

; 80486 test - Bit 21 in EFLAGS is not settable on a 486, but is
;   changeable on the Pentium CPU.  If bit 21 is changeable, it 
;   indicates the CPU supports the CPUID instruction.  It's 
;   amazing it's only taken 10 years to implement the CPUID 
;   instruction, which should have been included from the start!  
;   During this test, we'll disable interrupts to ensure no 
;   interrupt will change any flags. 

up486:
        cli                        ; disable interrupts
        pushfd                     ; push flags to look at
        pop     eax                ; get eflags
        mov     ebx, eax           ; save for later
        xor     eax, 200000h       ; toggle bit 21
        push    eax                                    
        popfd                      ; load modified eflags to CPU
        pushfd                     ; push eflags to look at
        pop     eax                ; get current eflags
        push    ebx                ; push original onto stack
        popfd                      ; restore original flags
        sti                        ; enable interrupts
        xor     eax, ebx           ; check if bit changed 
        jnz     upPentium          ; changed, it's a Pentium
        mov     ax, 4              ; set 80486 flag
        jmp     uP_Exit

; Pentium - It's possible the CPUID instruction may appear on 
;   other CPU chips, so run the CPUID instruction to see what 
;   CPU type it indicates.  The CPUID returns a family number 
;   0 to 5 for the processor type.  As of this date, only the 
;   Pentium supports the CPUID instruction and it is assigned 
;   type 5.

upPentium:
        push    ecx                ; CPUID changes eax to edx
        push    edx
        mov     eax, 1             ; get family info function
        CPUID                      ; macro for CPUID instruction
        and     eax, 0F00h         ; find family info
        shr     eax, 8             ; move to al
        mov     ah, 1              ; set flag that CPUID ok
        pop     edx
        pop     ecx
       
up_Exit:
        pop     es
        pop     ds
        pop     dx
        pop     cx
        ret
cpuvalue endp
.8086                              ; return to 8086 instructions


;
;    CPU MODE
;       Check if the 286 or later CPU is in real, protected or
;       V86 mode.  It is assumed that if the 80386 or later
;       processor is in protected mode, we must be in V86 mode.
;
;       Call with:      ds:[cpu_val] set
;
;       Returns:        al = 0 protected mode not supported 
;                            1 if real mode 
;                            2 if protected mode
;                            3 if V86 mode
;                       ah = privilege level 0 to 3 
;
;       Regs used:      ax

.386P                              ; allow 286/386 instructions

;
;    CPU MODE
;       Check if the 286 or later CPU is in real, protected or
;       V86 mode.  It is assumed that if the 80386 or later
;       processor is in protected mode, we must be in V86 mode.
;
;       Call with:      ds:[cpu_val] set
;
;       Returns:        al = 0 protected mode not supported 
;                            1 if real mode 
;                            2 if protected mode
;                            3 if V86 mode
;                       ah = privilege level 0 to 3 
;
;       Regs used:      ax

.386P                              ; allow 286/386 instructions

cpumode proc    near
        push    cx
        xor     cx, cx             ; assume no protected mode
        cmp     [cpu_val], 2       ; 286 CPU or later ?
        jb      cpum_Exit          ; jump if not
        mov     cx, 1              ; assume real mode flag
        smsw    ax                 ; get machine status word
        test    ax, 1              ; in protected mode ?
        jz      cpum_Exit          ; jump if not (real mode)

cpu_not_real:
        mov     cl, 2              ; protected mode
        pushf
        pop     ax                 ; get flags
        and     ax, 3000h          ; get I/O privilege level
        shl     ax,12
        mov     ch, al             ; save privilege
        cmp     [cpu_val], 2       ; if 286, then protected
        je      cpum_Exit          ; jump if so

; On 386 or later, we have to assume V86 mode.  Note that the 
;  next four lines of code (commented out) might seem the 
;  correct way to detect V86 mode.  It will not work, since the 
;  PUSHFD instruction clears the VM bit before placing it on the 
;  stack.  This is undocumented on the 386 and 486, but 
;  documented on the Pentium. 

;        pushfd
;        pop     eax                ; get extended flags
;        test    eax, 20000h        ; V86 mode ?
;        jz      cpum_out_mode      ; jump if not

        mov     cl, 3              ; return V86 mode

cpum_Exit:
        mov     ax, cx             ; return status
        pop     cx
        ret
cpumode endp
.8086


cseg    ends

;================================================== stack ======

stacka  segment para stack

        db      192 dup (0)

stacka  ends

        end     start



