
;
;                            KEYVIEW
;
;
;    Display the current keyboard controller command byte, then 
;    turn off normal keyboard controller translation, and 
;    display the untranslated code when a key is pressed. 
;    Ignore key release codes. Escape key exits program.
;
;    For all computers equipped with a 8042 keyboard controller 
;    (not for PC/XT). 
;
;    (c) Copyright 1994  Frank van Gilluwe  All Rights Reserved.

include undocpc.inc


cseg    segment para public
        assume  cs:cseg, ds:cseg, ss:stack

keyview         proc near

message1 db     CR, LF
         db     'KEYBOARD INFORMATION AND SCAN CODES', CR, LF
         db     '  Displays the untranslated scan code for each'
         db     ' key pressed.', CR, LF
         db     '  To exit, press Escape', CR, LF
         db     CR, LF, '$'

message2 db     'New keyboard command byte = '
commandb db     '  ', CR, LF, CR, LF, '$'

message3 db     'Keypressed scan code = '
scanbyte db     '  ', CR, LF, '$'
         
message4 db     'Keyboard read timeout error', CR, LF, '$'


start:
        push    cs
        pop     ds
        OUTMSG  message1           ; display initial message

        mov     bl, 7              ; turn all LEDs on 
        call    setLEDs


; Send command to turn off keyboard interrupt IRQ1 (bit 0=0) so 
;   the BIOS will not handle anything from the keyboard. Also 
;   set keyboard translation off (bit 6=0) 

        cli                        ; disable interrupts
        mov     bl, 60h            ; set command byte function
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display if error (ah=1)
        mov     al, 24h            ; turn off use of IRQ 1 
                                   ;  (int 9), no translation
        call    keyboard_write     ; send command to keyboard
        call    error_write        ; if error, display

; flush any remaining keys out of the buffer 

flush_all:
        in      al, 64h            ; get status
        test    al, 1              ; any keys to read ?
        jz      read_command
        call    keyboard_read      ; read until buffer empty
        jmp     flush_all


; Now read the just programmed command byte and display it

read_command:
        mov     bl, 20h            ; get command byte function
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display if error (ah=1)
        call    keyboard_read      ; read command byte into al
        or      ah, ah             ; valid ?
        jnz     error              ; jump if not
        mov     bx, offset commandb
        call    hex                ; convert to ascii
        OUTMSG  message2           ; display new command byte

; Display each key pressed scan code screen or the release of 
;   the escape key exits program

next_key:
        call    keyboard_read      ; read command byte into al
        or      ah, ah             ; valid ?
        jnz     next_key           ; loop if no key
        cmp     al, 76h            ; Escape key ?
        je      escape             ; exit if so

display_key:
        push    dx
        mov     bx, offset scanbyte
        call    hex                ; convert to ascii
        OUTMSG  message3           ; display scan information
        pop     dx
        dec     dl
        jnz     next_key
        
; Escape pressed - so restore controller to normal operation

escape:
        mov     bl, 60h            ; set command byte function
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display if error (ah=1)
        mov     al, 45h            ; reset to normal
        call    keyboard_write     ; send command to keyboard
        call    error_write        ; if error, display
        jmp     exit


; Display keyboard read timeout error message

error:
        OUTMSG  message4           ; display error message

exit:
        sti
        mov     bl, 0AEh           ; enable keyboard
        call    keyboard_cmd       ; do-it
        call    error_cmd          ; display if error (ah=1)
        mov     ah, 4Ch
        int     21h                ; exit
keyview endp


;
;    SET LEDs
;       Send 3 bits from BL to the keyboard LEDs (handy for 
;       debugging TSRs). Does NOT update the keyboard flags. The 
;       next LED update will restore the 3 keyboard LEDs. In the 
;       unlikely event that an LED update is in progress, the 
;       update is skipped.
;       
;           40:97h = keyboard_flags2
;                 bit 7 = 1 if keyboard transmit error occurred
;                 bit 6 = 1 if LED update is in progress
;                 bit 5 = 1 if resend received
;                 bit 4 = 1 if acknowledgment received
;
;       This routine also assumes the standard int 9 BIOS 
;       keyboard handler maintains keyboard status in 
;       keyboard_flags2 at 40:97h.
;
;       Called with:    bl, bit 0=1 to turn on Scroll Lock LED
;                               1=1 to turn on Num Lock LED
;                               2=1 to turn on Caps Lock LED
;
;       Regs Used:      ax

setLEDs proc    near
        push    es
        mov     ax, 40h
        mov     es, ax                  ; es points to BIOS data
        cli                             ; ints off while update
        test    byte ptr es:[97h], 40h  ; already updating ?
        jnz     setLED_return3          ; return if so
        
        or      byte ptr es:[97h], 40h  ; set update in-progress
        mov     al, 0EDh                ; Update LED command
        call    keyboard_write          ; send keyboard command
        test    byte ptr es:[97h], 80h  ; xmit error ?
        jnz     setLED_return1          ; exit if xmit error
        mov     al, bl
        and     al, 7                   ; only send 3 LED bits
        call    keyboard_write          ; LED data to keyboard
        test    byte ptr es:[97h], 80h  ; xmit error ?
        jz      setLED_return2          ; jump if not
setLED_return1:
        mov     al, 0F4h                ; enable keyboard 
        call    keyboard_write          ;  since error occurred
setLED_return2:
        and     byte ptr es:[97h], 3Fh  ; error off & update
setLED_return3:
        sti                             ; enable interrupts 
        pop     es
        ret
setLEDs endp


;
;    KEYBOARD_READ
;       read a byte from the keyboard into al (port 60h).
;
;       Called with:    nothing
;
;       Returns:        if ah=0, al=byte read from keyboard
;                       if ah=1, no byte ready after timeout
;
;       Regs Used:      al

keyboard_read   proc    near
        push    cx
        push    dx

        xor     cx, cx             ; counter for timeout (64K)
key_read_loop:
        in      al, 64h            ; keyboard controller status
        IODELAY
        test    al, 1              ; is a data byte ready ?
        jnz     key_read_ready     ; jump if now ready to read
        loop    key_read_loop

        mov     ah, 1              ; return status - bad
        jmp     key_read_exit

key_read_ready:
        push    cx                 ; delay routine needed for 
        mov     cx, 16             ;   MCA Type 1 controller. 
key_read_delay:                    ;   Insures a 7 uS or longer
        IODELAY
        loop    key_read_delay     ;   Assumes CPU is 80486,
        pop     cx                 ;   66Mhz or slower

        in      al, 60h            ; now read the byte
        IODELAY
        xor     ah, ah             ; return status - ok
key_read_exit:
        pop     dx
        pop     cx
        ret
keyboard_read   endp



;
;    KEYBOARD_WRITE
;       Send byte AL to the keyboard controller (port 60h). 
;       Assumes no BIOS interrupt 9 handler active. 
;
;       If the routine times out due to the buffer remaining 
;       full, ah is non-zero. 
; 
;       Called with:    al = byte to send 
;                       ds = cs 
;
;       Returns:        if ah = 0, successful
;                       if ah = 1, failed
; 
;       Regs Used:      ax

keyboard_write  proc    near
        push    cx
        push    dx

        mov     dl, al             ; save data for keyboard
        xor     cx, cx             ; counter for timeout (64K)
kbd_wrt_loop:
        in      al, 64h            ; get keyboard status
        IODELAY
        jz      kbd_wrt_ok
        loop    kbd_wrt_loop       ; try again
                                   ; fall through, still busy
        mov     ah, 1              ; return status - failed
        jmp     kbd_wrt_exit      

kbd_wrt_ok:
        mov     al, dl
        out     60h, al            ; data to controller/keyboard
        IODELAY
        xor     ah, ah             ; return status ok

kbd_wrt_exit:
        pop     dx
        pop     cx
        ret
keyboard_write  endp


;
;    KEYBOARD_CMD
;       Send a command in register BL to the keyboard controller 
;       (port 64h). 
;
;       If the routine times out due to the buffer remaining 
;       full, ah is non-zero. 
;
;       Called with:    bl = command byte
;                       ds = cs  
;
;       Returns:        if ah = 0, successful
;                       if ah = 1, failed
;
;       Regs Used:      ax, cx


keyboard_cmd    proc    near
        xor     cx, cx             ; counter for timeout (64K)
cmd_wait:
        in      al, 64h            ; get controller status
        IODELAY
        test    al, 2              ; is input buffer full?
        jz      cmd_send           ; ready to accept command ?
        loop    cmd_wait           ; jump if not
                                   ; fall through, still busy
        jmp     cmd_error

cmd_send:                          ; send command byte
        mov     al, bl
        out     64h, al            ; send command 
        IODELAY

        xor     cx, cx             ; counter for timeout (64K)
cmd_accept:
        in      al, 64h            ; get controller status
        IODELAY
        test    al, 2              ; is input buffer full?
        jz      cmd_ok             ; jump if command accepted
        loop    cmd_accept         ; try again
                                   ; fall through, still busy
cmd_error:
        mov     ah, 1              ; return status - failed
        jmp     cmd_exit
cmd_ok:
        xor     ah, ah             ; return status ok
cmd_exit:
        ret
keyboard_cmd    endp


;
;    ERROR_WRITE
;       Check if ah=0, as returned from the keyboard_write 
;       routine and display message that a keyboard write failed 
;       if ah not zero. A real error handler might replace this 
;       routine. 
;
;       Regs Used:      ah

error_write     proc    near
        cmp     ah, 0              ; did an error occur ?
        je      error_w_exit       ; exit if not
        push    dx
        OUTMSG  kbd_wrt_errmsg     ; error handler goes here
        call    boop            
        pop     dx
error_w_exit:
        ret
error_write     endp

kbd_wrt_errmsg  db      'Keyboard_write - Input buffer full'
                db       CR, LF, '$'


;
;    ERROR_CMD
;       Check if ah=0, as returned from the keyboard_cmd routine
;       and display message that a keyboard command failed if 
;       ah not zero. A real error handler might replace this 
;       routine. 
; 
;       Regs Used:      ah 

error_cmd       proc    near
        cmp     ah, 0              ; did an error occur ?
        je      error_c_exit       ; exit if not
        push    dx
        OUTMSG  kbd_cmd_errmsg     ; error handler goes here
        call    boop            
        pop     dx
error_c_exit:
        ret
error_cmd       endp

kbd_cmd_errmsg  db      'Keyboard_cmd - Input buffer full'
                db      CR, LF, '$'


;
;   HEX SUBROUTINE
;       convert the hex number in al into two ascii characters 
;       at ptr ds:bx.
;
;       Called with:    al = input hex number 
;                       ds:bx = ptr where to put ascii
;
;       Regs Used:      al, bx

hex     proc    near
        push    bx
        mov     bl, al
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bh, al
             
        mov     al, bl             ; upper nibble
        shr     al, 1
        shr     al, 1
        shr     al, 1
        shr     al, 1
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bl, al
        mov     ax, bx
        pop     bx
        mov     [bx], ax           ; transfer ascii bytes
        ret
hex     endp


;
;    BOOP ERROR SOUND SUBROUTINE
;       Send a short tone to the speaker
;
;       Called with:    nothing
;
;       Regs used:      none

boop    proc    near
        push    ax
        push    bx
        push    cx
        in      al, 61h            ; read 8255 port B
        and     al, 0FEh           ; turn off the 8253 timer bit
        mov     bx, 150            ; loop bx times 

; begin loops of tone on and tone off to create frequency

bbcycle:                                        
        or      al, 2              ; turn on speaker bit
        out     61h, al            ; output to port B
        mov     cx, 300            ; on cycle time duration
bbeepon:    
        loop    bbeepon            ; delay
        and     al, 0FDh           ; turn off the speaker bit
        out     61h, al            ; output to port B
        mov     cx, 300            ; off cycle time duration
bbeepoff:
        loop    bbeepoff           ; delay
        dec     bx
        jnz     bbcycle            ; loop
        pop     cx
        pop     bx
        pop     ax
        ret
boop    endp


cseg    ends
             
;================================================== stack ======

stack   segment para stack
        
        db      192 dup (0)
             
stack   ends


        end     start

