
;
;                          MAX_RECV
;
;
;   This routine works with a companion routine MAX_XMIT to find
;   the maximum sustained serial transfer rate between two 
;   systems, connected by a null modem cable.  Serial port 1 is 
;   assumed unless command line indicates otherwise.  From the
;   command specify port number x from 1 to 4:   MAX_RECV  x
;
;   (c) Copyright 1994  Frank van Gilluwe  All Rights Reserved.

include undocpc.inc

cseg    segment para public
        assume  cs:cseg, ds:cseg, ss:tsrstk
    
max_recv        proc    far

msg1    db      CR, LF
        db      'MAX_RECV - Finds Maximum speed on serial link'
        db      CR, LF, '$'

msg2    db      CR, LF, '  Ready to receive data.'
        db      '  Waiting for transmitter MAX_XMIT connection.'
        db      CR, LF
        db      '  (Press Ctrl-Break to exit)', CR, LF, '$'

msg3    db      'Testing at baud rate: $'
msgok   db      '  Test OK', CR, LF, '$'
msgfail db      '  Test failed - $'

msgovr  db      'Overrun error', CR, LF, '$'
msgpar  db      'Parity error', CR, LF, '$'
msgfra  db      'Framing error', CR, LF, '$'
msgdata db      'Data is invalid', CR, LF, '$'

msgerr  db      '  Serial port not present', CR, LF, '$'

divisor db      30h, '2,400  $'  ; Table of divisors and baud
div2nd  db      18h, '4,800  $'  ;  rates to test
        db      0Ch, '9,600  $'
        db      8,   '14,400 $'
        db      6,   '19,200 $'
        db      3,   '38,400 $'
        db      2,   '57,600 $'
        db      1,   '115,200$'
divisor_end     label  byte

buffer  db      512 dup (0h)       ; buffer of bytes to send

ctrlb   db      0                  ; 1=control break occurred


start:
        mov     bx, 0              ; assume port 1
        cmp     byte ptr ds:[80h], 0  ; command line options ?
        je      max_skp1           ; jump if none
        mov     al, ds:[82h]       ; get command line value
        sub     al, 31h            ; convert to digit 0-3
        cmp     al, 3
        ja      max_skp1           ; skip if out of range
        mov     bl, al             ; save port number 
max_skp1:
        push    cs
        pop     ds
        mov     ax, 40h            ; BIOS data area segment
        mov     es, ax
        mov     byte ptr es:[71h], 0  ; force break off
        OUTMSG  msg1               ; output initial message

; get the serial base I/O port address and set up pointer

        shl     bx, 1              ; setup to get I/O port
        mov     dx, es:[bx]        ; base I/O port address
        cmp     dx, 0
        jne     valid_port
        jmp     no_serial_port     ; exit if zero

valid_port:
        mov     bp, dx             ; save base port address
        mov     si, offset divisor ; pointer to divisor table

; inform the remote system that we are ready

        add     dx, 4              ; modem control register
        mov     al, 1              ;  set Data Terminal Ready
        out     dx, al             ;  but not Ready To Send
        IODELAY

        OUTMSG  msg2               ; output "waiting message"

; Now check if the remote system is ready.  Data Set Ready
;   will be high.
                                   
        mov     dx, bp
        add     dx, 6              ; modem status register

max_loop1:
        in      al, dx
        test    al, 20h            ; is remote system ready ?
        jnz     ready_to_recv      ; jump if so (DSR = 1)
        test    byte ptr es:[71h], 80h  ; did break occur ?
        jz      max_loop1          ; jump if not
        jmp     max_recv_end       ; exit, ctrl-break

; Set up the modem with 8-bit, 1 stop bit, odd parity, and
;   set the current test baud rate divisor

ready_to_recv:
        mov     dx, bp
        add     dx, 3              ; line control register
        mov     al, 10001011b      ;  8-bit, 1 stop, odd parity
        out     dx, al             ;  and DLAB=1 to load baud
        IODELAY
        mov     dx, bp             ; divisor latch LSB
        mov     al, [si]           ;  get divisor LSB
        out     dx, al             ;  set divisor LSB
        IODELAY
        inc     dx
        mov     al, 0              ;  divisor MSB is 0 
        out     dx, al             ;  set divisor MSB
        IODELAY
        add     dx, 2              ; line control register
        mov     al, 00001011b      ;  turn DLAB off
        out     dx, al
        IODELAY

; flush anything in input buffer

        mov     dx, bp             ; load base address
        add     dx, 5              ; line status register
        in      al, dx
        test    al, 1              ; any received byte ?
        jz      skip_input         ; skip if not
        mov     dx, bp             ; receive register
        in      al, dx             ; flush receive buffer
skip_input:
        OUTMSG  msg3               ; output Testing ...
        mov     dx, si
        inc     dx                 ; point to baud rate text
        mov     ah, 9
        int     21h                ; output baud rate number

        mov     dx, bp
        add     dx, 5              ; line status register
        in      al, dx             ; a read clears error flags
        mov     di, offset buffer  ; buffer of bytes to receive
        mov     cx, 512            ; 512 bytes expected
        call    get_block          ; receive 512 bytes into ds:[di]
                                  
; check if any errors occurred and exit if so (al=line status)

        test    byte ptr es:[71h], 80h  ; did break occur ?
        jnz     max_recv_end       ; exit if so
        test    al, 0Eh            ; any errors on reception ?
        jnz     max_recv_fail      ; jump if so

        mov     cx, 512            ; check that the data is valid
        mov     di, offset buffer
        mov     al, 0

max_loop2:
        cmp     byte ptr [di], 31h ; correct value in buffer ?
        jne     max_recv_fail      ; jump if not
        mov     byte ptr [di], al  ; zero location for next test
        inc     di                 
        loop    max_loop2

; test successful, set RTS=0 output success and set next baud rate 

        mov     dx, bp
        add     dx, 4              ; modem control register
        mov     al, 1              ;  set Data Terminal Ready
        out     dx, al             ;  but Request to Send now off

        OUTMSG  msgok
        add     si, offset div2nd - offset divisor
        cmp     si, offset divisor_end
        jae     delay_a_bit        ; jump if last divisor complete
        jmp     ready_to_recv

delay_a_bit:
        mov     cx, 0FFFFh         ; short delay to be sure
max_delay:                         ;  transmitter gets RTS=0
        loop    max_delay          ;  signal before we set
        jmp     max_recv_end       ;  DTR=0 as well

max_recv_fail:
        push    ax
        OUTMSG  msgfail            ; output test failed message
        pop     ax
        mov     dx, offset msgovr  ; assume overrun error
        test    al, 2              ; overrun error ?
        jnz     max_recv_fail2     ; jump if so
        mov     dx, offset msgfra  ; assume framing error
        test    al, 8              ; framing error ?
        jnz     max_recv_fail2     ; jump if so
        mov     dx, offset msgpar  ; assume parity error
        test    al, 4              ; parity error ?
        jnz     max_recv_fail2     ; jump if so
        mov     dx, offset msgdata ; must be data error (al=0)
max_recv_fail2:
        mov     ah, 9
        int     21h                ; output type of error

max_recv_end:
        mov     dx, bp
        add     dx, 4              ; modem control register
        mov     al, 0              ;  set DTR & RTS off to 
        out     dx, al             ;  tell transmitter to stop
        jmp     max_recv_done

no_serial_port:
        OUTMSG  msgerr             ; output error message

max_recv_done:
        mov     ah, 4Ch
        int     21h                ; exit to DOS
max_recv endp

;
;    GET_BLOCK
;       Get a block of data as fast as possible over the 
;       serial link.  Exit if Control-break or errors are 
;       detected.  There are no timeouts.
;
;       Called with:    cx = number of bytes expected
;                       bp = base port I/O address
;                       es:di = destination of data
;
;       Returns:        bytes received on serial link, unless
;                         control-break occurred
;
;       Regs used:      ax, bx, cx, dx, di

get_block proc near
        push    si
        push    ds
        push    es

        mov     ax, ds
        mov     es, ax             ; es = buffer segment
        mov     ax, 40h
        mov     ds, ax             ; ds = BIOS data segment
        mov     si, 71h            ; offset of BIOS break flag
        mov     ah, 5              ; for adjusting DX    
        mov     bh, 8Eh            ; test mask for errors & break
        mov     bl, 1              ; test mask if byte available

        mov     dx, bp
        add     dx, 4              ; modem control register
        mov     al, 3              ;  set Data Terminal Ready
        out     dx, al             ;  and Request to Send

        cld                        ; clear direction flag
        mov     dx, bp             ; base I/O address

; The following loop is tightly optimized, with many fixed values 
; pre-loaded into registers.  In addition, the error conditions 
; (bits 3-1 of the line status register) and the BIOS break bit 7 
; at 40:71h are combined for a single exit test.  This assumes no
; FIFO mode, where bit 7 of the line status register is always 
; zero.  The STOSB instruction loads AL into es:[di] and inc's DI.

get_next: 
        add     dl, ah             ; line status register
wait_loop1:
        in      al, dx             ; get error & data ready bits
        or      al, [si]           ; insert break bit 7
        test    al, bh             ; errors or break (bh=8E) ?
        jnz     get_exit           ; jump if so
        test    al, bl             ; any received byte (bl=1) ?
        jz      wait_loop1         ; loop if not
        mov     dx, bp             ; receive register
        in      al, dx             ; get byte
        stosb                      ; store byte into es:[di]
        loop    get_next

        mov     al, 0              ; no errors
get_exit:
        pop     es
        pop     ds
        pop     si
        ret
get_block endp


cseg    ends

;===================================================== stack ====

tsrstk  segment para stack
        db      150 dup (0)
tsrstk  ends
        end     start

